/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { LinkRelation } from '@dcx/assets';
import {
    AdobeDCXBranch,
    AdobeDCXComponent,
    AdobeDCXComponentData,
    AdobeDCXElement,
    AdobeDCXError,
    AdobeDCXNode,
    AdobeDCXNodeData,
    AdobeDCXRootNodeData,
    AdobeUploadResults,
    ComponentCallback,
    LinkRelationKey,
    LinkSet,
    NodeCallback,
} from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import Logger from '@dcx/logger';
import { uploadResultsFromComponentDescriptor } from '@dcx/repo-api-session';
import { appendPathElements, deepCopy, flatCopy, generateUuid, isValidAbsolutePath, validateParams } from '@dcx/util';
import DCXBranch from './AdobeDCXBranch';
import { AdobeDCXComponent as DCXComponent } from './AdobeDCXComponent';
import DCXElement from './AdobeDCXElement';
import { AdobeDCXLocalNodeData, AdobeDCXNode as DCXNode } from './AdobeDCXNode';
import { COMPOSITE_ARCHIVAL_STATES, COMPOSITE_STATES } from './enum';

export interface SourceAssetInfoEntry {
    componentPath: string;
    compositeAssetId: string;
    componentId: string;
    componentVersion: string;
    repositoryId?: string;
}

type CopyComponentCallback = (
    originalComponent: DCXComponent,
    copiedComponent: DCXComponent,
    localData?: AdobeDCXLocalNodeData['local'],
) => void;

/**
 * @class
 * @hideconstructor
 * Internal helper class. Implements the core functionality shared by both AdobeDCXBranch and AdobeDCXElement.
 * @constructor
 * @private
 * @param {Object}  [data]     private
 * @param {Object}  [owner]    The branch or the element.
 * @param {Boolean} [readOnly] private
 */
export class AdobeDCXBranchCore {
    /** @internal */
    _owner: DCXBranch | DCXElement;

    private _allNodes: Record<string, DCXNode> = {};
    private _allComponents: Record<string, DCXComponent> = {};
    private _absolutePaths: Record<string, DCXNode | DCXComponent> = {};

    /** @internal */
    _readOnly = false;

    /** @internal */
    _data!: AdobeDCXNodeData & AdobeDCXLocalNodeData;

    /** @internal */
    _isDirty = false;

    /** @internal */
    _compositeAssetId?: string;

    /** @internal */
    _compositeRepositoryId?: string;

    /** @internal */
    _compositeLinks?: LinkSet;

    /**
     * @internal
     * Dictionary of componentId to component/asset info.
     * Set during copy.
     */
    _sourceAssetInfoLookup: Record<string, SourceAssetInfoEntry> = {};

    /**
     * @internal
     * Source composite info.
     * Set during copy.
     */
    _sourceCompositeInfo: { links: LinkSet; assetId: string; repositoryId: string } | undefined;

    constructor(
        data: AdobeDCXNodeData | AdobeDCXRootNodeData,
        owner: AdobeDCXBranch | AdobeDCXElement,
        readOnly = false,
    ) {
        this._readOnly = readOnly;
        this._owner = owner as DCXElement | DCXBranch;
        this._setData(data);
    }

    //******************************************************************************
    // Getters/setters for properties
    //******************************************************************************

    /**
     * @internal
     */
    get allNodes() {
        return this._allNodes;
    }

    /**
     * An AdobeDCXNode object that represents the root node of the branch or element.
     * @type {AdobeDCXNode}
     * @internal
     * @readonly
     */
    get rootNode(): DCXNode {
        return this.getChildWithAbsolutePath(DCXNode.ROOT_PATH) as DCXNode;
    }
    set rootNode(_) {
        throw new DCXError(DCXError.READ_ONLY, 'Property "rootNode" is read-only.');
    }

    /**
     * Whether the branch or element has has been modified in memory.
     * @internal
     * @readonly
     * @type {Boolean}
     */
    get isDirty(): boolean {
        return this._isDirty;
    }
    set isDirty(_) {
        throw new DCXError(DCXError.READ_ONLY, 'Cannot set the property isDirty.');
    }

    /**
     * @internal
     */
    get changeCount(): number {
        return this._data.local ? this._data.local.change || 0 : 0;
    }
    set changeCount(_) {
        throw new DCXError(DCXError.READ_ONLY, 'Cannot set the property changeCount.');
    }

    //******************************************************************************
    // Children (Nodes)
    //******************************************************************************

    /**
     * Returns the node with the given id or undefined if not found.
     * @param   {String } id The id of the child node to look up.
     * @returns {AdobeDCXNode}
     */
    public getChildWithId(id: string): DCXNode | undefined {
        return this._allNodes[id];
    }

    /**
     * Returns the node with the given absolute path or undefined if not found.
     * @param   {String} path The absolute path.
     * @returns {AdobeDCXNode}
     */
    public getChildWithAbsolutePath(path: string): DCXNode | undefined {
        const item = this._absolutePaths[path.toLowerCase()];

        return item ? (item instanceof DCXNode ? item : undefined) : undefined;
    }

    /**
     * Generates and returns an array of the child nodes of the given parent node.
     * @example
     * var rootNodes = branch.getChildrenOf(branch.rootNode);
     * @param   {AdobeDCXNode} parentNode The parent node to return the children for.
     * @returns {Array}
     */
    public getChildrenOf(pParentNode?: AdobeDCXNode): AdobeDCXNode[] {
        const parentNode = (pParentNode ? this._allNodes[pParentNode.id] : this.rootNode) as DCXNode;

        if (!parentNode) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Unknown node');
        }

        const result: AdobeDCXNode[] = [];
        const children = parentNode._data.children;
        if (Array.isArray(children)) {
            for (let i = 0; i < children.length; i++) {
                const nodeData = children[i];
                const node = this._allNodes[nodeData.id];
                if (!node) {
                    throw new DCXError(DCXError.INVALID_DATA, 'Node not in cache');
                }
                result.push(node);
            }
        }

        return result;
    }

    /**
     * Creates a new node and inserts it into the children list of the given parent node or of the
     * root if no parent node is given.
     * Returns the new child node.
     * @param   {String}  [name]       The name of the new child. If undefined the child will not
     *                                 have a name.
     * @param   {String}  [nodeId]     The id of the new child. If undefined the new child node will
     *                                 get a random id.
     * @param   {Integer} [index]      If given and less than or equal to the current number of
     *                                 children than the node gets inserted at the given index.
     *                                 Otherwise it gets added to the end.
     * @param   {String}  [parentNode] The parent node to add the node to. Default parent is the
     *                                 root node.
     * @returns {AdobeDCXNode}
     */
    public addChild(name?: string, nodeId?: string, index?: number, parentNode?: AdobeDCXNode) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }

        nodeId = nodeId || generateUuid();
        if (this._allNodes[nodeId]) {
            throw new DCXError(DCXError.DUPLICATE_VALUE, 'Node already exists in branch.');
        }
        if (index && (typeof index !== 'number' || index % 1 !== 0)) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "index" must be an integer.');
        }

        const newParent = parentNode ? this._allNodes[parentNode.id] : this.rootNode;
        if (!newParent) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Unknown parent node.');
        }

        // Create the new node
        const newChild = new DCXNode();
        newChild.id = nodeId;
        if (name) {
            newChild.name = name;
        }
        newChild._parentPath = newParent.absolutePath || newParent._parentPath;

        const children = newParent._data.children;
        if (children) {
            if (!((index as number) >= 0 && (index as number) <= children.length)) {
                index = children.length;
            }
            if (index === children.length) {
                // Simple case: add to end
                children[index] = newChild._data;
            } else {
                // Insert
                children.splice(index as number, 0, newChild._data);
            }
        } else {
            newParent._data.children = [newChild._data];
        }

        this._allNodes[newChild.id] = newChild;
        newChild._owner = this._owner;

        this._setDirty();
        return newChild;
    }

    /**
     * Removes and returns the given child node from the branch.
     * @param   {AdobeDCXNode} node The child node to remove.
     * @returns {AdobeDCXNode}
     */
    public removeChild(pNode: AdobeDCXNode): DCXNode {
        let node = pNode as DCXNode;
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        if (!node) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting param "node".');
        }
        if (node.isRoot) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Cannot remove the root node.');
        }

        const nodeId = node.id;
        if (!nodeId) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "node" must have an id.');
        }

        const found = this._nodeDataOfParentOfNode(node);
        if (!found) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Node not found in this branch.');
        }

        // remove from data
        const children = found.parentNodeData.children;
        if (children) {
            children.splice(found.index, 1);
            if (children.length === 0) {
                // Remove empty children node
                delete found.parentNodeData.children;
            }
        }

        // update caches
        node = this._allNodes[nodeId];
        this._removeNodeFromCachesRecursively(node._data);

        this._setDirty();

        return node;
    }

    /**
     * Moves the existing child from its current parent/index to the given parent/index.
     * @param   {AdobeDCXNode} node         The child node to move
     * @param   {Integer}      index        If given and less than or equal to the current number of
     *                                      children than the node gets inserted at the given index.
     *                                      Otherwise it gets added to the end.
     * @param   {AdobeDCXNode} [parentNode] The parent node to move the node to. Default parent is
     *                                      the root.
     * @returns {AdobeDCXNode}
     */
    public moveChild(pNode: AdobeDCXNode, index?: number | null, pParentNode?: AdobeDCXNode | null): AdobeDCXNode {
        let node = pNode as DCXNode;
        const parentNode = pParentNode as DCXNode;

        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        if (!node) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting param "node".');
        }
        if (node.isRoot) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Cannot move the root node.');
        }
        if (index && (typeof index !== 'number' || index % 1 !== 0)) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "index" must be an integer.');
        }

        const nodeId = node.id;
        if (!nodeId) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "node" must have an id.');
        }

        const found = this._nodeDataOfParentOfNode(node);
        node = this._allNodes[nodeId];
        if (!found) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Node not found in this branch.');
        }

        const newParent = parentNode ? this._allNodes[parentNode.id] : this.rootNode;
        if (!newParent) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Unknown parent node.');
        }

        if (this._nodeIdIsDescendantOf(newParent.id, node._data)) {
            // Trying to move the node into a decsendant of itself.
            throw new DCXError(DCXError.INVALID_PARAMS, 'Must not create a cycle.');
        }

        // Remove from data from the old parent and add it to the new parent.
        // We are using try-catch to ensure that we do not end up losing the node.
        const data = found.parentNodeData.children!.splice(found.index, 1)[0] as AdobeDCXNodeData;
        try {
            const children = newParent._data.children;
            if (children) {
                if (!((index as number) >= 0 && (index as number) <= children.length)) {
                    index = children.length;
                }
                if (index === children.length) {
                    // Simple case: add to end
                    children[index] = data;
                } else {
                    // Insert
                    children.splice(index as number, 0, data);
                }
            } else {
                newParent._data.children = [data];
            }
        } catch (x) {
            // re-insert
            found.parentNodeData.children!.splice(found.index, 0, data);
            throw x;
        }

        if (found.parentNodeData.children!.length === 0) {
            // Remove empty children node
            delete found.parentNodeData.children;
        }

        // No need to update caches since this is only a move.
        this._setDirty();

        return this._allNodes[nodeId];
    }

    /**
     * Copies the given child node as a new node into this branch. The node can be from the same or
     * from a different composite.
     *
     * <p>This function will try reuse the ids of any children and components of the copied node,
     * in order to minimize the amount of data that will later have to be uploaded, however, clients
     * must not rely on these ids being preserved in the copied objects.</p>
     *
     * <p>Fails if a node with the same id or same absolute path already exists.</p>
     *
     * <p>Notice: This method does not work without local storage (e.g. browser environment) if
     * used to copy between two composites stored at different endpoints.</p>
     ** @param   {AdobeDCXNode} node         The child node to copy. If it is the root node then
     *                                      newPath must be provided.
     * @param   {AdobeDCXNode} [parentNode] The parent node to copy the child node to. If undefined
     *                                      then the new child node will be added to the root of the
     *                                      branch.
     * @param   {Integer}      [index]      If provided and less than or equal to the current number of
     *                                      children of the parentNode (or root) the child node gets
     *                                      inserted at the given index. Otherwise it gets added to
     *                                      the end.
     * @param   {String}       [newPath]    <p>If provided, the copy of the child node will be assigned
     *                                      this a its path property and it will also receive a new
     *                                      random id (unless one is provided with the newId param).
     *                                      If left undefined then the copy of the node will keep
     *                                      the path of the original. In either case the function will
     *                                      fail if the resulting absolute path of the child or any
     *                                      of its children/components conflicts with an already
     *                                      existing absolute path.</p>
     *                                      <p>You must provide a newPath if you are copying the root
     *                                      node of a branch or element.</p>
     * @param   {String}       [newId]      If provided, the copy of the child node will be assigned
     *                                      this a its id. If left undefined (and the newPath param
     *                                      is also undefined) then the copy will retain the id of
     *                                      the original. In either case the function will
     *                                      fail if the resulting id of the child or any
     *                                      of its children/components conflicts with an already
     *                                      existing id.
     * @param   {NodeCallback} [callback]   Optional when not copying between different composites or
     *                                      when not using local storage.
     *                                      Gets called when the copy is done or has failed.
     * @returns                             The copied Node if no callback specified.
     */
    public copyChild(
        node: AdobeDCXNode,
        parentNode?: AdobeDCXNode,
        index?: number,
        newPath?: string,
        newId?: string,
    ): AdobeDCXNode;
    public copyChild(
        node: AdobeDCXNode,
        parentNode: AdobeDCXNode | undefined,
        index: number | undefined,
        newPath: string | undefined,
        newId: string | undefined,
        callback: NodeCallback,
    ): void;
    public copyChild(
        node: AdobeDCXNode,
        parentNode?: AdobeDCXNode,
        index?: number,
        newPath?: string,
        newId?: string,
        callback?: NodeCallback,
    ): AdobeDCXNode | undefined {
        return this._copyChild(
            node as DCXNode,
            parentNode as DCXNode,
            index,
            newPath,
            newId,
            /*replaceExisting*/ false,
            callback,
        );
    }

    /**
     * <p>Replaces the child node in this branch with a copy of the given node or branch with the same
     * id (or, if provided, with the given id). Fails if the child node does not exist in this branch.</p>
     *
     * <p>This function will try reuse the ids of any children and components of the copied node,
     * in order to minimize the amount of data that will later have to be uploaded, however, clients
     * must not rely on these ids being preserved in the copied objects.</p>
     *
     * <p>Notice: This method does not work without local storage (e.g. browser environment) if
     * used to copy between two different composites.</p>
     *
     * @param   {AdobeDCXNode} node       The child node to update from.
     * @param   {String}       [newPath]  If provided the copy of the component will be assigned
     *                                    this a its path property. Otherwise it will retain its original path.
     * @param   {String}       [newId]    If provided the copy of the child node will be assigned
     *                                    this as its id. Otherwise it will retain its original id.
     * @param   {NodeCallback} [callback]   Optional when not copying between different composites or
     *                                    when not using local storage.
     *                                    Gets called when the copy is done or has failed.
     * @returns                           The replaced Node, or undefined if callback is
     *                                    given.
     */
    public replaceChild(node: AdobeDCXNode, newPath?: string, newId?: string): AdobeDCXNode;
    public replaceChild(
        node: AdobeDCXNode,
        newPath: string | undefined,
        newId: string | undefined,
        callback: NodeCallback,
    ): void;
    public replaceChild(
        node: AdobeDCXNode,
        newPath?: string,
        newId?: string,
        callback?: NodeCallback,
    ): AdobeDCXNode | undefined {
        return this._copyChild(
            node as DCXNode,
            /*parentNode*/ undefined,
            /*index*/ undefined,
            /*newPath*/ newPath,
            newId || node.id,
            /*replaceExisting*/ true,
            callback,
        );
    }

    //******************************************************************************
    // Components
    //******************************************************************************

    /**
     * Given an INCOMPLETE_COMPOSITE DCXError, this will attempt to return the invalid components so they can be removed or updated
     * @param error An IMCOMPLETE_COMPOSITE error
     * @returns Array of components from this branch that are declared missing in the INCOMPLETE_COMPOSITE error report
     */
    public getMissingComponentsFromError(error: AdobeDCXError): AdobeDCXComponent[] {
        if (error.code !== DCXError.INCOMPLETE_COMPOSITE) {
            return [];
        }
        const missingComponentReports =
            error.response!.response.report.failures.filter((failure) => failure.rule === 'MissingComponent') ?? [];
        return missingComponentReports
            .map(
                (report) =>
                    this.getComponentWithId(report.component_id) ||
                    this.getComponentWithAbsolutePath(report.component_path),
            )
            .filter((componentFound) => componentFound);
    }

    /**
     * Returns an array of all components in the branch.
     * @returns {Array}.
     */
    public allComponents(): DCXComponent[] {
        const result: DCXComponent[] = [];
        for (const id in this._allComponents) {
            if (Object.prototype.hasOwnProperty.call(this._allComponents, id)) {
                result.push(this._allComponents[id]);
            }
        }
        return result;
    }

    /**
     * Returns the component with the given id or undefined if not found.
     * @param   {String} id The id of the component to look up.
     * @returns {AdobeDCXComponent}
     */
    public getComponentWithId(id: string): DCXComponent | undefined {
        return this._allComponents[id];
    }

    /**
     * Returns the component with the given absolute path or undefined if not found.
     * @param   {String} path The absolute path of the desired component.
     * @returns {AdobeDCXComponent}
     */
    public getComponentWithAbsolutePath(path: string): DCXComponent | undefined {
        const item = this._absolutePaths[path.toLowerCase()];
        return item ? (item instanceof DCXComponent ? item : undefined) : undefined;
    }

    /**
     * Returns an array containing the components of the given node.
     * @param   {AdobeDCXNode} parentNode The node whose components to return.
     * @returns {Array}
     */
    public getComponentsOf(pParentNode?: AdobeDCXNode): DCXComponent[] {
        const parentNode = pParentNode ? this._allNodes[pParentNode.id] : this.rootNode;
        if (!parentNode) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Unknown node');
        }

        const result: DCXComponent[] = [];
        const components = parentNode._data.components;
        if (Array.isArray(components)) {
            let i;
            for (i = 0; i < components.length; i++) {
                const componentData = components[i];
                const component = this._allComponents[componentData.id];
                if (!component) {
                    throw new DCXError(DCXError.INVALID_DATA, 'Component not in cache');
                }
                result[result.length] = component;
            }
        }

        return result;
    }

    /**
     * Gets passed into addComponent(), updateComponent(), copyComponent(), replaceComponent() and
     * called back whenever the operation has finished.
     * @callback ComponentCallback
     *    @param {Error}                error
     *    @param {AdobeDCXComponent}    component
     */

    /**
     * <p>Creates and adds a component to the given parent node or to the root if no parent node is
     * given.</p>
     *
     * @param componentDescriptor The serialized component descriptor to use.
     * @param name          The name of the new component.
     * @param path          The path of the new component. Must satisfy uniquenes rules for components.
     * @param relationship  The relationship of the new component.
     * @param parentNode  The node to add the node to. Defaults to the root.
     * @returns {AdobeDCXComponent}               The new component.
     */
    public addComponentWithComponentDescriptor(
        componentDescriptor: string,
        name: string,
        path: string,
        relationship: LinkRelationKey = LinkRelation.COMPONENT,
        parentNode?: AdobeDCXNode,
    ) {
        // The ordering of parameters for addComponentWithUploadResults
        // differs from this method
        // The intention with the signature for addComponentWithComponentDescriptor
        // is to enable the caller to pass in minimal positional parameters by
        // moving required parameters to the front, optional parameters to the back
        // and providing a default value for the relation type.
        return this.addComponentWithUploadResults(
            name,
            relationship,
            path,
            parentNode,
            uploadResultsFromComponentDescriptor(componentDescriptor),
        );
    }

    /**
     * <p>Creates and adds a component to the given parent node or to the root if no parent node is
     * given.</p>
     *
     * @param   {String}            name          The name of the new component.
     * @param   {String}            relationship  The relationship of the new component.
     * @param   {String}            path          The path of the new component. Must satisfy uniquenes
     *                                            rules for components.
     * @param   {AdobeDCXNode}      [parentNode]  The node to add the node to. Defaults to the root.
     * @param   {Object}            uploadResults The upload results object returned by a previous call
     *                                            to AdobeDCXCompositeXfer.uploadAssetForComponent().
     * @returns {AdobeDCXComponent}               The new component.
     */
    public addComponentWithUploadResults(
        name: string,
        relationship: string,
        path: string,
        parentNode: AdobeDCXNode | undefined,
        uploadResults: AdobeUploadResults,
    ): DCXComponent {
        // check preconditions
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        if (uploadResults.compositeId !== (this._owner as AdobeDCXBranch).compositeId) {
            throw new DCXError(
                DCXError.INVALID_PARAMS,
                'Param uploadResults does not appear valid to be for this composite.',
            );
        }
        const recordKeys = Object.keys(uploadResults.records);
        if (recordKeys.length !== 1) {
            throw new DCXError(
                DCXError.INVALID_PARAMS,
                'Param uploadResults must contain records of exactly one component upload.',
            );
        }
        const uploadRecord = uploadResults.records[recordKeys[0]];
        const componentId = uploadRecord.id;

        if (this._allComponents[componentId]) {
            throw new DCXError(DCXError.DUPLICATE_VALUE, 'Duplicate component id: ' + componentId);
        }

        const newParent = parentNode ? this._allNodes[parentNode.id] : this.rootNode;
        if (!newParent) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Unknown parent node.');
        }

        // create new component
        const newComponent = new DCXComponent();
        newComponent.id = componentId;
        newComponent.name = name;
        newComponent.type = uploadRecord.type;
        newComponent.relationship = relationship;
        newComponent.path = path;
        newComponent.state = COMPOSITE_STATES.unmodified;
        newComponent.etag = uploadRecord.etag;
        newComponent.length = uploadRecord.length;
        newComponent.version = uploadRecord.version;
        newComponent.md5 = uploadRecord.md5;

        // Check that the resulting absolute path of the component will be unique
        const absPath = this._normalizedAbsolutePathForItem(newComponent, newParent);
        if (this._absolutePaths[absPath]) {
            throw new DCXError(DCXError.DUPLICATE_VALUE, 'Duplicate absolute path: ' + absPath);
        }
        newComponent._parentPath = newParent.absolutePath || newParent._parentPath;

        // add to parent node
        const components = newParent._data.components;
        if (components) {
            components.push(newComponent.data);
        } else {
            newParent._data.components = [newComponent.data];
        }
        newComponent._owner = this._owner;

        // Update caches
        this._allComponents[componentId] = newComponent;
        this._absolutePaths[absPath] = newComponent;

        this._setDirty();

        return newComponent;
    }

    /**
     * <p>Updates the component record with the results of a recent upload of said component.</p>
     *
     * @param   {AdobeDCXComponent} component     The component.
     * @param   {Object}            uploadResults The upload results object returned by a previous
     *                                            call to AdobeDCXCompositeXfer.uploadAssetForComponent().
     * @returns {AdobeDCXComponent} The updated component.
     */
    public updateComponentWithUploadResults(component: DCXComponent, uploadResults: AdobeUploadResults): DCXComponent {
        // check preconditions
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        component = this._allComponents[component.id];
        if (!component) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Unknown component.');
        }
        if (uploadResults.compositeId !== (this._owner as AdobeDCXBranch).compositeId) {
            throw new DCXError(
                DCXError.INVALID_PARAMS,
                'Param uploadResults does not appear to be valid for this composite.',
            );
        }
        const uploadRecord = uploadResults.records[component.id];
        if (!uploadRecord) {
            throw new DCXError(
                DCXError.INVALID_PARAMS,
                'Param uploadResults does not contain an upload record for the given component.',
            );
        }

        component.etag = uploadRecord.etag;
        component.version = uploadRecord.version;
        component.md5 = uploadRecord.md5;
        component.length = uploadRecord.length;

        // Clear any source asset info for this component since its own copy now exists in the cloud
        this._setSourceAssetInfoOfComponent(undefined, component);
        // Reset the component state to be unmodified for the same reason
        component.state = COMPOSITE_STATES.unmodified;

        this._setDirty();

        return component;
    }

    /**
     * Removes the component from the branch.
     * @param   {AdobeDCXComponent} component The component to remove.
     * @returns {AdobeDCXComponent} The removed component.
     */
    public removeComponent(component: DCXComponent): DCXComponent {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        if (!component) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting param "component".');
        }

        const componentId = component.id;
        if (!componentId) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "component" must have an id.');
        }

        const found = this._nodeDataOfParentOfComponent(component);
        if (!found) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Component not found in this branch.');
        }

        // remove from data
        found.parentNodeData.components!.splice(found.index, 1);
        if (found.parentNodeData.components!.length === 0) {
            // Remove empty components node
            delete found.parentNodeData.components;
        }

        // update caches
        component = this._allComponents[componentId];
        delete this._allComponents[componentId];
        delete this._absolutePaths[this._normalizedAbsolutePathForItem(component)];

        component._owner = undefined;
        component._parentPath = '';

        this._setDirty();

        return component;
    }

    /**
     * Moves the component to the given node or the root if node is undefined
     * @param   {AdobeDCXComponent} component    The component to move.
     * @param   {AdobeDCXNode}      [parentNode] The node to move the component to.
     * @returns {AdobeDCXComponent} The moved component.
     */
    public moveComponent(component: DCXComponent, parentNode?: DCXNode): DCXComponent {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        if (!component) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting param "component".');
        }

        const componentId = component.id;
        if (!componentId) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "component" must have an id.');
        }

        const found = this._nodeDataOfParentOfComponent(component);
        if (!found) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Component not found in this branch.');
        }

        const newParent = parentNode ? this._allNodes[parentNode.id] : this.rootNode;
        if (!newParent) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Unknown parent node.');
        }

        // Check that the resulting absolute path of the component will be unique
        const absPath = this._normalizedAbsolutePathForItem(component, newParent);
        if (this._absolutePaths[absPath]) {
            throw new DCXError(DCXError.DUPLICATE_VALUE, 'Duplicate absolute path: ' + absPath);
        }

        // move
        const data = found.parentNodeData.components!.splice(found.index, 1)[0] as AdobeDCXComponentData;
        const components = newParent._data.components;
        if (components) {
            components.push(data);
        } else {
            newParent._data.components = [data];
        }
        if (found.parentNodeData.components!.length === 0) {
            // Remove empty components node
            delete found.parentNodeData.components;
        }

        // update component and caches
        component = this._allComponents[componentId];
        delete this._absolutePaths[this._normalizedAbsolutePathForItem(component)];
        component._parentPath = newParent.absolutePath || newParent._parentPath;
        this._absolutePaths[absPath] = component;

        this._setDirty();

        return component;
    }

    /**
     * Copies the given component and adds it as a new component to this branch. Fails if the
     * component already exists.
     *
     * @param   {AdobeDCXComponent} component    The component to copy.
     * @param   {AdobeDCXNode}      [parentNode] The node to copy the component to. If none is
     *                                           provided then the component will be added to the
     *                                           root.
     * @param   {String}            [newPath]    If provided the copy of the component will be
     *                                           assigned this a its path property and it will also
     *                                           get assigned a new id if none is provided via the
     *                                           <em>newId</em> param.
     * @param   {String}            [newId]      If provided the copy of the component will be
     *                                           assigned this a its id property.
     * @param   {ComponentCallback} [callback]   Optional when not copying between different
     *                                           composites. Gets called when the copy is done or
     *                                           has failed.
     * @returns {AdobeDCXComponent}              Only returns the new component when called without
     *                                           a callback.
     */
    public copyComponent(
        component: DCXComponent,
        parentNode?: DCXNode,
        newPath?: string,
        newId?: string,
        callback?: any,
    ) {
        return this._copyComponent(component, parentNode, newPath, newId, /*replaceExisting*/ false, callback);
    }

    /**
     * Replaces the matching component (same id) in this branch with a copy of the given component.
     * Fails if the component can't be found.
     *
     * @param   {AdobeDCXComponent} component  The component to copy.
     * @param   {String}            [newPath]  If provided the copy of the component will be
     *                                         assigned this a its path property and it will also
     *                                         get assigned a new id if none is provided via the
     *                                         newId param.
     * @param   {String}            [newId]    If provided the copy of the component will be
     *                                         assigned this a its id property.
     * @param   {ComponentCallback} [callback] Optional when not copying between different
     *                                         composites. Gets called when the copy is done or has
     *                                         failed.
     * @returns {AdobeDCXComponent}            Only returns the new component when called without
     *                                         a callback.
     */
    public replaceComponent(component: DCXComponent, newPath?: string, newId?: string, callback?: ComponentCallback) {
        const newComponent = this._copyComponent(
            component,
            /*parentNode*/ undefined,
            newPath,
            newId || component.id,
            /*replaceExisting*/ true,
            callback,
        );
        if (newId && newId !== component.id) {
            delete this._allComponents[component.id];
        }
        return newComponent;
    }

    /**
     * Gets passed into assetOfComponent() and called back whenever the operation has finished.
     * @callback FilePathCallback
     *    @param {Error}     error
     *    @param {String}    filePath
     */

    //******************************************************************************
    // Private
    //******************************************************************************

    /**
     * @private
     */
    private _getHrefOfComponent(component: DCXComponent) {
        // HACK: We need the session in order to determine the source href for the component
        // since we currently do not have that available we just construct it here

        let href;
        const branch = this._getBranchOf(component);
        const compositeHref = branch.compositeHref;

        if (compositeHref && typeof component.version !== 'undefined') {
            href = appendPathElements(compositeHref, component.id);
            href += ';version=' + component.version;
        }

        return href;
    }

    /**
     * Sets or clears the source asset info of the given component.
     * @internal
     * @param {Object}            sourceAssetInfo The asset and component ids to be used for S2SC
     * @param {AdobeDCXComponent} component  The component.
     * @param {Object}            [local]    Optional: The local data object to use.
     */
    _setSourceAssetInfoOfComponent(
        sourceAssetInfo: SourceAssetInfoEntry | undefined,
        component: DCXComponent | AdobeDCXComponentData,
    ) {
        const id = component.id;
        const lookup = this._sourceAssetInfoLookup;
        if (sourceAssetInfo) {
            lookup[id] = sourceAssetInfo;
        } else if (lookup[id]) {
            delete lookup[id];
        }
    }

    /**
     * Returns the source href of the given component if one has previously been recorded.
     * @internal
     * @param   {AdobeDCXComponent} component The component.
     * @param {Object}              [local]   Optional: The local data object to use.
     * @returns {Object|undefined}  The source asset info or undefined.
     */
    _getSourceAssetInfoOfComponent(component: DCXComponent | AdobeDCXComponentData): SourceAssetInfoEntry | undefined {
        const id = component.id;
        const lookup = this._sourceAssetInfoLookup;
        if (lookup) {
            return lookup[id];
        }
        return undefined;
    }

    /**
     * Copies over any source asset info from the given branch core. Overwrites or clears any
     * existing info.
     * @internal
     * @param {AdobeDCXBranchCore} otherCore The core to copy from
     */
    _copySourceHrefsFrom(otherCore: AdobeDCXBranchCore) {
        const otherLookup = otherCore._sourceAssetInfoLookup;
        const myComponents = this.allComponents();
        const numComponents = myComponents.length;

        if (numComponents && otherLookup) {
            const myLookup = this._sourceAssetInfoLookup;
            for (let i = 0; i < numComponents; i++) {
                const componentId = myComponents[i].id;
                myLookup[componentId] = otherLookup[componentId];
            }
        }
    }

    /**
     * @private
     */
    private _getBranchOf(item: DCXComponent | DCXBranch | DCXElement | AdobeDCXBranchCore): DCXBranch {
        if ((item as AdobeDCXBranchCore)._owner) {
            return this._getBranchOf((item as AdobeDCXBranchCore)._owner);
        }
        return item as DCXBranch;
    }

    /**
     * Returns true if the given core is of the same composite.
     * @private
     * @param   {AdobeDCXBranchCore} otherCore The other core.
     * @returns {Boolean}            Whether the composites match.
     */
    private _isSameComposite(otherCore: AdobeDCXBranchCore) {
        const thisBranch = this._getBranchOf(this);
        const otherBranch = this._getBranchOf(otherCore);

        if (thisBranch._data.id !== otherBranch._data.id) {
            // Different composite ids
            return false;
        }

        if (thisBranch.compositeAssetId !== otherBranch.compositeAssetId) {
            // Different identity on the server
            return false;
        }

        if (thisBranch.compositeRepositoryId !== otherBranch.compositeRepositoryId) {
            // Different repository
            return false;
        }

        return true;
    }

    /**
     * Returns a stringified representation of the branch that can be used to persist
     * and later restore it.
     * @internal
     * @param   {Boolean} stripOutLocalData Whether to exclude the local data node.
     * @param   {Boolean} pretty            Whether to pretty print the JSON output.
     * @returns {String}
     */
    _stringify(stripOutLocalData: boolean, pretty = false): string {
        if (stripOutLocalData) {
            // temporarily remove the local node
            const local = this._data.local;
            let remoteData: any = null;
            try {
                delete this._data.local;
                remoteData = JSON.stringify(this._data, undefined, pretty ? 2 : undefined);
            } finally {
                if (local) {
                    this._data.local = local;
                }
            }
            return remoteData;
        }
        return JSON.stringify(this._data, undefined, pretty ? 2 : undefined);
    }

    /**
     * Verifies and sets the new data and creates the caches. Throws if invalid data.
     * @private
     * @param   {Object}         data Data
     * @returns {AdobeDCXBranch} The branch.
     */
    private _setData(data: AdobeDCXNodeData | AdobeDCXRootNodeData): AdobeDCXBranchCore {
        // Create the root node
        const rootNode = new DCXNode(data, this._readOnly, true);
        rootNode._owner = this._owner;
        rootNode._parentPath = '';
        const id = rootNode.id;

        // Create new caches and lookups
        const allComponents: AdobeDCXBranchCore['_allComponents'] = {};
        const allNodes: AdobeDCXBranchCore['_allNodes'] = {};
        const absPaths: AdobeDCXBranchCore['_absolutePaths'] = {};
        allNodes[id] = rootNode;
        absPaths[DCXNode.ROOT_PATH] = rootNode;

        // function to recurse down the hierarchy
        const buildDOMandCachesRecursively = (
            thisNodeData: AdobeDCXRootNodeData | AdobeDCXNodeData,
            parentPath: string,
        ) => {
            let path: string | undefined;
            let absPath: string;
            // Capture the components of this node
            const components = thisNodeData.components;
            if (Array.isArray(components)) {
                for (let i = 0; i < components.length; i++) {
                    const componentData = components[i];
                    const component = new DCXComponent(componentData, this._readOnly);
                    if (allComponents[component.id]) {
                        throw new DCXError(DCXError.INVALID_DATA, 'Duplicate component id: ' + component.id);
                    }
                    component._owner = this._owner;
                    component._parentPath = parentPath;
                    absPath = this._normalizedAbsolutePathForItem(component);
                    if (absPaths[absPath]) {
                        throw new DCXError(DCXError.INVALID_DATA, 'Duplicate absolute path: ' + absPath);
                    }
                    if (!isValidAbsolutePath(absPath)) {
                        throw new DCXError(DCXError.INVALID_DATA, 'Invalid absolute component path: ' + absPath);
                    }
                    allComponents[component.id] = component;
                    absPaths[absPath] = component;
                }
            }

            // Capture the child nodes and recurse down
            const children = thisNodeData.children;
            if (Array.isArray(children)) {
                for (let i = 0; i < children.length; i++) {
                    const nodeData = children[i];
                    const node = new DCXNode(nodeData, this._readOnly);
                    if (allNodes[node.id]) {
                        throw new DCXError(DCXError.INVALID_DATA, 'Duplicate node id: ' + node.id);
                    }
                    path = node.path;
                    node._owner = this._owner;
                    node._parentPath = parentPath;
                    if (path) {
                        absPath = this._normalizedAbsolutePathForItem(node);
                        if (absPaths[absPath]) {
                            throw new DCXError(DCXError.INVALID_DATA, 'Duplicate absolute path: ' + absPath);
                        }
                        if (!isValidAbsolutePath(absPath)) {
                            throw new DCXError(DCXError.INVALID_DATA, 'Invalid node absolute path: ' + absPath);
                        }
                        absPaths[absPath] = node;
                    }
                    allNodes[node.id] = node;

                    // recurse down
                    buildDOMandCachesRecursively(
                        nodeData,
                        node.path ? appendPathElements(parentPath, node.path) : parentPath,
                    );
                }
            }
        };

        // Create components and child nodes while populating the new caches.
        buildDOMandCachesRecursively(data, DCXNode.ROOT_PATH);

        this._data = data;
        this._allComponents = allComponents;
        this._allNodes = allNodes;
        this._absolutePaths = absPaths;

        this._isDirty = false;
        return this;
    }

    /**
     * Recursively removes the given node and all its sub nodes and components from the caches.
     * @private
     * @param {Object} thisNodeData Data
     */
    private _removeNodeFromCachesRecursively(thisNodeData: any) {
        // Remove the components
        const components = thisNodeData.components;
        if (Array.isArray(components)) {
            for (let i = 0; i < components.length; i++) {
                const componentId = components[i].id;
                const component = this._allComponents[componentId];
                delete this._allComponents[componentId];
                delete this._absolutePaths[this._normalizedAbsolutePathForItem(component)];
                component._owner = undefined;
            }
        }

        // Remove the child nodes
        const children = thisNodeData.children;
        if (Array.isArray(children)) {
            for (let i = 0; i < children.length; i++) {
                this._removeNodeFromCachesRecursively(children[i]);
            }
        }

        // Remove this node
        const nodeId = thisNodeData.id;
        const node = this._allNodes[nodeId];
        delete this._allNodes[nodeId];
        if (node.path) {
            delete this._absolutePaths[this._normalizedAbsolutePathForItem(node)];
        }
        node._owner = undefined;
    }

    /**
     * Returns the local node, creating it if necessary.
     * @internal
     * @returns {Object}
     */
    _local(): Required<AdobeDCXLocalNodeData>['local'] {
        if (!this._data.local) {
            this._data.local = { version: 2 };
        }
        return this._data.local;
    }

    /**
     * Recursively resets components.
     * @internal
     * @param {AdobeDCXNodeData} node
     */
    _recursiveReset(node: AdobeDCXNodeData, componentCallback?: (component: AdobeDCXComponentData) => unknown) {
        const components = node.components;

        if (components) {
            for (let i = components.length - 1; i >= 0; i--) {
                // iterating back to front so that we can delete items
                const component = components[i];

                if (component.state === COMPOSITE_STATES.committedDelete) {
                    // the component is no longer used
                    delete components[i];
                    delete this._allComponents[component.id];
                    delete (component as unknown as DCXComponent)._owner;

                    this._setSourceAssetInfoOfComponent(undefined, component);
                } else {
                    delete (component as Partial<typeof component>).etag;
                    delete (component as Partial<typeof component>).version;
                    component.state = COMPOSITE_STATES.modified;
                    if (componentCallback) {
                        componentCallback(component);
                    }
                }
            }
        }

        const children = node.children;

        if (children) {
            for (let i = 0; i < children.length; i++) {
                this._recursiveReset(children[i], componentCallback);
            }
        }
    }

    /**
     * Figures out whether a node with id nodeId is a sub node of nodeData.
     * @private
     * @param   {String}  nodeId
     * @param   {Object}  nodeData
     * @returns {Boolean}
     */
    private _nodeIdIsDescendantOf(nodeId: string, nodeData: AdobeDCXNodeData) {
        const children = nodeData.children;

        if (children) {
            for (let i = 0; i < children.length; i++) {
                const childData = children[i];
                if (childData.id === nodeId || this._nodeIdIsDescendantOf(nodeId, childData)) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Returns an array with all components of the given node and its descendants.
     * @internal
     * @param   {Object} node               The node.
     * @param   {Object} [allComponents=[]] USed for recursion. Leave undefined.
     * @returns {Object}
     */
    _recursivelyGetAllComponentsOfChild(
        node: DCXNode | AdobeDCXNodeData,
        allComponents: DCXComponent[] = [],
    ): DCXComponent[] {
        const data: AdobeDCXNodeData = (node as DCXNode)._data || (node as AdobeDCXNodeData);
        const components = data.components;
        if (components) {
            const c = components.length;
            for (let i = 0; i < c; i++) {
                allComponents[allComponents.length] = this._allComponents[components[i].id];
            }
        }

        const children = data.children;
        if (children) {
            const c = children.length;
            for (let i = 0; i < c; i++) {
                allComponents = this._recursivelyGetAllComponentsOfChild(children[i], allComponents);
            }
        }

        return allComponents;
    }

    /**
     * Returns the node data of the parent node of the given node.
     * @private
     * @param   {Object}   node
     * @param   {Object}   [currentNodeData=this._data] Used for recursion. Don't set.
     * @returns {Object}     found an object of the form { parentNodeData: dataObject, index: integer }. Otherwise null.
     */
    private _nodeDataOfParentOfNode(
        node: DCXNode,
        currentNodeData: AdobeDCXNodeData = this._data,
    ): { parentNodeData: AdobeDCXNodeData; index: number } | null {
        const id = node._data.id,
            children = currentNodeData.children;

        if (children) {
            for (let i = 0; i < children.length; i++) {
                const childData = children[i];
                if (childData.id === id) {
                    return { parentNodeData: currentNodeData, index: i };
                }
                const found = this._nodeDataOfParentOfNode(node, childData);
                if (found) {
                    return found;
                }
            }
        }

        return null;
    }

    /**
     * Returns the node data of the parent node of the given component.
     * @private
     * @param   {Object}   component
     * @param   {Object} [currentNodeData=this._data] Used for recursion. Don't set.
     * @returns {Object} found an object of the form { parentNodeData: dataObject, index: integer }. Otherwise null.
     */
    private _nodeDataOfParentOfComponent(
        component: DCXComponent,
        currentNodeData: AdobeDCXNodeData = this._data,
    ): { parentNodeData: AdobeDCXNodeData; index: number } | null {
        const id = component._data.id,
            components = currentNodeData.components;
        if (components) {
            for (let i = 0; i < components.length; i++) {
                const componentData = components[i];
                if (componentData.id === id) {
                    return { parentNodeData: currentNodeData, index: i };
                }
            }
        }

        const children = currentNodeData.children;
        if (children) {
            for (let i = 0; i < children.length; i++) {
                const childData = children[i];
                const found = this._nodeDataOfParentOfComponent(component, childData);
                if (found) {
                    return found;
                }
            }
        }

        return null;
    }

    /**
     * Collects all the updates necessary to update the absolute paths of all sub nodes and components.
     * @private
     * @param {Object} currentNodeData
     * @param {String} currentAbsolutePath
     * @param {Array}  updates
     */
    private _determineAbsolutePathChangesRecursively(
        currentNodeData: AdobeDCXNodeData,
        currentAbsolutePath: string | undefined,
        updates: { item: DCXNode | DCXComponent; parentPath?: string; absPath?: string }[],
    ) {
        const components = currentNodeData.components;
        if (components) {
            for (let i = 0; i < components.length; i++) {
                const componentData = components[i];
                const component = this._allComponents[componentData.id];

                updates.push({
                    item: component,
                    parentPath: currentAbsolutePath,
                    absPath: appendPathElements(currentAbsolutePath, componentData.path),
                });
            }
        }

        const children = currentNodeData.children;
        if (children) {
            for (let i = 0; i < children.length; i++) {
                const nodeData = children[i];
                const node = this._allNodes[nodeData.id];
                const path = nodeData.path;

                const newAbsPath = path ? appendPathElements(currentAbsolutePath, path) : currentAbsolutePath;
                updates.push({ item: node, parentPath: currentAbsolutePath, absPath: path ? newAbsPath : undefined });

                // recurse down
                this._determineAbsolutePathChangesRecursively(nodeData, newAbsPath, updates);
            }
        }
    }

    /**
     * Updates the path of a node. Caller must dirty the branch.
     *
     * @throws {AdobeDCXError}
     * @internal
     * @param {Object} node
     * @param {String} newPath
     */
    _setPathOfNode(node: DCXNode, newPath?: string) {
        // We need to recalculate the absolute and parent paths of the node and its components as well as
        // of any of its sub nodes and their components.
        // At the same time we need to guarantee that all absolute paths are unique and leave the
        // DOM untouched in case of an error.

        const newAbsPath = newPath ? appendPathElements(node._parentPath, newPath) : node._parentPath;

        const updates: { item: DCXNode | DCXComponent; parentPath?: string; absPath?: string }[] = [
            { item: node, absPath: newPath ? newAbsPath : undefined },
        ];
        this._determineAbsolutePathChangesRecursively(node._data, newAbsPath, updates);

        // Create a new cache by duplicating the old cache, removing the old paths of the changed items and then adding
        // the new paths while checking for duplicates.
        let update;
        const newAbsolutePaths = flatCopy(this._absolutePaths);
        for (let i = 0; i < updates.length; i++) {
            const item = updates[i].item;
            if (item._data.path) {
                delete newAbsolutePaths[this._normalizedAbsolutePathForItem(item)];
            }
        }
        for (let i = 0; i < updates.length; i++) {
            update = updates[i];
            if (update.absPath) {
                const absPath = update.absPath.toLowerCase();
                if (newAbsolutePaths[absPath]) {
                    throw new DCXError(DCXError.DUPLICATE_VALUE, 'Duplicate absolute path: ' + absPath);
                }
                if (!isValidAbsolutePath(absPath)) {
                    throw new DCXError(DCXError.INVALID_PARAMS, 'Invalid absolute path: ' + absPath);
                }
                newAbsolutePaths[absPath] = update.item;
            }
        }

        // Now we can update the parent paths of the items since we know that they are all unique
        for (let i = 0; i < updates.length; i++) {
            update = updates[i];
            if (update.parentPath) {
                update.item._parentPath = update.parentPath;
            }
        }
        node._data.path = newPath;
        this._absolutePaths = newAbsolutePaths;
    }

    /**
     * Updates the path of a component. Caller must dirty the branch.
     *
     * @throws {AdobeDCXError}
     *
     * @internal
     * @param {AdobeDCXComponent} component
     * @param {String} newPath
     */
    _setPathOfComponent(component: DCXComponent, newPath: string) {
        if (!this._allComponents[component.id]) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Unknown component.');
        }

        const newAbsPath = appendPathElements(component._parentPath, newPath).toLowerCase();
        if (this._absolutePaths[newAbsPath]) {
            throw new DCXError(DCXError.DUPLICATE_VALUE, 'Duplicate absolute path for component: ' + newAbsPath);
        }
        if (!isValidAbsolutePath(newAbsPath)) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Invalid absolute path: ' + newAbsPath);
        }

        delete this._absolutePaths[this._normalizedAbsolutePathForItem(component)];
        this._absolutePaths[newAbsPath] = component;

        component._data.path = newPath;
    }

    /**
     * Sets the dirty flag and (optionally) the compositeState
     *
     * @throws {AdobeDCXError}
     *
     * @internal
     * @param {Boolean} [preserveCompositeState]
     */
    _setDirty(preserveCompositeState = false) {
        if (!preserveCompositeState) {
            if (
                this._local().archivalState === COMPOSITE_ARCHIVAL_STATES.pending ||
                this._local().archivalState === COMPOSITE_ARCHIVAL_STATES.archived
            ) {
                throw new DCXError(DCXError.INVALID_STATE, 'Cannot modify an archived composite.');
            }
            if (
                this._data.state === COMPOSITE_STATES.pendingDelete ||
                this._data.state === COMPOSITE_STATES.committedDelete
            ) {
                Logger.warn('Modifying deleted composite');
            }
        }
        this._isDirty = true;
        this._local().change = this.changeCount + 1;
        if (!preserveCompositeState && (this._data.state === COMPOSITE_STATES.unmodified || !this._data.state)) {
            this._data.state = COMPOSITE_STATES.modified;
        }
    }

    /**
     * Returns a normalized and lowercased copy of the absolute path of item.
     * @private
     * @param   {Object}   item
     * @param   {Object} newParent Optional. If given this parent will be used to determine the absolute path.
     * @returns {string}
     */
    private _normalizedAbsolutePathForItem(item: DCXNode | DCXComponent, newParent?: DCXNode): string {
        let path;
        if (newParent) {
            path = appendPathElements(newParent.absolutePath || newParent._parentPath, item.path);
        } else {
            path = item.absolutePath;
        }
        return path.toLowerCase();
    }

    /**
     * @private
     */
    private _hasSameEndpoint(item: unknown) {
        // Composites are no longer associated with hrefs so for now we
        // are assuming that endpoints are the same
        // TODO: add compositeEndpoint property?
        return true;
    }

    /**
     * <p>Deep copies the given component and adds it as a new component to this branch.</p>
     *
     * <p>This function gets called in various different situations and thus needs to work correctly or
     * fail with proper errors in the possible permutations of:</p>
     *
     * <p>- locality: whether the source and target are from the same branch, different branches of the
     * same composite, different composites or even composites with different endpoints.</p>
     * <p>- identity: whether the node should be a new node or replace (update) an existing one.</p>
     * <p>- state: whether the component is unmodified, or modified with a source href.</p>
     *
     * <p>After some initial verification and preparation the function first starts any necessary
     * asynchronous operations (i.e. the file copy). If none are necessary or once they all have
     * succeeded it calls _copyComponentModel to do the actual model changes, passing in a callback that
     * gets called for each component (one in this case) and that takes care of udating local storage
     * and/or source href for a later server-to-server copy request.</p>
     *
     * @private
     */
    private _copyComponent(
        component: DCXComponent,
        parentNode?: DCXNode,
        newPath?: string,
        newId?: string,
        replaceExisting?: boolean,
        callback?: (error?: AdobeDCXError, component?: AdobeDCXComponent) => unknown,
    ) {
        validateParams(['component', component, 'object'], ['callback', callback, 'function', true]);

        const sourceCore = component._owner!._core;
        if (!sourceCore) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Cannot copy, undefined component owner');
        }
        const isSameComposite = this._isSameComposite(sourceCore);
        const isSameEndpoint = isSameComposite || this._hasSameEndpoint(component);
        const sourceLocalData = sourceCore._local();

        // Verify parameters
        if (!isSameEndpoint) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Cannot copy between two different endpoints.');
        }

        // This gets passed as copyComponentCallback into the call to _copyComponentCore. It updates
        // the local file mapping and source href of the copied component as necessary.
        const copyComponentCallback = function (
            originalComponent: DCXComponent,
            copiedComponent: DCXComponent,
            localData: AdobeDCXLocalNodeData['local'],
        ) {
            updateStorageForCopiedComponent(
                originalComponent,
                copiedComponent,
                isSameComposite,
                isSameEndpoint,
                /*canChangeId*/ !newId,
                localData,
                sourceLocalData,
            );
        };

        // Special case: If no callback is given we perform the copy synchronously since all we have to do is
        // to copy the local storage mapping for each component.
        if (!callback) {
            return this._copyComponentModel(
                component,
                parentNode as DCXNode,
                replaceExisting,
                newPath,
                newId,
                copyComponentCallback,
            );
        }

        try {
            callback(
                undefined,
                this._copyComponentModel(
                    component,
                    parentNode as DCXNode,
                    replaceExisting,
                    newPath,
                    newId,
                    copyComponentCallback,
                ),
            );
        } catch (x) {
            callback(
                DCXError.wrapError(
                    DCXError.UNEXPECTED,
                    'Unexpected error attempting to copy component model',
                    x as Error,
                ),
            );
        }
    }

    /**
     * Copies the given component from a different branch of the same composite into this branch.
     * Notice that this method and its call to copyComponentCallback are synchronous. If you need to
     * do anything asynchronous you will need to do that before (preferably) or after calling this
     * method.
     *
     * Throws if it runs into an error.
     * @private
     * @param   {AdobeDCXComponent} component               The component to copy.
     * @param   {AdobeDCXNode}      [parentNode]            Optional: The node to copy the component to.
     *                                                      If none if provided then the component will
     *                                                      be added to the root.
     * @param   {Boolean}           [replaceExisting]       Optional: Default is false. Whether to
     *                                                      replace an existing component.
     * @param   {String}            [newPath]               Optional: If provided the copy of the
     *                                                      component will be assigned this a its
     *                                                      path property and it will also get
     *                                                      assigned a new id.
     * @param   {String}            [newId]                 Optional: If provided the copy of the
     *                                                      component will be assigned this as its id.
     * @param   {ComponentCallback} [copyComponentCallback] Optional. Gets called for each component
     *                                                      that gets copied. Caller can use this to
     *                                                      tweak the component before it gets added
     *                                                      to the DOM.
     *                                                      Signature: function (originalComponent,
     *                                                      copiedCoponent, localDataOfTargetBranch)
     * @returns {AdobeDCXComponent}                         The new component
     */
    private _copyComponentModel(
        component: DCXComponent,
        parentNode?: DCXNode,
        replaceExisting?: boolean,
        newPath?: string,
        newId?: string,
        copyComponentCallback?: (
            originalComponent: DCXComponent,
            copiedComponent: DCXComponent,
            localData: AdobeDCXLocalNodeData['local'],
        ) => unknown,
    ) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        if (!component) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting param "component".');
        }

        const sourceBranch = component._owner as DCXBranch;

        // Create the new component as a copy of component
        const newComponent = new DCXComponent(JSON.parse(JSON.stringify(component._data)));
        if (newPath) {
            newComponent.path = newPath;
            newComponent.id = newId || generateUuid();
        } else if (newId) {
            newComponent.id = newId;
        }
        newComponent._owner = this._owner;
        if (component.id !== newComponent.id || !this._isSameComposite(sourceBranch._core)) {
            newComponent.state = COMPOSITE_STATES.modified;
            newComponent.etag = undefined;
            newComponent.version = undefined;
            newComponent.md5 = undefined;
            newComponent.length = undefined;
        }

        if (this._allComponents[newComponent.id] && !replaceExisting) {
            throw new DCXError(DCXError.DUPLICATE_VALUE, 'Component already exists.');
        }

        const existingComponent = this._allComponents[component.id];
        if (replaceExisting && !existingComponent) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Could not find existing component.');
        }

        // Determine the new parent
        let index, newParent;
        if (!replaceExisting) {
            newParent = parentNode ? this._allNodes[parentNode.id] : this.rootNode;
            if (!newParent) {
                throw new DCXError(DCXError.INVALID_PARAMS, 'Unknown parent node.');
            }
        } else {
            const found = this._nodeDataOfParentOfComponent(existingComponent);
            if (!found) {
                throw new DCXError(
                    DCXError.INVALID_PARAMS,
                    'Parent node of existing component not found in this branch.',
                );
            }
            const parentId = found.parentNodeData.id;
            newParent = this._allNodes[parentId];
            if (!newParent) {
                throw new DCXError(DCXError.INVALID_STATE, 'Unknown parent node.');
            }
            index = found.index;
        }
        newComponent._parentPath = newParent.absolutePath || newParent._parentPath;

        // Check path
        const absPath = this._normalizedAbsolutePathForItem(newComponent);
        const itemWithSamePath = this._absolutePaths[absPath];
        if (itemWithSamePath && itemWithSamePath !== existingComponent) {
            throw new DCXError(DCXError.DUPLICATE_VALUE, 'Duplicate absolute path: ' + absPath);
        }
        if (!isValidAbsolutePath(absPath)) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Component path must be a valid path for a component.');
        }

        let newLocalData = this._local();
        if (copyComponentCallback) {
            // Call back to give the caller a chance to tweak the copied component before we insert it.
            newLocalData = deepCopy(newLocalData);
            copyComponentCallback(component, newComponent, newLocalData);
        }

        // Now we can make the actual model changes:

        // Insert new component
        const components = newParent._data.components;
        if (replaceExisting) {
            components[index] = newComponent.data;
        } else {
            if (components) {
                components.push(newComponent.data);
            } else {
                newParent._data.components = [newComponent.data];
            }
        }

        // Update caches
        this._data.local = newLocalData;
        this._allComponents[newComponent.id] = newComponent;
        this._absolutePaths[absPath] = newComponent;

        this._setDirty();

        // SUCCESS
        return newComponent;
    }

    /**
     * <p>Deep copies the given child node and its nodes and components into this branch as a new child.
     * The node can be from the same or from a different composite.
     * If it is a root node then you must specify newPath since it can't keep its original path.</p>
     *
     * <p>This function gets called in various different situations and thus needs to work correctly or
     * fail with proper errors in the various possible permutations:</p>
     *
     * <p>- whether the source and target are from the same branch, different branches of the
     * same composite, different composites or even composites with different endpoints.</p>
     * <p>- whether the node should be a new node or replace (update) an existing one.</p>
     * <p>- whether the components are unmodified, or modified with a source href.</p>
     *
     * <p>After some initial verification and preparation the function first starts any necessary
     * asynchronous operations (i.e. file copies). If none are necessary or once they all have
     * succeeded it calls _copyChildModel to do the actual model changes, passing in a callback that
     * gets called for each component and that takes care of udating local storage and/or source hrefs
     * for a later server-to-server copy request.</p>
     *
     * <p>The function creates copies of the lookup tables and manipulates those during the copy
     * operation. Only once everything has succeeded it inserts the new node (and removes the one to replace)
     * and replaces the lookup tables with its copies so that it can bail out at any time before that
     * without having actually modified the DOM.</p>
     *
     * @private
     */
    private _copyChild(
        node: DCXNode,
        parentNode?: DCXNode,
        index?: number,
        newPath?: string,
        newId?: string,
        replaceExisting?: boolean,
        callback?: any,
    ) {
        if (!node) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting param "node".');
        }

        const sourceCore = node._owner!._core as AdobeDCXBranchCore;
        // if (!sourceCore) {
        //     throw new DCXError(DCXError.INVALID_PARAMS, 'Cannot copy component without owner.');
        // }

        const isSameComposite = this._isSameComposite(sourceCore);
        const isSameEndpoint = isSameComposite || this._hasSameEndpoint(node);

        const sourceLocalData = sourceCore._local();
        const sourceComponents = sourceCore._recursivelyGetAllComponentsOfChild(node);

        if (sourceComponents.length > 0) {
            if (!isSameEndpoint) {
                throw new DCXError(DCXError.INVALID_PARAMS, 'Cannot copy components between two different endpoints.');
            }
        }

        // This gets passed as copyComponentCallback into the call to _copyComponentCore. It updates
        // the local file mapping and source href of the copied component as necessary.
        const copyComponentCallback: CopyComponentCallback = (originalComponent, copiedComponent, localData) => {
            updateStorageForCopiedComponent(
                originalComponent,
                copiedComponent,
                isSameComposite,
                isSameEndpoint,
                /*canChangeId*/ true,
                localData,
                sourceLocalData,
            );
        };

        if (!callback) {
            return this._copyChildModel(
                node,
                parentNode,
                index,
                replaceExisting,
                newPath,
                newId,
                copyComponentCallback,
            );
        }

        try {
            callback(
                undefined,
                this._copyChildModel(node, parentNode, index, replaceExisting, newPath, newId, copyComponentCallback),
            );
        } catch (x) {
            callback(x);
        }
    }

    /**
     * Copies the given child node with all its nodes and components. Notice that this method and
     * its call to copyComponentCallback are synchronous. If you need to do anything asynchronous you
     * will need to do that before (preferably) or after calling this method.
     *
     * Throws if it runs into an error.
     * @private
     * @param   {AdobeDCXNode} node                  The child node (or whole branch) to copy.
     * @param   {AdobeDCXNode} parentNode            The parent node to copy the child node to.
     *                                               Gets ignored if replaceExisting is true.
     * @param   {Integer}      index                 Optional: If given and less than or equal to the
     *                                               current number of children of the parentNode the
     *                                               node gets inserted at the given index. Otherwise
     *                                               it gets added to the end. Gets ignored if
     *                                               replaceExisting is true.
     * @param   {Boolean}      replaceExisting       Optional: Default is false.
     * @param   {String}       newPath               Optional: If provided the copy of the component
     *                                               will be assigned this a its path property.
     * @param   {String}       newId                 Optional: If provided the copy of the component
     *                                               will be assigned this as its id.
     * @param   {Boolean}      reuseIds              If true we try to reuse ids.
     * @param   {NodeCallback} copyComponentCallback Optional. Gets called for each component that
     *                                               gets copied. Caller can use this to tweak the
     *                                               component before it gets added to the DOM.
     *                                               Should throw if something goes wrong.
     *                                               Signature: function (originalComponent,
     *                                               copiedCoponent, localDataOfTargetBranch)
     * @returns {AdobeDCXNode}          new child node.
     */
    private _copyChildModel(
        node: DCXNode,
        parentNode?: DCXNode,
        index?: number,
        replaceExisting?: boolean,
        newPath?: string,
        newId?: string,
        copyComponentCallback?: CopyComponentCallback,
    ) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }

        // TODO: verify that undefined owner can't reach here
        const sourceBranch = node._owner as DCXBranch;

        // Create the new node
        const newNode = new DCXNode(JSON.parse(JSON.stringify(node._data)));
        if (newPath) {
            newNode.path = newPath;
            newNode.id = newId || generateUuid();
        } else if (newId) {
            newNode.id = newId;
        }

        newNode._owner = this._owner;
        if (node.isRoot) {
            // Remove unnecessary properties from an embedded composite a.k.a element
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            delete newNode._data.local;
            delete (newNode._data as unknown as Record<string, unknown>).state;
            delete (newNode._data as unknown as Record<string, unknown>)['manifest-format-version'];
        }

        // Verify pre-existing node
        const existingNode = this._allNodes[newNode.id];
        if (existingNode && !replaceExisting) {
            throw new DCXError(DCXError.DUPLICATE_VALUE, 'Child node already exists.');
        }
        if (replaceExisting && !existingNode) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Could not find existing node to replace.');
        }

        // Determine the new parent
        let newParent: DCXNode;
        if (!replaceExisting) {
            newParent = parentNode ? this._allNodes[parentNode.id] : this.rootNode;
            if (!newParent) {
                throw new DCXError(DCXError.INVALID_PARAMS, 'Unknown parent node.');
            }
        } else {
            // We ignore any parent node that was passed in and just use the parent node of the node to replace.
            const found = this._nodeDataOfParentOfNode(existingNode);
            if (!found) {
                throw new DCXError(DCXError.INVALID_PARAMS, 'Parent of existing node not found in this branch.');
            }
            const parentId = found.parentNodeData.id;
            newParent = this._allNodes[parentId];
            if (!newParent) {
                throw new DCXError(DCXError.INVALID_STATE, 'Unknown parent node.');
            }
            index = found.index;
        }
        newNode._parentPath = newParent.absolutePath || newParent._parentPath;

        // Verify path
        if (newNode.path) {
            // Check that the resulting absolute path of the node will be unique
            const absPath = this._normalizedAbsolutePathForItem(newNode);
            const itemWithSamePath = this._absolutePaths[absPath];
            if (itemWithSamePath && itemWithSamePath !== existingNode) {
                throw new DCXError(DCXError.DUPLICATE_VALUE, 'Duplicate absolute path: ' + absPath);
            }
            if (!isValidAbsolutePath(absPath)) {
                throw new DCXError(DCXError.INVALID_PARAMS, 'Node path must be a valid path for a node.');
            }
        }

        // Now that we have passed all the checks above we can start making the necessary changes.
        // We do so by creating copies of our lookup tables and the local storage data so that we can
        // back out cleanly if we run into a problem while traversing the new node's hiearchy.

        const newAllNodes = flatCopy(this._allNodes);
        const newAllComponents = flatCopy(this._allComponents);
        const newAbsolutePaths = flatCopy(this._absolutePaths);
        const newLocalData = deepCopy(this._local());

        // Recurses through the node and its descendents and removes its contents from all the lookup tables
        // that are being passed in.
        const recursivelyRemoveNodeFromLookups = function (
            recursiveNode: DCXNode,
            core: AdobeDCXBranchCore,
            allNodes: Record<string, DCXNode>,
            allComponents: Record<string, DCXComponent>,
            absolutePaths: Record<string, DCXNode | DCXComponent>,
            localData: AdobeDCXLocalNodeData['local'],
        ) {
            delete allNodes[recursiveNode.id];
            if (recursiveNode.path) {
                delete absolutePaths[core._normalizedAbsolutePathForItem(recursiveNode)];
            }

            let i, count;
            const children = recursiveNode._data.children;
            if (children) {
                count = children.length;
                for (i = 0; i < count; i++) {
                    recursivelyRemoveNodeFromLookups(
                        allNodes[children[i].id],
                        core,
                        allNodes,
                        allComponents,
                        absolutePaths,
                        localData,
                    );
                }
            }

            const components = recursiveNode._data.components;
            if (components) {
                count = components.length;
                for (i = 0; i < count; i++) {
                    const component = allComponents[components[i].id];
                    delete allComponents[component.id];
                    delete absolutePaths[core._normalizedAbsolutePathForItem(component)];
                }
            }
        };

        if (existingNode) {
            recursivelyRemoveNodeFromLookups(
                existingNode,
                this,
                newAllNodes,
                newAllComponents,
                newAbsolutePaths,
                newLocalData,
            );
        }

        const recursivelyAddNodeToLookups = (
            recursiveNode: DCXNode,
            owner: DCXBranch | DCXElement,
            allNodes: Record<string, DCXNode>,
            allComponents: Record<string, DCXComponent>,
            absolutePaths: Record<string, DCXNode | DCXComponent>,
            localData: AdobeDCXLocalNodeData['local'],
            sourceLocalData: AdobeDCXLocalNodeData['local'],
        ) => {
            if (allNodes[recursiveNode.id]) {
                throw new DCXError(DCXError.DUPLICATE_VALUE, 'Duplicate node id: ' + recursiveNode.id);
            }
            allNodes[recursiveNode.id] = recursiveNode;

            let newAbsPath;
            if (recursiveNode.path) {
                newAbsPath = owner._core._normalizedAbsolutePathForItem(recursiveNode);
                if (absolutePaths[newAbsPath]) {
                    throw new DCXError(DCXError.DUPLICATE_VALUE, 'Duplicate absolute path: ' + newAbsPath);
                }
                absolutePaths[newAbsPath] = recursiveNode;
            }

            const children = recursiveNode._data.children;
            if (children) {
                const count = children.length;
                for (let i = 0; i < count; i++) {
                    const child = new DCXNode(children[i]);
                    if (allNodes[child.id]) {
                        child.id = generateUuid();
                    }
                    child._owner = owner;
                    child._parentPath = recursiveNode.absolutePath || recursiveNode._parentPath;
                    recursivelyAddNodeToLookups(
                        child,
                        owner,
                        allNodes,
                        allComponents,
                        absolutePaths,
                        localData,
                        sourceLocalData,
                    );
                }
            }

            const components = recursiveNode._data.components;
            if (components) {
                const count = components.length;
                for (let i = 0; i < count; i++) {
                    const component = new DCXComponent(components[i]);
                    const originalComponentId = component.id;
                    if (allComponents[originalComponentId]) {
                        component.id = generateUuid();
                        component.state = COMPOSITE_STATES.modified;
                        component.etag = undefined;
                        component.version = undefined;
                        component.md5 = undefined;
                        component.length = undefined;
                    }
                    component._owner = owner;
                    component._parentPath = recursiveNode.absolutePath || recursiveNode._parentPath;
                    if (copyComponentCallback) {
                        copyComponentCallback(
                            sourceBranch.getComponentWithId(originalComponentId) as DCXComponent,
                            component,
                            localData,
                        );
                    }
                    if (allComponents[component.id]) {
                        throw new DCXError(DCXError.DUPLICATE_VALUE, 'Duplicate component id: ' + component.id);
                    }
                    allComponents[component.id] = component;
                    newAbsPath = owner._core._normalizedAbsolutePathForItem(component);
                    if (absolutePaths[newAbsPath]) {
                        throw new DCXError(DCXError.DUPLICATE_VALUE, 'Duplicate absolute path: ' + newAbsPath);
                    }
                    absolutePaths[newAbsPath] = component;
                }
            }
        };

        recursivelyAddNodeToLookups(
            newNode,
            this._owner,
            newAllNodes,
            newAllComponents,
            newAbsolutePaths,
            newLocalData,
            sourceBranch._local(),
        );

        // If none of the previous code has thrown we can actually add the new node to the model and upadate the caches.

        if (replaceExisting) {
            (newParent as any)._data.children[index as number] = newNode._data;
        } else {
            const children = newParent._data.children;
            if (children) {
                if (!((index as number) >= 0 && (index as number) <= children.length)) {
                    index = children.length;
                }
                if (index === children.length) {
                    // Simple case: add to end
                    children[index as number] = newNode._data;
                } else {
                    // Insert
                    children.splice(index as number, 0, newNode._data);
                }
            } else {
                newParent._data.children = [newNode._data];
            }
        }

        // Update caches, local data and lookups
        this._allNodes = newAllNodes;
        this._allComponents = newAllComponents;
        this._absolutePaths = newAbsolutePaths;
        this._data.local = newLocalData;

        this._setDirty();

        // SUCCESS
        return newNode;
    }

    /**
     * Verifies the integrity of the in-memory structures of the branch.
     * Looks for incorrect caches/lookup tables, incorrect
     * object references and cycles/duplicate objects.
     * Also, optionally, verifies that all asset files exist locally.
     *
     * @internal
     * @param   {Boolean}  shouldBeComplete Whether to check for the existence of all component assets.
     *                                      @warning This check is using synchronous file system calls.
     * @param   {Function} [logger]         A function that gets called for every error found.
     *                                      Signature: function (string)
     * @param   {Object}   [fs]             The file system object to use in the shouldBeComplete check
     * @returns {Array}    Array of errors or null if everything is OK.
     */
    _verifyIntegrity(shouldBeComplete?: boolean, logger?: (log?: string) => unknown, fs?: any): AdobeDCXError[] | null {
        const errors: AdobeDCXError[] = [];
        let startedLog = false;

        const logError = (message: string) => {
            errors.push(new DCXError(DCXError.INVALID_STATE, message));
            if (logger) {
                if (!startedLog) {
                    logger('Branch for composite ' + this.rootNode.id + ' has the following errors:');
                    startedLog = true;
                }
                logger('   ' + message);
            }
            return false;
        };

        const assert = function (condition: unknown, message: string) {
            if (!condition) {
                return logError(message);
            }
            return true;
        };

        const itemsEncountered: (DCXNode | DCXComponent)[] = [];
        const itemEncountered = function (item: DCXNode | DCXComponent) {
            if (itemsEncountered.indexOf(item) < 0 || logError('Item ' + item.id + ' encountered more than once')) {
                itemsEncountered.push(item);
            }
        };

        // Make copies of the caches
        const allComponents = flatCopy(this._allComponents);
        const allNodes = flatCopy(this._allNodes);
        const absolutePaths = flatCopy(this._absolutePaths);

        const absPathArray = Object.keys(absolutePaths);
        const count = absPathArray.length;
        for (let i = 0; i < count; i++) {
            const absPath = absPathArray[i];
            assert(absPath.charAt(0) === '/', 'Absolute path ' + absPath + ' does not start with a slash.');
        }

        const recurse = (nodeData: AdobeDCXNodeData, parentPath: string) => {
            let j, newAbsPath;
            const children = nodeData.children;
            if (children) {
                for (j = 0; j < children.length; j++) {
                    const childData = children[j];
                    const node = allNodes[childData.id];
                    if (node || logError('Node ' + childData.id + ' is not in cache.')) {
                        itemEncountered(node);
                        assert(node._data === childData, 'Node ' + node.id + ' _data property incorrect.');
                        delete allNodes[node.id];
                        if (node.path && node._owner) {
                            newAbsPath = this._normalizedAbsolutePathForItem(node);
                            if (
                                absolutePaths[newAbsPath] ||
                                logError('Absolute path of node ' + node.id + ' (' + newAbsPath + ') is not in cache.')
                            ) {
                                delete absolutePaths[newAbsPath];
                            }
                        }
                        assert(
                            node._parentPath === parentPath,
                            'Parent path of node ' +
                                node.id +
                                ' should be ' +
                                parentPath +
                                ' but is ' +
                                node._parentPath,
                        );
                        assert(node._owner === this._owner, 'Node ' + node.id + ' _owner property is not correct.');
                    }
                    recurse(childData, childData.path ? appendPathElements(parentPath, childData.path) : parentPath);
                }
            }
            const components = nodeData.components;
            if (components) {
                for (j = 0; j < components.length; j++) {
                    const componentData = components[j];
                    const component = allComponents[componentData.id];
                    if (component || logError('Component ' + componentData.id + ' is not in cache.')) {
                        itemEncountered(component);
                        assert(
                            component._data === componentData,
                            'Component ' + component.id + ' _data property incorrect.',
                        );
                        delete allComponents[component.id];
                        if (component._owner) {
                            newAbsPath = this._normalizedAbsolutePathForItem(component);
                            if (
                                absolutePaths[newAbsPath] ||
                                logError('Absolute path ' + newAbsPath + ' is not in cache.')
                            ) {
                                delete absolutePaths[newAbsPath];
                            }
                        }
                        assert(
                            component._parentPath === parentPath,
                            'Parent path of component ' +
                                component.id +
                                ' should be ' +
                                parentPath +
                                ' but is ' +
                                component._parentPath,
                        );
                        assert(
                            component._owner === this._owner,
                            'Component ' + component.id + ' _owner property is not correct.',
                        );
                    }
                }
            }
        };

        // Verify that the root node is there
        const rootNode = absolutePaths[DCXNode.ROOT_PATH];
        if (assert(rootNode, 'Cannot find root node via path')) {
            assert(rootNode.id === this._data.id, 'Root node has correct id');
            assert(rootNode.path === DCXNode.ROOT_PATH, 'Root node has correct path');
            assert(rootNode._parentPath === '', 'Root node has correct parent path');
            assert(rootNode._owner === this._owner, 'Root node has correct owner');
            assert(rootNode._data === this._data, 'Root node has correct data');
            if (assert(allNodes[rootNode.id] === rootNode, 'Cannot find root node via id')) {
                delete allNodes[rootNode.id];
            }
            delete absolutePaths[DCXNode.ROOT_PATH];
        }

        recurse(this._data, DCXNode.ROOT_PATH);

        // Ensure that all our chaches are empty
        let keysLeft = Object.keys(allComponents);
        for (let i = 0; i < keysLeft.length; i++) {
            logError('Component ' + keysLeft[i] + ' is in cache but could not be found.');
        }
        keysLeft = Object.keys(allNodes);
        for (let i = 0; i < keysLeft.length; i++) {
            logError('Node ' + keysLeft[i] + ' is in cache but could not be found.');
        }
        keysLeft = Object.keys(absolutePaths);
        for (let i = 0; i < keysLeft.length; i++) {
            logError('Absolute path ' + keysLeft[i] + ' is in cache but could not be found.');
        }

        return errors.length ? errors : null;
    }
}

/**
 * This gets called in the context of the component callback in both _copyChild and _copyComponent.
 * It updates local storage, source hrefs and state of the copied component based on the circumstances.
 *
 * @private
 *
 * @param {Object}  originalComponent The component that got copied.
 * @param {Object}  copiedComponent   The copy of the component.
 * @param {Boolean} isSameComposite   Whether the original and copied component are from the same
 *                                    composite.
 * @param {Boolean} isSameEndpoint    Whether the composite of original and copy reside at the same
 *                                    endpoint.
 * @param {Boolean} canChangeId       Whether it is ok to change the id of the component.
 * @param {Object}  localData         The new local data section for the target branch.
 * @param {Object}  sourceLocalData   The local data section of the soure branch to be used when
 *                                    copying within the same composite.
 */
function updateStorageForCopiedComponent(
    originalComponent: DCXComponent,
    copiedComponent: DCXComponent,
    isSameComposite: boolean,
    isSameEndpoint: boolean,
    canChangeId: boolean,
    localData: any,
    sourceLocalData: any,
) {
    if (!copiedComponent._owner || !originalComponent._owner) {
        throw new DCXError(
            DCXError.INVALID_PARAMS,
            'Failed to update storage for copied component, missing target or source branch core.',
        );
    }
    const targetCore = copiedComponent._owner._core;
    const sourceCore = originalComponent._owner._core;

    // If the component gets copied within a composite and if it retains its id it by
    // definition retains its identity.
    const retainsIdentity = isSameComposite && originalComponent.id === copiedComponent.id;
    // See whether the original component already has a source asset data
    let sourceAssetInfo = sourceCore._getSourceAssetInfoOfComponent(originalComponent);
    // See whether we have an existing component with the same id that we are going to update.
    const existingComponentWithSameId = targetCore.getComponentWithId(copiedComponent.id);

    // Currently S2SC is only possible if all of the below is true:
    const s2scIsPossible =
        isSameEndpoint &&
        (!existingComponentWithSameId || !existingComponentWithSameId.etag) &&
        originalComponent.state === COMPOSITE_STATES.unmodified;

    // Regardless of whether S2SC is possible or not, we only want to do it if we are dealing
    // with a new component and the original doesn't already have a source href.
    const s2scIsDesired = !retainsIdentity && !sourceAssetInfo;

    // We assume that S2SC is required if it is desired. We set this to false later if we find
    // that we have a valid local file that we could upload instead.
    const s2scIsRequired = s2scIsDesired;

    if (canChangeId && !existingComponentWithSameId) {
        if (sourceAssetInfo || s2scIsDesired) {
            // TODO: Figure out whether we already have upload results from a previously
            // interrupted push. If so then change the id so that we don't have to copy
            // the asset again.

            if (originalComponent.id === copiedComponent.id) {
                // TODO: See whether we can remove this when we get full s2sc support.

                // Currently the storage service does not support updates via s2sc. If we are
                // using s2sc we need to ensure that we do not reuse a component id that might
                // have existed previously. An example would be if I copy a library element into
                // a composite, push, delete the element and then copy it again.
                copiedComponent._data.id = generateUuid();
                copiedComponent.state = COMPOSITE_STATES.modified;
                copiedComponent.etag = undefined;
                copiedComponent.version = undefined;
                copiedComponent.md5 = undefined;
                copiedComponent.length = undefined;
            }
        }
    }

    // Update the source href for S2SC
    if (s2scIsDesired) {
        if (s2scIsPossible && originalComponent._owner.compositeAssetId) {
            sourceAssetInfo = {
                compositeAssetId: originalComponent._owner.compositeAssetId,
                componentId: originalComponent.id,
                componentVersion: originalComponent.version as string,
                componentPath: originalComponent.absolutePath as string,
                repositoryId: originalComponent._owner.compositeRepositoryId,
            };

            targetCore._setSourceAssetInfoOfComponent(sourceAssetInfo, copiedComponent);
        } else if (s2scIsRequired) {
            throw new DCXError(
                DCXError.INVALID_STATE,
                'Could not use server-to-server copy for component ' + originalComponent.id,
            );
        }
    } else {
        // We need to set the source href whether it is defined or not.
        targetCore._setSourceAssetInfoOfComponent(sourceAssetInfo, copiedComponent);
        // 10/25/20 remove localData param since called fn doesn't use it
        // targetCore._setSourceAssetInfoOfComponent(sourceAssetInfo, copiedComponent, localData);
    }

    if (sourceAssetInfo) {
        copiedComponent.state = COMPOSITE_STATES.modified;
    }
}

export default AdobeDCXBranchCore;
