/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { isAdobeDCXBranchLike } from '@dcx/assets';
import {
    AdobeDCXBranch,
    AdobeDCXComponentData,
    AdobeDCXElement,
    AdobeDCXError,
    AdobeDCXState,
    AdobeUploadRecord,
    HalLink,
    AdobeDCXComponent as IAdobeDCXComponent,
} from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { CURRENT_COMPONENT_DESCRIPTOR_HASH_TYPE, CURRENT_COMPONENT_DESCRIPTOR_VERSION } from '@dcx/repo-api-session';
import { appendPathElements, isValidPath, objectsEqual } from '@dcx/util';
import DCXBranch from './AdobeDCXBranch';
import DCXElement from './AdobeDCXElement';
import { COMPOSITE_STATES } from './enum';

const reservedKeys: Record<string, boolean> = {
    name: true,
    id: true,
    state: true,
    path: true,
    rel: true,
    type: true,
    etag: true,
    length: true,
    version: true,
    md5: true,
    width: true,
    height: true,
};

/**
 * @class
 * @classdesc AdobeDCXComponent represents a component of a DCX manifest.
 * <p>The constructor for AdobeDCXComponent is private. Refer to {@link AdobeDCXBranch} to
 * learn how to access existing components or create new ones.
 * @hideconstructor
 * @param {Object}  data
 * @param {Boolean} readOnly
 */
export class AdobeDCXComponent implements IAdobeDCXComponent {
    /** @internal */
    private _readOnly = false;

    /** @internal */
    private _compositeId?: string;

    /** @internal */
    _data: AdobeDCXComponentData = {} as AdobeDCXComponentData;

    /** @internal */
    _owner?: DCXBranch | DCXElement;

    /** @internal */
    _parentPath?: string;

    /** @internal */
    _assetId?: string;

    /** @internal */
    _repoId?: string;

    public records: Record<string, AdobeUploadRecord> = {}; // map: originalComponentId : UploadRecord

    static readonly STATES = COMPOSITE_STATES;

    constructor(data?: AdobeDCXComponentData, readOnly = false) {
        if (data) {
            this._setData(data);
        }
        this._readOnly = readOnly;
    }

    //******************************************************************************
    // Getters/setters for properties
    //******************************************************************************

    public get compositeId() {
        return this._compositeId;
    }

    public get owner(): AdobeDCXBranch | AdobeDCXElement | undefined {
        return this._owner;
    }

    public get data() {
        return this._data;
    }

    /**
     * The id of the component. Must be a unique among the components of the composite.
     *
     * <p>Cannot be changed for a component that is part of a branch.</p>
     * @memberof AdobeDCXComponent#
     * @type {String}
     */
    public get id(): string {
        return this._data.id;
    }
    public set id(id: string) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (this._owner) {
            throw new DCXError(
                DCXError.READ_ONLY,
                'Cannot change the id of a component that is part of a branch or element.',
            );
        } else if (typeof id !== 'string' || id === '') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a non-empty string.');
        } else {
            this._data.id = id;
            this._setDirty();
        }
    }

    /**
     * The name of the component.
     *
     * @memberof AdobeDCXComponent#
     * @type {String}
     */
    public get name(): string | undefined {
        return this._data.name;
    }
    public set name(name: string | undefined) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof name !== 'string' && typeof name !== 'undefined') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string or undefined.');
        } else {
            this._data.name = name;
            this._setDirty();
        }
    }

    /**
     * The type of the component.
     *
     * @memberof AdobeDCXComponent#
     * @type {String}
     */
    public get type(): string {
        return this._data.type;
    }
    public set type(type: string) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof type !== 'string' || type === '') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a non-empty string.');
        } else {
            this._data.type = type;
            this._setDirty();
        }
    }

    /**
     * The path of the component.
     *
     * @memberof AdobeDCXComponent#
     * @type {String}
     */
    public get path(): string {
        return this._data.path as string;
    }
    public set path(path: string) {
        if (this._data.path !== path) {
            if (this._readOnly) {
                throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
            } else if (!isValidPath(path)) {
                throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a valid path.');
            } else {
                if (this._owner) {
                    // We need to call the owner to update the path and handle
                    // all neccessary updates to absolute paths
                    this._owner._core._setPathOfComponent(this, path);
                } else {
                    this._data.path = path;
                    this._parentPath = '';
                }
                this._setDirty();
            }
        }
    }

    /**
     * The absolute path of the parent of the component.
     *
     * @memberof AdobeDCXComponent#
     * @readonly
     * @type {String}
     */
    public get parentPath(): string | undefined {
        return this._parentPath;
    }
    public set parentPath(_) {
        throw new DCXError(DCXError.READ_ONLY, 'parentPath is read-only.');
    }

    /**
     * The absolute path of the component.
     *
     * @memberof AdobeDCXComponent#
     * @readonly
     * @type {String}
     */
    public get absolutePath(): string | undefined {
        return this._data.path && this._owner ? appendPathElements(this._parentPath, this._data.path) : undefined;
    }
    public set absolutePath(_) {
        throw new DCXError(DCXError.READ_ONLY, 'absolutePath is read-only.');
    }

    /**
     * The relationship of the component to its parent node and its sibling components.
     *
     * @memberof AdobeDCXComponent#
     * @type {String}
     */
    public get relationship(): string | undefined {
        return this._data.rel;
    }
    public set relationship(relationship: string | undefined) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof relationship !== 'string' && typeof relationship !== 'undefined') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string or undefined.');
        } else {
            this._data.rel = relationship;
            this._setDirty();
        }
    }

    /**
     * The edit state of the component.
     *
     * @memberof AdobeDCXComponent#
     * @readonly
     * @private
     * @type {String}
     */
    public get state(): AdobeDCXState {
        return this._data.state;
    }
    public set state(state: AdobeDCXState) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof state !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string.');
        } else if (!Object.keys(COMPOSITE_STATES).includes(state)) {
            throw new DCXError(
                DCXError.INVALID_PARAMS,
                'State must be "modified", "unmodified", "pendingDelete", or "committedDelete".',
            );
        } else {
            this._data.state = state;
            this._setDirty();
        }
    }

    /**
     * The etag of the component.
     *
     * @memberof AdobeDCXComponent#
     * @readonly
     * @private
     * @type {String}
     */
    public get etag(): string | undefined {
        return this._data.etag;
    }
    public set etag(etag: string | undefined) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof etag !== 'string' && typeof etag !== 'undefined') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string or undefined.');
        } else {
            this._data.etag = etag;
            this._setDirty();
        }
    }

    /**
     * The md5 of the component.
     *
     * @memberof AdobeDCXComponent#
     * @readonly
     * @private
     * @type {String}
     */
    public get md5(): string | undefined {
        return this._data.md5;
    }
    public set md5(md5: string | undefined) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof md5 !== 'string' && typeof md5 !== 'undefined') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string or undefined.');
        } else {
            this._data.md5 = md5 as string;
            this._setDirty();
        }
    }

    /**
     * The version of the component.
     *
     * @memberof AdobeDCXComponent#
     * @readonly
     * @private
     * @type {String}
     */
    public get version(): string | undefined {
        return this._data.version;
    }
    public set version(version: string | undefined) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof version !== 'undefined' && typeof version !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string or undefined.');
        } else {
            this._data.version = version;
            this._setDirty();
        }
    }

    /**
     * The content length of the component.
     *
     * @memberof AdobeDCXComponent#
     * @readonly
     * @type {Integer}
     */
    public get length(): number | undefined {
        return this._data.length;
    }
    public set length(length: number | undefined) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof length !== 'number' && typeof length !== 'undefined') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a number or undefined.');
        } else {
            this._data.length = length;
            this._setDirty();
        }
    }

    /**
     * The width property of the component.
     *
     * @memberof AdobeDCXComponent#
     * @type {Number}
     */
    public get width(): number | undefined {
        return this._data.width;
    }
    public set width(width: number | undefined) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof width !== 'number' && typeof width !== 'undefined') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a number or undefined.');
        } else {
            this._data.width = width;
            this._setDirty();
        }
    }

    /**
     * The height property of the component.
     *
     * @memberof AdobeDCXComponent#
     * @type {Number}
     */
    public get height(): number | undefined {
        return this._data.height;
    }
    public set height(height: number | undefined) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof height !== 'number' && typeof height !== 'undefined') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a number or undefined.');
        } else {
            this._data.height = height;
            this._setDirty();
        }
    }

    /**
     * The asset id of the component in cloud storage
     *
     * @memberof AdobeDCXComponent#
     * @type {String}
     */
    public get assetId(): string | undefined {
        return this._assetId;
    }
    public set assetId(assetId: string | undefined) {
        if (assetId !== this._assetId) {
            if (this._assetId) {
                throw new DCXError(DCXError.INVALID_STATE, 'assetId property cannot be changed.');
            }
            this._assetId = assetId;
        }
    }

    /**
     * The repo id of the component in cloud storage
     *
     * @memberof AdobeDCXComponent#
     * @type {String}
     */
    public get repositoryId(): string | undefined {
        return this._repoId;
    }
    public set repositoryId(repoId: string | undefined) {
        if (repoId !== this._repoId) {
            if (this._repoId) {
                throw new DCXError(DCXError.INVALID_STATE, 'repoId property cannot be changed.');
            }
            this._repoId = repoId;
        }
    }

    /**
     * Get a serialized component descriptor.
     *
     * @returns {string}
     *
     * ```ts
     * const componentDescriptor = component.getComponentDescriptor();
     * ```
     */
    public getComponentDescriptor(): string {
        if (!isAdobeDCXBranchLike(this._owner)) {
            throw new DCXError(DCXError.INVALID_STATE, 'Component must be part of a composite to get a descriptor.');
        }
        return JSON.stringify({
            versionId: CURRENT_COMPONENT_DESCRIPTOR_VERSION, // when format changes, this must be incremented
            componentId: this.id,
            compositeId: this._owner.compositeId,
            cloudAssetId: this._owner.compositeAssetId,
            repositoryId: this.repositoryId || this._owner.compositeRepositoryId,
            componentRevisionId: this.version,
            type: this.type,
            size: this.length,
            etag: this.etag,
            hashType: CURRENT_COMPONENT_DESCRIPTOR_HASH_TYPE, // may change in later versions
            hashValue: this.md5, // but for now we lock to the md5 property
        });
    }

    /**
     * Returns an array of custom keys that are present at this component. Custom keys are manifest
     * properties that are not considered standard DCX properties.
     * @returns {Array}
     */
    public getCustomKeys(): string[] {
        const customKeys: string[] = [];
        const keys = Object.keys(this._data);
        const count = keys.length;
        for (let i = 0; i < count; i++) {
            const key = keys[i];
            if (!reservedKeys[key]) {
                customKeys.push(key);
            }
        }
        return customKeys;
    }

    /**
     * Returns the object or value for the given custom key.
     * @param   {String} key The custom key to look up.
     * @returns {*}
     */
    public getValue<T = any>(key: string): T {
        if (typeof key !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "key" must be a string');
        }

        return this._data[key] as T;
    }

    /**
     * Sets the object or value for the given custom key.
     * @param {String} key   The custom key to set the value for.
     * @param {*}      value The value to set.
     */
    public setValue<T = any>(key: string, value: T): void {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        if (typeof key !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "key" must be a string');
        }

        this._data[key] = value;
        this._setDirty();
    }

    /**
     * Removes the object or value for the given custom key.
     * @param {String} key The custom key to remove.
     */
    public removeValue(key: string): void {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        if (typeof key !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "key" must be a string');
        }

        delete this._data[key];
        this._setDirty();
    }

    //******************************************************************************
    // Links
    //******************************************************************************

    /**
     * Returns the link with the given relationship as a JS object or undefined if the component
     * doesn't have such a link.
     * @param   {String} relationship The relationship of the link to the component.
     * @returns {Object} The link object with the given relationship as a JS object or undefined if
     *                   the component doesn't have such a link.
     */
    public getLink(relationship: string): HalLink | undefined {
        if (typeof relationship !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "relationship" must be a string');
        }
        return this._data._links ? this._data._links[relationship] : undefined;
    }

    /**
     * Sets the link with the given relationship to the given object.
     * @param {Object} link         A JS object representing the link.
     * @param {String} relationship The relationship of the link to the component.
     */
    public setLink(link: HalLink, relationship: string): void {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        if (typeof link !== 'object') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "link" must be an object.');
        }
        if (typeof relationship !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "relationship" must be a string');
        }

        let links = this._data._links;
        if (!links) {
            links = this._data._links = {};
        }

        links[relationship] = link;
        this._setDirty();
    }

    /**
     * Removes the link with the given relationship.
     * @param {String} relationship The relationship of the link to the component.
     *
     */
    public removeLink(relationship: string): void {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        if (typeof relationship !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "relationship" must be a string');
        }

        const links = this._data._links;
        if (links) {
            delete links[relationship];
            if (Object.keys(links).length < 1) {
                delete this._data._links;
            }
            this._setDirty();
        }
    }

    //******************************************************************************
    // Miscellaneous
    //******************************************************************************

    /**
     * Returns false if any of the properties of the given component is different from the properties
     * of this component. Recurses both data structures.
     *
     * @example
     * if (!component.isEqualTo(otherComponent, { unimportantProperty: true })) {
     *      // The components have different properties
     * }
     * @param   {AdobeDCXComponent} component            The component to compare with.
     * @param   {Object}            [propertiesToIgnore] An object having the properties that should
     *                                                 not be compared.
     * @returns {Boolean}
     */
    public isEqualTo(component: AdobeDCXComponent, propertiesToIgnore?: Record<string, boolean>): boolean {
        return objectsEqual(this._data, component._data, propertiesToIgnore);
    }

    //******************************************************************************
    // Private
    //******************************************************************************

    /**
     * Verifies and sets the new data. Throws if invalid data.
     * @private
     * @param   {Object}   data
     * @returns {AdobeDCXComponent}
     */
    private _setData(data: AdobeDCXComponentData): AdobeDCXComponent {
        const err = this._verify(data);
        if (err === null) {
            this._data = data;
            return this;
        }
        throw err;
    }

    /**
     * Sets the dirty flag on the node's branch (if set)
     * @internal
     */
    _setDirty() {
        if (this._owner) {
            this._owner._setDirty();
        }
    }

    /**
     * Returns an error if the passed in data is not a valid component. Returns null if everything is OK.
     * @private
     * @param   {Object} data
     * @returns {Error}
     */
    private _verify(data: AdobeDCXComponentData): AdobeDCXError | null {
        if (typeof data.id !== 'string') {
            return new DCXError(DCXError.INVALID_DATA, 'Component is missing an id of type string');
        }

        return null;
    }
}

export default AdobeDCXComponent;
