/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { AdobeAsset, Composite, isMinimalAdobeAsset, parseLinksFromResponseHeader } from '@dcx/assets';
import {
    AdobeDCXBranch,
    AdobeDCXElement,
    AdobeDCXError,
    AdobeDCXRootNodeData,
    AdobeResponse,
    BranchCallback,
    GetSliceCallback,
    AdobeDCXComposite as IAdobeDCXComposite,
    Link,
    LinkSet,
    ResourceDesignator,
    SliceableData,
} from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import { AdobeRepoAPISession } from '@dcx/repo-api-session';
import { generateUuid, isObject, validateParams } from '@dcx/util';
import DCXBranch from './AdobeDCXBranch';
import AdobeDCXPushJournal from './AdobeDCXPushJournal';
import { COMPOSITE_STATES } from './enum';
import { DerivationType } from './util/xmp';

const dbg = newDebug('dcx:dcxjs:dcxcomposite');

const MAX_CLIENT_DATA_LENGTH = 1024; // Maximum length of clientDataString is 1KB by default.

export const COLLABORATION: Record<'PRIVATE' | 'SHARED_WITH_USER' | 'SHARED_BY_USER', string> = {
    PRIVATE: undefined as unknown as string,
    SHARED_BY_USER: 'sharedByUser',
    SHARED_WITH_USER: 'sharedWithUser',
};

export interface AdobeDCXCompositeOptions {
    xhrBaseBranchSupport?: boolean;
    maxClientDataLength?: number;
}

/**
 * @class
 * @classdesc AdobeDCXComposite represents a DCX composite.
 * <p>The constructor for AdobeDCXComposite is private. Refer to {@link AdobeDCX} to learn how
 * to create instances of AdobeDCXComposite.
 * @hideconstructor
 * @param {String} name
 * @param {String} type
 * @param {String} id
 * @param {String} assetId
 * @param {Object} options
 */
export class AdobeDCXComposite implements IAdobeDCXComposite {
    /**
     * Constants for the collaborationType property.
     * @private
     */
    static readonly COLLABORATION = COLLABORATION;

    /** @internal */
    private _clientDataString?: string;

    /** @internal */
    private _baseBranchData?: string;

    /** @internal */
    private _name!: string;

    /** @internal */
    private _href?: string;

    /** @internal */
    private _assetId?: string;

    /** @internal */
    private _type!: string;

    /** @internal */
    private _id!: string;

    /** @internal */
    _current!: DCXBranch;

    /** @internal */
    private _path!: string;

    /** @internal */
    private _repoId!: string;

    /** @internal */
    private _collaborationType: string = COLLABORATION.PRIVATE;

    /** @internal */
    _options: AdobeDCXCompositeOptions;

    /** @internal */
    _pushJournal?: AdobeDCXPushJournal;

    /** @internal */
    _links?: LinkSet;

    /** @internal */
    _pushedBranchData?: string;

    /** @internal */
    _pulledBranchData?: string;

    constructor(
        name?: string,
        type?: string,
        id?: string,
        assetId?: string,
        repoId?: string | AdobeDCXCompositeOptions,
        options?: AdobeDCXCompositeOptions,
    );
    constructor(
        name?: string,
        type?: string,
        id?: string,
        assetId?: string,
        repoId?: string,
        options: AdobeDCXCompositeOptions = {},
    ) {
        if (repoId != null && typeof repoId === 'object') {
            options = repoId;
            repoId = undefined;
        }

        this._repoId = repoId as string;

        this._current = new DCXBranch();
        if (repoId) {
            this._current.compositeRepositoryId = repoId as string;
        }
        if (name || name === '') {
            this._current.data.name = name;
        }
        if (type) {
            this._current.data.type = type;
        }
        if (id) {
            this._current.compositeId = id;
        }

        this._current._collaborationType = this._collaborationType;
        this._current.data.state = COMPOSITE_STATES.modified;
        this._current._setDirty(true);
        this._options = options;
        if (assetId) {
            // exists on server but not yet locally
            this._assetId = assetId;
            this._current.compositeAssetId = assetId;
            if (name || name === '') {
                this._name = name;
            }
            if (type) {
                this._type = type;
            }
            if (id) {
                this._id = id;
            }
        }

        if (this._options.xhrBaseBranchSupport) {
            this._baseBranchData = undefined;
            this._pushedBranchData = undefined;
            this._pulledBranchData = undefined;
        }

        // wrap callbacks for public APIs if initialized with a promise library
        // Promisify.wrapAll<typeof AdobeDCXComposite>(['loadBaseBranch'], this, AdobeDCXComposite);
    }

    //******************************************************************************
    // Getters/setters for properties
    //******************************************************************************

    public get href() {
        return this._href as string;
    }
    public set href(href: string) {
        this._href = href;
    }

    public get links(): LinkSet | undefined {
        return this._links;
    }
    public set links(links: LinkSet | undefined) {
        validateParams(['links', links, 'object']);

        if (this._current) {
            this._current.compositeLinks = links;
        }
        this._links = links;
    }

    public get repositoryId(): string | undefined {
        return this._current ? this._current.compositeRepositoryId : this._repoId;
    }
    public set repositoryId(repoId: string | undefined) {
        validateParams(['repoId', repoId, 'string']);

        if (this._current) {
            this._current.compositeRepositoryId = repoId;
        }
        this._repoId = repoId as string;
    }

    /**
     * The id of the composite.
     *
     * <p>While not strictly read-only most clients do not ever have to modify this property.</p>
     * @memberof AdobeDCXComposite#
     * @type {String}
     */
    public get id(): string {
        return this._current ? this._current.compositeId : (this._id as string);
    }
    public set id(id: string) {
        if (typeof id !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string.');
        } else {
            if (this._current) {
                this._current.compositeId = id;
            }
            this._id = id;
        }
    }

    /**
     * The name of the composite.
     *
     * @memberof AdobeDCXComposite#
     * @type {String}
     */
    public get name(): string {
        return this._current ? (this._current.name as string) : this._name;
    }
    public set name(name: string) {
        if (typeof name !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string.');
        } else {
            if (this._current) {
                this._current.name = name;
            }
            this._name = name;
        }
    }

    /**
     * The type of the composite.
     *
     * @memberof AdobeDCXComposite#
     * @type {String}
     */
    public get type(): string {
        return this._current ? this._current.type : this._type;
    }
    public set type(type: string) {
        if (typeof type !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string.');
        } else {
            if (this._current) {
                this._current.type = type;
            }
            this._type = type;
        }
    }

    /**
     * The asset id of the composite that can be used to pull and push the composite.
     * <strong>Do not modify this for a bound composite.</strong>
     * @memberof AdobeDCXComposite#
     * @type {String}
     */
    public get assetId(): string | undefined {
        return this._current ? this._current.compositeAssetId : this._assetId;
    }
    public set assetId(assetId: string | undefined) {
        if (typeof assetId !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string.');
        } else {
            if (this._current) {
                this._current.compositeAssetId = assetId;
            }
            this._assetId = assetId;
        }
    }

    /**
     * Whether (and how) the composite is shared. See
     * {@link AdobeDCX.COLLABORATION}.
     * It is the client's responsibility to correctly set this property so that DCX can push
     * the composite correctly.
     * <p><strong>Composites with a collaboration type of AdobeDCX.COLLABORATION.SHARED_WITH_USER
     * must not be deleted. Use AdobeStorageSession.leaveSharedComposite() to remove a shared
     * composite from the user's sync group.</p>
     * // @TODO: Determine if this is still used/necessary in RAPI.
     * @memberof AdobeDCXComposite#
     * @type {String}
     * @default [COLLABORATION.PRIVATE]
     */
    public get collaborationType(): string | undefined {
        return this._current ? this._current._collaborationType : this._collaborationType;
    }
    public set collaborationType(value: string | undefined) {
        if (
            value !== COLLABORATION.PRIVATE &&
            value !== COLLABORATION.SHARED_BY_USER &&
            value !== COLLABORATION.SHARED_WITH_USER
        ) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Invalid value: .' + value);
        } else {
            if (this._current) {
                this._current._collaborationType = value;
            }
            this._collaborationType = value;
        }
    }

    /**
     * User-defined data to associate with the composite - this can be an arbitrary string, up to a limit of 1KB.
     * This data is not synchronized to the server, but it is persisted locally to disk.
     *
     * @memberof AdobeDCXComposite#
     * @type {String}
     * @default [undefined]
     */
    public get clientDataString(): string | undefined {
        return this._current ? this._current._clientDataString : this._clientDataString;
    }
    public set clientDataString(value: string | undefined) {
        // Value must be a string, or undefined.
        if (typeof value !== 'string' && value !== undefined) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Invalid clientDataString: must be a string or undefined.');
        }
        // Check that the string length is below the limit (1KB by default).
        const maxLength = this._options.maxClientDataLength || MAX_CLIENT_DATA_LENGTH;
        if (value && value.length > maxLength) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Invalid clientDataString: cannot be greater than 1KB');
        }

        if (this._current) {
            this._current._clientDataString = value;
        }
        this._clientDataString = value;
    }

    /**
     * Whether the composite is bound to a composite on a server. If <em>false</em> for newly
     * created empty composites that have never been pulled from or pushed to the server.
     * @readonly
     * @memberof AdobeDCXComposite#
     * @type {Boolean}
     */
    public get isBound(): boolean {
        return !!((this._current && this._current.isBound) || (!this._current && this._assetId));
    }
    public set isBound(_: boolean) {
        throw new DCXError(
            DCXError.READ_ONLY,
            'Cannot set the property isBound. Use resetIdentity or resetBinding instead.',
        );
    }

    /**
     * The editable current branch the composite. Is undefined in case of a composite that hasn't
     * been pulled from the server yet.
     * @readonly
     * @memberof AdobeDCXComposite#
     * @type {AdobeDCXBranch}
     */
    public get current(): AdobeDCXBranch {
        return this._current;
    }
    /**
     * @private
     */
    public set current(_: AdobeDCXBranch) {
        throw new DCXError(DCXError.READ_ONLY, 'Property "current" is read-only.');
    }

    /**
     * <p>Creates and returns a new composite object that is a copy of compositeBranchOrElement.</p>
     *
     * <p> The resulting composite will be in memory only and you can push it to the server, which
     * will result in the component files being copied on the server. </p>
     *
     * @param   {Object}            compositeOrBranchOrElement As the name implies this param is expected to be
     *                                                         either a composite (with a valid current branch),
     *                                                         a branch or an element. Furthermore it may (at
     *                                                         this point in time) not contain an new or modified
     *                                                         components.
     * @param   {String}            [name]                     An optional new name for the new composite. If undefined
     *                                                         the composite will have the same name as the original.
     * @param   {String}            [type]                     An optional new type for the new composite. If undefined
     *                                                         the composite will have the same type as the original.
     * @param   {String}            [id]                       An optional new id for the new composite. If undefined
     *                                                         the composite will get a random new id.
     * @param   {Object}            [options]                  See constructor.
     * @returns {AdobeDCXComposite} The new composite
     */
    static newCompositeAsCopyOf(
        compositeOrBranchOrElement: AdobeDCXBranch | IAdobeDCXComposite | AdobeDCXElement,
        name?: string,
        type?: string,
        id?: string,
        options?: AdobeDCXCompositeOptions,
    ): AdobeDCXComposite {
        let core = (compositeOrBranchOrElement as DCXBranch)._core;

        if (!core) {
            // TODO: make duck-typing function for DCX DOM classes
            if ((compositeOrBranchOrElement as AdobeDCXComposite).current) {
                core = ((compositeOrBranchOrElement as AdobeDCXComposite).current as DCXBranch)._core;
            }
        }
        if (!core) {
            throw new DCXError(
                DCXError.INVALID_PARAMS,
                'compositeBranchOrElement must be a branch, an element or a composite with a current branch.',
            );
        }

        const composite = new AdobeDCXComposite(undefined, undefined, undefined, undefined, undefined, options);

        composite._current = DCXBranch._newBranchAsCopyOfCore(core);
        composite._current._derivationType = DerivationType.COPY;
        composite._current._derivationDatetime = new Date().toISOString();

        composite.id = id || generateUuid();
        if (name) {
            composite.name = name;
        }
        if (type) {
            composite.type = type;
        }

        return composite;
    }

    //******************************************************************************
    // Branches
    //******************************************************************************

    /**
     * Loads the base manifest and instantiates a read-only branch if the manifest exists.
     *
     * <p>This method can only be called if the AdobeDCXComposite instance has been initialized with the
     * <em>xhrBaseBranchSupport</em> option.</p>
     *
     * @param {BranchCallback} callback Gets called on success or failure.
     */
    public loadBaseBranch(): AdobePromise<AdobeDCXBranch, AdobeDCXError>;
    public loadBaseBranch(callback: BranchCallback): void;
    public loadBaseBranch(callback?: BranchCallback): AdobePromise<AdobeDCXBranch, AdobeDCXError> | void {
        if (!this._baseBranchData) {
            throw new DCXError(DCXError.NO_BASE_BRANCH_DATA, 'No base branch data.');
        }

        let error: AdobeDCXError | undefined = undefined;
        let branch: AdobeDCXBranch | undefined = undefined;
        try {
            branch = new DCXBranch(undefined, true, this.assetId).parse(this._baseBranchData);
            if (!callback) {
                return AdobePromise.resolve(branch);
            }
        } catch (x) {
            if (!callback) {
                return AdobePromise.reject(x);
            }
            error = x as AdobeDCXError;
        }

        callback(error, branch);
    }

    /**
     * Callback function used for operations that return a branch.
     * @callback BranchCallback
     *    @param {Error}           error
     *    @param {AdobeDCXBranch}  branch
     */

    /**
     * Callback function used for operations that don't return anything other than a potential error.
     * @callback ErrorCallback
     *    @param {Error}           error
     */

    /**
     * Callback function used for operations that don't return anything other than an array if error
     * objects.
     * @callback ErrorsCallback
     *    @param {Array}           Array of error objects or undefined.
     */

    /**
     * Accepts the given merged branch as the new current branch, discards any temporary pull data
     * and also updates the base manifest. Is a no-op if no branch is provided and if there
     * is no pulled manifest in local storage (as may be the case if a pull didn't find any
     * modification on the server).
     *
     * @example
     * // NJS
     * pullComposite(composite, session, function (error, pulledBranch) {
     *      if (!error) {
     *          // The pull has succeeded. Now you need to incorporate the changes from the server
     *          // into your current branch.
     *          // Start the merge by making a mutable copy of pulledBranch
     *          var mergedBranch = pulledBranch.copy();
     *          // Now merge any changes you have made to current since the last successful push or
     *          // pull. You might want to also load the base branch to help you figure out what has
     *          // changed in either branch.
     *          ...
     *          // When done merging you need to make this call:
     *          composite.resolvePullWithBranch(mergedBranch, function (error, newCurrent) {
     *              if (!error) {
     *                  // Now you are done!
     *              }
     *          });
     *      }
     * });
     *
     * @example
     * // XHR
     * pullCompositeManifestOnly(composite, session, function (error, pulledBranch) {
     *      if (!error) {
     *          // The pull has succeeded. Now you need to incorporate the changes from the server
     *          // into your current branch. If you follow the recommendations for working in an
     *          // online environment you will not have modified current so you can just call:
     *          composite.resolvePullWithBranch(pulledBranch);
     *      }
     * });
     *
     *
     * <p><strong>The branch passed in must be derived from the pulled branch.</strong></p>
     *
     * @param {AdobeDCXBranch} branch     A branch which should become the new current branch. It
     *                                    should be the result of merging the previously pulled
     *                                    branch with any changes in current.
     * @param {BranchCallback} [callback] Optional if running in an <strong>XHR</strong>
     *                                    environment. Gets called on completion.
     * @return {AdobeDCXBranch}           Only returns the new current branch if called without
     *                                     a callback.
     */
    public resolvePullWithBranch(branch: AdobeDCXBranch): void;
    public resolvePullWithBranch(branch: AdobeDCXBranch, callback: BranchCallback): void;
    public resolvePullWithBranch(pBranch: AdobeDCXBranch, callback?: BranchCallback): AdobeDCXBranch | void {
        const branch = pBranch as DCXBranch;
        if (!branch) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Need a branch.');
        }
        if (branch === this._current) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Cannot resolve current.');
        }

        const swapCurrent = () => {
            // Swaps out the current branch with the one we're accepting, and makes sure that any local data is copied over.
            // This prevents a race-condition if the local data was changed on the composite between the pulled branch being
            // loaded, and it being resolved.
            branch.readOnly = false;
            branch._collaborationType = this.collaborationType;
            branch._clientDataString = this.clientDataString;

            // Finally, set the new branch (has to be after the above steps)
            this._current = branch as DCXBranch;
        };

        if (this._options.xhrBaseBranchSupport) {
            // We need to preserve the pulled branch as base branch.
            // The pull logic in the the xfer class is responsible for setting this._pulledBranchData.
            this._baseBranchData = this._pulledBranchData;
            this._pulledBranchData = undefined;
        }
        swapCurrent();
        if (callback) {
            callback(undefined, this._current);
        } else {
            return this._current;
        }
    }

    /**
     * Updates the current branch (both in memory and -- if applicable -- in local storage) with the
     * results of the most recent push, discards any temporary push data and updates base. Is a
     * no-op if there are no push results (because e.g. there were no changes to push).
     *
     * @example
     * // NJS
     * pushComposite(composite, false, session, function (error) {
     *      if (!error) {
     *          // The push has succeeded. You now need to accept it to update your current branch
     *          composite.acceptPush(function (error) {
     *              if (!error) {
     *                  // Now you are done!
     *              }
     *          });
     *      }
     * });
     *
     * @example
     * // XHR
     * pushComposite(composite, false, session, function (error) {
     *      if (!error) {
     *          // The push has succeeded. If you want to continue to work with the composite you
     *          // must accept the push -- otherwise you are done now
     *          composite.acceptPush();
     *      }
     * });
     *
     * @param   {ErrorCallback} [callback]  Optional when calling in an <strong>XHR</strong>
     *                                      environment. Gets called on completion.
     * @returns {AdobeDCXBranch}            Only returns the new current branch if called without
     *                                      a callback.
     */
    public acceptPush(callback?: (err?: Error) => unknown): void | AdobeDCXBranch {
        const pushJournal = this._pushJournal;
        let error;

        // If pushJournal is undefined this means that it doesn't exist, which is OK
        if (pushJournal) {
            try {
                pushJournal.applyToBranch(this._current, /*preserveDirtyState*/ true);
                if (this._options.xhrBaseBranchSupport) {
                    // We need to preserve the pulled branch as base branch.
                    // The push logic in the the xfer class is responsible for setting self._pushedBranchData.
                    this._baseBranchData = this._pushedBranchData;
                    this._pushedBranchData = undefined;
                }
                this._pushJournal = undefined;
                if (callback) {
                    callback();
                }
            } catch (x) {
                error = x;
            }
        } else if (callback) {
            callback();
        }

        if (error) {
            if (callback) {
                callback(error);
            } else {
                throw error;
            }
        }

        if (!callback) {
            return this._current;
        }
    }

    //******************************************************************************
    // Miscellaneous
    //******************************************************************************

    /**
     * Removes all service-related data from the current banch and resets all other branches so that
     * the composite can be pushed again to the same or a different service.
     * @param {ErrorCallback} [callback] Optional if called from an <strong>XHR</strong>
     *                                   environment. Gets called on success or failure.
     */
    public resetBinding(callback?: (err?: Error) => unknown) {
        delete this._assetId;
        if (this._current) {
            this._current._resetBinding();
        }

        this._baseBranchData = undefined;
        this._pushedBranchData = undefined;
        this._pulledBranchData = undefined;

        if (callback) {
            callback();
        }
    }

    /**
     * Assigns a new id to the current branch and resets all other branches. Also removes all
     * service-related data from the current branch so that it can be pushed again to the same or
     * a different service.
     * @param {ErrorCallback} [callback] Optional if called from an <strong>XHR</strong>
     *                                   environment. Gets called on success or failure.
     */
    public resetIdentity(callback?: (err?: Error) => unknown) {
        delete this._assetId;
        if (this._current) {
            this._current._resetIdentity();
            this._id = this._current.compositeId;
        } else {
            this._id = generateUuid();
        }

        this._baseBranchData = undefined;
        this._pushedBranchData = undefined;
        this._pulledBranchData = undefined;

        if (callback) {
            callback();
        }
    }

    //******************************************************************************
    // Private
    //******************************************************************************

    /** Sets the path property and creates or updates local storage. */
    /**
     * @private
     * @param {String} path
     */
    private _setPath(path: string) {
        this._path = path;
    }

    /**
     * @internal
     */
    get _isRepoComposite(): boolean {
        return isObject(this._current) && typeof this._current._isRepoComposite === 'boolean'
            ? this._current._isRepoComposite
            : !!this.repositoryId;
    }
}

export default AdobeDCXComposite;

/**
 * Convenience APIs
 */

/**
 * Factory method for creating a R-API based DCXComposite.
 * To create a bound composite, provide `repositoryId` and `assetId` and/or `links`.
 * To create an unbound composite, provide neither `repositoryId` or `assetId`. Must provide `name` and `type`.
 *
 * @experimental
 *
 * @param {string}                  [assetId]           - Asset ID of the composite if it already exists in the cloud.
 * @param {string}                  [repositoryId]      - Repository ID of the composite if it already exists in the cloud.
 * @param {string}                  [name]              - Composite name.
 * @param {string}                  [id]                - Composite ID.
 * @param {string}                  [type]              - Content Type of the composite, once created this cannot be changed.
 * @param {Record<string, Link>}    [links]             - Links associated with the composite.
 * @param {AdobeDCXCompositeOptions}              [options]           - Additional configuration options.
 */
export function newDCXComposite(
    assetId?: string,
    repositoryId?: string,
    name?: string,
    id?: string,
    type?: string,
    links?: Record<string, Link>,
    options: AdobeDCXCompositeOptions = {},
): AdobeDCXComposite {
    if (assetId == null && links == null) {
        if (typeof type !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Unbound DCXComposite must have a type.');
        }
        if (typeof name !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Unbound DCXComposite must have a name.');
        }
    }
    const composite = new AdobeDCXComposite(name, type, id, assetId, repositoryId, options);
    if (links) {
        composite.links = links;
    }

    return composite;
}

/**
 * Create a new composite cloud asset at the specified path.
 *
 * <p> After creating a new, local-only composite the client must first bind it to a specific repository, path,
 * and asset id in the cloud by calling this method. If successful then the composite object's assetId property will
 * be assigned and the client is now free to upload or copy new components for the composite and call pushComposite.
 *
 * @param session               The repo session to use.
 * @param composite             New composite object whose assetId property is still undefined.
 * @param parentDir             Parent directory in which to create the composite.
 * @param relPath               Path at which new composite will be created in the cloud, relative to parentDir
 */
export function createComposite(
    session: AdobeRepoAPISession,
    composite: AdobeDCXComposite,
    parentDir: AdobeAsset,
    relPath: string,
): AdobePromise<AdobeDCXComposite, AdobeDCXError>;

/**
 * Create a new composite cloud asset at the specified path.
 *
 * <p> After creating a new, local-only composite the client must first bind it to a specific repository, path,
 * and asset id in the cloud by calling this method. If successful then the composite object's assetId property will
 * be assigned and the client is now free to upload or copy new components for the composite and call pushComposite.
 *
 * @param session               The repo session to use.
 * @param composite             New composite object whose assetId property is still undefined.
 * @param parentDir             Parent directory in which to create the composite.
 * @param relPath               Path at which new composite will be created in the cloud, relative to parentDir
 * @param respondWith           undefined
 * @param additionalHeaders     Additional headers to be applied to HTTP requests
 */
export function createComposite(
    session: AdobeRepoAPISession,
    composite: AdobeDCXComposite,
    parentDir: AdobeAsset,
    relPath: string,
    respondWith: undefined,
    additionalHeaders?: Record<string, string>,
    snapshot?: SliceableData | GetSliceCallback,
    snapshotSize?: number,
): AdobePromise<AdobeDCXComposite, AdobeDCXError>;

/**
 * Create a new composite cloud asset at the specified path and respond with the requested resource if possible.
 * <p> After creating a new, local-only composite the client must first bind it to a specific repository, path,
 * and asset id in the cloud by calling this method. If successful then the composite object's assetId property will
 * be assigned and the client is now free to upload or copy new components for the composite and call pushComposite.
 * The resolved value of the promise when a resource designator is provided is the requested resource itself.
 *
 * @see {@link https://git.corp.adobe.com/pages/caf/api-spec/chapters/basics/create.html#alternate-form-create-with-response}
 *
 * @param session               The repo session to use.
 * @param composite             New composite object whose assetId property is still undefined.
 * @param parentDir             Parent directory in which to create the composite.
 * @param relPath               Path at which new composite will be created in the cloud, relative to parentDir
 * @param respondWith           A resource to respond with. This is useful for resolving with the repository metadata
 * @param additionalHeaders     Additional headers to be applied to HTTP requests
 *
 * @returns {AdobePromise<AdobeAsset, AdobeDCXError>}
 */
export function createComposite(
    session: AdobeRepoAPISession,
    composite: AdobeDCXComposite,
    parentDir: AdobeAsset,
    relPath: string,
    respondWith: ResourceDesignator,
    additionalHeaders?: Record<string, string>,
    snapshot?: SliceableData | GetSliceCallback,
    snapshotSize?: number,
): AdobePromise<AdobeAsset, AdobeDCXError>;

/**
 * Create a new composite cloud asset at the specified path and respond with the requested resource if possible.
 * <p> After creating a new, local-only composite the client must first bind it to a specific repository, path,
 * and asset id in the cloud by calling this method. If successful then the composite object's assetId property will
 * be assigned and the client is now free to upload or copy new components for the composite and call pushComposite.
 * The resolved value of the promise when a resource designator is provided is the requested resource itself.
 *
 * @see {@link https://git.corp.adobe.com/pages/caf/api-spec/chapters/basics/create.html#alternate-form-create-with-response}
 *
 * @param session               The repo session to use.
 * @param composite             New composite object whose assetId property is still undefined.
 * @param parentDir             Parent directory in which to create the composite.
 * @param relPath               Path at which new composite will be created in the cloud, relative to parentDir
 * @param respondWith           A resource to respond with. This is useful for resolving with the repository metadata
 * @param additionalHeaders     Additional headers to be applied to HTTP requests
 *
 * @returns {AdobePromise<AdobeAsset, AdobeDCXError>}
 */
export function createComposite(
    session: AdobeRepoAPISession,
    composite: AdobeDCXComposite,
    parentDir: AdobeAsset,
    relPath: string,
    respondWith: ResourceDesignator,
    additionalHeaders?: Record<string, string>,
    snapshot?: SliceableData | GetSliceCallback,
    snapshotSize?: number,
): AdobePromise<AdobeAsset, AdobeDCXError>;

export function createComposite(
    session: AdobeRepoAPISession,
    composite: AdobeDCXComposite,
    parentDir: AdobeAsset,
    relPath: string,
    respondWith?: ResourceDesignator,
    additionalHeaders?: Record<string, string>,
    snapshot?: SliceableData | GetSliceCallback,
    snapshotSize?: number,
): AdobePromise<AdobeDCXComposite | AdobeAsset, AdobeDCXError> {
    dbg('createComposite()');

    validateParams(
        ['session', session, 'object'],
        ['composite', composite, 'object'],
        ['parentDir', parentDir, 'object'],
        ['relPath', relPath, 'string'],
    );

    if (composite.assetId) {
        throw new DCXError(DCXError.INVALID_STATE, 'Composite must not already have an assigned asset id.');
    }

    if (typeof parentDir.repositoryId !== 'string' && typeof composite.repositoryId !== 'string') {
        throw new DCXError(DCXError.INVALID_PARAMS, 'Either composite or parentDir must contain valid repositoryId.');
    }

    if (
        typeof parentDir.repositoryId === 'string' &&
        typeof composite.repositoryId === 'string' &&
        parentDir.repositoryId !== composite.repositoryId
    ) {
        throw new DCXError(DCXError.INVALID_PARAMS, 'Composite contains a repositoryId that does not match parentDir.');
    }

    if (typeof composite.type !== 'string') {
        throw new DCXError(DCXError.INVALID_STATE, 'Composite must contain a valid type.');
    }

    // allow repoId to come from parentDir or DCXComposite
    const repositoryId = typeof composite.repositoryId === 'string' ? composite.repositoryId : parentDir.repositoryId;
    const parent = parentDir;
    Object.getOwnPropertyNames(parentDir).forEach((name) => {
        parent[name] = parentDir[name];
        parent['repositoryId'] = repositoryId;
    });

    if (!isMinimalAdobeAsset(parent)) {
        throw new DCXError(DCXError.INVALID_PARAMS, 'parentDir must contain links or repositoryId & assetId or path');
    }
    // respondWith is optional, but overloads that utilize it determine which type the promise is resolved to
    // because of this, typecsript appears to prefer that the calls be broken out individually,
    // one with the respondWith parameter and one without, in order to properly parse the types without error.
    if (respondWith) {
        return _createCompositeAtPath(
            session,
            composite,
            parent,
            relPath,
            composite.type,
            respondWith,
            additionalHeaders,
            snapshot,
            snapshotSize,
        );
    }
    return _createCompositeAtPath(
        session,
        composite,
        parent,
        relPath,
        composite.type,
        undefined,
        additionalHeaders,
        snapshot,
        snapshotSize,
    );
}

/**
 * Convert an HLA asset class instance, or AdobeAsset compatible data object to an AdobeDCXComposite.
 *
 * @note Does not validate that incoming asset has any specific properties.
 *
 * @param {AdobeAsset}                  asset           - Asset to convert.
 * @param {string}                      [id]            - Composite ID.
 * @param {AdobeDCXCompositeOptions}    [opts={}]       - Additional configuration options
 */
export function convertToDCXComposite(
    asset: AdobeAsset | Composite,
    id?: string,
    opts?: AdobeDCXCompositeOptions,
): AdobeDCXComposite {
    dbg('convertToDCXComposite()');

    if (asset.format && !asset.format.endsWith('+dcx')) {
        throw new DCXError(DCXError.INVALID_PARAMS, 'Format must end in "+dcx"');
    }

    return new AdobeDCXComposite(asset.name, asset.format, id, asset.assetId, asset.repositoryId, opts);
}

/**
 * Internal methods
 */

/**
 * Create a DCXComposite at a path relative to a parent directory.
 *
 * @param {AdobeRepoAPISession}    session      - Session to use.
 * @param {AdobeDCXComposite}      composite    - DCXComposite to create, must not contain assetId.
 * @param {AdobeAsset}             parentDir    - Parent directory, must contain assetId and repositoryId or links.
 * @param {string}                 relPath      - Path to composite, relative to parent directory.
 * @param {string}                 type         - Content type of composite
 */
function _createCompositeAtPath(
    session: AdobeRepoAPISession,
    composite: AdobeDCXComposite,
    parentDir: AdobeAsset,
    relPath: string,
    type: string,
    respondWith: undefined, // promise is resolved to AdobeDCXComposite when undefined
    additionalHeaders?: Record<string, string>,
    snapshot?: SliceableData | GetSliceCallback,
    snapshotSize?: number,
): AdobePromise<AdobeDCXComposite, AdobeDCXError>;
function _createCompositeAtPath(
    session: AdobeRepoAPISession,
    composite: AdobeDCXComposite,
    parentDir: AdobeAsset,
    relPath: string,
    type: string,
    respondWith: ResourceDesignator, // promise is resolved to AdobeAsset when provided
    additionalHeaders?: Record<string, string>,
    snapshot?: SliceableData | GetSliceCallback,
    snapshotSize?: number,
): AdobePromise<AdobeAsset, AdobeDCXError>;
function _createCompositeAtPath(
    session: AdobeRepoAPISession,
    composite: AdobeDCXComposite,
    parentDir: AdobeAsset,
    relPath: string,
    type: string,
    respondWith?: ResourceDesignator,
    additionalHeaders?: Record<string, string>,
    snapshot?: SliceableData | GetSliceCallback,
    snapshotSize?: number,
): AdobePromise<AdobeDCXComposite | AdobeAsset, AdobeDCXError> {
    const { repositoryId } = parentDir;

    return session
        .createComposite(parentDir, relPath, type, respondWith, additionalHeaders, snapshot, snapshotSize)
        .then(async (res) => {
            const { assetId, links } = res.result;
            composite.assetId = assetId;
            composite.repositoryId = repositoryId;
            composite.links = links;
            if (snapshot) {
                const { manifestData } = await session.getCompositeManifest(composite);
                composite.resolvePullWithBranch(new DCXBranch(manifestData as unknown as AdobeDCXRootNodeData));
            }
            if (respondWith) {
                return res.result;
            }
            return composite;
        })
        .catch((e) => {
            // on error, still cache links if possible then rethrow error
            try {
                if (e.response && isObject(e.response.headers)) {
                    const assetId = e.response.headers['asset-id'] || e.response.headers['x-resource-id'];
                    const links = parseLinksFromResponseHeader(e.response as AdobeResponse);
                    composite.links = links;

                    if (typeof assetId === 'string' && isObject(links)) {
                        session.updateCachedAssetLinks({ assetId, repositoryId, links });
                    }
                }
            } catch (_) {
                // no op
            }
            throw e;
        });
}
