/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    AdobeDCXBranch,
    AdobeDCXElement,
    AdobeDCXError,
    AdobeDCXNodeData,
    AdobeDCXRootNodeData,
    HalLink,
    AdobeDCXNode as IAdobeDCXNode,
} from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { appendPathElements, isValidPath, merge, objectsEqual } from '@dcx/util';
import DCXBranch from './AdobeDCXBranch';
import DCXElement from './AdobeDCXElement';

export interface AdobeDCXLocalNodeData {
    local?: {
        version?: number;
        change?: number;
        collaborationType?: string;
        clientDataString?: string;
        archivalState?: 'pending' | 'archived';
        manifestEtag?: string;
    };
}

/**
 * @class
 * @classdesc AdobeDCXNode represents a node of a DCX manifest.
 * <p>The constructor for AdobeDCXNode is private. Refer to {@link AdobeDCXBranch} to
 * learn how to access existing nodes or create new ones.
 * @hideconstructor
 * @param {Object}  data
 * @param {Boolean} readOnly
 * @param {Boolean} isRoot
 */
export class AdobeDCXNode implements IAdobeDCXNode {
    public static ROOT_PATH = '/';

    /** @internal  */
    _parentPath?: string;

    /** @internal  */
    _owner?: DCXBranch | DCXElement;

    /** @internal  */
    _data: (AdobeDCXNodeData | AdobeDCXRootNodeData) & AdobeDCXLocalNodeData = {} as AdobeDCXNodeData;

    /** @internal  */
    _readOnly: boolean;

    readonly _isRoot: boolean;

    constructor(data?: AdobeDCXNodeData | AdobeDCXRootNodeData, readOnly = false, isRoot = false) {
        if (data) {
            this._setData(data);
        }
        this._readOnly = readOnly;
        this._isRoot = isRoot;
    }

    //******************************************************************************
    // Getters/setters for properties
    //******************************************************************************

    public get owner(): AdobeDCXElement | AdobeDCXBranch | undefined {
        return this._owner;
    }

    /**
     * The id of the node. Must be a unique among the nodes of the composite.
     * <p>Cannot be changed for a node that is part of a branch or element.</p>
     * @memberof AdobeDCXNode#
     * @type {String}
     */
    public get id(): string {
        return this._data.id;
    }
    public set id(id: string) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (this._owner) {
            throw new DCXError(
                DCXError.READ_ONLY,
                'Cannot change the id of a node that is part of a branch or element.',
            );
        } else if (typeof id !== 'string' || id === '') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a non-empty string.');
        } else {
            this._data.id = id;
            this._setDirty();
        }
    }

    /**
     * The name of the node.
     *
     * @memberof AdobeDCXNode#
     * @type {String}
     */
    public get name(): string | undefined {
        return this._data.name;
    }
    public set name(name: string | undefined) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof name !== 'string' && typeof name !== 'undefined') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string or undefined.');
        } else {
            this._data.name = name;
            this._setDirty();
        }
    }

    /**
     * The type of the node.
     *
     * @memberof AdobeDCXNode#
     * @type {String}
     */
    public get type(): string {
        return this._data.type;
    }
    public set type(type: string) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof type !== 'string' && typeof type !== 'undefined') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string or undefined.');
        } else {
            this._data.type = type;
            this._setDirty();
        }
    }

    /**
     * The relationship of the node to its parent node.
     *
     * @memberof AdobeDCXNode#
     * @type {String}
     */
    public get relationship(): string | undefined {
        return this._data.rel;
    }
    public set relationship(relationship: string | undefined) {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof relationship !== 'string' && typeof relationship !== 'undefined') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string or undefined.');
        } else {
            this._data.rel = relationship;
            this._setDirty();
        }
    }

    /**
     * The path property of the node.
     *
     * @memberof AdobeDCXNode#
     * @type {String}
     */
    public get path(): string | undefined {
        return this._isRoot ? AdobeDCXNode.ROOT_PATH : this._data.path;
    }
    public set path(path: string | undefined) {
        if (path !== this.path) {
            if (this._readOnly) {
                throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
            } else if (this._isRoot) {
                throw new DCXError(DCXError.READ_ONLY, 'Cannot change the path of the root node.');
            } else if (path && !isValidPath(path)) {
                throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a valid path or undefined.');
            } else {
                if (this._owner) {
                    // We need to call the parent branch/element to update the path and handle
                    // all neccessary updates to absolute paths
                    this._owner._setPathOfNode(this, path || undefined);
                } else {
                    this._data.path = path || undefined;
                    this._parentPath = '';
                }
                this._setDirty();
            }
        }
    }

    /**
     * The absolute path of the parent of the node.
     *
     * @memberof AdobeDCXNode#
     * @readonly
     * @type {String}
     */
    public get parentPath(): string | undefined {
        return this._parentPath;
    }
    public set parentPath(_) {
        throw new DCXError(DCXError.READ_ONLY, 'parentPath is read-only.');
    }

    /**
     * The absolute path of the node.
     *
     * @memberof AdobeDCXNode#
     * @readonly
     * @type {String}
     */
    public get absolutePath(): string | undefined {
        const path = this.path;
        if (this._isRoot) {
            return path;
        }
        return path && this._owner ? appendPathElements(this._parentPath, path) : undefined;
    }
    public set absolutePath(_) {
        throw new DCXError(DCXError.READ_ONLY, 'absolutePath is read-only.');
    }

    /**
     * Whether this node is the root of its branch or element.
     *
     * @memberof AdobeDCXNode#
     * @readonly
     * @type {Boolean}
     */
    public get isRoot(): boolean {
        return this._isRoot;
    }
    public set isRoot(_) {
        throw new DCXError(DCXError.READ_ONLY, 'isRoot is read-only.');
    }

    //******************************************************************************
    // Links
    //******************************************************************************

    /**
     * Returns the link with the given relationship as a JS object or undefined if the node
     * doesn't have such a link.
     * @param   {String} relationship The relationship of the link to the node.
     * @returns {Object} The link with the given relationship as a JS object or undefined if
     *                   the node doesn't have such a link.
     */
    public getLink(relationship: string): HalLink | undefined {
        if (typeof relationship !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "relationship" must be a string');
        }
        return this._data._links ? this._data._links[relationship] : undefined;
    }

    /**
     * Sets the link with the given relationship to the given object.
     * @param {Object} link         A JS object representing the link.
     * @param {String} relationship The relationship of the link to the node.
     */
    public setLink(link: HalLink, relationship: string): void {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        if (typeof link !== 'object') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "link" must be an object.');
        }
        if (typeof relationship !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "relationship" must be a string');
        }

        let links = this._data._links;
        if (!links) {
            links = this._data._links = {};
        }

        links[relationship] = link;
        this._setDirty();
    }

    /**
     * Removes the link with the given relationship.
     * @param {String} relationship The relationship of the link to the node.
     *
     */
    public removeLink(relationship: string): void {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        if (typeof relationship !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "relationship" must be a string');
        }

        const links = this._data._links;
        if (links) {
            delete links[relationship];
            if (Object.keys(links).length < 1) {
                delete this._data._links;
            }
            this._setDirty();
        }
    }

    /**
     * Returns an array of non-standard keys that are present at this node.
     * @returns {Array} An array of all non-standard property keys.
     */
    public getCustomKeys(): string[] {
        const customKeys: string[] = [];
        const keys = Object.keys(this._data);
        const count = keys.length;
        for (let i = 0; i < count; i++) {
            const key = keys[i];
            if (this._isRoot ? !reservedKeysOfRoot[key] : !reservedKeys[key]) {
                customKeys.push(key);
            }
        }
        return customKeys;
    }

    /**
     * Returns the object or value for the given key.
     * @param   {String} key The custom key to look up.
     * @returns {*} The value or object for the key.
     */
    public getValue<T = any>(key: string): T {
        if (typeof key !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "key" must be a string');
        }
        if (key === 'children' || key === 'components') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Illegal key');
        }

        return this._data[key] as T;
    }

    /**
     * Returns the object or value for the given key.
     * @param {String}   key   The custom key to set the value for.
     * @param {*} value The value or object.
     */
    public setValue<T = any>(key: string, value: T): void {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        if (typeof key !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "key" must be a string');
        }
        if (key === 'children' || key === 'components') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Illegal key');
        }

        this._data[key] = value;
        this._setDirty();
    }

    /**
     * Removes the object or value for the given key.
     * @param {String} key The custom key to remove.
     */
    public removeValue(key: string): void {
        if (this._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        if (typeof key !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "key" must be a string');
        }
        if (key === 'children' || key === 'components') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Illegal key');
        }

        delete this._data[key];
        this._setDirty();
    }

    //******************************************************************************
    // Miscellaneous
    //******************************************************************************

    /**
     * Creates and returns a deep copy of the node sans any of its components or children.
     * @returns {AdobeDCXNode} The copy.
     */
    public copy(): IAdobeDCXNode {
        let copy: IAdobeDCXNode;
        const children = this._data.children;
        const components = this._data.components;
        try {
            if (children) {
                delete this._data.children;
            }
            if (components) {
                delete this._data.components;
            }
            const newData = JSON.parse(JSON.stringify(this._data));
            copy = new AdobeDCXNode(newData);
        } finally {
            if (children) {
                this._data.children = children;
            }
            if (components) {
                this._data.components = components;
            }
        }
        return copy;
    }

    /**
     * Returns false if any of the properties of the given node is different from the properties
     * of this node. Recurses both data structures.
     * @param   {AdobeDCXNode} node                          The node to compare with.
     * @param   {Array}        [nodePropertiesToIgnore]      Optional. An object having the properties
     *                                                       that should not be compared for node.
     * @param   {Array}        [componentPropertiesToIgnore] Optional. An object having the properties
     *                                                       that should not be compared for components.
     * @returns {Boolean}
     */
    public isEqualTo(
        node: IAdobeDCXNode,
        nodePropertiesToIgnore?: Record<string, boolean>,
        componentPropertiesToIgnore?: Record<string, boolean>,
    ) {
        return this._isEqual(
            this._data,
            (node as AdobeDCXNode)._data,
            merge({ children: true, components: true }, nodePropertiesToIgnore),
            componentPropertiesToIgnore,
        );
    }

    //******************************************************************************
    // Private
    //******************************************************************************

    /**
     * Internal. Compares nodeData
     * @private
     * @param   {Object}  nodeData1
     * @param   {Object}  nodeData2
     * @param   {Object}  nodePropertiesToIgnore
     * @param   {Object}  componentPropertiesToIgnore
     * @returns {Boolean}
     */
    private _isEqual(
        nodeData1: AdobeDCXNodeData,
        nodeData2: AdobeDCXNodeData,
        nodePropertiesToIgnore: Record<string, boolean>,
        componentPropertiesToIgnore?: Record<string, boolean>,
    ) {
        let i, array1, array2, count1, count2;

        if (!objectsEqual(nodeData1, nodeData2, nodePropertiesToIgnore)) {
            return false;
        }

        // compare components
        array1 = nodeData1.components;
        array2 = nodeData2.components;
        count1 = array1 ? array1.length : 0;
        count2 = array2 ? array2.length : 0;

        if (count1 !== count2) {
            return false;
        }
        if (count1) {
            let component1, component2, j;
            for (i = 0; i < count1; i++) {
                // The components list is not ordered so we need to find the component by id
                component1 = array1[i];
                component2 = null;
                for (j = 0; j < count1; j++) {
                    if (component1.id === array2[j].id) {
                        component2 = array2[j];
                        break;
                    }
                }
                if (!component2) {
                    return false;
                }
                if (!objectsEqual(component1, component2, componentPropertiesToIgnore)) {
                    return false;
                }
            }
        }

        // compare children
        array1 = nodeData1.children;
        array2 = nodeData2.children;
        count1 = array1 ? array1.length : 0;
        count2 = array2 ? array2.length : 0;

        if (count1 !== count2) {
            return false;
        }
        if (count1) {
            for (i = 0; i < count1; i++) {
                if (!this._isEqual(array1[i], array2[i], nodePropertiesToIgnore, componentPropertiesToIgnore)) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Verifies and sets the new data. Throws if invalid data.
     * @private
     * @param   {Object} data
     * @returns {AdobeDCXNode}
     *
     * @throws {AdobeDCXError}
     */
    private _setData(data: AdobeDCXNodeData): AdobeDCXNode {
        const err = this._verify(data);
        if (err === null) {
            this._data = data;
            return this;
        }
        throw err;
    }

    /**
     * Returns an error if the passed in data is not a valid node. Returns null if everything is OK.
     * @private
     * @param   {Object} data
     * @returns {AdobeDCXError | null}
     */
    private _verify(data: AdobeDCXNodeData): AdobeDCXError | null {
        if (typeof data.id !== 'string') {
            return new DCXError(DCXError.INVALID_DATA, 'Node is missing an id of type string');
        }

        return null;
    }

    /**
     * Sets the dirty flag on the node's branch or element (if set)
     * @private
     */
    private _setDirty() {
        if (this._owner) {
            this._owner._setDirty();
        }
    }
}

// TODO: implement getter for absolute index

//******************************************************************************
// Custom Properties
//******************************************************************************

const reservedKeysOfRoot: Record<string, boolean> = {
    components: true,
    children: true,
    'manifest-format-version': true,
    id: true,
    name: true,
    type: true,
    state: true,
    local: true,
};
const reservedKeys: Record<string, boolean> = {
    components: true,
    children: true,
    rel: true,
    path: true,
    id: true,
    name: true,
    type: true,
};

export default AdobeDCXNode;
