/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { AdobeDCXBranch, AdobeDCXComposite } from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import DCXBranch from './AdobeDCXBranch';
import { SourceAssetInfoEntry } from './AdobeDCXBranchCore';
import DCXComponent from './AdobeDCXComponent';
import { COMPOSITE_ARCHIVAL_STATES, COMPOSITE_STATES } from './enum';
import { DerivationType } from './util/xmp';

const UPLOAD_AGE_THRESHOLD = 1000 * 60 * 60 * 24 * 7 - 1000 * 60 * 60; // 7 days - 1 hour in milliseconds

export interface UploadedComponentRecord {
    etag: string;
    length: number;
    version: string;
    md5: string;
    timestamp: string;
    'source-asset-info'?: any;
}

interface PushJournalInternalData {
    'composite-archived'?: boolean;
    'composite-deleted'?: boolean;
    'composite-href': string;
    'uploaded-components': Record<string, UploadedComponentRecord>;
    'current-branch-etag'?: string;
    etag?: string;
    change?: number;
    versionId?: string;
    'derivation-type'?: DerivationType;
}

/**
 * Captures the state and progess of a composite push operation
 * which can be used to resume a failed push at a later time.
 * @private
 */
export class AdobeDCXPushJournal {
    private _data: PushJournalInternalData;

    // The following three properties are there to handle simultaneous commit requests
    private _waitingCallbacks: unknown[] = [];
    private _commitInProgress = false;
    private _needAnotherCommit = false;

    constructor(composite: AdobeDCXComposite) {
        this._data = { 'composite-href': composite.href as string, 'uploaded-components': {} };
    }

    //******************************************************************************
    // Getters/setters for properties
    //******************************************************************************

    public get compositeHref(): string {
        return this._data['composite-href'];
    }
    public set compositeHref(id: string) {
        throw new DCXError(DCXError.READ_ONLY, 'This property is read-only.');
    }

    public get derivationType(): DerivationType | undefined {
        return this._data['derivation-type'];
    }
    public set derivationType(derivationType: DerivationType | undefined) {
        this._data['derivation-type'] = derivationType;
    }

    // The new etag of the manifest as returned by the upload.
    public get manifestEtag(): string | undefined {
        return this._data.etag;
    }
    public set manifestEtag(etag: string | undefined) {
        if (etag) {
            this._data.etag = etag;
        } else {
            delete this._data.etag;
        }
    }

    public get versionId(): string | undefined {
        return this._data.versionId;
    }
    public set versionId(v: string | undefined) {
        if (v) {
            this._data.versionId = v;
        } else {
            delete this._data.versionId;
        }
    }

    // The etag of the current manifest that was pushed.
    public get currentBranchEtag(): string | undefined {
        return this._data['current-branch-etag'];
    }
    public set currentBranchEtag(etag: string | undefined) {
        if (etag) {
            this._data['current-branch-etag'] = etag;
        } else {
            delete this._data['current-branch-etag'];
        }
    }

    // The change count of the manifest that go uploaded.
    public get changeCount(): number | undefined {
        return this._data.change;
    }
    public set changeCount(change: number | undefined) {
        if (change) {
            this._data.change = change;
        } else {
            delete this._data.change;
        }
    }

    // Whether the composite has been deleted during the push.
    public get compositeHasBeenDeleted(): boolean {
        return this._data['composite-deleted'] ? true : false;
    }
    public set compositeHasBeenDeleted(hasBeenDeleted: boolean) {
        if (hasBeenDeleted) {
            this._data['composite-deleted'] = true;
        } else {
            delete this._data['composite-deleted'];
        }
    }

    // Whether the composite has been archived during the push.
    public get compositeHasBeenArchived(): boolean {
        return this._data['composite-archived'] ? true : false;
    }
    public set compositeHasBeenArchived(hasBeenArchived: boolean) {
        if (hasBeenArchived) {
            this._data['composite-archived'] = true;
        } else {
            delete this._data['composite-archived'];
        }
    }

    // Is true if the journal doesn't contain any information worth keeping.
    public get isEmpty(): boolean {
        return (
            Object.keys(this._data['uploaded-components']).length === 0 &&
            !this.compositeHasBeenDeleted &&
            !this.compositeHasBeenArchived &&
            !this.manifestEtag
        );
    }
    public set isEmpty(_) {
        throw new DCXError(DCXError.READ_ONLY, 'This property is read-only.');
    }

    // Is true if the journal represents a completed push
    public get isComplete(): boolean {
        return (this._data.etag && this._data.change) ||
            this._data['composite-deleted'] ||
            this._data['composite-archived']
            ? true
            : false;
    }
    public set isComplete(_) {
        throw new DCXError(DCXError.READ_ONLY, 'This property is read-only.');
    }

    //******************************************************************************
    // Component uploads
    //******************************************************************************

    /**
     * Records the upload of the given component.
     * @internal
     * @param {String}  componentId
     * @param {String}  etag
     * @param {String}  version
     * @param {String}  md5
     * @param {Integer} length
     * @param {Object}  sourceAssetInfo
     */
    public recordUploadedComponent(
        componentId: string,
        etag: string,
        version: string,
        md5: string,
        length: number,
        sourceAssetInfo?: SourceAssetInfoEntry,
    ): void {
        const record: UploadedComponentRecord = {
            etag: etag,
            length: length,
            version: version,
            md5: md5,
            timestamp: new Date().toISOString(),
        };

        if (sourceAssetInfo) {
            record['source-asset-info'] = sourceAssetInfo;
        }

        this._data['uploaded-components'][componentId] = record;
    }

    /**
     * Returns an array of the ids of all uploaded components.
     * @internal
     * @returns {Array}
     */
    public idsOfAllUploadedComponents(): string[] {
        return Object.keys(this._data['uploaded-components']);
    }

    /**
     * Returns an upload record of the given component or undefined if it has not been uploaded or
     * has aged out.
     * @param   {String}   componentId
     * @returns {Object}
     * @example
     *      {
     *           etag: componentEtag,
     *           length: componentLength,
     *           version: component.Version,
     *           md5: componentMd5,
     *           'source-asset-info': {
     *              'compositeAssetId': assetId,
     *              'componentId': componentId,
     *              'componentVersion': version
     *           }
     *      }
     */
    public getRecordForUploadedComponent(componentId: string): UploadedComponentRecord | undefined {
        const record = this._data['uploaded-components'][componentId];

        if (record && !this.isComplete) {
            let then: Date = record.timestamp as unknown as Date;
            if (!then) {
                // no timestamp
                this.removeRecordForUploadedComponent(componentId);
                return;
            }

            then = new Date(then);
            if (isNaN(then.getTime())) {
                // invalid timestamp
                this.removeRecordForUploadedComponent(componentId);
                return;
            }
            const millisecondsPassed = Date.now() - then.getTime();
            if (millisecondsPassed >= UPLOAD_AGE_THRESHOLD) {
                // too old
                this.removeRecordForUploadedComponent(componentId);
                return;
            }
        }

        return record;
    }

    /**
     * Deletes the upload record for the given component.
     */
    public removeRecordForUploadedComponent(componentId: string): void {
        delete this._data['uploaded-components'][componentId];
    }

    //******************************************************************************
    // Merge
    //******************************************************************************

    /**
     * Applies the journaled data to the provided branch.
     */
    public applyToBranch(pBranch: AdobeDCXBranch, preserveDirtyState?: boolean): void {
        const branch = pBranch as DCXBranch;

        if (!this.isComplete) {
            throw new DCXError(DCXError.INVALID_STATE, 'Journal is not complete.');
        }
        const branchWasDirty = branch.isDirty;
        if (this.compositeHasBeenDeleted) {
            branch._data.state = COMPOSITE_STATES.committedDelete;
            branch.manifestEtag = this.manifestEtag; // This bumps the change count on branch
        } else {
            if (this.compositeHasBeenArchived) {
                branch.compositeArchivalState = COMPOSITE_ARCHIVAL_STATES.archived;
            }

            const branchHasChangedSincePush =
                branch.compositeState !== COMPOSITE_STATES.unmodified && this.changeCount !== branch.changeCount;

            // Iterate over all uploaded components and update their counterparts in branch.
            const componentIds = this.idsOfAllUploadedComponents();
            const count = componentIds.length;
            for (let i = 0; i < count; i++) {
                const id = componentIds[i];
                const component = branch.getComponentWithId(id) as DCXComponent;

                if (component) {
                    // we ignore components that have been deleted since the push
                    const uploadRecord = this.getRecordForUploadedComponent(component.id) as UploadedComponentRecord;

                    component._data.etag = uploadRecord.etag;
                    component._data.length = uploadRecord.length;
                    component._data.version = uploadRecord.version;
                    component._data.md5 = uploadRecord.md5;

                    if (uploadRecord['source-asset-info']) {
                        const sourceAssetInfo = branch._core._getSourceAssetInfoOfComponent(component);
                        const uploadedAssetInfo = uploadRecord['source-asset-info'];
                        if (
                            sourceAssetInfo &&
                            uploadedAssetInfo &&
                            sourceAssetInfo.compositeAssetId === uploadedAssetInfo.compositeAssetId &&
                            sourceAssetInfo.componentId === uploadedAssetInfo.componentId &&
                            sourceAssetInfo.componentVersion === uploadedAssetInfo.componentVersion
                        ) {
                            // If the source asset info matches than the component is no longer modified
                            component._data.state = COMPOSITE_STATES.unmodified;
                            branch._core._setSourceAssetInfoOfComponent(undefined, component);
                        }
                    } else {
                        component._data.state = COMPOSITE_STATES.unmodified;
                    }
                }
            }

            // Update the branch root.
            branch.manifestEtag = this.manifestEtag; // This bumps the change count on branch

            // Update the versionId
            if (typeof this.versionId === 'string') {
                branch.versionId = this.versionId;
            }

            if (typeof this.derivationType !== 'undefined') {
                branch._derivationType = this.derivationType;
            }

            // Do this last:
            if (!branchHasChangedSincePush) {
                // If the change count matches what was pushed then the composite is no longer modified
                branch._data.state = COMPOSITE_STATES.unmodified;
            }
        }
        if (preserveDirtyState) {
            branch._isDirty = branchWasDirty;
        }
    }
}

export default AdobeDCXPushJournal;
