/* istanbul ignore file */

/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

// import { Link } from '@dcx/util';
import { HTTPMethods } from '@dcx/assets';
import { AdobeDCXError, AdobeResponse, AdobeSessionBase, BasicLink, InvokeOptions, Link } from '@dcx/common-types';
import DCXError, { networkError, unexpectedResponse } from '@dcx/error';
import { AdobeHTTPService, AdobeResponseType, RequestDescriptor, RequestOptions, ResponseCallback } from '@dcx/http';
import { log, newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import {
    appendPathElements,
    endPointOf,
    expandURITemplate,
    generateUuid,
    parse as linkParse,
    merge,
    noOp,
    parseHeaders,
    parseURI,
    pruneUndefined,
} from '@dcx/util';

const dbg = newDebug('dcx:dcxjs:sb');

const ifNoneMatchHeader = 'if-none-match';

// override for storage/community session
// all statuses resolve, none reject.
const isStatusValid = () => true;
const retryOptions = { disableRetry: true };
const autoParseJson = false;

export const SYNC_ASYNC_DEFAULT_DELAY = 5;
export const ASYNC_DEFAULT_DELAY = 1;
export const DEFAULT_POLL_DELAY = 10;

const primaryTemplateKey = 'primary';
const manifestTemplateKey = 'http://ns.adobe.com/ccapi/manifest';
const manifest2TemplateKey = 'http://ns.adobe.com/ccapi/manifest2';
const componentTemplateKey = 'http://ns.adobe.com/ccapi/component';
const versionHistoryTemplateKey = 'version-history';
const renditionTemplateKey = 'rendition';

let rootReqDesc: AdobePromise<any, AdobeDCXError, RequestDescriptor> | null = null;

let _promiseCache: AdobePromise<any, AdobeDCXError, RequestDescriptor> | null = null;

// documentation for CachedAssetInfo
interface RenditionTemplate {
    uri: string;
}

/**
 * Contains cached content location and link data previously retrieved by issuing
 * a head request.
 * @typedef {Object} CachedAssetInfo
 *   @property {String}  primaryTemplate  Primary ID-based URI template to the composite in aggregate
 *   @property {String}  manifestTemplate URI template that can be expanded to locate the asset's manifest
 *   @property {String}  componentTemplate URI template that can be expanded to locate the asset's components
 *   @property {String}  versionHistory
 */
export interface CachedAssetInfo {
    primaryTemplate: string;
    manifestTemplate: string;
    componentTemplate: string;
    versionHistory: string;
    renditionTemplates: RenditionTemplate[];
}

// documentation for GetCachedAssetInfoCallback
/**
 * Gets passed into `getCachedAssetInfo()`.
 * @callback GetCachedAssetInfoCallback
 *   @param {Error}             error       If not undefined the request has failed.
 *   @param {CachedAssetInfo}   assetInfo
 */
interface GetCachedAssetInfoCallback {
    (error?: undefined, info?: CachedAssetInfo): any;
    (error?: AdobeDCXError, info?: undefined): any;
}

//******************************************************************************
// Public API
//******************************************************************************

/**
 * @class
 * @classdesc Base class for the different sessions.
 * @param httpService the HTTP service
 * @param server      the url for the storage server
 */
export class SessionBase implements AdobeSessionBase {
    protected _server: string;
    public _endPoint?: string;
    protected _maxRedirects = 5;
    protected _authenticationAllowList: string[] = ['adobe.com', 'adobe.io', 'adobelogin.com', 'fotolia.net'];
    protected _assetInfoCache: any = {};
    protected _assetIdResolutionTemplate: any = null;

    protected readonly SYNC_ASYNC_DEFAULT_DELAY = 5;
    protected readonly ASYNC_DEFAULT_DELAY = 1;
    protected readonly DEFAULT_POLL_DELAY = 10;

    /** @internal */
    _service: AdobeHTTPService;

    public constructor(httpService: AdobeHTTPService, server: string) {
        this._service = httpService;
        this._server = server;
        if (server) {
            this._endPoint = endPointOf(server);
        }
    }

    /**
     * The maximum number of redirects the session will follow *if* the underlying service
     * doesn't handle redirects. Gets ignored for services that handle redirects (e.g. XHR).
     * Set this to 0 if the session should not follow redirects. Default value: 5
     * @memberof AdobeSessionBase#
     * @type Integer
     */
    get maxRedirects() {
        return this._maxRedirects;
    }
    set maxRedirects(value) {
        if (typeof value !== 'number') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a number.');
        }
        this._maxRedirects = Math.floor(value);
    }
    /**
     * A list of domain names that should get sent the auth credentials of the underlying service.
     * @memberof AdobeSessionBase#
     * @type Array
     */
    get authenticationAllowList() {
        return this._authenticationAllowList;
    }
    set authenticationAllowList(value: string[]) {
        if (!Array.isArray(value)) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting an array.');
        }
        this._authenticationAllowList = value;
    }

    /**
     * Given an href string this method returns true if it is either relative or it refers
     * to the same endpoint as the session's server.
     * @param {String} href The href to check
     */
    public isValidHref(href: string): boolean {
        const endPoint = endPointOf(href);

        return !endPoint || endPoint === this._endPoint;
    }

    public isDomainOnAllowList(href: string): boolean {
        dbg('isDomainOnAllowList');

        const host = parseURI(href).authority;

        if (!host || host.length === 0) {
            // href is relative
            return true;
        }

        const hostLength = host.length;

        const count = this._authenticationAllowList.length;
        for (let i = 0; i < count; i++) {
            const domain = this._authenticationAllowList[i];
            const domainLength = domain.length;
            let hostDomain;
            // The host name must end with the domain:
            if (hostLength > domainLength) {
                hostDomain = host.slice(-domainLength);
            } else {
                hostDomain = host;
            }

            if (hostDomain === domain) {
                dbg('iDOAL true');
                return true;
            }
        }

        dbg('iDOAL false');
        return false;
    }

    /**
     * Gets an asset from the service.
     * @param {String}          href                The full URL of the asset to retrieve.
     * @param {String|Object}   [etagOrHeaders]     If a string, the etag to use, if an object a set of additional headers, if undefined. ignored
     * @param {Function}        callback            Called back when the asset has been fetched. Signature: `callback(error, httpResponse)`.
     * @return {Object}         The http requestDescriptor.
     */
    public getAsset(
        href: string,
        etagOrHeaders: string | Record<string, string>,
        responseType: AdobeResponseType,
        callback: ResponseCallback,
    ): RequestDescriptor {
        return this._getAsset(href, etagOrHeaders, responseType, undefined, callback);
    }

    /**
     * @private
     * @param {Object}          reuseRequestDesc    A RequestDescriptor object to reuse for additional HTTP requests
     */
    protected _getAsset(
        href: string,
        etagOrHeaders: string | Record<string, string> = {},
        responseType: AdobeResponseType,
        reuseRequestDesc: RequestDescriptor | undefined,
        callback: ResponseCallback,
    ): RequestDescriptor {
        dbg('_getAsset()');

        href = this._resolveUrl(href);
        if (!href) {
            callback(new DCXError(DCXError.WRONG_ENDPOINT, 'Wrong endpoint: ' + href));
            return undefined as unknown as RequestDescriptor;
        }

        let requestDesc = reuseRequestDesc;
        const headers: { [key: string]: string } = {};
        if (typeof etagOrHeaders === 'string') {
            headers[ifNoneMatchHeader] = etagOrHeaders;
        } else {
            merge(headers, etagOrHeaders);
        }
        let numRedirects = 0;
        // eslint-disable-next-line prefer-const
        let invokeRequest: (redirectTo?: string) => RequestDescriptor; // forward declare
        const respHandler: ResponseCallback = (error, response) => {
            dbg('_getAsset() respHandler', error, response);

            if (error) {
                callback(networkError('Error downloading an asset', error, response));
                return;
            }
            const statusCode = response.statusCode;
            if (statusCode === 200) {
                callback(undefined, response);
            } else {
                if (statusCode === 304) {
                    callback(undefined, response);
                } else if (statusCode === 301 || statusCode === 302 || statusCode === 303 || statusCode === 307) {
                    if (!this._service.handlesRedirects && this.maxRedirects) {
                        if (numRedirects < this.maxRedirects) {
                            // The underlying service doesn't handle redirects so we do that ourselves
                            const redirectTo = response.headers.location;
                            if (redirectTo) {
                                invokeRequest(redirectTo);
                            } else {
                                /* istanbul ignore next */
                                callback(
                                    unexpectedResponse('Missing location header in redirect response', error, response),
                                );
                            }
                        } else {
                            callback(unexpectedResponse('Too many redirects', error, response));
                        }
                    } else {
                        /* istanbul ignore next */
                        callback(unexpectedResponse('Unexpected response getting an asset', error, response));
                    }
                } else {
                    callback(unexpectedResponse('Unexpected response getting an asset', error, response));
                }
            }
        };
        invokeRequest = (redirectTo?: string) => {
            const options: InvokeOptions = { responseType: responseType, isStatusValid, retryOptions, autoParseJson };
            if (redirectTo) {
                numRedirects++;
                // reuse request desc
                options.reuseRequestDesc = requestDesc;
                if (!this.isDomainOnAllowList(redirectTo)) {
                    // We are being redirected to a different endpoint.
                    (headers as Record<string, unknown>).authorization = null; // A value of null instructs the service to not inject the auth header
                }
            }
            return this._service.invoke(HTTPMethods.GET, redirectTo || href, headers, undefined, options, respHandler);
        };
        requestDesc = invokeRequest();
        requestDesc.progress = noOp;
        return requestDesc;
    }

    /**
     * Get asset data in specified responseType result
     * @param {String}          href                The full URL of the asset to retrieve.
     * @param {String}          responseType        The desired response type.
     * @param {String|Object}   [etagOrHeaders]     If a string, the etag to use, if an object a set of additional headers, if undefined. ignored
     * @param {Function}        callback            Called back when the asset has been fetched. Signature: `callback(error, buffer, etag, response)`. The buffer is "null" if the etags match.
     *
     * @return               the requestDescriptor for the request.
     */
    public getAssetAsType(
        href: string,
        responseType: AdobeResponseType,
        etagOrHeaders: string | Record<string, string>,
        callback: (error?: AdobeDCXError, buffer?: Buffer | null, etag?: string, response?: AdobeResponse) => void,
    ): RequestDescriptor {
        return this._getAssetAsType(href, responseType, etagOrHeaders, undefined, undefined, undefined, callback);
    }

    private _cacheLinksFromResponse(assetId: string | undefined, response: AdobeResponse) {
        if (typeof assetId === 'string') {
            const headerInfo = this._parseCachableLinks(response);
            if (Object.keys(headerInfo).length > 0) {
                const alreadyCached = this._assetInfoCache[assetId];
                const newLinks = Object.assign({}, alreadyCached || {}, pruneUndefined(headerInfo));
                this._cacheInfoForAssetId(assetId, undefined, newLinks);
            }
        }
    }

    /**
     * @private
     * @param {Object}          reuseRequestDesc    A RequestDescriptor object to reuse for additional HTTP requests
     */
    public _getAssetAsType(
        href: string,
        responseType: AdobeResponseType,
        etagOrHeaders: string | Record<string, string> | undefined,
        reuseRequestDesc: RequestDescriptor | undefined,
        assetId: string | undefined,
        versionId: string | undefined,
        callback: (error?: AdobeDCXError, buffer?: Buffer | null, etag?: string, response?: AdobeResponse) => void,
    ): RequestDescriptor {
        dbg('_getAssetAsType');
        const onComplete: ResponseCallback = (error, response) => {
            if (error) {
                return callback(error); // error is already wrapped
            }
            if (response.statusCode === 200) {
                const buf = response.response || (response as any).responseText;
                if (versionId == null) {
                    this._cacheLinksFromResponse(assetId, response);
                }
                callback(undefined, buf, response.headers.etag, response);
            } else if (response.statusCode === 304) {
                // null signals "no change"
                callback(undefined, null, response.headers.etag, response);
            } else {
                /* istanbul ignore next */
                callback(unexpectedResponse('Unexpected response getting asset', error, response));
            }
        };
        return this._getAsset(href, etagOrHeaders, responseType, reuseRequestDesc, onComplete);
    }

    public _getResourcePathFromHref(
        href: string,
        callback: (err?: AdobeDCXError, path?: string) => void,
    ): RequestDescriptor {
        dbg('_getResourcePathFromHref');
        const promise = this._service.invoke(
            HTTPMethods.HEAD,
            href,
            {},
            undefined,
            { isStatusValid, retryOptions, autoParseJson },
            (headError, response) => {
                dbg('_gRPFH invoked', headError, response);

                if (headError) {
                    return callback(headError);
                }
                const statusCode = response.statusCode;
                if (statusCode === 200) {
                    const linkHeader = response.headers['link'];
                    if (linkHeader) {
                        const parsedLinks = linkParse(linkHeader as string);

                        // Lookup path-based URI - this isn't cached since it may be changed independently
                        const link = parsedLinks.get('rel', 'http://ns.adobe.com/ccapi/path');
                        if (link && link[0] && link[0].uri) {
                            return callback(undefined, link[0].uri);
                        }
                    }
                }
                /* istanbul ignore next */
                return callback(
                    unexpectedResponse('Unexpected response trying to retrieve path to asset.', undefined, response),
                );
            },
        );

        // return (promise as any).props as RequestDescriptor;
        return promise as RequestDescriptor;
    }

    public resolveRootURLAsync(): AdobePromise<any, AdobeDCXError, RequestDescriptor> {
        dbg('resolveRootURLAsync()');

        // return new Promise((resolve, reject) => {
        const rootHref = this._resolveUrl('/');
        dbg('rRUA() rootHref', rootHref);
        let responseError;
        rootReqDesc = this._service
            .invoke(HTTPMethods.GET, rootHref, {}, undefined, {
                responseType: 'text',
                isStatusValid,
                retryOptions,
                autoParseJson,
            })
            .then((response) => {
                dbg('rRUA() resolved', response);

                // if (getResolveUrlError) {
                //     reject(getResolveUrlError);
                // }

                if (response.statusCode === 200) {
                    const buf = (response as any).responseText || response.response;
                    try {
                        // In this case the URI templates are included in the response body
                        const json = JSON.parse(buf);
                        const resolutionLink = json._links
                            ? json._links['http://ns.adobe.com/ccapi/resolve/id']
                            : undefined;
                        if (resolutionLink) {
                            return { href: resolutionLink.href, self: this };
                        }
                    } catch (x) {
                        responseError = new DCXError(
                            DCXError.INVALID_DATA,
                            'Invalid JSON returned by server',
                            x,
                            response,
                        );
                        throw responseError;
                    }
                } else {
                    responseError = new DCXError(
                        DCXError.UNEXPECTED_RESPONSE,
                        'Resolve root URL did not succeed with 200',
                        undefined,
                        response,
                    );
                    throw responseError;
                }
            })
            .catch((e) => {
                dbg('rRUA() rejected', e);

                throw e;
            });
        dbg('rRUA() rootReqDesc', rootReqDesc);
        return rootReqDesc;

        // return rootReqDesc;
        // });
    }

    public getPromiseCacheInfo(
        fn: (...args: any[]) => AdobePromise<any, AdobeDCXError, RequestDescriptor>,
    ): AdobePromise<any, AdobeDCXError, RequestDescriptor> {
        dbg('getPromiseCacheInfo()');

        // if we have an in-flight request, return it
        if (_promiseCache != null) {
            return _promiseCache;
        }
        _promiseCache = fn();
        // clear the cache for the next set of requests
        _promiseCache &&
            _promiseCache.then &&
            _promiseCache.then(
                () => {
                    dbg('gPCI() resolved');
                    _promiseCache = null;
                },
                () => {
                    dbg('gPCI() rejected');
                    _promiseCache = null;
                },
            );
        return _promiseCache;
    }

    private _parseCachableLinks(response: AdobeResponse): Record<string, string> {
        const headerInfo: { [key: string]: string } = {};
        const contentLocation = response.headers['content-location'];
        const linkHeader = response.headers['link'];
        if (linkHeader) {
            const parsedLinks = linkParse(linkHeader as string);
            // Primary URI
            let link = parsedLinks.get('rel', primaryTemplateKey);
            if (link) {
                headerInfo.primaryTemplate = link[0].uri;
            }
            // Manifest template
            link = parsedLinks.get('rel', manifest2TemplateKey);
            if (link) {
                headerInfo.manifestTemplate = link[0].uri;
            } else {
                link = parsedLinks.get('rel', manifestTemplateKey);
                if (link) {
                    headerInfo.manifestTemplate = link[0].uri;
                }
            }
            // Component template
            link = parsedLinks.get('rel', componentTemplateKey);
            if (link) {
                headerInfo.componentTemplate = link[0].uri;
            }
            // Version history
            link = parsedLinks.get('rel', versionHistoryTemplateKey);
            if (link) {
                headerInfo.versionHistory = link[0].uri;
            }
            // Rendition template (in case we are retrieving links for a component asset)
            link = parsedLinks.get('rel', renditionTemplateKey);
            if (link) {
                headerInfo.renditionTemplates = link as any;
            }
        }
        return headerInfo;
    }

    /**
     * Returns CachedAssetInfo for the specified asset id
     *
     * @param {String}  assetId     The unique id of the asset
     * @param {GetCachedAssetInfoCallback} Callback function to call with the result
     * @returns {RequestDescriptor} A RequestDescriptor if an HTTP request was made or undefined if none was required
     */
    public getCachedAssetInfo(assetId: string, getCachedInfoCallback: GetCachedAssetInfoCallback) {
        dbg('getCachedAssetInfo()');
        const info = this._infoForAssetId(assetId);
        if (info) {
            dbg('gCAI() cached');
            return getCachedInfoCallback(undefined, info);
        }

        const needsHeadRequest = this._awaitInfoForAssetId(assetId, getCachedInfoCallback);
        dbg('gCAI() needs head req', needsHeadRequest);

        if (needsHeadRequest) {
            let requestDesc: RequestDescriptor;
            const resolveAssetId = (assetIdResolutionTemplate: string) => {
                let href = expandURITemplate(assetIdResolutionTemplate, { asset_id: assetId });
                href = this._resolveUrl(href);
                if (!href) {
                    dbg('gCAI() no href');
                    return getCachedInfoCallback(new DCXError(DCXError.WRONG_ENDPOINT, 'Wrong endpoint: ' + href));
                }
                let requestOptions = {};
                if (requestDesc) {
                    requestOptions = { reuseRequestDesc: requestDesc };
                    dbg('gCAI() reuse RD', requestOptions);
                }
                requestDesc = this._service.invoke(
                    HTTPMethods.HEAD,
                    href,
                    {},
                    undefined,
                    { ...requestOptions, isStatusValid, retryOptions, autoParseJson },
                    (invokeError, response) => {
                        dbg('gCAI() resolveAssetId() resolved', invokeError, response);
                        if (invokeError) {
                            return this._cacheInfoForAssetId(assetId, invokeError); // return raw error from http
                        }

                        const statusCode = response.statusCode;
                        if (statusCode === 200) {
                            const headerInfo = this._parseCachableLinks(response);
                            return this._cacheInfoForAssetId(assetId, undefined, headerInfo);
                        }
                        if (statusCode === 404) {
                            return this._cacheInfoForAssetId(
                                assetId,
                                new DCXError(DCXError.ASSET_NOT_FOUND, 'Asset not found', invokeError, response),
                            );
                        }
                        /* istanbul ignore next */
                        return this._cacheInfoForAssetId(
                            assetId,
                            unexpectedResponse('Unable to resolve asset id.', invokeError, response),
                        );
                    },
                ) as RequestDescriptor;

                return requestDesc;
            };

            if (this._assetIdResolutionTemplate) {
                dbg('gCAI() has resolution template');

                return resolveAssetId(this._assetIdResolutionTemplate);
            }

            dbg('gCAI() get resolution template');
            rootReqDesc = this.getPromiseCacheInfo(this.resolveRootURLAsync.bind(this))
                .then((ret) => {
                    dbg('gCAI() getPromiseCacheInfo() resolved', ret);

                    this._assetIdResolutionTemplate = ret.href;
                    return resolveAssetId(ret.self._assetIdResolutionTemplate);
                })
                .catch((respErr) => {
                    dbg('gCAI() getPromiseCacheInfo() rejected', respErr);

                    const responseError = new DCXError(
                        DCXError.UNEXPECTED_RESPONSE,
                        'Unable to retrieve asset id resolution link from root resource.',
                        respErr,
                        respErr.response,
                    );
                    return getCachedInfoCallback(responseError);
                });
            dbg('gCAI() rootReqDesc', rootReqDesc);

            return rootReqDesc as unknown as RequestDescriptor;
        }
    }

    /**
     * Registers a set of links and returns a (fake) assetId
     *
     * @param {Object}  links   The links to register (body of _links object)
     * @returns {String}        Unique (but fake) assetId
     */
    public registerLinks(links: Record<string, Link>): string {
        links = (links as any)._links || links;
        const info: any = {};
        info.assetId = 'urn:aaid:faux:' + generateUuid();
        let link = links[primaryTemplateKey] as BasicLink;
        if (link) {
            info.primaryTemplate = link.href;
        }
        link = links[manifestTemplateKey] as BasicLink;
        if (link) {
            info.manifestTemplate = link.href;
        }
        link = links[manifest2TemplateKey] as BasicLink;
        if (link) {
            info.manifestTemplate = link.href;
        }
        link = links[componentTemplateKey] as BasicLink;
        if (link) {
            info.componentTemplate = link.href;
        }
        link = links[versionHistoryTemplateKey] as BasicLink;
        if (link) {
            info.versionHistory = link.href;
        }
        link = links[renditionTemplateKey] as BasicLink;
        if (link) {
            info.renditionTemplates = [{ rel: 'rendition', uri: link.href }];
        }
        // The assetId is a generated uuid, so it can't already be in the cache
        this._assetInfoCache[info.assetId] = info;
        return info.assetId;
    }

    //******************************************************************************
    // Internal Methods
    //******************************************************************************

    /** @internal */
    _infoForAssetId(assetId: string) {
        const info = this._assetInfoCache[assetId];
        if (info && !info.callbacks) {
            // If callbacks is defined then we are still waiting for a HEAD
            // request to complete
            return info;
        }
        return undefined;
    }

    private _awaitInfoForAssetId(assetId: string, getCachedInfoCallback: GetCachedAssetInfoCallback) {
        dbg('_awaitInfoForAssetId');
        let info = this._assetInfoCache[assetId];
        if (info) {
            dbg('_aIFA pending');
            info.callbacks.push(getCachedInfoCallback);
            return false; // a head request is already pending
        }

        dbg('_aIFA not pending');
        info = {};
        info.assetId = assetId;
        info.callbacks = [getCachedInfoCallback];
        this._assetInfoCache[assetId] = info;
        return true; // caller should issue head request for retrieve asset info
    }

    protected _cacheInfoForAssetId(assetId: string, error?: Error, info?: any) {
        dbg('_cacheInfoForAssetId');

        let cachedInfo = this._assetInfoCache[assetId];
        if (cachedInfo) {
            if (error) {
                const callbacks = cachedInfo.callbacks;
                this._assetInfoCache[assetId] = undefined;
                for (let i = 0; i < callbacks.length; i++) {
                    callbacks[i](error);
                }
            }
        } else if (!error && info) {
            this._assetInfoCache[assetId] = cachedInfo = {};
        }

        if (!error && info) {
            if (!cachedInfo.assetId) {
                cachedInfo.assetId = assetId;
            }
            cachedInfo.primaryTemplate = info.primaryTemplate;
            cachedInfo.manifestTemplate = info.manifestTemplate;
            cachedInfo.componentTemplate = info.componentTemplate;
            cachedInfo.versionHistory = info.versionHistory;
            cachedInfo.renditionTemplates = info.renditionTemplates;

            if (cachedInfo.callbacks) {
                // Inform callers awaiting results
                for (let i = 0; i < cachedInfo.callbacks.length; i++) {
                    cachedInfo.callbacks[i](undefined, cachedInfo);
                }

                // implicitly marks cached object as being valid
                cachedInfo.callbacks = undefined;
            }
        }
    }

    /**
     * A _very_ naive URL resolver whose function is to prepend the a base to a relative URL
     * or to simply return the href directly if it's absolute.
     * Nothing is done about . or .. elements.
     */
    public _resolveUrl(href: string): string {
        const endPoint = endPointOf(href);

        if (endPoint) {
            // href is absolute
            return href;
        }

        // the href is relative so we just append it to our base
        return appendPathElements(this._server, href);
    }

    /**
     * @private
     * @param {String} href The href
     */
    protected _makeRelativeUrl(href: string) {
        const endPoint = endPointOf(href);

        if (endPoint) {
            const uriItems = parseURI(href);

            if (uriItems.scheme || uriItems.authority) {
                href = uriItems.path;
                if (uriItems.query) {
                    href += '?' + uriItems.query;
                }
                if (uriItems.fragment) {
                    href += '#' + uriItems.fragment;
                }
            }
        }

        return href;
    }

    //******************************************************************************
    // Internal Methods for Asynchronous Requests
    //******************************************************************************

    /**
     * Helper function to parse a response when polling for asynchronous results.
     * The body returned from these requests includes the response to the original request
     * including its own status code and headers.
     *
     * @notice This methods throws if it fails to parse at least the status code.
     *
     * @param asyncResponse The respone received from a successful (status code 200) poll request.
     * @return A fake response object with a statusCode and headers property set.
     *
     * @notice For now this methods drops the response body of the embedded response.
     * @internal
     */
    public _parseAsyncResponse(asyncResponse: any) {
        dbg('_parseAsyncResponse');
        const body = asyncResponse.responseText || asyncResponse.response;
        if (!body) {
            throw new DCXError(DCXError.INVALID_DATA, 'No body data.');
        }

        // TODO: use Response type
        const response: any = {};

        // We are not using split(CRLF) but rather use a combination of indexOf() and slice() calls to inspect
        // the response line by line since the body of the response could be very big and we are only interested
        // in the first few lines containing the status code and response headers.

        // Get the first line (i.e. the status line)
        const endOfLine = body.indexOf('\n');
        if (endOfLine === -1) {
            throw new DCXError(DCXError.INVALID_DATA, 'Could not find status line.');
        }
        const line = body.slice(0, body.charCodeAt(endOfLine - 1) === 13 ? endOfLine - 1 : endOfLine);
        const parts = line.split(' '); // The status line contains multiple elements separated by spaces
        response.statusCode = parseInt(parts[1], 10); // The status code is the second element
        if (!response.statusCode) {
            throw new DCXError(DCXError.INVALID_DATA, 'Could not find status code.');
        }
        if (parts.length > 2) {
            response.statusText = parts[2];
        }

        // Now we extract the headers by removing the first line and chopping off the response body (if there is one)
        let endOfHeaders = body.search(/\r?\n\r?\n/);
        if (endOfHeaders === -1) {
            endOfHeaders = body.length;
        }
        const allHeaders = body.slice(endOfLine + 1, endOfHeaders);
        response.headers = parseHeaders(allHeaders);

        // The remainder should be the response
        const bodyStartsAt = body.charAt(endOfHeaders) === '\r' ? endOfHeaders + 4 : endOfHeaders + 2;
        response.response = body.slice(bodyStartsAt);

        return response;
    }

    /**
     * Issues and handles the responses to an async poll reuqest.
     * Calls itself recursively if it gets another 202 response.
     *
     * @private
     * @param pollHref The href returned by the original 202 response.
     * @param pollFrequencyInSeconds Self explanatory.
     * @param requestDesc The request descriptor to reuse.
     * @param callback Gets called upon success or failure. Signature: function (error, response)
     */
    private _pollForAsyncResponse(
        pollHref: string,
        pollFrequencyInSeconds: number,
        requestDesc: RequestDescriptor,
        callback: (err?: AdobeDCXError, response?: any) => void,
    ) {
        dbg('_pollForAsyncResponse');

        const when = Date.now() + pollFrequencyInSeconds * 1000;
        const options = <RequestOptions>{
            responseType: 'text',
            reuseRequestDesc: requestDesc,
            noSoonerThen: when,
            isStatusValid,
            retryOptions,
            autoParseJson,
        };
        log('asynchronous request - polling');
        const promise = this._service.invoke(
            HTTPMethods.GET,
            pollHref,
            undefined,
            undefined,
            options,
            (error, response) => {
                if (error) {
                    return callback(networkError('Error polling for an asynchronous reponse', error, response));
                }
                let statusCode = response.statusCode;
                if (statusCode === 202) {
                    // poll again
                    log('asynchronous request - not yet ready');
                    let retryAfter: number | undefined = undefined;
                    if (response.headers['retry-after']) {
                        retryAfter = parseInt(response.headers['retry-after'] as string, 10);
                    }
                    this._pollForAsyncResponse(pollHref, retryAfter || DEFAULT_POLL_DELAY, requestDesc, callback);
                } else if (statusCode === 200) {
                    log('asynchronous request - success');
                    // extract the status code and other headers from the response body and fake a response using the original reponse
                    try {
                        response = this._parseAsyncResponse(response);
                    } catch (x) {
                        /* istanbul ignore next */
                        return callback(unexpectedResponse('Error parsing response body.', x, response));
                    }
                    statusCode = response.statusCode;
                    if (statusCode === 200 || statusCode === 201 || statusCode === 204) {
                        callback(undefined, response);
                    } else {
                        /* istanbul ignore next */
                        callback(
                            unexpectedResponse(
                                'Unexpected response polling for an asynchronous response',
                                error,
                                response,
                            ),
                        );
                    }
                } else {
                    log('asynchronous request - error');
                    /* istanbul ignore next */
                    callback(
                        unexpectedResponse('Unexpected response polling for an asynchronous response', error, response),
                    );
                }
            },
        );

        // requestDesc = (promise as any).props;
        requestDesc = promise as RequestDescriptor;
    }

    /**
     * Handles a 202 response by scheduling a poll request reusing the same requestDesc.
     *
     * @protected
     * @param response The original 202 response.
     * @param requestDesc The request descriptor to reuse.
     * @param defaultSecondsToWaitForRetry Integer >= 0. If the reply doesn't contain a retry-after directive this is being used as the initial polling delay.
     * @param callback Gets called upon success or failure. Signature: function (error, response)
     */
    protected _handle202Response(
        response: any,
        requestDesc: RequestDescriptor,
        defaultSecondsToWaitForRetry: number,
        callback: (err?: AdobeDCXError, result?: any) => void,
    ) {
        dbg('_handle202Response');

        const responseText = response.responseText || response.response;
        let parsedResponse;
        try {
            parsedResponse = JSON.parse(responseText);
        } catch (x) {
            /* istanbul ignore next */
            return callback(unexpectedResponse('Error parsing 202 response body.', x, response));
        }
        const pollHref = parsedResponse.href;
        if (!pollHref) {
            /* istanbul ignore next */
            return callback(unexpectedResponse('202 response missing an href.', undefined, response));
        }
        let pollFrequencyInSeconds = response.headers['retry-after'];
        if (pollFrequencyInSeconds) {
            pollFrequencyInSeconds = parseInt(pollFrequencyInSeconds, 10);
        } else {
            pollFrequencyInSeconds = defaultSecondsToWaitForRetry;
        }
        this._pollForAsyncResponse(this._resolveUrl(pollHref), pollFrequencyInSeconds, requestDesc, callback);
    }
}
