/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { AdobeDCXComposite, FailedComponent, ProgressCallback } from '@dcx/common-types';
import { newDebug } from '@dcx/logger';
import { AdobeRepoAPISession } from '@dcx/repo-api-session';
import { isAnyFunction } from '@dcx/util';
import DCXBranch from '../AdobeDCXBranch';
import DCXComposite from '../AdobeDCXComposite';
import AdobeDCXPushJournal from '../AdobeDCXPushJournal';

const dbg = newDebug('dcx:dcxjs:repocompositexfer:xferctx');

/**
 * Public XferContext API.
 */
export interface AdobeXferContext {
    /**
     * Components failed during an xfer, used for pushComposite.
     * If none failed, this will be an empty array.
     */
    failedComponents: FailedComponent[];

    /**
     * Whether the xfer has been aborted.
     */
    aborted: boolean;

    /**
     * Total bytes currently transferred at a given time.
     */
    bytesTransfered: number;

    /**
     * Total estimated bytes for the xfer.
     */
    bytesTotal: number;

    /**
     * Register callback to be called after aborting xfer.
     * @param {AbortHandler} cb - The handler to be called.
     */
    onAbort: (cb: AbortHandler) => void;

    /**
     * Set a callback to be called on each progress event of the xfer.
     * Unlike onAbort, this is a single handler
     */
    onProgress: ProgressCallback | undefined;

    /**
     * Abort the xfer, canceling pending requests where possible.
     * @param {Error|string} reason - Reason for the abort, will become the error message for each canceled promise.
     */
    abort(reason?: Error | string): void;
}

export type AbortHandler = (reason?: string | Error) => void;

/**
 * R-API implementation of XferContext.
 * This implementation is private, not intended to be used directly by clients.
 *
 * Public APIs that return an XferContext-like object (for example, `AdobePromise<T, U, Context>`)
 * should use the interface `AdobeXferContext` as opposed to this class.
 *
 * Internal methods can use this to strongly-type the additional data required for the xfer operation.
 * This is done through the use of the `T` generic, which defined the type of the `_additionalData` internal property.
 */
export class XferContext<T = void> implements AdobeXferContext {
    private _aborted = false;
    private _bytesTransfered = 0;
    private _abortHandlers: AbortHandler[] = [];
    // private _progressHandlers: ProgressCallback[] = [];
    private _progressHandler?: ProgressCallback;

    /** @internal */
    _bytesTotal = 0;

    /** @internal */
    _additionalData: T = {} as T;

    /** @internal */
    _failedComponents: FailedComponent[] = [];

    /** @internal */
    _journal!: AdobeDCXPushJournal;

    /** @internal */
    _referenceBranch?: DCXBranch;

    /** @internal */
    _composite: DCXComposite;

    /** @internal */
    _session: AdobeRepoAPISession;

    /** @internal */
    _indeterminateTotalBytes?: boolean;

    /** @internal */
    _blockSize?: number;

    constructor(
        session: AdobeRepoAPISession,
        composite: AdobeDCXComposite,
        additionalData?: T,
        progressCb?: ProgressCallback,
        readonly additionalHeaders?: Record<string, string>,
        blockSize?: number,
    ) {
        this._session = session;

        this._composite = composite as DCXComposite;
        this.onProgress = progressCb;
        if (additionalData) {
            this._additionalData = additionalData;
        }
        this._blockSize = blockSize;
    }

    get aborted(): boolean {
        return this._aborted;
    }
    get bytesTransfered(): number {
        return this._bytesTransfered;
    }

    get bytesTotal(): number {
        return this._bytesTotal;
    }

    get failedComponents(): FailedComponent[] {
        return this._failedComponents;
    }
    get hasProgressHandler(): boolean {
        return this._progressHandler !== undefined;
    }
    set onProgress(cb: ProgressCallback | undefined) {
        if (isAnyFunction(cb)) {
            this._progressHandler = cb;
        }
    }
    get onProgress(): ProgressCallback | undefined {
        return this._progressHandler;
    }

    public onAbort(h: AbortHandler): void {
        this._abortHandlers.push(h);
    }

    public abort(reason?: string | Error): void {
        if (!this._aborted) {
            this._abortHandlers.forEach((c) => isAnyFunction(c) && c.call(undefined, reason));

            (
                this._session as unknown as {
                    _service: { abortAllWithToken: (token: unknown) => void };
                }
            )._service.abortAllWithToken(this);

            this._aborted = true;
        }
    }

    /**
     * Report a progress event.
     *
     * If the event has an indeterminate length, the rest of the XferContext is also indeterminate.
     * This may occur in the case of BlockUpload where the size estimate by the client was wrong,
     * and there was a call to extend the transfer. At that point a client may upload unknowable
     * amount of data.
     *
     * @internal
     *
     * @param {number} incr             - Delta of bytes sent or received
     * @param {boolean} [indeterminate] - True if the total size of the transfer can not be determined
     */
    _reportProgress(incr: number, indeterminate?: boolean) {
        dbg('progress', incr);
        if (indeterminate) {
            this._indeterminateTotalBytes = true;
        }

        if (incr > 0) {
            this._bytesTransfered += incr;
            if (isAnyFunction(this._progressHandler)) {
                try {
                    this._progressHandler(this._bytesTransfered, this._bytesTotal, this._indeterminateTotalBytes);
                } catch (error) {
                    console.error('Error in progress callback', error);
                }
            }
            // this._progressHandlers.forEach((c) => isAnyFunction(c) && c(this._bytesTransfered, this._bytesTotal));
        }
    }
}
