/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { parseLinksFromResponseHeader } from '@dcx/assets';
import { AdobeDCXBranch, AdobeDCXError, AdobeDCXRootNodeData, AdobeResponse } from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import { isObject } from '@dcx/util';
import DCXBranch from '../AdobeDCXBranch';
import { COMPOSITE_STATES } from '../enum';
import { AdobeXferContext, XferContext } from './XferContext';
import { _reportIncrementalProgress } from './common';

const dbg = newDebug('dcx:dcxjs:compositexfer:pull');

/**
 * Interface defining additional context data for pull
 * @internal
 */
export interface AdditionalDataForPull {
    versionId?: string;
    response?: AdobeResponse;
    referenceBranch?: DCXBranch;
}

/** @private */
export function _pullCompositeManifest(
    pullContext: XferContext<AdditionalDataForPull>,
): AdobePromise<DCXBranch, AdobeDCXError, AdobeXferContext> {
    dbg('_pullCompositeManifest()');

    const {
        _composite: composite,
        _session: session,
        _additionalData: { versionId, referenceBranch },
        additionalHeaders,
    } = pullContext;

    // If we've been provided with links, register them with the session and get an assetId (almost certainly fake)
    if (composite._links) {
        const { assetId, repositoryId } = session.registerLinks(
            composite._links,
            composite.repositoryId,
            composite.assetId,
        );
        composite.assetId = assetId;
        composite.repositoryId = repositoryId;
        delete composite._links;
    }
    if (typeof composite.assetId !== 'string') {
        throw new DCXError(DCXError.INVALID_STATE, 'Composite must have an asset id.');
    }

    // Initiates a manifest pull
    dbg('_pCM() pullManifest()', pullContext);

    let etag: string | undefined;
    if (referenceBranch && !versionId) {
        etag = referenceBranch.manifestEtag;
    }

    return AdobePromise.resolve<AdobeDCXBranch, AdobeDCXError, AdobeXferContext>(undefined, pullContext)
        .then(() => {
            const req = session.getCompositeManifest(composite, versionId, etag, additionalHeaders);
            if (pullContext.hasProgressHandler) {
                req.progress = _reportIncrementalProgress(pullContext._reportProgress.bind(pullContext));
            }

            return req;
        })
        .then((res) => {
            const { manifestData, manifestEtag, response } = res;
            pullContext._additionalData.response = response;

            try {
                const links = parseLinksFromResponseHeader(response);
                composite.links = links;
            } catch (_) {
                // noop
            }

            dbg('_pCM() pM() getCompositeManifest() cb', manifestEtag);

            if (manifestData === null) {
                // This implies we had an etags match and there is nothing to pull
                return;
            }

            let branch: DCXBranch;
            // No 304 response
            try {
                if (isObject(manifestData)) {
                    branch = new DCXBranch(manifestData as unknown as AdobeDCXRootNodeData);
                } else {
                    branch = new DCXBranch().parse(manifestData as string);
                }
                branch.compositeAssetId = composite.assetId;
                branch.compositeRepositoryId = composite.repositoryId;

                branch._collaborationType = composite.collaborationType;
                branch._clientDataString = composite.clientDataString;
                branch.versionId = response.headers['version'] != null ? response.headers['version'] : versionId;
                branch.manifestEtag = manifestEtag;
            } catch (e) {
                dbg('_pCM() pM() gCM() bad manifest');
                throw new DCXError(DCXError.INVALID_JSON, 'Corrupted manifest', e as Error);
            }

            if (
                branch.compositeState === COMPOSITE_STATES.committedDelete ||
                branch.compositeState === COMPOSITE_STATES.pendingDelete
            ) {
                throw new DCXError(
                    DCXError.INVALID_STATE,
                    'R-API composites should be deleted using a single step with RepoAPISession#deleteAsset',
                );
            }

            // clean up deleted and modified components
            const components = branch._core.allComponents();
            for (let i = 0; i < components.length; ++i) {
                const component = components[i];
                if (
                    component.state === COMPOSITE_STATES.pendingDelete ||
                    component.state === COMPOSITE_STATES.committedDelete
                ) {
                    // Get rid of any deleted components
                    branch.removeComponent(component);
                } else {
                    component.state = COMPOSITE_STATES.unmodified;
                }
            }

            // We are done
            return branch;
        })
        .then((branch) => {
            if (branch == null) {
                return undefined as unknown as DCXBranch;
            }

            // letious accessors will mark an unmodified manifest modified but even if the manifest
            // comes to us marked as "modified" on the server, we want to treat it as unmodified.
            branch.compositeState = COMPOSITE_STATES.unmodified;
            branch.readOnly = true;
            // And, finally, save the pulled manifest
            if (composite._options.xhrBaseBranchSupport && !versionId) {
                // make a copy of the pushed manifest data so that it can be used for a base branch
                composite._pulledBranchData = branch.localData;
            }
            return branch;
        })
        .catch((manifestError) => {
            if (manifestError.code === DCXError.NOT_FOUND) {
                throw new DCXError(
                    DCXError.NO_COMPOSITE,
                    'Composite missing or deleted',
                    manifestError.underlyingError || manifestError,
                );
            }
            throw manifestError;
        });
}
