/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    COPY_RESOURCES_RESOURCE_LIMIT,
    CopyResourcesOperationResult,
    EmbeddedMetadataMediaTypes,
    HeaderKeys,
    HTTPMethods,
    JSONPatchMediaType,
    LinkRelation,
    ManifestMediaType,
    newOperationDocBuilder,
    OperationDocumentBuilder,
    PathOrIdAssetDesignator,
    RepoResponseResult,
} from '@dcx/assets';
import {
    AdobeDCXBranch,
    AdobeDCXComponentData,
    AdobeDCXError,
    AdobeResponse,
    FailedComponent,
    InternalFunction,
    JSONPatchDocument,
    RequestDescriptor,
} from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import AdobePromise, { SettledPromise } from '@dcx/promise';
import { AdobeRepoAPISession } from '@dcx/repo-api-session';
import { generateUuid, isObject, pruneUndefined } from '@dcx/util';
import {
    appendHistoryEvent,
    initializeXMPXML,
    makeDerivedWithAction,
    makeHistoryEventXML,
    repairXMPXML,
} from '@dcx/xmp-util';
import DCXBranch from '../AdobeDCXBranch';
import { SourceAssetInfoEntry } from '../AdobeDCXBranchCore';
import DCXComponent, { AdobeDCXComponent } from '../AdobeDCXComponent';
import DCXComposite from '../AdobeDCXComposite';
import { COMPOSITE_ARCHIVAL_STATES, COMPOSITE_STATES } from '../enum';
import { DerivationType } from '../util/xmp';
import { _reportIncrementalProgress } from './common';
import { AdobeXferContext, XferContext } from './XferContext';

const dbg = newDebug('dcx:dcxjs:compositexfer:push');

const MAX_CONCURRENT_BATCHES = 4;
const MAX_OPERATIONS_PER_BATCH = 50; // Set the constrain to 50 based on https://developers.corp.adobe.com/storage-api/docs/reference/operations-copy.md

const XMP_COMPONENT_PATH = 'META-INF/metadata.xml';

export enum XMPModes {
    MANAGED,
    PARTIALLY_MANAGED,
    CLIENT_MANAGED,
    UNMANAGED,
}

type XMPConfigClientManaged = { mode: XMPModes.CLIENT_MANAGED; initialXMPXML?: string; xmpPatch?: JSONPatchDocument };
type XMPConfigManaged = {
    mode?: XMPModes.MANAGED | XMPModes.PARTIALLY_MANAGED;
    /**
     * Client app using dcx-js to push the metadata.
     * If undefined will use `dcx-js`.
     */
    creatorTool?: string;
    /**
     * Datetime string in ISO8601 format.
     */
    modifyDate?: string;
};
type XMPConfigUnmanaged = {
    mode: XMPModes.UNMANAGED;
};
export type XMPConfig = XMPConfigClientManaged | XMPConfigManaged | XMPConfigUnmanaged;

// export type alias incase we add more options in the future
export type PushCompositeOptions = XMPConfig;

export interface AdditionalDataForPush {
    compositeIsNew: boolean;
    overwriteExisting: boolean;
    owner: DCXBranch;
    validationLevel: number;
    manifestResponse?: AdobeResponse<'json'>;
    xmpConfig: Required<XMPConfigClientManaged> | Required<XMPConfigManaged> | XMPConfigUnmanaged;
    shouldPushWithXMP: boolean;
    xmpPushed: boolean;
}

interface CopyOperationCollector {
    // Doc builder associated with an individual copy operation.
    // Once it reaches 100 entries, a new doc builder will be added to the `copyOperations` array.
    docBuilder: OperationDocumentBuilder;
    // Array of tuples of DCXComponent and sourceAssetInfo for components
    // that are being copied, in the order that they are placed in the copy document,
    // and so, the order that they will return from the server.
    orderedPendingComponents: [DCXComponent, SourceAssetInfoEntry][];
}

// A result from a single batch. Since multiple batches of copy operations may occur during a
// single pushComposite() call.
// This interface represents:
// 1. the result from a batch (transformed to include DCXErrors instead of problem codes)
// 2. the response object from the HTTPService invoke
// 3. the ordered list of components and their sourceAssetInfo for only that specific batch
interface OrderedBatchCopyResult {
    results: CopyResourcesOperationResult[];
    response: AdobeResponse<'json'>;
    orderedPendingComponents: [DCXComponent, SourceAssetInfoEntry][];
}

/**
 * Internal implementation of push composite.
 *
 * @private
 *
 * @param {XferContext<AdditionalDataForPush>} pushContext  - Push context, used as `this` scope for other internals.
 * @param {DCXComposite} composite                          - Composite to push
 */
export function _pushComposite(
    pushContext: XferContext<AdditionalDataForPush>,
    composite: DCXComposite,
): AdobePromise<AdobeDCXBranch, AdobeDCXError<{ failedComponents?: FailedComponent[] }>, AdobeXferContext> {
    dbg('_pushComposite()');

    if (!composite.assetId || !composite.repositoryId) {
        throw new DCXError(
            DCXError.INVALID_STATE,
            `Composite is not bound to a cloud asset. ` +
                `createComposite() must be called to assign an asset ID and repository ID before calling push.`,
        );
    }

    const { owner: branch, compositeIsNew, shouldPushWithXMP } = pushContext._additionalData;

    // nothing to do, return immediately resolving promise
    if (_isNoOpPush(branch, compositeIsNew)) {
        return AdobePromise.resolve<AdobeDCXBranch, AdobeDCXError, AdobeXferContext>(branch, pushContext);
    }

    _assertStateIsValidForPush(branch);

    // initially the total size to push is unknown,
    // so set to indeterminate and totalBytes to 0
    pushContext._bytesTotal = 0;
    pushContext._indeterminateTotalBytes = true;

    return _pushComponents
        .call(pushContext)
        .then(_handleFailedComponents.bind(pushContext))
        .then(
            /* istanbul ignore next */
            // does not contain metadata component but should push XMP, create it
            shouldPushWithXMP && !_getMetadataComponent(branch) ? _pushMetadataComponent.bind(pushContext) : undefined,
        )
        .then(_pushManifest.bind(pushContext))
        .then(_handleManifestResponse.bind(pushContext))
        .catch(_handleError.bind(pushContext));
}

/**
 * Get metadata component.
 *
 * @private
 *
 * @param branch
 */
function _getMetadataComponent(branch: DCXBranch): DCXComponent | undefined {
    const component = branch.getComponentWithAbsolutePath(`/${XMP_COMPONENT_PATH}`);
    return component as DCXComponent;
}

/**
 * Add a component for metadata to the manifest.
 *
 * This occurs on push of a composite that doesn't contain a
 * metadata component in it's manifest.
 *
 * We have to create the component explicitly and add it to the manifest
 * since JSON-LD PUT will update the manifest on the server, but not locally;
 * meaning we are out of sync and need to pull the manifest again.
 *
 * @this {XferContext<AdditionalDataForPush>}
 */
/* istanbul ignore next */
/** letting code coverage to ignore this function because of DCX-4473 */
function _pushMetadataComponent(this: XferContext<AdditionalDataForPush>) {
    const {
        _composite: composite,
        _session: session,
        _additionalData: { xmpConfig: _xmpConfig, owner: branch, compositeIsNew },
    } = this;
    const xmpConfig = _xmpConfig as XMPConfigClientManaged | XMPConfigManaged;
    const xmpXmlStr = _buildXMPPutBody(branch, xmpConfig, compositeIsNew);
    return session
        .putCompositeComponent(
            composite,
            generateUuid(),
            xmpXmlStr,
            EmbeddedMetadataMediaTypes.XML,
            true,
            xmpXmlStr.length,
        )
        .then((res) => session.uploadResultsFromAdobeRepoUploadResult(res, composite.id))
        .then((uploadResults) => {
            // add to composite
            this._additionalData.xmpPushed = true;
            return branch.addComponentWithUploadResults(
                'xmp-metadata',
                'metadata',
                XMP_COMPONENT_PATH,
                undefined,
                uploadResults,
            );
        });
}

/**
 * Handle error that occurred during push.
 *
 * Errors pushing to deleted composites can be very messy and can look like
 * other kinds of things. So here we check to see if there's a manifest.
 * If not, then we call it a NO_COMPOSITE error. Otherwise just pass it along.
 *
 * @private
 *
 * @this XferContext<AdditionalDataForPush>
 *
 * @param {AdobeDCXError} originalError
 */
function _handleError(
    this: XferContext<AdditionalDataForPush>,
    originalError: AdobeDCXError<{ failedComponents?: FailedComponent[]; bulkResponse?: AdobeResponse[] }>,
): AdobePromise<never> {
    dbg('_handleError()', originalError);

    const {
        _session: session,
        _composite: composite,
        _additionalData: { compositeIsNew },
    } = this;

    // In some cases we want to skip checking if the composite exists:
    // 1. The composite is new (NO_COMPOSITE isn't helpful here)
    // 2. No failed components AND the code is 400 (validation issue with the manifest, may be new)
    // 3. No failed components AND the failure came from a bulk request (bulk will return correct error)
    // In those cases, pass along the original error.
    /* istanbul ignore next */
    if (
        compositeIsNew ||
        (this.failedComponents.length === 0 &&
            ((originalError.response && originalError.response.statusCode === 400) ||
                (originalError.additionalData && originalError.additionalData.bulkResponse)))
    ) {
        throw originalError;
    }

    let exists = false;
    return session
        .headCompositeManifest(composite)
        .then(() => {
            exists = true;
            throw originalError;
        })
        .catch((e) => {
            if (!exists && e.response.statusCode === 404) {
                throw new DCXError<{ failedComponents?: FailedComponent[] }>(
                    DCXError.NO_COMPOSITE,
                    'Composite does not exist; may have been deleted',
                    originalError,
                    originalError.response,
                    { failedComponents: this.failedComponents },
                );
            }

            originalError.additionalData = { failedComponents: this.failedComponents };
            throw originalError;
        });
}

/**
 * Push the manifest if the composite has been modified.
 *
 * @this XferContext<AdditionalDataForPush>
 *
 * @param {DCXBranch} branch
 */
function _pushManifest(
    this: XferContext<AdditionalDataForPush>,
): AdobePromise<AdobeResponse<'json'>, AdobeDCXError> | undefined {
    dbg('_pushManifest()');
    const branch = this._additionalData.owner;

    if (branch.compositeState !== COMPOSITE_STATES.modified) {
        return undefined;
    }

    const {
        _session: session,
        _additionalData: {
            overwriteExisting,
            validationLevel,
            xmpConfig,
            compositeIsNew,
            shouldPushWithXMP,
            xmpPushed,
        },
    } = this;

    branch.compositeState = COMPOSITE_STATES.unmodified;

    const manifestData = branch.remoteData;

    const manifestSizeEstimate = manifestData.length;
    // approximate size of eventual manifest
    // only the manifest reports progress events, as components being copied are not known upfront
    this._bytesTotal += manifestSizeEstimate;
    this._indeterminateTotalBytes = false;

    // Etag for use with if-match header
    const manifestEtag = _updateManifestEtag(this, branch);

    let p: AdobePromise<AdobeResponse<'json'>, AdobeDCXError, RequestDescriptor>;
    // unmanaged, client managed without a payload to use,
    // or xmp pushed already using XML component => update manifest alone
    if (!shouldPushWithXMP || xmpPushed) {
        p = session.updateCompositeManifest(branch, manifestData, overwriteExisting, validationLevel, manifestEtag);
        if (this.hasProgressHandler) {
            p.progress = _reportIncrementalProgress(this._reportProgress.bind(this));
        }
        return p;
    }

    /* istanbul ignore next */
    /** letting code coverage to ignore this function because of DCX-4473 */
    return _pushManifestWithXMP(
        this,
        branch,
        manifestData,
        overwriteExisting,
        validationLevel,
        manifestEtag /* manifest etag */,
        compositeIsNew,
        xmpConfig as XMPConfigManaged | XMPConfigClientManaged,
    );
}

/**
 * Push manifest using a bulk request,
 * with a second request to PATCH embedded metadata.
 *
 * 1. Determine if the composite being pushed is new
 *      if new -> using PUT with initialized XMP
 *      if not -> using PATCH
 * 2. Build Patch document or initial XMP body
 * 3. Get URL & headers for api/:embedded call
 * 4. Get URL & headers for api/:manifest call
 * 5. Issue bulk request
 * (6a). Handle PUT api/:manifest failure
 *      -> if 409/412, retry tweaking if-match
 *      -> if 424, fix embedded PATCH if possible
 * (6b). Handle PATCH api/:embedded failure
 *      -> if 404, retry bulk with PUT
 *      -> if 424, fix manifest PUT if possible
 *      -> if 400, GET api/:embedded as JSON-LD, try to repair, redo with PUT
 *      -> otherwise, fail (or continue with only pushing manifest without embedded update)
 */
/* istanbul ignore next */
/** letting code coverage to ignore this function because of DCX-4473 */
function _pushManifestWithXMP(
    ctx: XferContext<AdditionalDataForPush>,
    branch: DCXBranch,
    manifestData: string,
    overwriteExisting: boolean,
    validationLevel: number,
    manifestEtag: string | undefined,
    compositeIsNew: boolean,
    xmpConfig: XMPConfigManaged | XMPConfigClientManaged,
    retryCount = 0,
): AdobePromise<AdobeResponse<'json'>> {
    const { _session: session, _composite: composite } = ctx;

    // get PATCH
    const patchDoc = _buildXMPPatchDoc(branch, xmpConfig);

    // Get URL & headers for api/:embedded call
    const embeddedUrlPromise = session.getLinkHrefForAsset(composite, LinkRelation.EMBEDDED_METADATA);
    const embeddedHeaders: Record<string, string> = pruneUndefined({
        [HeaderKeys.IF_MATCH]: '*',
        [HeaderKeys.CONTENT_TYPE]: JSONPatchMediaType,
    });

    // Get URL & headers for api/:manifest call
    const manifestUrlPromise = session.getLinkHrefForAsset(composite, LinkRelation.MANIFEST);
    const manifestHeaders: Record<string, string> = pruneUndefined({
        // [HeaderKeys.IF_MATCH]: '*' as string,
        [HeaderKeys.IF_MATCH]: manifestEtag as string,
        [HeaderKeys.CONTENT_TYPE]: `${ManifestMediaType};validation-level=${validationLevel}`,
    });

    // Issue bulk request
    return AdobePromise.allSettled([manifestUrlPromise, embeddedUrlPromise])
        .then(([manifestRes, embeddedRes]) => {
            /* istanbul ignore if */
            if (embeddedRes.status === 'rejected' || manifestRes.status === 'rejected') {
                // This should never happen in practice.
                // Both links should already exist in the composite or cache before reaching
                // this function, but we keep the check anyways.
                throw new DCXError(DCXError.INVALID_STATE, 'Could not retrieve required links.');
            }

            return session.performBulkRequest(composite, [
                // manifest req
                {
                    method: HTTPMethods.PUT,
                    href: manifestRes.value,
                    headers: manifestHeaders,
                    body: manifestData,
                },
                // embedded req
                {
                    method: HTTPMethods.PATCH,
                    href: embeddedRes.value,
                    headers: embeddedHeaders,
                    body: JSON.stringify(patchDoc),
                },
            ]);
        })
        .then((res) =>
            /* istanbul ignore next */
            /** letting code coverage to ignore this function because of DCX-4473 */
            _handleManifestAndXMPResponse(
                ctx,
                branch,
                manifestData,
                overwriteExisting,
                validationLevel,
                manifestEtag,
                compositeIsNew,
                xmpConfig,
                res,
                retryCount,
            ),
        );
}

/**
 * This function attempts to repair the XMP metadata
 * of a composite that is both bound and contains a
 * metadata component.
 *
 * Repairing the XMP metadata could be done in two ways:
 * 1. By using JSON-LD and embedded APIs
 * 2. By pulling the component, parsing/repairing XML, pushing
 * the component and updating the component in the manifest.
 *
 * In the case of some bound composites, the only way to repair
 * is through method #2, since PUTs against the embedded APIs will
 * always fail if the component is broken.
 */
/* istanbul ignore next */
function _repairXMPComponent(
    ctx: XferContext<AdditionalDataForPush>,
    component: DCXComponent,
    xmpConfig: XMPConfigManaged,
): AdobePromise<string | undefined> {
    const {
        _session: session,
        _composite: composite,
        _additionalData: { owner: branch },
    } = ctx;

    return session
        .getCompositeComponent(composite, component.id, component.version, 'text')
        .then(({ response: xmpStr }) => {
            const repaired = repairXMPXML(xmpStr, xmpConfig.creatorTool, xmpConfig.modifyDate);
            if (xmpStr !== repaired) {
                return session.putCompositeComponent(
                    composite,
                    component.id,
                    repaired,
                    EmbeddedMetadataMediaTypes.XML,
                    false,
                    repaired.length,
                );
            }
        })
        .then((res) => res && session.uploadResultsFromAdobeRepoUploadResult(res, composite.id))
        .then((uploadResults) => {
            if (!uploadResults) {
                return;
            }
            branch.updateComponentWithUploadResults(component, uploadResults);
            return uploadResults.records[component.id].etag;
        });
}

/**
 * Handle bulk responses.
 * Since bulk will return a 200, we need to check each response for failure codes.
 * And since we're using version sub requests, if one request fails, both will.
 * See: {@link https://git.corp.adobe.com/pages/caf/api-spec/single.html#version-sub-requests | Version Sub-requests}
 *
 * @this {XferContext<AdditionalDataForPush>}
 *
 * @param bulkResponse
 * @returns The manifest response
 */
/* istanbul ignore next */
/** letting code coverage to ignore this function because of DCX-4473 */
export function _handleManifestAndXMPResponse(
    ctx: XferContext<AdditionalDataForPush>,
    branch: DCXBranch,
    manifestData: string,
    overwrite: boolean,
    validationLevel: number,
    manifestEtag: string | undefined,
    compositeIsNew: boolean,
    xmpConfig: XMPConfigManaged | XMPConfigClientManaged,
    bulkResponse: RepoResponseResult<AdobeResponse[]>,
    retryCount = 0,
): AdobePromise<AdobeResponse<'json'>> | AdobeResponse<'json'> {
    dbg('_handleManifestAndXMPResponse()');
    const [manifestRes, embeddedRes] = bulkResponse.result;
    const { statusCode: manifestCode } = manifestRes;
    const { statusCode: embeddedCode } = embeddedRes;
    dbg('_hMAXR() manifest, embedded status: ', manifestCode, embeddedCode);

    if (retryCount > 2 && (manifestCode >= 400 || embeddedCode >= 400)) {
        // Break out of any loops.
        // This shouldn't happen, but we circuit break just in case.
        // Possible ways it could happen:
        // 1. XMP is corrupted (ie. a history sequence of the wrong type),
        //    repairing will "succeed" but continue to give 400s on PATCH.
        // 2. Server failures
        // ...
        // The 3 request limit is chosen because there could realistically be 3 retries
        // in a valid request:
        // 1. PUT api/:manifest overwrite=true and doesn't exist yet, if-match: "*" -> 412, retry with no if-match
        // 2. PATCH api/:embedded with broken XMP content -> 400, repair XMP and retry
        // 3. Should succeed
        dbg('_hMAXR() circuit break retries');
        throw new DCXError(
            DCXError.INVALID_STATE,
            'Repairing failed or repeated failures during XMP/manifest push. Check additionalData.',
            undefined,
            bulkResponse[0],
            { bulkResponse },
        );
    }

    // Handle failures
    // It's possible either request fails.
    let newOverwrite = overwrite;
    let newCompositeIsNew = compositeIsNew;
    let newManifestEtag = manifestEtag;

    function retry(data = manifestData) {
        dbg('_hMAXR() retry bulk; isNew, etag, overwrite: ', newCompositeIsNew, newManifestEtag, newOverwrite);
        /* istanbul ignore next */
        return _pushManifestWithXMP(
            ctx,
            branch,
            data,
            newOverwrite,
            validationLevel,
            newManifestEtag,
            newCompositeIsNew,
            xmpConfig,
            retryCount + 1,
        );
    }

    if (manifestCode >= 400) {
        if (manifestCode === 404) {
            throw new DCXError(DCXError.NO_COMPOSITE, 'Composite not found.', undefined, manifestRes, {
                bulkResponse,
            });
        } else if (manifestCode === 424) {
            // only embedded request failed, nothing to change here
        } else if (overwrite && manifestCode === 409) {
            throw new DCXError(DCXError.UPDATE_CONFLICT, 'Manifest has been changed', undefined, manifestRes, {
                bulkResponse,
            });
        } else if (overwrite && manifestCode === 412) {
            // NOTE: This logic pulls and repairs a component by parsing the existing XMP XML
            // A bug may come about where a composite enters a corrupted state that includes a
            // metadata component, but that metadata component has a mismatched version/etag.
            // It seems that the XMP PATCH in a bulk request uses the etag of the metadata
            // component's etag (from the manifest), so this fails with a 412 on the manifest PUT.

            // const metaComponent = _getMetadataComponent(branch);
            // if (!isRetry && (!compositeIsNew || branch._derivationType === DerivationType.COPY) && metaComponent) {
            // Already bound or copied and metadata component exists in manifest.
            // Try to parse/repair XMP as XML.
            // return _repairXMPComponent(ctx, metaComponent, xmpConfig as any).then(retry);
            // }
            throw new DCXError(DCXError.PRECONDITION_FAILED, 'Precondition failed', undefined, manifestRes, {
                bulkResponse,
            });
        } else if (manifestCode === 409) {
            newOverwrite = false;
        } else if (manifestCode === 412) {
            newOverwrite = false;
            newManifestEtag = undefined;
        } else {
            throw new DCXError(
                DCXError.UNEXPECTED_RESPONSE,
                'Unexpected response from manifest PUT in bulk.',
                undefined,
                manifestRes,
                { bulkResponse },
            );
        }
    }

    if (embeddedCode >= 400) {
        if (embeddedCode === 424) {
            // only know that precondition failed, so retry without changing embedded request
        } else if (embeddedCode === 404) {
            // try with PUT
            newCompositeIsNew = true;
        } else if (embeddedCode === 400) {
            // pull and repair metadata component
            const metaComponent = _getMetadataComponent(branch);
            if (metaComponent) {
                // repair component, retry with the new manifest data
                return _repairXMPComponent(ctx, metaComponent, xmpConfig as XMPConfigManaged).then(() =>
                    retry(branch.remoteData),
                );
            }
            throw new DCXError(DCXError.INVALID_STATE, 'Manifest appears to be corrupted.', undefined, manifestRes, {
                bulkResponse,
            });
        } else {
            throw new DCXError(
                DCXError.UNEXPECTED_RESPONSE,
                'Unexpected response from embedded request in bulk.',
                undefined,
                manifestRes,
                { bulkResponse },
            );
        }

        // Retry the request.
        // This should always be entered even if PUT manifest was the
        // request that failed, since in that case the sub request for
        // embedded should fail (424) as well.

        //Getting test coverage of this retry logic should be attempted again once
        //https://jira.corp.adobe.com/browse/DCX-4473 is fixed
        return retry();
    }

    // We have 2 successful codes
    // Update the etag and version of the metadata component

    // TODO: Remove this workaround (see: https://jira.corp.adobe.com/browse/DCX-4473)
    // Currently there's a bug in how bulk requests are performed with
    // XMP Patches. The response headers contain the manifest etag and version,
    // and the content-md5 header is missing. Updating the manifest with these
    // values leads to a manifest where the metadata component is tracking the manifest
    // etag/version, which likely doesn't exist, and results in a 412 on the next push.
    // The 412 is caused by a second bug: the XMP Patch should be using a wildcard etag
    // but instead it is using the metadata component's etag from the manifest.
    // // NOTE: This workaround is unfortunate, since it adds and entire pull for each push, but
    // there's no way around it without significant changes to the push logic and public API.
    // (ie. rewriting without bulk requests and JSON Patch & including full XMP parsing, using
    // the same logic as dcx-cpp/java)
    // return _reloadManifestWorkaround(ctx);

    // NOTE: This is workaround option #2.
    // By updating the version manually, and using wildcard in the manifest push
    // we can avoid an extra call. The downside is that the manifest data will be out
    // of date; the metadata component contains the old etag/length. We're also
    // relying on the version being incremented by exactly one.
    const metaComponent = _getMetadataComponent(branch) as DCXComponent;
    ctx._journal.recordUploadedComponent(
        metaComponent.id,
        '*' /* etag, using * as the etag forces wildcard update on future push */,
        `${parseInt(metaComponent.version as string) + 1}` /* version, manually incremented */,
        embeddedRes.headers['content-md5'] || (metaComponent.md5 as string),
        metaComponent.length as number,
    );

    // TODO: Uncomment this (correct) way to update the manifest
    // from the bulk XMP response. (see: https://jira.corp.adobe.com/browse/DCX-4473)
    // const metaComponent = _getMetadataComponent(branch) as DCXComponent;
    // ctx._journal.recordUploadedComponent(
    //     metaComponent.id,
    //     embeddedRes.headers['etag'],
    //     embeddedRes.headers['version'],
    //     embeddedRes.headers['content-md5'] || (metaComponent.md5 as string),
    //     metaComponent.length as number
    // );

    return manifestRes;
}

/**
 * TODO: Remove this function when the workaround is definitely no longer needed.
 *
 * See: https://jira.corp.adobe.com/browse/DCX-4473
 *
 * @param ctx
 * @returns
 */
/* istanbul ignore next */
function _reloadManifestWorkaround(ctx: XferContext<AdditionalDataForPush>) {
    const {
        _session,
        _composite,
        _journal,
        _additionalData: { owner: branch },
    } = ctx;
    return _session.getCompositeManifest(_composite).then(({ manifestData, response }) => {
        const metaComponent = _getMetadataComponent(branch);
        if (!metaComponent) {
            return response;
        }

        const newComponents = (manifestData.components as AdobeDCXComponentData[]) || [];
        const len = newComponents.length;
        let newMetaCompData: AdobeDCXComponentData = metaComponent._data;
        for (let i = 0; i < len; i++) {
            const c = newComponents[i];
            if (c.path === XMP_COMPONENT_PATH) {
                newMetaCompData = c;
                break;
            }
        }

        _journal.recordUploadedComponent(
            metaComponent.id,
            newMetaCompData.etag as string,
            newMetaCompData.version as string,
            newMetaCompData.md5 || (metaComponent.md5 as string),
            newMetaCompData.length || (metaComponent.length as number),
        );

        return response;
    });
}

/**
 * Create an XMP body in RDF+XML format.
 *
 * @param xmpConfig
 * @param type
 * @returns
 */
/* istanbul ignore next */
/** letting code coverage to ignore this function because of DCX-4473 */
function _buildXMPPutBody(
    branch: DCXBranch,
    xmpConfig: XMPConfigClientManaged | XMPConfigManaged,
    compositeIsNew: boolean,
): string {
    if (xmpConfig.mode === XMPModes.CLIENT_MANAGED && xmpConfig.initialXMPXML) {
        return xmpConfig.initialXMPXML;
    }

    const { creatorTool, modifyDate } = xmpConfig as XMPConfigManaged;
    const docId = generateUuid();

    // let derivedFromData: string | undefined = undefined;
    const historyEvents: string[] = [];
    if (branch._derivationType === DerivationType.COPY) {
        // Composite was copied, but the source composite didn't have XMP metadata.
        const copyDateTime = branch._derivationDatetime as string;

        // TODO:? add DerivedFrom data, even though it doesn't contain any source data
        // derivedFromData = makeDerivedWithXML(docId, docId, docId, copyDateTime);

        // Add copied event
        const evt = makeHistoryEventXML(creatorTool, 'copied', docId, copyDateTime);
        historyEvents.push(evt);
    } else if (compositeIsNew) {
        // New composite, created event
        const evt = makeHistoryEventXML(creatorTool, 'created', docId, modifyDate);
        historyEvents.push(evt);
    }

    if (!compositeIsNew || branch._derivationType === DerivationType.COPY) {
        // Existing or copied.
        // Since copying a composite in the online operational model does not
        // immediately create a derivedFrom event, it's possible changes are made to
        // the copied composite. For that reason, we add a saved event to the history
        // as well as (and after) the copied event.
        const evt = makeHistoryEventXML(creatorTool, 'saved', docId, modifyDate);
        historyEvents.push(evt);
    }

    return initializeXMPXML(creatorTool, docId, modifyDate, historyEvents, undefined /** derivedFromData */);
}

/**
 * Get or build XMP patch document.
 * If in client managed mode, the xmpPatch object has already been checked and exists.
 *
 * @param branch
 * @param xmpConfig
 *
 * @returns The JSON Patch document for XMP update
 */
function _buildXMPPatchDoc(branch: DCXBranch, xmpConfig: XMPConfigClientManaged | XMPConfigManaged): JSONPatchDocument {
    if (xmpConfig.mode === XMPModes.CLIENT_MANAGED) {
        return xmpConfig.xmpPatch as JSONPatchDocument;
    }

    // Otherwise, it's partially managed or managed mode.
    // Either way, at this point the xmpConfig object has both
    // creatorTool and modifyDate defined.
    const { creatorTool, modifyDate } = xmpConfig as XMPConfigManaged;
    const patchDocument: JSONPatchDocument = [];

    if (branch._derivationType === DerivationType.COPY) {
        // Composite was copied, metadata component should have been copied already.
        // Add DerivedFrom and add a copied event.
        makeDerivedWithAction.call({ patchDocument }, creatorTool, 'copied', undefined, modifyDate);
    }
    // Add a saved event.
    // Since copying a composite in the online operational model does not
    // immediately create a derivedFrom event, it's possible changes are made to
    // the copied composite. For that reason, we add a saved event to the history
    // as well as (and after) the copied event.
    appendHistoryEvent.call({ patchDocument }, 'saved', creatorTool, modifyDate);
    return patchDocument;
}

/**
 * Parse response data from manifest push.
 * Update journal and composite.
 *
 * @this XferContext<AdditionalDataForPush>
 *
 * @param {AdobeResponse} response
 */
function _handleManifestResponse(
    this: XferContext<AdditionalDataForPush>,
    responseOrUndef: AdobeResponse<'json'> | undefined,
) {
    if (typeof responseOrUndef === 'undefined') {
        // This means branch was unmodified and no push occurred.
        // in practice this should never happen.
        return this._additionalData.owner;
    }

    this._additionalData.manifestResponse = responseOrUndef;
    const {
        _journal: journal,
        _composite: composite,
        _additionalData: { owner: branch },
    } = this;

    const etag = responseOrUndef.headers.etag;
    const versionId = responseOrUndef.headers['version'];
    const originalChangeCount = branch.changeCount;

    journal.manifestEtag = etag;
    journal.derivationType = DerivationType.NONE;
    journal.currentBranchEtag = branch.manifestEtag;
    journal.changeCount = originalChangeCount;
    journal.versionId = versionId;

    journal.applyToBranch(branch);
    branch.compositeState = COMPOSITE_STATES.unmodified;

    /* istanbul ignore if */
    if (composite._options.xhrBaseBranchSupport) {
        // make a copy of the pushed manifest data so that it can be used for a base branch
        composite._pushedBranchData = branch.localData;
    }

    return branch;
}

/**
 * Check for failed components, providing a more meaningful error if possible.
 *
 * @this XferContext<AdditionalDataForPush>
 *
 * @param {FailedComponent[]} failedComponents
 */
function _handleFailedComponents(this: XferContext<AdditionalDataForPush>, failedComponents: FailedComponent[]) {
    dbg('_handleFailedComponents()');

    // Pipe branch through if there are no errors
    if (failedComponents.length === 0) {
        return this._additionalData.owner;
    }

    // If there's an exceeds quota error code in any of the failed components
    // use that as the more specific error.
    const exceedsQuotaErrors = failedComponents.filter((fc) => fc.error && fc.error.code === DCXError.EXCEEDS_QUOTA);
    if (exceedsQuotaErrors.length > 0) {
        throw new DCXError<{ failedComponents: FailedComponent[] }>(
            DCXError.EXCEEDS_QUOTA,
            'One or more components failed to upload.',
            undefined,
            undefined,
            { failedComponents },
        );
    }

    // Otherwise, throw a generic error.
    throw new DCXError<{ failedComponents: FailedComponent[] }>(
        DCXError.COMPONENT_UPLOAD_ERROR,
        'One or more components failed to upload.',
        undefined,
        undefined,
        { failedComponents },
    );
}

/**
 * Return true if the composite is all 3 of:
 * 1. not new
 * 2. not modified
 * 3. not pending archival
 *
 * There is no push to perform in this case.
 *
 * @private
 *
 * @param {DCXBranch} branch
 * @param {boolean} compositeIsNew
 * @param {boolean} discardWhenDone
 * @returns {boolean}
 */
function _isNoOpPush(branch: DCXBranch, compositeIsNew: boolean): boolean {
    dbg('_isNoOpPush()');

    if (!compositeIsNew && branch.compositeState === COMPOSITE_STATES.unmodified) {
        return true;
    }
    return false;
}

/**
 * Validate current branch state.
 * Throw an error if the current state clashes with the push operation.
 *
 * @throws {AdobeDCXError}
 *
 * @param {DCXBranch} branch
 * @returns {void}
 */
function _assertStateIsValidForPush(branch: DCXBranch): void {
    dbg('_assertStateIsValidForPush()');

    if (branch.compositeState === COMPOSITE_STATES.committedDelete) {
        throw new DCXError(DCXError.DELETED_COMPOSITE, 'Attempt to push a deleted composite');
    }

    if (branch.compositeState === COMPOSITE_STATES.pendingDelete) {
        throw new DCXError(
            DCXError.INVALID_STATE,
            'R-API composites should be deleted using a single step with RepoAPISession#deleteAsset',
        );
    }

    if (
        branch.compositeArchivalState === COMPOSITE_ARCHIVAL_STATES.pending ||
        branch.compositeArchivalState === COMPOSITE_ARCHIVAL_STATES.archived
    ) {
        throw new DCXError(DCXError.INVALID_STATE, 'R-API composites cannot be archived');
    }
}

/**
 * Get the current copy op collector in the array.
 * If the doc builder has reached it's limit, create a new one and push to the end.
 *
 * @param {CopyOperationCollector[]} copyOpCollectors
 *
 * @this {XferContext<AdditionalDataForPush>}
 */
function _getCurrentCopyOpCollector(
    this: XferContext<AdditionalDataForPush>,
    copyOpCollectors: CopyOperationCollector[],
): CopyOperationCollector {
    dbg('_getCurrentCopyOpCollector()');

    if (
        copyOpCollectors.length === 0 ||
        copyOpCollectors[copyOpCollectors.length - 1].docBuilder.entryCount >=
            _getMaxOperationsPerBatch(this as XferContext<never>)
    ) {
        dbg('_gCCOC() new copy op collector');

        copyOpCollectors.push({
            docBuilder: newOperationDocBuilder(),
            orderedPendingComponents: [],
        });
    }

    return copyOpCollectors[copyOpCollectors.length - 1];
}

/**
 * Conditionally add a copy operation for a component to component copy to the operation doc builder.
 * Cleans up the branch when encountering new & deleted/pending delete components.
 * Cleans up the journal when encountering non-new & unmodified components.
 *
 * Return a promise that resolves when the operation is added.
 *
 * If no operation is required for the component
 * (already copied, errored, pending delete) then this returns undefined.
 *
 * @this XferContext<AdditionalDataForPush>
 *
 * @param {OperationDocumentBuilder} docBuilder                                     - Operation document builder to use
 * @param {[AdobeDCXComponent, SourceAssetInfoEntry][]} orderedPendingComponents    - List of tuples representing the component and their source info.
 *                                                                                  This is added to whenever a component is added to the copy document.
 *                                                                                  The order of this array represents the order of the copy document,
 *                                                                                  and therefore, the order of the responses from the server.
 * @param {AdobeDCXComponent} component                                             - Component to possibly add to the copy document
 */
function _reduceComponent(
    this: XferContext<AdditionalDataForPush>,
    copyOpCollectors: CopyOperationCollector[],
    component: AdobeDCXComponent,
): void {
    dbg('_reduceComponent()');

    const {
        _journal: journal,
        _composite: composite,
        _additionalData: { owner: branch },
    } = this;
    const { state: componentState = COMPOSITE_STATES.unmodified, etag } = component;

    const componentIsNew = etag == null;

    if (!componentIsNew && componentState === COMPOSITE_STATES.unmodified) {
        dbg('_rC() not new, not modified');

        journal.removeRecordForUploadedComponent(component.id);
        return;
    }

    // Clean up components from manifests that assumed the old component deletion model
    if (
        !componentIsNew &&
        (componentState === COMPOSITE_STATES.committedDelete || componentState === COMPOSITE_STATES.pendingDelete)
    ) {
        dbg('_rC() not new, committedDelete or pendingDelete');

        branch.removeComponent(component);
        return;
    }

    const sourceAssetInfo = branch._core._getSourceAssetInfoOfComponent(component);

    dbg('_rC() source asset', sourceAssetInfo);

    // Unexpected state, source asset info should be defined.
    // Continue with the other copies, including the failed component in the ultimate rejection.
    if (sourceAssetInfo == null) {
        this._failedComponents.push({
            component: component,
            error: new DCXError(
                DCXError.INVALID_STATE,
                'Component should not be modified or new without local storage',
            ),
        });
        return;
    }

    const journalRecord = journal.getRecordForUploadedComponent(component.id);
    if (sourceAssetInfo && journalRecord) {
        const journalSrcAssetInfo = journalRecord['source-asset-info'];
        if (
            journalSrcAssetInfo &&
            journalSrcAssetInfo.compositeAssetId === sourceAssetInfo.compositeAssetId &&
            journalSrcAssetInfo.componentId === sourceAssetInfo.componentId &&
            journalSrcAssetInfo.componentVersion === sourceAssetInfo.componentVersion
        ) {
            // Already uploaded, no need to copy.
            // TODO: possibly? extend journal entry to mark time of upload
            // 7 days valid, 30 if copied
            // also timestamp on component in composite DOM
            component.etag = journalRecord.etag;
            component.version = journalRecord.version;
            component.md5 = journalRecord.md5;
            component.length = journalRecord.length;
            component.state = COMPOSITE_STATES.unmodified;
            return;
        }
    }

    const currentOpCollector = _getCurrentCopyOpCollector.call(this, copyOpCollectors);

    // Once both complete, add document to doc builder.
    currentOpCollector.docBuilder.copyResources(
        {
            assetId: sourceAssetInfo.compositeAssetId,
            repositoryId: sourceAssetInfo.repositoryId,
        } as PathOrIdAssetDesignator,
        composite,
        [
            {
                source: {
                    component_id: sourceAssetInfo.componentId,
                    revision: sourceAssetInfo.componentVersion,
                    reltype: LinkRelation.COMPONENT,
                },
                target: {
                    component_id: component.id,
                    reltype: LinkRelation.COMPONENT,
                },
            },
        ],
        true,
    );

    // At this point, we're going to attempt a copy of the component,
    // and it has it's order in the document, so add to ordered pending components.
    currentOpCollector.orderedPendingComponents.push([component, sourceAssetInfo]);

    return;
}

/**
 * Process a set of batch copy results from a settled (either rejected or resolved) promise.
 * If an error is encountered, add that component to failedComponents.
 * If the entire promise was rejected, throw immediately.
 *
 * @this {XferContext<AdditionalDataForPush>}
 *
 * @param {SettledPromise<OrderedBatchCopyResult, AdobeDCXError>[]} results - Settled promises from batch copy.
 */
function _handleBatchResults(
    this: XferContext<AdditionalDataForPush>,
    results: SettledPromise<OrderedBatchCopyResult, AdobeDCXError>[],
) {
    dbg('_handleBatchResults(): ', results);

    const { _journal: journal, _failedComponents: failedComponents } = this;

    results.map((r) => {
        if (r.status === 'rejected') {
            dbg('_hBR() error', r.reason);
            throw r.reason;
        }

        const { orderedPendingComponents, results: batchResults, response } = r.value;
        batchResults.forEach(({ resources }) => {
            resources.forEach((result) => {
                const matchingComponent = orderedPendingComponents.find(
                    ([, source]) => source.componentId === result.target.component_id,
                );
                if (matchingComponent) {
                    const [component, sourceAssetInfo] = matchingComponent;
                    try {
                        const { etag, id, version, md5, length } = _validateCopyResponse(
                            result.target,
                            response,
                            component,
                        );
                        journal.recordUploadedComponent(id, etag, version, md5, length, sourceAssetInfo);
                    } catch (error) /* istanbul ignore next */ {
                        failedComponents.push({
                            component,
                            error: error as AdobeDCXError<unknown>,
                        });
                    }
                }
            });
        });
    });
}

/**
 * Issue parallel batch copy operations for each set of components.
 * Limited to 100 per batch, 3 batches at a time, as defined by the generator.
 *
 * @this {XferContext<AdditionalDataForPush>}
 *
 * @param {AdobeRepoAPISession} session - Repo session to use.
 * @param {Generator} copyOpCollectors - Generator yielding batches of ordered copy documents.
 */
function _parallelBatchOperations(
    this: XferContext<AdditionalDataForPush>,
    session: AdobeRepoAPISession,
    copyOpGenerator: ReturnType<typeof _makeCopyOpGenerator>,
): AdobePromise<FailedComponent[], AdobeDCXError> {
    dbg('_parallelBatchOperations()');

    const { value, done } = copyOpGenerator.next();

    dbg('_pBO() next generator: ', value, done);

    return AdobePromise.resolve<CopyOperationCollector[], AdobeDCXError>(value)
        .then((copyOpCollectors) => {
            const ps = copyOpCollectors
                .filter((collector) => collector.orderedPendingComponents.length)
                .map((collector) => {
                    // For each batch of copy operations, get the document,
                    // convert to string upfront to find the size estimate,
                    // and add that to the bytesTotal. The total bytes are set as
                    // indeterminate until all batch operations are issued,
                    // at which point the indeterminate flag is switched to false.
                    const document = collector.docBuilder.getDocument();
                    const docString = JSON.stringify(document);
                    this._bytesTotal += docString.length;

                    const req = session.performBatchOperation(docString).then(({ result: results, response }) => {
                        return {
                            results: results as CopyResourcesOperationResult[],
                            response,
                            orderedPendingComponents: collector.orderedPendingComponents,
                        };
                    });
                    if (this.hasProgressHandler) {
                        // attach progress handler for each copy op
                        req.progress = _reportIncrementalProgress(this._reportProgress.bind(this));
                    }

                    return req;
                });
            return AdobePromise.allSettled<OrderedBatchCopyResult, AdobeDCXError>(ps);
        })
        .then(_handleBatchResults.bind(this))
        .then(() => {
            if (done) {
                return this.failedComponents;
            }

            // recurse
            return _parallelBatchOperations.call(this, session, copyOpGenerator);
        });
}

/**
 * Make a generator that yields a limited number of batched copy operations for parallel requests.
 *
 * @this {XferContext<AdditionalDataForPush>}
 *
 * @param {DCXComponent[]} components - Components to process, possibly adding to a copy document.
 */
function* _makeCopyOpGenerator(
    this: XferContext<AdditionalDataForPush>,
    components: DCXComponent[],
): Generator<CopyOperationCollector[], CopyOperationCollector[]> {
    dbg('_makeCopyOpGenerator()');
    // Array of objects that each contain a document builder and the ordered components
    // involved in that copy. Once one of the document builders reaches 100 copies (platform batch limit),
    // create a new document builder and append to this array (see: _getCurrentCopyOpCollector()).
    let copyOperationCollectors: CopyOperationCollector[] = [];
    for (let i = 0; i < components.length; i++) {
        const component = components[i];
        _reduceComponent.call(this, copyOperationCollectors, component);
        const currentOpCollector = _getCurrentCopyOpCollector.call(this, copyOperationCollectors);
        if (currentOpCollector.orderedPendingComponents.length === COPY_RESOURCES_RESOURCE_LIMIT) {
            // yield the current batch and start a new one
            yield [currentOpCollector];
            copyOperationCollectors = [];
        }
    }

    return copyOperationCollectors;
}

/**
 * Push components.
 * For each component: if state is modified, and we have valid source asset data, add it to an operation document.
 * Then send the operation, adding failed components to the context's `_failedComponent` property.
 *
 * If no components need pushing, resolves immediately.
 *
 * @note
 * It's important for this method to always return the correct AdobePromise signature.
 * The source object must always be set to the pushContext, since this is the first promise in the chain.
 * Later "thenned" promises may return an undefined source object, in which case the source object is propagated.
 *
 * @private
 *
 * @this XferContext<AdditionalDataForPush>
 */
function _pushComponents(
    this: XferContext<AdditionalDataForPush>,
): AdobePromise<FailedComponent[], AdobeDCXError, AdobeXferContext> {
    dbg('_pushComponents()');

    const {
        _session: session,
        _additionalData: { owner: branch },
    } = this;

    const components = branch.allComponents();
    dbg('_pC() component count: ', components.length);
    const copyOpGenerator = _makeCopyOpGenerator.call(this, components as DCXComponent[]);

    return AdobePromise.resolve<void, AdobeDCXError, XferContext<AdditionalDataForPush>>(undefined, this).then(() =>
        _parallelBatchOperations.call(this, session, copyOpGenerator),
    );
}

/**
 * Clean an asset from a copy response, returning the required key/values.
 * If the result contains an error, throw it.
 * If the result isn't valid, throw an error.
 * If one of the required keys doesn't exist, throw an error.
 *
 * @throws {AdobeDCXError}
 *
 * @private
 *
 * @param {AdobeOperationResult} result - Operation result for a single asset
 * @param {AdobeResponse<'json'>} response - Overall response from the copy operation
 */
function _validateCopyResponse(
    result: CopyResourcesOperationResult['resources'][number]['target'],
    response: AdobeResponse<'json'>,
    component: AdobeDCXComponent,
): AdobeDCXComponent & { etag: string; version: string; md5: string; length: number } {
    dbg('_validateCopyResponse()');

    if (!isObject(result)) {
        throw new DCXError(DCXError.INVALID_DATA, 'Invalid result', undefined, response);
    }

    const cleaned = {
        etag: result.etag!,
        version: result.revision,
        md5: component.md5 as string,
        length: component.length!,
        state: COMPOSITE_STATES.unmodified,
    };
    for (const key in cleaned) {
        if (cleaned[key] == null) {
            throw new DCXError(DCXError.INVALID_DATA, `No ${key}`, undefined, response);
        }
    }
    return Object.assign(component, cleaned);
}

/**
 * Sets the manifestEtag on the branch if needed and returns the etag
 * to use for any calls to `updateCompositeManifest()`.
 *
 * If the composite has been pushed successfully before, we need to
 * use the etag from that push. *Unless* the client has pulled/resolved
 * a new version that has a newer etag.
 *
 * @private
 *
 * @param {XferContext<AdditionalDataForPush>} pushContext
 * @param {DCXBranch} branch
 */
function _updateManifestEtag(pushContext: XferContext<AdditionalDataForPush>, branch: DCXBranch) {
    dbg('_updateManifestEtag()');

    // journal manifestEtag being set means a push has occurred before.
    if (pushContext._journal.manifestEtag) {
        if (pushContext._journal.currentBranchEtag === branch.manifestEtag) {
            dbg('_uME() updating etag, previous: ', branch.manifestEtag);

            branch.manifestEtag = pushContext._journal.manifestEtag;
        }
    }

    dbg('_uME() using if-match: ', branch.manifestEtag);

    return branch.manifestEtag;
}

// eslint-disable-next-line camelcase
function _getMaxOperationsPerBatch(ctx: XferContext<{ _test_operationsPerBatch?: number }>): number {
    return ctx._additionalData._test_operationsPerBatch || MAX_OPERATIONS_PER_BATCH;
}

// eslint-disable-next-line camelcase
function _getMaxConcurrentBatches(ctx: XferContext<{ _test_maxConcurrentBatches?: number }>): number {
    return ctx._additionalData._test_maxConcurrentBatches || MAX_CONCURRENT_BATCHES;
}

/**
 * Internals for testing
 *
 * @private
 * @internal
 */
export interface InternalTypes {
    _isNoOpPush: InternalFunction<typeof _isNoOpPush>;
    _handleFailedComponents: InternalFunction<typeof _handleFailedComponents>;
    _pushComponents: InternalFunction<typeof _pushComponents>;
    _reduceComponent: InternalFunction<typeof _reduceComponent>;
    _assertStateIsValidForPush: InternalFunction<typeof _assertStateIsValidForPush>;
    _updateManifestEtag: InternalFunction<typeof _updateManifestEtag>;
    _validateCopyResponse: InternalFunction<typeof _validateCopyResponse>;
    _getCurrentCopyOpCollector: InternalFunction<typeof _getCurrentCopyOpCollector>;
    _makeCopyOpGenerator: InternalFunction<typeof _makeCopyOpGenerator>;
    _parallelBatchOperations: InternalFunction<typeof _parallelBatchOperations>;
    _handleBatchResults: InternalFunction<typeof _handleBatchResults>;
    _pushComposite: InternalFunction<typeof _pushComposite>;
    _handleError: InternalFunction<typeof _handleError>;
    _pushManifest: InternalFunction<typeof _pushManifest>;
    _handleManifestAndXMPResponse: InternalFunction<typeof _handleManifestAndXMPResponse>;
}

let _internals: InternalTypes = undefined as unknown as InternalTypes;
/* istanbul ignore next */
if (process.env.NODE_ENV === 'development') {
    _internals = {
        _isNoOpPush,
        _handleFailedComponents,
        _pushComponents,
        _reduceComponent,
        _assertStateIsValidForPush,
        _updateManifestEtag,
        _validateCopyResponse,
        _getCurrentCopyOpCollector,
        _makeCopyOpGenerator,
        _parallelBatchOperations,
        _handleBatchResults,
        _pushComposite,
        _handleError,
        _pushManifest,
        _handleManifestAndXMPResponse,
    };
}

/**
 * @private
 * @internal
 **/
export const internals = _internals;
