/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    AdobeBlockUpload,
    AdobeDCXError,
    AdobeRepoUploadResult,
    AdobeResponse,
    AdobeUploadRecord,
    AdobeUploadResults,
    GetSliceCallback,
    InternalFunction,
} from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import { createUploadRecord, createUploadResults } from '@dcx/repo-api-session';
import { validateObject, validateParams } from '@dcx/util';
import { AdobeXferContext, XferContext } from './XferContext';
import { _reportIncrementalProgress } from './common';

const dbg = newDebug('dcx:dcxjs:compositexfer:upload');

/**
 * Interface defining additional context data for upload
 * @internal
 */
export interface AdditionalDataForUpload {
    componentType: string;
    componentIsNew: boolean;
    componentId: string;
    size?: number;
    md5?: string;
    response?: AdobeResponse;
}

/**
 * Internal implementation of upload component.
 *
 * @internal
 *
 * @param {XferContext<AdditionalDataForUpload>} pushContext    - Push context, used as `this` scope for other internals.
 * @param {any} data                                            - Data or slicer function to upload
 */
export function _uploadComponent(
    uploadContext: XferContext<AdditionalDataForUpload>,
    dataOrSliceCallback: Buffer | ArrayBuffer | Blob | string | GetSliceCallback,
): AdobePromise<AdobeUploadResults, AdobeDCXError, AdobeXferContext & { blockUpload?: AdobeBlockUpload | undefined }> {
    dbg('_uploadComponent()');

    const {
        additionalHeaders,
        _additionalData: { componentId, componentType, componentIsNew, size, md5 },
        _composite: composite,
        _session: session,
        _blockSize: blockSize,
    } = uploadContext;

    // validate composite has required properties
    validateObject(composite, 'composite', ['id', 'string'], ['repositoryId', 'string'], ['assetId', 'string']);

    // validate other params
    validateParams(
        ['session', session, 'object'],
        ['dataOrSliceCallback', dataOrSliceCallback, ['object', 'function', 'string']],
        ['componentType', componentType, 'string'],
        ['componentId', componentId, 'string', true],
        ['size', size, '+number', true],
        ['md5', md5, 'string', true],
    );

    const { id: compositeId, repositoryId } = composite;

    // Prepare the upload
    const dataLength =
        size ||
        (dataOrSliceCallback as Buffer).length ||
        (dataOrSliceCallback as ArrayBuffer).byteLength ||
        (dataOrSliceCallback as Blob).size;
    uploadContext._bytesTotal = dataLength;

    // Upload the component asset
    return AdobePromise.resolve<void, AdobeDCXError, XferContext<AdditionalDataForUpload>>(undefined, uploadContext)
        .then(() =>
            session.putCompositeComponent(
                composite,
                componentId,
                dataOrSliceCallback,
                componentType,
                componentIsNew,
                dataLength,
                md5,
                uploadContext.hasProgressHandler
                    ? _reportIncrementalProgress(uploadContext._reportProgress.bind(uploadContext))
                    : undefined,
                additionalHeaders,
                blockSize,
            ),
        )
        .then(({ result, response }) => {
            uploadContext._additionalData.response = response;
            const results = createUploadResults(compositeId, composite.assetId as string, repositoryId);
            results.records[componentId] = _validateUploadResults(componentId, dataLength, componentType, result);
            return results;
        });
}

/**
 * Clean an asset from an upload results, returning the required key/values.
 * If a required property is missing, throw an error.
 *
 * @throws {AdobeDCXError}
 *
 * @private
 *
 * @returns {UploadRecord}
 */
export function _validateUploadResults(
    componentId: string,
    pLength: number,
    type: string,
    result: Partial<AdobeRepoUploadResult['result']>,
): AdobeUploadRecord {
    if (typeof result !== 'object') {
        throw new DCXError(DCXError.UNEXPECTED_RESPONSE, 'Invalid upload result, expecting object');
    }

    const { md5, etag, revision, length = pLength } = result;

    if (length == null) {
        throw new DCXError(DCXError.UNEXPECTED_RESPONSE, 'No length');
    }
    if (etag == null) {
        throw new DCXError(DCXError.UNEXPECTED_RESPONSE, 'No etag');
    }
    if (revision == null) {
        throw new DCXError(DCXError.UNEXPECTED_RESPONSE, 'No version');
    }
    if (md5 == null) {
        throw new DCXError(DCXError.UNEXPECTED_RESPONSE, 'No md5');
    }
    return createUploadRecord(componentId, etag, revision, md5, length, type);
}

/**
 * Internal types for testing
 *
 * @internal
 * @private
 */
export interface InternalTypes {
    _validateUploadResults: InternalFunction<typeof _validateUploadResults>;
}

/* istanbul ignore next */
let _internals: InternalTypes = undefined as unknown as InternalTypes;
if (process.env.NODE_ENV === 'development') {
    _internals = {
        _validateUploadResults,
    };
}

/**
 * @internal
 * @private
 */
export const internals = _internals;
