/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
import { AdobeDCXBranch, AdobeDCXComponent, AdobeDCXComponentData, AdobeDCXElement, AdobeDCXError, AdobeDCXNode, AdobeDCXNodeData, AdobeDCXRootNodeData, AdobeUploadResults, ComponentCallback, LinkRelationKey, LinkSet, NodeCallback } from '@dcx/common-types';
import DCXBranch from './AdobeDCXBranch';
import { AdobeDCXComponent as DCXComponent } from './AdobeDCXComponent';
import DCXElement from './AdobeDCXElement';
import { AdobeDCXLocalNodeData, AdobeDCXNode as DCXNode } from './AdobeDCXNode';
export interface SourceAssetInfoEntry {
    componentPath: string;
    compositeAssetId: string;
    componentId: string;
    componentVersion: string;
    repositoryId?: string;
}
/**
 * @class
 * @hideconstructor
 * Internal helper class. Implements the core functionality shared by both AdobeDCXBranch and AdobeDCXElement.
 * @constructor
 * @private
 * @param {Object}  [data]     private
 * @param {Object}  [owner]    The branch or the element.
 * @param {Boolean} [readOnly] private
 */
export declare class AdobeDCXBranchCore {
    /** @internal */
    _owner: DCXBranch | DCXElement;
    private _allNodes;
    private _allComponents;
    private _absolutePaths;
    /** @internal */
    _readOnly: boolean;
    /** @internal */
    _data: AdobeDCXNodeData & AdobeDCXLocalNodeData;
    /** @internal */
    _isDirty: boolean;
    /** @internal */
    _compositeAssetId?: string;
    /** @internal */
    _compositeRepositoryId?: string;
    /** @internal */
    _compositeLinks?: LinkSet;
    /**
     * @internal
     * Dictionary of componentId to component/asset info.
     * Set during copy.
     */
    _sourceAssetInfoLookup: Record<string, SourceAssetInfoEntry>;
    /**
     * @internal
     * Source composite info.
     * Set during copy.
     */
    _sourceCompositeInfo: {
        links: LinkSet;
        assetId: string;
        repositoryId: string;
    } | undefined;
    constructor(data: AdobeDCXNodeData | AdobeDCXRootNodeData, owner: AdobeDCXBranch | AdobeDCXElement, readOnly?: boolean);
    /**
     * @internal
     */
    get allNodes(): Record<string, DCXNode>;
    /**
     * An AdobeDCXNode object that represents the root node of the branch or element.
     * @type {AdobeDCXNode}
     * @internal
     * @readonly
     */
    get rootNode(): DCXNode;
    set rootNode(_: DCXNode);
    /**
     * Whether the branch or element has has been modified in memory.
     * @internal
     * @readonly
     * @type {Boolean}
     */
    get isDirty(): boolean;
    set isDirty(_: boolean);
    /**
     * @internal
     */
    get changeCount(): number;
    set changeCount(_: number);
    /**
     * Returns the node with the given id or undefined if not found.
     * @param   {String } id The id of the child node to look up.
     * @returns {AdobeDCXNode}
     */
    getChildWithId(id: string): DCXNode | undefined;
    /**
     * Returns the node with the given absolute path or undefined if not found.
     * @param   {String} path The absolute path.
     * @returns {AdobeDCXNode}
     */
    getChildWithAbsolutePath(path: string): DCXNode | undefined;
    /**
     * Generates and returns an array of the child nodes of the given parent node.
     * @example
     * var rootNodes = branch.getChildrenOf(branch.rootNode);
     * @param   {AdobeDCXNode} parentNode The parent node to return the children for.
     * @returns {Array}
     */
    getChildrenOf(pParentNode?: AdobeDCXNode): AdobeDCXNode[];
    /**
     * Creates a new node and inserts it into the children list of the given parent node or of the
     * root if no parent node is given.
     * Returns the new child node.
     * @param   {String}  [name]       The name of the new child. If undefined the child will not
     *                                 have a name.
     * @param   {String}  [nodeId]     The id of the new child. If undefined the new child node will
     *                                 get a random id.
     * @param   {Integer} [index]      If given and less than or equal to the current number of
     *                                 children than the node gets inserted at the given index.
     *                                 Otherwise it gets added to the end.
     * @param   {String}  [parentNode] The parent node to add the node to. Default parent is the
     *                                 root node.
     * @returns {AdobeDCXNode}
     */
    addChild(name?: string, nodeId?: string, index?: number, parentNode?: AdobeDCXNode): DCXNode;
    /**
     * Removes and returns the given child node from the branch.
     * @param   {AdobeDCXNode} node The child node to remove.
     * @returns {AdobeDCXNode}
     */
    removeChild(pNode: AdobeDCXNode): DCXNode;
    /**
     * Moves the existing child from its current parent/index to the given parent/index.
     * @param   {AdobeDCXNode} node         The child node to move
     * @param   {Integer}      index        If given and less than or equal to the current number of
     *                                      children than the node gets inserted at the given index.
     *                                      Otherwise it gets added to the end.
     * @param   {AdobeDCXNode} [parentNode] The parent node to move the node to. Default parent is
     *                                      the root.
     * @returns {AdobeDCXNode}
     */
    moveChild(pNode: AdobeDCXNode, index?: number | null, pParentNode?: AdobeDCXNode | null): AdobeDCXNode;
    /**
     * Copies the given child node as a new node into this branch. The node can be from the same or
     * from a different composite.
     *
     * <p>This function will try reuse the ids of any children and components of the copied node,
     * in order to minimize the amount of data that will later have to be uploaded, however, clients
     * must not rely on these ids being preserved in the copied objects.</p>
     *
     * <p>Fails if a node with the same id or same absolute path already exists.</p>
     *
     * <p>Notice: This method does not work without local storage (e.g. browser environment) if
     * used to copy between two composites stored at different endpoints.</p>
     ** @param   {AdobeDCXNode} node         The child node to copy. If it is the root node then
     *                                      newPath must be provided.
     * @param   {AdobeDCXNode} [parentNode] The parent node to copy the child node to. If undefined
     *                                      then the new child node will be added to the root of the
     *                                      branch.
     * @param   {Integer}      [index]      If provided and less than or equal to the current number of
     *                                      children of the parentNode (or root) the child node gets
     *                                      inserted at the given index. Otherwise it gets added to
     *                                      the end.
     * @param   {String}       [newPath]    <p>If provided, the copy of the child node will be assigned
     *                                      this a its path property and it will also receive a new
     *                                      random id (unless one is provided with the newId param).
     *                                      If left undefined then the copy of the node will keep
     *                                      the path of the original. In either case the function will
     *                                      fail if the resulting absolute path of the child or any
     *                                      of its children/components conflicts with an already
     *                                      existing absolute path.</p>
     *                                      <p>You must provide a newPath if you are copying the root
     *                                      node of a branch or element.</p>
     * @param   {String}       [newId]      If provided, the copy of the child node will be assigned
     *                                      this a its id. If left undefined (and the newPath param
     *                                      is also undefined) then the copy will retain the id of
     *                                      the original. In either case the function will
     *                                      fail if the resulting id of the child or any
     *                                      of its children/components conflicts with an already
     *                                      existing id.
     * @param   {NodeCallback} [callback]   Optional when not copying between different composites or
     *                                      when not using local storage.
     *                                      Gets called when the copy is done or has failed.
     * @returns                             The copied Node if no callback specified.
     */
    copyChild(node: AdobeDCXNode, parentNode?: AdobeDCXNode, index?: number, newPath?: string, newId?: string): AdobeDCXNode;
    copyChild(node: AdobeDCXNode, parentNode: AdobeDCXNode | undefined, index: number | undefined, newPath: string | undefined, newId: string | undefined, callback: NodeCallback): void;
    /**
     * <p>Replaces the child node in this branch with a copy of the given node or branch with the same
     * id (or, if provided, with the given id). Fails if the child node does not exist in this branch.</p>
     *
     * <p>This function will try reuse the ids of any children and components of the copied node,
     * in order to minimize the amount of data that will later have to be uploaded, however, clients
     * must not rely on these ids being preserved in the copied objects.</p>
     *
     * <p>Notice: This method does not work without local storage (e.g. browser environment) if
     * used to copy between two different composites.</p>
     *
     * @param   {AdobeDCXNode} node       The child node to update from.
     * @param   {String}       [newPath]  If provided the copy of the component will be assigned
     *                                    this a its path property. Otherwise it will retain its original path.
     * @param   {String}       [newId]    If provided the copy of the child node will be assigned
     *                                    this as its id. Otherwise it will retain its original id.
     * @param   {NodeCallback} [callback]   Optional when not copying between different composites or
     *                                    when not using local storage.
     *                                    Gets called when the copy is done or has failed.
     * @returns                           The replaced Node, or undefined if callback is
     *                                    given.
     */
    replaceChild(node: AdobeDCXNode, newPath?: string, newId?: string): AdobeDCXNode;
    replaceChild(node: AdobeDCXNode, newPath: string | undefined, newId: string | undefined, callback: NodeCallback): void;
    /**
     * Given an INCOMPLETE_COMPOSITE DCXError, this will attempt to return the invalid components so they can be removed or updated
     * @param error An IMCOMPLETE_COMPOSITE error
     * @returns Array of components from this branch that are declared missing in the INCOMPLETE_COMPOSITE error report
     */
    getMissingComponentsFromError(error: AdobeDCXError): AdobeDCXComponent[];
    /**
     * Returns an array of all components in the branch.
     * @returns {Array}.
     */
    allComponents(): DCXComponent[];
    /**
     * Returns the component with the given id or undefined if not found.
     * @param   {String} id The id of the component to look up.
     * @returns {AdobeDCXComponent}
     */
    getComponentWithId(id: string): DCXComponent | undefined;
    /**
     * Returns the component with the given absolute path or undefined if not found.
     * @param   {String} path The absolute path of the desired component.
     * @returns {AdobeDCXComponent}
     */
    getComponentWithAbsolutePath(path: string): DCXComponent | undefined;
    /**
     * Returns an array containing the components of the given node.
     * @param   {AdobeDCXNode} parentNode The node whose components to return.
     * @returns {Array}
     */
    getComponentsOf(pParentNode?: AdobeDCXNode): DCXComponent[];
    /**
     * Gets passed into addComponent(), updateComponent(), copyComponent(), replaceComponent() and
     * called back whenever the operation has finished.
     * @callback ComponentCallback
     *    @param {Error}                error
     *    @param {AdobeDCXComponent}    component
     */
    /**
     * <p>Creates and adds a component to the given parent node or to the root if no parent node is
     * given.</p>
     *
     * @param componentDescriptor The serialized component descriptor to use.
     * @param name          The name of the new component.
     * @param path          The path of the new component. Must satisfy uniquenes rules for components.
     * @param relationship  The relationship of the new component.
     * @param parentNode  The node to add the node to. Defaults to the root.
     * @returns {AdobeDCXComponent}               The new component.
     */
    addComponentWithComponentDescriptor(componentDescriptor: string, name: string, path: string, relationship?: LinkRelationKey, parentNode?: AdobeDCXNode): DCXComponent;
    /**
     * <p>Creates and adds a component to the given parent node or to the root if no parent node is
     * given.</p>
     *
     * @param   {String}            name          The name of the new component.
     * @param   {String}            relationship  The relationship of the new component.
     * @param   {String}            path          The path of the new component. Must satisfy uniquenes
     *                                            rules for components.
     * @param   {AdobeDCXNode}      [parentNode]  The node to add the node to. Defaults to the root.
     * @param   {Object}            uploadResults The upload results object returned by a previous call
     *                                            to AdobeDCXCompositeXfer.uploadAssetForComponent().
     * @returns {AdobeDCXComponent}               The new component.
     */
    addComponentWithUploadResults(name: string, relationship: string, path: string, parentNode: AdobeDCXNode | undefined, uploadResults: AdobeUploadResults): DCXComponent;
    /**
     * <p>Updates the component record with the results of a recent upload of said component.</p>
     *
     * @param   {AdobeDCXComponent} component     The component.
     * @param   {Object}            uploadResults The upload results object returned by a previous
     *                                            call to AdobeDCXCompositeXfer.uploadAssetForComponent().
     * @returns {AdobeDCXComponent} The updated component.
     */
    updateComponentWithUploadResults(component: DCXComponent, uploadResults: AdobeUploadResults): DCXComponent;
    /**
     * Removes the component from the branch.
     * @param   {AdobeDCXComponent} component The component to remove.
     * @returns {AdobeDCXComponent} The removed component.
     */
    removeComponent(component: DCXComponent): DCXComponent;
    /**
     * Moves the component to the given node or the root if node is undefined
     * @param   {AdobeDCXComponent} component    The component to move.
     * @param   {AdobeDCXNode}      [parentNode] The node to move the component to.
     * @returns {AdobeDCXComponent} The moved component.
     */
    moveComponent(component: DCXComponent, parentNode?: DCXNode): DCXComponent;
    /**
     * Copies the given component and adds it as a new component to this branch. Fails if the
     * component already exists.
     *
     * @param   {AdobeDCXComponent} component    The component to copy.
     * @param   {AdobeDCXNode}      [parentNode] The node to copy the component to. If none is
     *                                           provided then the component will be added to the
     *                                           root.
     * @param   {String}            [newPath]    If provided the copy of the component will be
     *                                           assigned this a its path property and it will also
     *                                           get assigned a new id if none is provided via the
     *                                           <em>newId</em> param.
     * @param   {String}            [newId]      If provided the copy of the component will be
     *                                           assigned this a its id property.
     * @param   {ComponentCallback} [callback]   Optional when not copying between different
     *                                           composites. Gets called when the copy is done or
     *                                           has failed.
     * @returns {AdobeDCXComponent}              Only returns the new component when called without
     *                                           a callback.
     */
    copyComponent(component: DCXComponent, parentNode?: DCXNode, newPath?: string, newId?: string, callback?: any): DCXComponent | undefined;
    /**
     * Replaces the matching component (same id) in this branch with a copy of the given component.
     * Fails if the component can't be found.
     *
     * @param   {AdobeDCXComponent} component  The component to copy.
     * @param   {String}            [newPath]  If provided the copy of the component will be
     *                                         assigned this a its path property and it will also
     *                                         get assigned a new id if none is provided via the
     *                                         newId param.
     * @param   {String}            [newId]    If provided the copy of the component will be
     *                                         assigned this a its id property.
     * @param   {ComponentCallback} [callback] Optional when not copying between different
     *                                         composites. Gets called when the copy is done or has
     *                                         failed.
     * @returns {AdobeDCXComponent}            Only returns the new component when called without
     *                                         a callback.
     */
    replaceComponent(component: DCXComponent, newPath?: string, newId?: string, callback?: ComponentCallback): DCXComponent | undefined;
    /**
     * Gets passed into assetOfComponent() and called back whenever the operation has finished.
     * @callback FilePathCallback
     *    @param {Error}     error
     *    @param {String}    filePath
     */
    /**
     * @private
     */
    private _getHrefOfComponent;
    /**
     * Sets or clears the source asset info of the given component.
     * @internal
     * @param {Object}            sourceAssetInfo The asset and component ids to be used for S2SC
     * @param {AdobeDCXComponent} component  The component.
     * @param {Object}            [local]    Optional: The local data object to use.
     */
    _setSourceAssetInfoOfComponent(sourceAssetInfo: SourceAssetInfoEntry | undefined, component: DCXComponent | AdobeDCXComponentData): void;
    /**
     * Returns the source href of the given component if one has previously been recorded.
     * @internal
     * @param   {AdobeDCXComponent} component The component.
     * @param {Object}              [local]   Optional: The local data object to use.
     * @returns {Object|undefined}  The source asset info or undefined.
     */
    _getSourceAssetInfoOfComponent(component: DCXComponent | AdobeDCXComponentData): SourceAssetInfoEntry | undefined;
    /**
     * Copies over any source asset info from the given branch core. Overwrites or clears any
     * existing info.
     * @internal
     * @param {AdobeDCXBranchCore} otherCore The core to copy from
     */
    _copySourceHrefsFrom(otherCore: AdobeDCXBranchCore): void;
    /**
     * @private
     */
    private _getBranchOf;
    /**
     * Returns true if the given core is of the same composite.
     * @private
     * @param   {AdobeDCXBranchCore} otherCore The other core.
     * @returns {Boolean}            Whether the composites match.
     */
    private _isSameComposite;
    /**
     * Returns a stringified representation of the branch that can be used to persist
     * and later restore it.
     * @internal
     * @param   {Boolean} stripOutLocalData Whether to exclude the local data node.
     * @param   {Boolean} pretty            Whether to pretty print the JSON output.
     * @returns {String}
     */
    _stringify(stripOutLocalData: boolean, pretty?: boolean): string;
    /**
     * Verifies and sets the new data and creates the caches. Throws if invalid data.
     * @private
     * @param   {Object}         data Data
     * @returns {AdobeDCXBranch} The branch.
     */
    private _setData;
    /**
     * Recursively removes the given node and all its sub nodes and components from the caches.
     * @private
     * @param {Object} thisNodeData Data
     */
    private _removeNodeFromCachesRecursively;
    /**
     * Returns the local node, creating it if necessary.
     * @internal
     * @returns {Object}
     */
    _local(): Required<AdobeDCXLocalNodeData>['local'];
    /**
     * Recursively resets components.
     * @internal
     * @param {AdobeDCXNodeData} node
     */
    _recursiveReset(node: AdobeDCXNodeData, componentCallback?: (component: AdobeDCXComponentData) => unknown): void;
    /**
     * Figures out whether a node with id nodeId is a sub node of nodeData.
     * @private
     * @param   {String}  nodeId
     * @param   {Object}  nodeData
     * @returns {Boolean}
     */
    private _nodeIdIsDescendantOf;
    /**
     * Returns an array with all components of the given node and its descendants.
     * @internal
     * @param   {Object} node               The node.
     * @param   {Object} [allComponents=[]] USed for recursion. Leave undefined.
     * @returns {Object}
     */
    _recursivelyGetAllComponentsOfChild(node: DCXNode | AdobeDCXNodeData, allComponents?: DCXComponent[]): DCXComponent[];
    /**
     * Returns the node data of the parent node of the given node.
     * @private
     * @param   {Object}   node
     * @param   {Object}   [currentNodeData=this._data] Used for recursion. Don't set.
     * @returns {Object}     found an object of the form { parentNodeData: dataObject, index: integer }. Otherwise null.
     */
    private _nodeDataOfParentOfNode;
    /**
     * Returns the node data of the parent node of the given component.
     * @private
     * @param   {Object}   component
     * @param   {Object} [currentNodeData=this._data] Used for recursion. Don't set.
     * @returns {Object} found an object of the form { parentNodeData: dataObject, index: integer }. Otherwise null.
     */
    private _nodeDataOfParentOfComponent;
    /**
     * Collects all the updates necessary to update the absolute paths of all sub nodes and components.
     * @private
     * @param {Object} currentNodeData
     * @param {String} currentAbsolutePath
     * @param {Array}  updates
     */
    private _determineAbsolutePathChangesRecursively;
    /**
     * Updates the path of a node. Caller must dirty the branch.
     *
     * @throws {AdobeDCXError}
     * @internal
     * @param {Object} node
     * @param {String} newPath
     */
    _setPathOfNode(node: DCXNode, newPath?: string): void;
    /**
     * Updates the path of a component. Caller must dirty the branch.
     *
     * @throws {AdobeDCXError}
     *
     * @internal
     * @param {AdobeDCXComponent} component
     * @param {String} newPath
     */
    _setPathOfComponent(component: DCXComponent, newPath: string): void;
    /**
     * Sets the dirty flag and (optionally) the compositeState
     *
     * @throws {AdobeDCXError}
     *
     * @internal
     * @param {Boolean} [preserveCompositeState]
     */
    _setDirty(preserveCompositeState?: boolean): void;
    /**
     * Returns a normalized and lowercased copy of the absolute path of item.
     * @private
     * @param   {Object}   item
     * @param   {Object} newParent Optional. If given this parent will be used to determine the absolute path.
     * @returns {string}
     */
    private _normalizedAbsolutePathForItem;
    /**
     * @private
     */
    private _hasSameEndpoint;
    /**
     * <p>Deep copies the given component and adds it as a new component to this branch.</p>
     *
     * <p>This function gets called in various different situations and thus needs to work correctly or
     * fail with proper errors in the possible permutations of:</p>
     *
     * <p>- locality: whether the source and target are from the same branch, different branches of the
     * same composite, different composites or even composites with different endpoints.</p>
     * <p>- identity: whether the node should be a new node or replace (update) an existing one.</p>
     * <p>- state: whether the component is unmodified, or modified with a source href.</p>
     *
     * <p>After some initial verification and preparation the function first starts any necessary
     * asynchronous operations (i.e. the file copy). If none are necessary or once they all have
     * succeeded it calls _copyComponentModel to do the actual model changes, passing in a callback that
     * gets called for each component (one in this case) and that takes care of udating local storage
     * and/or source href for a later server-to-server copy request.</p>
     *
     * @private
     */
    private _copyComponent;
    /**
     * Copies the given component from a different branch of the same composite into this branch.
     * Notice that this method and its call to copyComponentCallback are synchronous. If you need to
     * do anything asynchronous you will need to do that before (preferably) or after calling this
     * method.
     *
     * Throws if it runs into an error.
     * @private
     * @param   {AdobeDCXComponent} component               The component to copy.
     * @param   {AdobeDCXNode}      [parentNode]            Optional: The node to copy the component to.
     *                                                      If none if provided then the component will
     *                                                      be added to the root.
     * @param   {Boolean}           [replaceExisting]       Optional: Default is false. Whether to
     *                                                      replace an existing component.
     * @param   {String}            [newPath]               Optional: If provided the copy of the
     *                                                      component will be assigned this a its
     *                                                      path property and it will also get
     *                                                      assigned a new id.
     * @param   {String}            [newId]                 Optional: If provided the copy of the
     *                                                      component will be assigned this as its id.
     * @param   {ComponentCallback} [copyComponentCallback] Optional. Gets called for each component
     *                                                      that gets copied. Caller can use this to
     *                                                      tweak the component before it gets added
     *                                                      to the DOM.
     *                                                      Signature: function (originalComponent,
     *                                                      copiedCoponent, localDataOfTargetBranch)
     * @returns {AdobeDCXComponent}                         The new component
     */
    private _copyComponentModel;
    /**
     * <p>Deep copies the given child node and its nodes and components into this branch as a new child.
     * The node can be from the same or from a different composite.
     * If it is a root node then you must specify newPath since it can't keep its original path.</p>
     *
     * <p>This function gets called in various different situations and thus needs to work correctly or
     * fail with proper errors in the various possible permutations:</p>
     *
     * <p>- whether the source and target are from the same branch, different branches of the
     * same composite, different composites or even composites with different endpoints.</p>
     * <p>- whether the node should be a new node or replace (update) an existing one.</p>
     * <p>- whether the components are unmodified, or modified with a source href.</p>
     *
     * <p>After some initial verification and preparation the function first starts any necessary
     * asynchronous operations (i.e. file copies). If none are necessary or once they all have
     * succeeded it calls _copyChildModel to do the actual model changes, passing in a callback that
     * gets called for each component and that takes care of udating local storage and/or source hrefs
     * for a later server-to-server copy request.</p>
     *
     * <p>The function creates copies of the lookup tables and manipulates those during the copy
     * operation. Only once everything has succeeded it inserts the new node (and removes the one to replace)
     * and replaces the lookup tables with its copies so that it can bail out at any time before that
     * without having actually modified the DOM.</p>
     *
     * @private
     */
    private _copyChild;
    /**
     * Copies the given child node with all its nodes and components. Notice that this method and
     * its call to copyComponentCallback are synchronous. If you need to do anything asynchronous you
     * will need to do that before (preferably) or after calling this method.
     *
     * Throws if it runs into an error.
     * @private
     * @param   {AdobeDCXNode} node                  The child node (or whole branch) to copy.
     * @param   {AdobeDCXNode} parentNode            The parent node to copy the child node to.
     *                                               Gets ignored if replaceExisting is true.
     * @param   {Integer}      index                 Optional: If given and less than or equal to the
     *                                               current number of children of the parentNode the
     *                                               node gets inserted at the given index. Otherwise
     *                                               it gets added to the end. Gets ignored if
     *                                               replaceExisting is true.
     * @param   {Boolean}      replaceExisting       Optional: Default is false.
     * @param   {String}       newPath               Optional: If provided the copy of the component
     *                                               will be assigned this a its path property.
     * @param   {String}       newId                 Optional: If provided the copy of the component
     *                                               will be assigned this as its id.
     * @param   {Boolean}      reuseIds              If true we try to reuse ids.
     * @param   {NodeCallback} copyComponentCallback Optional. Gets called for each component that
     *                                               gets copied. Caller can use this to tweak the
     *                                               component before it gets added to the DOM.
     *                                               Should throw if something goes wrong.
     *                                               Signature: function (originalComponent,
     *                                               copiedCoponent, localDataOfTargetBranch)
     * @returns {AdobeDCXNode}          new child node.
     */
    private _copyChildModel;
    /**
     * Verifies the integrity of the in-memory structures of the branch.
     * Looks for incorrect caches/lookup tables, incorrect
     * object references and cycles/duplicate objects.
     * Also, optionally, verifies that all asset files exist locally.
     *
     * @internal
     * @param   {Boolean}  shouldBeComplete Whether to check for the existence of all component assets.
     *                                      @warning This check is using synchronous file system calls.
     * @param   {Function} [logger]         A function that gets called for every error found.
     *                                      Signature: function (string)
     * @param   {Object}   [fs]             The file system object to use in the shouldBeComplete check
     * @returns {Array}    Array of errors or null if everything is OK.
     */
    _verifyIntegrity(shouldBeComplete?: boolean, logger?: (log?: string) => unknown, fs?: any): AdobeDCXError[] | null;
}
export default AdobeDCXBranchCore;
//# sourceMappingURL=AdobeDCXBranchCore.d.ts.map