/// <reference types="node" />
import { AdobeDCXError, AdobeResponse, AdobeSessionBase, Link } from '@dcx/common-types';
import { AdobeHTTPService, AdobeResponseType, RequestDescriptor, ResponseCallback } from '@dcx/http';
import AdobePromise from '@dcx/promise';
export declare const SYNC_ASYNC_DEFAULT_DELAY = 5;
export declare const ASYNC_DEFAULT_DELAY = 1;
export declare const DEFAULT_POLL_DELAY = 10;
interface RenditionTemplate {
    uri: string;
}
/**
 * Contains cached content location and link data previously retrieved by issuing
 * a head request.
 * @typedef {Object} CachedAssetInfo
 *   @property {String}  primaryTemplate  Primary ID-based URI template to the composite in aggregate
 *   @property {String}  manifestTemplate URI template that can be expanded to locate the asset's manifest
 *   @property {String}  componentTemplate URI template that can be expanded to locate the asset's components
 *   @property {String}  versionHistory
 */
export interface CachedAssetInfo {
    primaryTemplate: string;
    manifestTemplate: string;
    componentTemplate: string;
    versionHistory: string;
    renditionTemplates: RenditionTemplate[];
}
/**
 * Gets passed into `getCachedAssetInfo()`.
 * @callback GetCachedAssetInfoCallback
 *   @param {Error}             error       If not undefined the request has failed.
 *   @param {CachedAssetInfo}   assetInfo
 */
interface GetCachedAssetInfoCallback {
    (error?: undefined, info?: CachedAssetInfo): any;
    (error?: AdobeDCXError, info?: undefined): any;
}
/**
 * @class
 * @classdesc Base class for the different sessions.
 * @param httpService the HTTP service
 * @param server      the url for the storage server
 */
export declare class SessionBase implements AdobeSessionBase {
    protected _server: string;
    _endPoint?: string;
    protected _maxRedirects: number;
    protected _authenticationAllowList: string[];
    protected _assetInfoCache: any;
    protected _assetIdResolutionTemplate: any;
    protected readonly SYNC_ASYNC_DEFAULT_DELAY = 5;
    protected readonly ASYNC_DEFAULT_DELAY = 1;
    protected readonly DEFAULT_POLL_DELAY = 10;
    /** @internal */
    _service: AdobeHTTPService;
    constructor(httpService: AdobeHTTPService, server: string);
    /**
     * The maximum number of redirects the session will follow *if* the underlying service
     * doesn't handle redirects. Gets ignored for services that handle redirects (e.g. XHR).
     * Set this to 0 if the session should not follow redirects. Default value: 5
     * @memberof AdobeSessionBase#
     * @type Integer
     */
    get maxRedirects(): number;
    set maxRedirects(value: number);
    /**
     * A list of domain names that should get sent the auth credentials of the underlying service.
     * @memberof AdobeSessionBase#
     * @type Array
     */
    get authenticationAllowList(): string[];
    set authenticationAllowList(value: string[]);
    /**
     * Given an href string this method returns true if it is either relative or it refers
     * to the same endpoint as the session's server.
     * @param {String} href The href to check
     */
    isValidHref(href: string): boolean;
    isDomainOnAllowList(href: string): boolean;
    /**
     * Gets an asset from the service.
     * @param {String}          href                The full URL of the asset to retrieve.
     * @param {String|Object}   [etagOrHeaders]     If a string, the etag to use, if an object a set of additional headers, if undefined. ignored
     * @param {Function}        callback            Called back when the asset has been fetched. Signature: `callback(error, httpResponse)`.
     * @return {Object}         The http requestDescriptor.
     */
    getAsset(href: string, etagOrHeaders: string | Record<string, string>, responseType: AdobeResponseType, callback: ResponseCallback): RequestDescriptor;
    /**
     * @private
     * @param {Object}          reuseRequestDesc    A RequestDescriptor object to reuse for additional HTTP requests
     */
    protected _getAsset(href: string, etagOrHeaders: string | Record<string, string> | undefined, responseType: AdobeResponseType, reuseRequestDesc: RequestDescriptor | undefined, callback: ResponseCallback): RequestDescriptor;
    /**
     * Get asset data in specified responseType result
     * @param {String}          href                The full URL of the asset to retrieve.
     * @param {String}          responseType        The desired response type.
     * @param {String|Object}   [etagOrHeaders]     If a string, the etag to use, if an object a set of additional headers, if undefined. ignored
     * @param {Function}        callback            Called back when the asset has been fetched. Signature: `callback(error, buffer, etag, response)`. The buffer is "null" if the etags match.
     *
     * @return               the requestDescriptor for the request.
     */
    getAssetAsType(href: string, responseType: AdobeResponseType, etagOrHeaders: string | Record<string, string>, callback: (error?: AdobeDCXError, buffer?: Buffer | null, etag?: string, response?: AdobeResponse) => void): RequestDescriptor;
    private _cacheLinksFromResponse;
    /**
     * @private
     * @param {Object}          reuseRequestDesc    A RequestDescriptor object to reuse for additional HTTP requests
     */
    _getAssetAsType(href: string, responseType: AdobeResponseType, etagOrHeaders: string | Record<string, string> | undefined, reuseRequestDesc: RequestDescriptor | undefined, assetId: string | undefined, versionId: string | undefined, callback: (error?: AdobeDCXError, buffer?: Buffer | null, etag?: string, response?: AdobeResponse) => void): RequestDescriptor;
    _getResourcePathFromHref(href: string, callback: (err?: AdobeDCXError, path?: string) => void): RequestDescriptor;
    resolveRootURLAsync(): AdobePromise<any, AdobeDCXError, RequestDescriptor>;
    getPromiseCacheInfo(fn: (...args: any[]) => AdobePromise<any, AdobeDCXError, RequestDescriptor>): AdobePromise<any, AdobeDCXError, RequestDescriptor>;
    private _parseCachableLinks;
    /**
     * Returns CachedAssetInfo for the specified asset id
     *
     * @param {String}  assetId     The unique id of the asset
     * @param {GetCachedAssetInfoCallback} Callback function to call with the result
     * @returns {RequestDescriptor} A RequestDescriptor if an HTTP request was made or undefined if none was required
     */
    getCachedAssetInfo(assetId: string, getCachedInfoCallback: GetCachedAssetInfoCallback): any;
    /**
     * Registers a set of links and returns a (fake) assetId
     *
     * @param {Object}  links   The links to register (body of _links object)
     * @returns {String}        Unique (but fake) assetId
     */
    registerLinks(links: Record<string, Link>): string;
    /** @internal */
    _infoForAssetId(assetId: string): any;
    private _awaitInfoForAssetId;
    protected _cacheInfoForAssetId(assetId: string, error?: Error, info?: any): void;
    /**
     * A _very_ naive URL resolver whose function is to prepend the a base to a relative URL
     * or to simply return the href directly if it's absolute.
     * Nothing is done about . or .. elements.
     */
    _resolveUrl(href: string): string;
    /**
     * @private
     * @param {String} href The href
     */
    protected _makeRelativeUrl(href: string): string;
    /**
     * Helper function to parse a response when polling for asynchronous results.
     * The body returned from these requests includes the response to the original request
     * including its own status code and headers.
     *
     * @notice This methods throws if it fails to parse at least the status code.
     *
     * @param asyncResponse The respone received from a successful (status code 200) poll request.
     * @return A fake response object with a statusCode and headers property set.
     *
     * @notice For now this methods drops the response body of the embedded response.
     * @internal
     */
    _parseAsyncResponse(asyncResponse: any): any;
    /**
     * Issues and handles the responses to an async poll reuqest.
     * Calls itself recursively if it gets another 202 response.
     *
     * @private
     * @param pollHref The href returned by the original 202 response.
     * @param pollFrequencyInSeconds Self explanatory.
     * @param requestDesc The request descriptor to reuse.
     * @param callback Gets called upon success or failure. Signature: function (error, response)
     */
    private _pollForAsyncResponse;
    /**
     * Handles a 202 response by scheduling a poll request reusing the same requestDesc.
     *
     * @protected
     * @param response The original 202 response.
     * @param requestDesc The request descriptor to reuse.
     * @param defaultSecondsToWaitForRetry Integer >= 0. If the reply doesn't contain a retry-after directive this is being used as the initial polling delay.
     * @param callback Gets called upon success or failure. Signature: function (error, response)
     */
    protected _handle202Response(response: any, requestDesc: RequestDescriptor, defaultSecondsToWaitForRetry: number, callback: (err?: AdobeDCXError, result?: any) => void): void;
}
export {};
//# sourceMappingURL=AdobeSessionBase.d.ts.map