/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
/**
 * Public namespace for compositeXfer.
 */
import { ComponentRequestByPath, CopyResourcesOperationResult, ManifestAndComponentsByPathResponse, PathOrIdAssetDesignator, RepoResponseResult } from '@dcx/assets';
import { AdobeAsset, AdobeDCXBranch, AdobeDCXComponent, AdobeDCXComposite, AdobeDCXError, AdobeUploadResults, CopyResourceDesignator, FailedComponent, GetSliceCallback, JSONPatchDocument, ProgressCallback, SliceableData } from '@dcx/common-types';
import AdobePromise from '@dcx/promise';
import { AdobeRepoAPISession } from '@dcx/repo-api-session';
import DCXBranch from '../AdobeDCXBranch';
import { PushCompositeOptions } from './push.internal';
import { AdobeXferContext } from './XferContext';
export { convertToDCXComposite } from '../AdobeDCXComposite';
export { XMPModes } from './push.internal';
type CompositeWithManifestAndComponents<C extends ComponentRequestByPath[]> = {
    composite: AdobeDCXComposite;
} & ManifestAndComponentsByPathResponse<C>;
/**
 * Create a pull branch with just the manifest updated.
 *
 * @param   {AdobeRepoAPISession}       session     - The repo session to pull from.
 * @param   {AdobeDCXComposite}         composite   - The composite to pull.
 * @returns {AdobePromise<DCXBranch, AdobeDCXError, AdobeXferContext>}
 *
 */
export declare function pullCompositeManifestOnly(session: AdobeRepoAPISession, composite: AdobeDCXComposite, additionalHeaders?: Record<string, string>): AdobePromise<DCXBranch, AdobeDCXError, AdobeXferContext>;
/**
 * Download the manifest of a specific version and return a read-only branch.
 *
 * @param session   The repo session to pull from.
 * @param composite The composite to pull.
 * @param versionId Composite version
 * @param additionalHeaders Additional headers to attach to HTTP requests
 */
export declare function pullCompositeVersionManifestOnly(session: AdobeRepoAPISession, composite: AdobeDCXComposite, versionId: string, additionalHeaders?: Record<string, string>): AdobePromise<DCXBranch, AdobeDCXError, AdobeXferContext>;
/**
 * Returns a URL that points to the component asset on the server.
 * When using the result URL for for creating a new component via PUT, `null`
 * should be provided for the revision parameter.
 *
 * @param session           The repo session to use.
 * @param component         The existing component. Must be part of a composite branch.
 * @param componentRevision Specific revision of the component to get.
 *                              If not provided, will use the revision defined in the component, or undefined.
 *                              To bypass the component's revision, pass null as revision.
 * @param additionalHeaders Additional headers to attach to HTTP requests
 */
export declare function getURLForComponent(session: AdobeRepoAPISession, component: AdobeDCXComponent, componentRevision?: string, additionalHeaders?: Record<string, string>): AdobePromise<string, AdobeDCXError>;
/**
 * <p>Uploads the given data as the asset for a new component in the current branch of the given
 * composite. Does not add the component in the composite branch. Instead it returns
 * (via a promise) an object containing the upload results which then can be used to
 * add the component to the branch.</p>
 *
 * @remarks Service can override the block size client specified @see {@link https://git.corp.adobe.com/caf/api-spec/blob/gh-pages/schemas/repository/transfer-document.schema.json#L45}
 *
 * @param session               The repo session to use for the upload.
 * @param composite             The composite to add the new component asset to.
 * @param dataOrSliceCallback   The data to upload, or a method that returns slices of data.
 * @param type                  The content type of the component.
 * @param componentId           The id to use for the new component. Must be
 *                                  unique within the composite. If not provided
 *                                  the component will get assigned a random uuid.
 * @param size                  The estimated size of the component data to upload.
 * @param md5                   The MD5 of the component data to upload.
 * @param progressCb            A callback to be invoked with progress updates
 * @param additionalHeaders     Additional headers to attach to HTTP requests
 * @param blockSize             Desired block size for the upload in bytes
 */
export declare function uploadNewComponent(session: AdobeRepoAPISession, composite: AdobeDCXComposite, dataOrSliceCallback: SliceableData | GetSliceCallback, type: string, componentId?: string, size?: number, md5?: string, progressCb?: ProgressCallback, additionalHeaders?: Record<string, string>, blockSize?: number): AdobePromise<AdobeUploadResults, AdobeDCXError, AdobeXferContext>;
/**
 * <p>Uploads the given data as the new asset for the given component. Does not update the
 * component in the composite branch. Instead it returns (via a promise) an object
 * containing the upload results which then can be used to update the component in the branch.</p>
 *
 * @param   {AdobeRepoAPISession}           session                 The repo session to use for the upload.
 * @param   {AdobeDCXComposite}             composite               The composite of the component.
 * @param   {AdobeDCXComponent}             component               The component to upload the new asset for.
 * @param   {Buffer|Function}               dataOrSliceCallback     The data to upload, or a method that returns slices of data.                                                                  unique within the composite. If not provided
 * @param   {number}                        [size]                  The estimated size of the component data to upload.
 * @param   {string}                        [md5]                   The MD5 of the component data to upload.
 *
 * @returns {AdobePromise<AdobeUploadResults, AdobeDCXError, AdobeXferContext>}
 */
export declare function uploadComponent(session: AdobeRepoAPISession, composite: AdobeDCXComposite, component: AdobeDCXComponent, dataOrSliceCallback: SliceableData | GetSliceCallback, size?: number, md5?: string, progressCb?: ProgressCallback, additionalHeaders?: Record<string, string>): AdobePromise<AdobeUploadResults, AdobeDCXError, AdobeXferContext>;
/**
 * @param session - The repo session to use.
 * @param asset - Asset from which the manifest and components should be retrieved. May be a DCXComposite or minimal AdobeAsset.
 * If a composite is provided, the same composite will returned on the response object.
 * @param components An array of ComponentRequestsByPath. NOTE: Paths are restricted to well-known paths only as determined by specific composite media type.
 * @see {@link https://wiki.corp.adobe.com/display/CA/Proposal%3A+Download+Components+by+DCX+Path Proposal: Download DCX Components by Path}
 * If using components that do not exist on the given well-known paths, please use @see performBulkRequest
 * @example
 *  ```
 *  const components = [
 *    {
 *      "/model/database.json",
 *      responseType: "json" // default value "defaultbuffer"
 *    }
 *  ]
 *  ```
 * @param version           Version of the composite manifest to fetch (components all come from this version of the manifest)
 * @param etag              Etag to use within the if-match header of the request
 * @param additionalHeaders Additional headers to attach to HTTP requests
 * @example
 * ```
 * {
 *   composite: AdobeDCXComposite, // The supplied composite is returned here, if it was not supplied a new one is created
 *   manifest: {
 *     data?: ManifestData, // JSON parsed manifest data
 *     error?: DCXError, // present if there was an error fetching the manifest
 *     response: AdobeResponse // response for the manifest sub-request (or secondary follow-up manifest request)
 *   },
 *   components: {
 *     "/model/database.json": { // component paths are the keys
 *       component_path: "/model/database.json",
 *       data?: ResponseTypeMap[ResponseType], // parsed into the requested responseType
 *       error?: DCXError, // present if there was an error fetching the manifest
 *       response: AdobeResponse, // response for component sub-request (or follow-up request)
 *     }
 *   },
 *   responses: AdobeResponse[] // array of Bulk request responses (1 response per bulk request issued)
 * }
 * ```
 */
export declare function getCompositeManifestAndComponentsByPath<C extends ComponentRequestByPath[]>(session: AdobeRepoAPISession, asset: AdobeAsset, components: C, version?: string, etag?: string, additionalHeaders?: Record<string, string>): AdobePromise<CompositeWithManifestAndComponents<C>, AdobeDCXError>;
/**
 * Push the composite.
 *
 * There are 4 modes for XMP metadata updates, determined by the `options` argument:
 * 1. Managed: all fields are managed by dcx-js. creatorTool is set to `dcx-js`
 * 2. Partially managed: Specific fields (eg. creatorTool) can be replaced by the caller,
 *    by passing them as key/values in an options object.
 * 3. Managed by client app via JSON Patch/initial XMP XML:
 *    On initial push: dcx-js will use the provided string `options.initialXMPXML` to create the component.
 *    On subsequent pushes: dcx-js will use the provided JSON Patch document `options.xmpPatch` to patch embedded
 *    metadata alongside the manifest push. This results in a single version, but leaves all validation and error \
 *    handling to the client application.
 *    See {@link https://git.corp.adobe.com/DMA/dcx-js/tree/dev/packages/xmp-util | @dcx/xmp-util} for
 *    convenience APIs for XMP Patching.
 * 4. Unmanaged: dcx-js skips the embedded metadata update entirely. It is the caller's responsbility
 *    to handle updates however they choose. If using this mode, a version will be created during dcx-js
 *    push; later updates by client applications will lead to an additional version being created.
 *
 * @note
 * To use mode 1 or 2, no mode needs to be specified. Properties that can be customized by clients
 * can be provided in the options object. If another mode is explicitly defined, those fields may be ignored.
 *
 * @note
 * If using mode 3, initialXMPXML is only used for initial pushes or when the manifest
 * does not yet contain a metadata component. If not provided, dcx-js uses the default XMP content.
 * Similarly, xmpPatch is only used for bound composites that contain a metadata component. If not
 * provided, no update will be performed on update pushes.
 *
 * @example
 * ```js
 * // 1. Managed mode
 * pushComposite(session, composite, overwrite, vLevel);
 * ```
 *
 * @example
 * ```js
 * // 2. Partially managed mode
 * pushComposite(session, composite, overwrite, vLevel, { creatorTool: 'your application name' });
 * ```
 *
 * @example
 * ```js
 * // 3. Client managed mode
 * pushComposite(session, composite, overwrite, vLevel, {
 *   mode: XMPModes.CLIENT_MANAGED,
 *   xmpPatch: [{ op: 'replace', path: '/xmp:ModifyDate', value: now}, ... ],
 *   initialXMPXML: '<x:xmpmeta ... />'
 * });
 * ```
 *
 * @example
 * ```js
 * // 4. Unmanaged mode
 * pushComposite(session, composite, overwrite, vLevel, { mode: XMPModes.UNMANAGED });
 * ```
 *
 * @note
 * <p>Notice that in the case of a rejected promise that error might represent multiple
 * errors since components are uploaded in parallel and each upload might fail individually.
 * Clients can get to per-component error information by checking the DCXError#additionalData#failedComponents
 * property on the error object. If set it will contain an array with one object per failed component
 * upload. Each component failure object has two properties: component and error.</p>
 *
 * @param   {AdobeRepoAPISession}                   session                 The repo session to use.
 * @param   {AdobeDCXComposite}                     composite               The composite to push.
 * @param   {Boolean}                               overwriteExisting       Whether to overwrite an existing composite.
 *                                                                          Only set this to true for an unbound composite
 *                                                                          if a previous push has failed with an
 *                                                                          INVALID_STATE error indicating that there is
 *                                                                          already a composite with the same id <em>and</em>
 *                                                                          if you are sure that you want to overwrite it.
 * @param   {number}                                [validationLevel=1]     Validation level for the manifest. Defaults to 1, must be >=1.
 * @param   {PushCompositeOptions}                  [options]               Push options
 * @returns {AdobePromise<DCXBranch, AdobeDCXError<{failedComponents?: FailedComponent[]}>, AdobeXferContext>}
 */
export declare function pushComposite(session: AdobeRepoAPISession, composite: AdobeDCXComposite, overwriteExisting: boolean, validationLevel?: number, options?: PushCompositeOptions, additionalHeaders?: Record<string, string>): AdobePromise<AdobeDCXBranch, AdobeDCXError<{
    failedComponents?: FailedComponent;
}>, AdobeXferContext>;
/**
 * Copies resources from source to target asset using the COPY_RESOURCES operation.
 * See {@link https://git.corp.adobe.com/pages/caf/api-spec/chapters/operations/copying_resources.html}
 *
 * @param {AdobeRepoAPISession} session                             Repository session obj
 * @param {PathOrIdAssetDesignator} sourceAsset                     The source asset
 * @param {PathOrIdAssetDesignator} targetAsset                     The destination asset
 * @param {CopyResourceDesignator[]} resources                      An array of resource designators to be copied from the source to the target asset.
 * @param {JSONPatchDocument} [manifestPatch]                       An optional JSON Patch document to be applied to the target asset's manifest.
 * @param {boolean} [intermediates]                                 If true, create intermediate assets if they do not exist.
 *
 * @returns {AdobePromise<RepoResponseResult<AdobeAsset, 'json'>, AdobeDCXError>}
 */
export declare function copyResources(session: AdobeRepoAPISession, sourceAsset: PathOrIdAssetDesignator | AdobeDCXComposite, targetAsset: PathOrIdAssetDesignator | AdobeDCXComposite, resources: CopyResourceDesignator[], intermediates?: boolean, manifestPatch?: JSONPatchDocument): AdobePromise<RepoResponseResult<CopyResourcesOperationResult, 'json'>, AdobeDCXError>;
//# sourceMappingURL=index.d.ts.map