/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
import { CopyResourcesOperationResult, OperationDocumentBuilder, RepoResponseResult } from '@dcx/assets';
import { AdobeDCXBranch, AdobeDCXError, AdobeResponse, FailedComponent, InternalFunction, JSONPatchDocument } from '@dcx/common-types';
import AdobePromise, { SettledPromise } from '@dcx/promise';
import { AdobeRepoAPISession } from '@dcx/repo-api-session';
import DCXBranch from '../AdobeDCXBranch';
import { SourceAssetInfoEntry } from '../AdobeDCXBranchCore';
import DCXComponent, { AdobeDCXComponent } from '../AdobeDCXComponent';
import DCXComposite from '../AdobeDCXComposite';
import { AdobeXferContext, XferContext } from './XferContext';
export declare enum XMPModes {
    MANAGED = 0,
    PARTIALLY_MANAGED = 1,
    CLIENT_MANAGED = 2,
    UNMANAGED = 3
}
type XMPConfigClientManaged = {
    mode: XMPModes.CLIENT_MANAGED;
    initialXMPXML?: string;
    xmpPatch?: JSONPatchDocument;
};
type XMPConfigManaged = {
    mode?: XMPModes.MANAGED | XMPModes.PARTIALLY_MANAGED;
    /**
     * Client app using dcx-js to push the metadata.
     * If undefined will use `dcx-js`.
     */
    creatorTool?: string;
    /**
     * Datetime string in ISO8601 format.
     */
    modifyDate?: string;
};
type XMPConfigUnmanaged = {
    mode: XMPModes.UNMANAGED;
};
export type XMPConfig = XMPConfigClientManaged | XMPConfigManaged | XMPConfigUnmanaged;
export type PushCompositeOptions = XMPConfig;
export interface AdditionalDataForPush {
    compositeIsNew: boolean;
    overwriteExisting: boolean;
    owner: DCXBranch;
    validationLevel: number;
    manifestResponse?: AdobeResponse<'json'>;
    xmpConfig: Required<XMPConfigClientManaged> | Required<XMPConfigManaged> | XMPConfigUnmanaged;
    shouldPushWithXMP: boolean;
    xmpPushed: boolean;
}
interface CopyOperationCollector {
    docBuilder: OperationDocumentBuilder;
    orderedPendingComponents: [DCXComponent, SourceAssetInfoEntry][];
}
interface OrderedBatchCopyResult {
    results: CopyResourcesOperationResult[];
    response: AdobeResponse<'json'>;
    orderedPendingComponents: [DCXComponent, SourceAssetInfoEntry][];
}
/**
 * Internal implementation of push composite.
 *
 * @private
 *
 * @param {XferContext<AdditionalDataForPush>} pushContext  - Push context, used as `this` scope for other internals.
 * @param {DCXComposite} composite                          - Composite to push
 */
export declare function _pushComposite(pushContext: XferContext<AdditionalDataForPush>, composite: DCXComposite): AdobePromise<AdobeDCXBranch, AdobeDCXError<{
    failedComponents?: FailedComponent[];
}>, AdobeXferContext>;
/**
 * Handle error that occurred during push.
 *
 * Errors pushing to deleted composites can be very messy and can look like
 * other kinds of things. So here we check to see if there's a manifest.
 * If not, then we call it a NO_COMPOSITE error. Otherwise just pass it along.
 *
 * @private
 *
 * @this XferContext<AdditionalDataForPush>
 *
 * @param {AdobeDCXError} originalError
 */
declare function _handleError(this: XferContext<AdditionalDataForPush>, originalError: AdobeDCXError<{
    failedComponents?: FailedComponent[];
    bulkResponse?: AdobeResponse[];
}>): AdobePromise<never>;
/**
 * Push the manifest if the composite has been modified.
 *
 * @this XferContext<AdditionalDataForPush>
 *
 * @param {DCXBranch} branch
 */
declare function _pushManifest(this: XferContext<AdditionalDataForPush>): AdobePromise<AdobeResponse<'json'>, AdobeDCXError> | undefined;
/**
 * Handle bulk responses.
 * Since bulk will return a 200, we need to check each response for failure codes.
 * And since we're using version sub requests, if one request fails, both will.
 * See: {@link https://git.corp.adobe.com/pages/caf/api-spec/single.html#version-sub-requests | Version Sub-requests}
 *
 * @this {XferContext<AdditionalDataForPush>}
 *
 * @param bulkResponse
 * @returns The manifest response
 */
/** letting code coverage to ignore this function because of DCX-4473 */
export declare function _handleManifestAndXMPResponse(ctx: XferContext<AdditionalDataForPush>, branch: DCXBranch, manifestData: string, overwrite: boolean, validationLevel: number, manifestEtag: string | undefined, compositeIsNew: boolean, xmpConfig: XMPConfigManaged | XMPConfigClientManaged, bulkResponse: RepoResponseResult<AdobeResponse[]>, retryCount?: number): AdobePromise<AdobeResponse<'json'>> | AdobeResponse<'json'>;
/**
 * Check for failed components, providing a more meaningful error if possible.
 *
 * @this XferContext<AdditionalDataForPush>
 *
 * @param {FailedComponent[]} failedComponents
 */
declare function _handleFailedComponents(this: XferContext<AdditionalDataForPush>, failedComponents: FailedComponent[]): DCXBranch;
/**
 * Return true if the composite is all 3 of:
 * 1. not new
 * 2. not modified
 * 3. not pending archival
 *
 * There is no push to perform in this case.
 *
 * @private
 *
 * @param {DCXBranch} branch
 * @param {boolean} compositeIsNew
 * @param {boolean} discardWhenDone
 * @returns {boolean}
 */
declare function _isNoOpPush(branch: DCXBranch, compositeIsNew: boolean): boolean;
/**
 * Validate current branch state.
 * Throw an error if the current state clashes with the push operation.
 *
 * @throws {AdobeDCXError}
 *
 * @param {DCXBranch} branch
 * @returns {void}
 */
declare function _assertStateIsValidForPush(branch: DCXBranch): void;
/**
 * Get the current copy op collector in the array.
 * If the doc builder has reached it's limit, create a new one and push to the end.
 *
 * @param {CopyOperationCollector[]} copyOpCollectors
 *
 * @this {XferContext<AdditionalDataForPush>}
 */
declare function _getCurrentCopyOpCollector(this: XferContext<AdditionalDataForPush>, copyOpCollectors: CopyOperationCollector[]): CopyOperationCollector;
/**
 * Conditionally add a copy operation for a component to component copy to the operation doc builder.
 * Cleans up the branch when encountering new & deleted/pending delete components.
 * Cleans up the journal when encountering non-new & unmodified components.
 *
 * Return a promise that resolves when the operation is added.
 *
 * If no operation is required for the component
 * (already copied, errored, pending delete) then this returns undefined.
 *
 * @this XferContext<AdditionalDataForPush>
 *
 * @param {OperationDocumentBuilder} docBuilder                                     - Operation document builder to use
 * @param {[AdobeDCXComponent, SourceAssetInfoEntry][]} orderedPendingComponents    - List of tuples representing the component and their source info.
 *                                                                                  This is added to whenever a component is added to the copy document.
 *                                                                                  The order of this array represents the order of the copy document,
 *                                                                                  and therefore, the order of the responses from the server.
 * @param {AdobeDCXComponent} component                                             - Component to possibly add to the copy document
 */
declare function _reduceComponent(this: XferContext<AdditionalDataForPush>, copyOpCollectors: CopyOperationCollector[], component: AdobeDCXComponent): void;
/**
 * Process a set of batch copy results from a settled (either rejected or resolved) promise.
 * If an error is encountered, add that component to failedComponents.
 * If the entire promise was rejected, throw immediately.
 *
 * @this {XferContext<AdditionalDataForPush>}
 *
 * @param {SettledPromise<OrderedBatchCopyResult, AdobeDCXError>[]} results - Settled promises from batch copy.
 */
declare function _handleBatchResults(this: XferContext<AdditionalDataForPush>, results: SettledPromise<OrderedBatchCopyResult, AdobeDCXError>[]): void;
/**
 * Issue parallel batch copy operations for each set of components.
 * Limited to 100 per batch, 3 batches at a time, as defined by the generator.
 *
 * @this {XferContext<AdditionalDataForPush>}
 *
 * @param {AdobeRepoAPISession} session - Repo session to use.
 * @param {Generator} copyOpCollectors - Generator yielding batches of ordered copy documents.
 */
declare function _parallelBatchOperations(this: XferContext<AdditionalDataForPush>, session: AdobeRepoAPISession, copyOpGenerator: ReturnType<typeof _makeCopyOpGenerator>): AdobePromise<FailedComponent[], AdobeDCXError>;
/**
 * Make a generator that yields a limited number of batched copy operations for parallel requests.
 *
 * @this {XferContext<AdditionalDataForPush>}
 *
 * @param {DCXComponent[]} components - Components to process, possibly adding to a copy document.
 */
declare function _makeCopyOpGenerator(this: XferContext<AdditionalDataForPush>, components: DCXComponent[]): Generator<CopyOperationCollector[], CopyOperationCollector[]>;
/**
 * Push components.
 * For each component: if state is modified, and we have valid source asset data, add it to an operation document.
 * Then send the operation, adding failed components to the context's `_failedComponent` property.
 *
 * If no components need pushing, resolves immediately.
 *
 * @note
 * It's important for this method to always return the correct AdobePromise signature.
 * The source object must always be set to the pushContext, since this is the first promise in the chain.
 * Later "thenned" promises may return an undefined source object, in which case the source object is propagated.
 *
 * @private
 *
 * @this XferContext<AdditionalDataForPush>
 */
declare function _pushComponents(this: XferContext<AdditionalDataForPush>): AdobePromise<FailedComponent[], AdobeDCXError, AdobeXferContext>;
/**
 * Clean an asset from a copy response, returning the required key/values.
 * If the result contains an error, throw it.
 * If the result isn't valid, throw an error.
 * If one of the required keys doesn't exist, throw an error.
 *
 * @throws {AdobeDCXError}
 *
 * @private
 *
 * @param {AdobeOperationResult} result - Operation result for a single asset
 * @param {AdobeResponse<'json'>} response - Overall response from the copy operation
 */
declare function _validateCopyResponse(result: CopyResourcesOperationResult['resources'][number]['target'], response: AdobeResponse<'json'>, component: AdobeDCXComponent): AdobeDCXComponent & {
    etag: string;
    version: string;
    md5: string;
    length: number;
};
/**
 * Sets the manifestEtag on the branch if needed and returns the etag
 * to use for any calls to `updateCompositeManifest()`.
 *
 * If the composite has been pushed successfully before, we need to
 * use the etag from that push. *Unless* the client has pulled/resolved
 * a new version that has a newer etag.
 *
 * @private
 *
 * @param {XferContext<AdditionalDataForPush>} pushContext
 * @param {DCXBranch} branch
 */
declare function _updateManifestEtag(pushContext: XferContext<AdditionalDataForPush>, branch: DCXBranch): string | undefined;
/**
 * Internals for testing
 *
 * @private
 * @internal
 */
export interface InternalTypes {
    _isNoOpPush: InternalFunction<typeof _isNoOpPush>;
    _handleFailedComponents: InternalFunction<typeof _handleFailedComponents>;
    _pushComponents: InternalFunction<typeof _pushComponents>;
    _reduceComponent: InternalFunction<typeof _reduceComponent>;
    _assertStateIsValidForPush: InternalFunction<typeof _assertStateIsValidForPush>;
    _updateManifestEtag: InternalFunction<typeof _updateManifestEtag>;
    _validateCopyResponse: InternalFunction<typeof _validateCopyResponse>;
    _getCurrentCopyOpCollector: InternalFunction<typeof _getCurrentCopyOpCollector>;
    _makeCopyOpGenerator: InternalFunction<typeof _makeCopyOpGenerator>;
    _parallelBatchOperations: InternalFunction<typeof _parallelBatchOperations>;
    _handleBatchResults: InternalFunction<typeof _handleBatchResults>;
    _pushComposite: InternalFunction<typeof _pushComposite>;
    _handleError: InternalFunction<typeof _handleError>;
    _pushManifest: InternalFunction<typeof _pushManifest>;
    _handleManifestAndXMPResponse: InternalFunction<typeof _handleManifestAndXMPResponse>;
}
/**
 * @private
 * @internal
 **/
export declare const internals: InternalTypes;
export {};
//# sourceMappingURL=push.internal.d.ts.map