import { AdobeDCXError as AdobeDCXError_2 } from '@dcx/common-types';
import { AdobeResponse } from '@dcx/common-types';

/**
 * @class
 * @classdesc `AdobeDCXError` is a subclass of `Error` which defines the DCX specific errors.
 * @augments Error
 * @hideconstructor
 * @param {String} code            The error code string.
 * @param {Str}    message         A string describing the error.
 * @param {Error}  underlyingError Optional underlying error.
 */
declare class AdobeDCXError<T = unknown> extends Error implements AdobeDCXError_2<T> {
    readonly code: string;
    readonly name = "AdobeDCXError";
    private _message;
    private _additionalData;
    private _response?;
    private _underlyingError?;
    stack: string | undefined;
    static readonly ABORTED: "ABORTED";
    static readonly INSECURE_REDIRECT: "INSECURE_REDIRECT";
    static readonly TOO_MANY_REDIRECTS: "TOO_MANY_REDIRECTS";
    static readonly INVALID_JSON: "INVALID_JSON";
    static readonly READ_ONLY: "READ_ONLY";
    static readonly INVALID_PARAMS: "";
    static readonly INVALID_DATA: "INVALID_DATA";
    static readonly DUPLICATE_VALUE: "DUPLICATE_VALUE";
    static readonly NO_BASE_BRANCH_DATA: "NO_BASE_BRANCH_DATA";
    static readonly INVALID_STATE: "INVALID_STATE";
    static readonly DELETED_COMPOSITE: "DELETED_COMPOSITE";
    static readonly INCOMPLETE_COMPOSITE: "INCOMPLETE_COMPOSITE";
    static readonly UNEXPECTED_RESPONSE: "UNEXPECTED_RESPONSE";
    static readonly NETWORK_ERROR: "NETWORK_ERROR";
    static readonly COMPONENT_DOWNLOAD_ERROR: "COMPONENT_DOWNLOAD_ERROR";
    static readonly COMPONENT_UPLOAD_ERROR: "COMPONENT_UPLOAD_ERROR";
    static readonly COMPONENT_MODIFIED_ERROR: "COMPONENT_MODIFIED_ERROR";
    static readonly UPDATE_CONFLICT: "UPDATE_CONFLICT";
    static readonly NO_COMPOSITE: "NO_COMPOSITE";
    static readonly ALREADY_EXISTS: "ALREADY_EXISTS";
    static readonly SERVICE_IS_INACTIVE: "SERVICE_IS_INACTIVE";
    static readonly EXCEEDS_QUOTA: "EXCEEDS_QUOTA";
    static readonly NOT_IMPLEMENTED: "NOT_IMPLEMENTED";
    static readonly RETRYABLE_SERVER_ERROR: "RETRYABLE_SERVER_ERROR";
    static readonly TIMED_OUT: "TIMED_OUT";
    static readonly UNEXPECTED: "UNEXPECTED";
    static readonly TERMINATED_INPUTSTREAM: "TERMINATED_INPUTSTREAM";
    static readonly WRONG_ENDPOINT: "WRONG_ENDPOINT";
    static readonly OUT_OF_SPACE: "ENOSPC";
    static readonly FILE_EXISTS_IN_CLOUD: "FILE_EXISTS_IN_CLOUD";
    static readonly ASSET_NOT_FOUND: "ASSET_NOT_FOUND";
    static readonly COMPOSITE_NOT_FOUND: "COMPOSITE_NOT_FOUND";
    static readonly NOT_FOUND: "NOT_FOUND";
    static readonly UNAUTHORIZED: "UNAUTHORIZED";
    static readonly FORBIDDEN: "FORBIDDEN";
    static readonly PRECONDITION_FAILED: "PRECONDITION_FAILED";
    constructor(code: string, message?: string, underlyingError?: unknown, response?: AdobeResponse, additionalData?: T);
    get response(): AdobeResponse<any> | undefined;
    get problemType(): any;
    get underlyingError(): Error | AdobeDCXError<unknown> | undefined;
    get additionalData(): T;
    /**
     * @internal
     */
    set additionalData(val: T);
    /**
     * For backwards compatibility.
     * Allow some properties to be accessible through their old property keys.
     *
     * Don't include in documentation or type declarations.
     *
     * @internal
     * @private
     */
    get failedComponents(): any;
    /** @private */
    static wrapError(code: string, message: string, error?: unknown, response?: AdobeResponse): AdobeDCXError;
    toString(): string;
    static networkError(message: string, error?: unknown, response?: AdobeResponse): AdobeDCXError<unknown>;
    static unexpectedResponse(message: string, error?: unknown, response?: AdobeResponse): AdobeDCXError<unknown>;
}
export { AdobeDCXError }
export { AdobeDCXError as DCXError }
export default AdobeDCXError;

/** @internal */
export declare const _defaultStatusValidator: (statusCode?: number, response?: AdobeResponse) => boolean | AdobeDCXError<unknown>;

export declare const ErrorCodes: {
    /**
     * This error indicates that the asset has changed regions and the existing links are no longer valid.
     * New links are sent along with a server generated response for asset moved errors.
     * Generally speaking, dcx-js will attempt to handle this error and update the links and retry the requested
     * operation using the new asset location. If this error is received by a client, dcx-js handling is likely
     * not yet implemented or does not have a viable resolution without further user input.
     */
    readonly ASSET_MOVED: "ASSET_MOVED";
    /**
     * Parsing JSON data has failed.
     */
    readonly INVALID_JSON: "INVALID_JSON";
    /** Trying to modify an immutable object or property.
     */
    readonly READ_ONLY: "READ_ONLY";
    /** The parameters passed to a function are not as expected.
     * @constant {String}
     */
    readonly INVALID_PARAMS: "";
    /**
     * Links are invalid, used with leaf functions when they don't receive valid links.
     * Links are either missing or do not contain the required relation.
     * @constant {string}
     */
    readonly INVALID_LINKS: "INVALID_LINKS";
    /**
     * A conditional check failed for the request
     * @constant {String}
     */
    readonly PRECONDITION_FAILED: "PRECONDITION_FAILED";
    /** Data is invalid. Usually this means that a document read from disk or from an http request is bad.
     * @constant {String}
     */
    readonly INVALID_DATA: "INVALID_DATA";
    /** Uniqueness constraint violated.
     * @constant {String}
     */
    readonly DUPLICATE_VALUE: "DUPLICATE_VALUE";
    /** Trying to invoke functionality that requires that AdobeDCX being initialized with the xhrBaseBranchSupport option.
     * @constant {String}
     */
    readonly NO_BASE_BRANCH_DATA: "NO_BASE_BRANCH_DATA";
    /** An object is not in the expected state.
     * @constant {String}
     */
    readonly INVALID_STATE: "INVALID_STATE";
    /** Invalid operation on a deleted composite.
     * @constant {String}
     */
    readonly DELETED_COMPOSITE: "DELETED_COMPOSITE";
    /** Composite did not pass integrity check. Incomplete composite.
     * @constant {String}
     */
    readonly INCOMPLETE_COMPOSITE: "INCOMPLETE_COMPOSITE";
    /** Unexpected Response
     * @constant {String}
     */
    readonly UNEXPECTED_RESPONSE: "UNEXPECTED_RESPONSE";
    /** Network error
     * @constant {String}
     */
    readonly NETWORK_ERROR: "NETWORK_ERROR";
    /** Component(s) download failure -- see error.failedComponents for more details.
     * @constant {String}
     */
    readonly COMPONENT_DOWNLOAD_ERROR: "COMPONENT_DOWNLOAD_ERROR";
    /** Component(s) upload failure -- see error.failedComponents for more details.
     * @constant {String}
     */
    readonly COMPONENT_UPLOAD_ERROR: "COMPONENT_UPLOAD_ERROR";
    /** Component file could not be removed since it was modified -- see error.componentId for the id of the component.
     * @constant {String}
     */
    readonly COMPONENT_MODIFIED_ERROR: "COMPONENT_MODIFIED_ERROR";
    /** Update Conflict
     * @constant {String}
     */
    readonly UPDATE_CONFLICT: "UPDATE_CONFLICT";
    /** No composite. Possibly deleted.
     * @constant {String}
     */
    readonly NO_COMPOSITE: "NO_COMPOSITE";
    /** Respouce already Exists.
     * @constant {String}
     */
    readonly ALREADY_EXISTS: "ALREADY_EXISTS";
    /** HTTP session is in inactive state.
     * @constant {String}
     */
    readonly SERVICE_IS_INACTIVE: "SERVICE_IS_INACTIVE";
    /** Exceeds quota
     * @constant {String}
     */
    readonly EXCEEDS_QUOTA: "EXCEEDS_QUOTA";
    /** Unimplemented server request
     * @constant {String}
     */
    readonly NOT_IMPLEMENTED: "NOT_IMPLEMENTED";
    /** Retryable server error
     * @constant {String}
     */
    readonly RETRYABLE_SERVER_ERROR: "RETRYABLE_SERVER_ERROR";
    /** Timed-out request
     * @constant {String}
     */
    readonly TIMED_OUT: "TIMED_OUT";
    /** Unexpected failure, usually a problem acquiring a local resource such as a file
     * @constant {String}
     */
    readonly UNEXPECTED: "UNEXPECTED";
    /** Input stream terminated abnormaly
     * @constant {String}
     */
    readonly TERMINATED_INPUTSTREAM: "TERMINATED_INPUTSTREAM";
    /** Trying to access an asset from a different endpoint
     * @constant {String}
     */
    readonly WRONG_ENDPOINT: "WRONG_ENDPOINT";
    /** A file operation failed because the disk is full
     * @constant {String}
     */
    readonly OUT_OF_SPACE: "ENOSPC";
    /** A composite could not be created because a regular file already exists at the specified path
     * @constant {String}
     */
    readonly FILE_EXISTS_IN_CLOUD: "FILE_EXISTS_IN_CLOUD";
    /** A resource (manifest, component or rendition) could not be found.
     * @constant {String}
     */
    readonly ASSET_NOT_FOUND: "ASSET_NOT_FOUND";
    /** A composite could not be found.
     * @constant {String}
     */
    readonly COMPOSITE_NOT_FOUND: "COMPOSITE_NOT_FOUND";
    /**
     * Generic not found error.
     * @constant {string}
     */
    readonly NOT_FOUND: "NOT_FOUND";
    /**
     * Unauthorized (401)
     * @constant {string}
     */
    readonly UNAUTHORIZED: "UNAUTHORIZED";
    /**
     * Forbidden (403)
     * @constant {string}
     */
    readonly FORBIDDEN: "FORBIDDEN";
    /**
     * Method Not Allowed (HTTP Status Code 405)
     */
    readonly METHOD_NOT_ALLOWED: "METHOD_NOT_ALLOWED";
    /**
     * Method Not Allowed (HTTP Status Code 406)
     */
    readonly NOT_ACCEPTABLE: "NOT_ACCEPTABLE";
    /**
     * Bandwidth Limit Exceeded (HTTP Status Code 509)
     */
    readonly BANDWIDTH_LIMIT_EXCEEDED: "BANDWIDTH_LIMIT_EXCEEDED";
    /**
     * Request aborted
     */
    readonly ABORTED: "ABORTED";
    /**
     * Request was redirected over set maximum redirects
     */
    readonly TOO_MANY_REDIRECTS: "TOO_MANY_REDIRECTS";
    /**
     * Request containing Authorization header redirected to HTTP
     */
    readonly INSECURE_REDIRECT: "INSECURE_REDIRECT";
};

/**
 *  This utility function assists the upper stream in extracting error messages from the server response.
 *  A server response with an error may be presented in two formats: as a property of the response object, or nested within an array in the response
 *  Please refer to https://git.corp.adobe.com/pages/caf/api-spec/chapters/responses/service_responses.html#Error-Responses for the specific format of the error.
 * @param response
 * @internal
 */
export declare function _handleErrorResponsePayload(response: AdobeResponse): AdobeResponse<any>;

/** @internal */
export declare const HTTP_STATUS_ERROR_MAP: Map<number, {
    code: "UNEXPECTED_RESPONSE";
    message: string;
} | {
    code: "UNAUTHORIZED";
    message: string;
} | {
    code: "FORBIDDEN";
    message: string;
} | {
    code: "NOT_FOUND";
    message: string;
} | {
    code: "METHOD_NOT_ALLOWED";
    message: string;
} | {
    code: "NOT_ACCEPTABLE";
    message: string;
} | {
    code: "ALREADY_EXISTS";
    message: string;
} | {
    code: "PRECONDITION_FAILED";
    message: string;
} | {
    code: "NOT_IMPLEMENTED";
    message: string;
} | {
    code: "EXCEEDS_QUOTA";
    message: string;
} | {
    code: "BANDWIDTH_LIMIT_EXCEEDED";
    message: string;
}>;

export declare function isAdobeDCXError<T = unknown>(p: unknown): p is AdobeDCXError<T>;

export declare function networkError(message: string, error?: unknown, response?: AdobeResponse): AdobeDCXError;

/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
/**
 * @see {@link https://git.corp.adobe.com/pages/caf/api-spec/single.html#problem-types RAPI Spec: Problem Types}
 */
export declare enum ProblemTypes {
    ACCESS_DENIED = "http://ns.adobe.com/adobecloud/problem/accessdenied",
    ASSET_LOCKED = "http://ns.adobe.com/adobecloud/problem/assetlocked",
    ASSET_MOVED = "http://ns.adobe.com/adobecloud/problem/assetmoved",
    ASSET_NAME_CONFLICT = "http://ns.adobe.com/adobecloud/problem/assetnameconflict",
    ASSET_NAME_INVALID = "http://ns.adobe.com/adobecloud/problem/assetnamenotvalid",
    ASSET_NOT_FOUND = "http://ns.adobe.com/adobecloud/problem/assetnotfound",
    ASSET_STATE_NOT_ALLOWED = "http://ns.adobe.com/adobecloud/problem/assetstatenotallowed",
    BAD_REQUEST = "http://ns.adobe.com/adobecloud/problem/badrequest",
    BULK_REQUEST_NOT_ATTEMPTED = "http://ns.adobe.com/adobecloud/problem/bulkrequestnotattempted",
    COMPOSITE_INTEGRITY = "http://ns.adobe.com/adobecloud/problem/compositeintegrity",
    DCX_VALIDATION = "http://ns.adobe.com/adobecloud/problem/dcxvalidation",
    DIRECTORY_NOT_EMPTY = "http://ns.adobe.com/adobecloud/problem/directorynotempty",
    EMBED_INVALID = "http://ns.adobe.com/adobecloud/problem/embedinvalid",
    EMBED_TOO_LARGE = "http://ns.adobe.com/adobecloud/problem/embedtoolarge",
    ENCRYPTION_KEY_INACCESSIBLE = "http://ns.adobe.com/adobecloud/problem/encryptionkeyinaccessible",
    INVALID_FRAGMENT = "http://ns.adobe.com/adobecloud/problem/invalidfragment",
    LIMIT_CHILDREN_COUNT = "http://ns.adobe.com/adobecloud/problem/limit/childrencount",
    LIMIT_COMPONENT_COUNT = "http://ns.adobe.com/adobecloud/problem/limit/componentcount",
    LIMIT_EMBED_SELECTOR_COUNT = "http://ns.adobe.com/adobecloud/problem/limit/embedspecifierselectorcount",
    LIMIT_MILESTONE_COUNT = "http://ns.adobe.com/adobecloud/problem/limit/milestonecount",
    LIMIT_MILESTONE_LABEL_LENGTH = "http://ns.adobe.com/adobecloud/problem/limit/milestonelabellength",
    LIMIT_NAME_LENGTH = "http://ns.adobe.com/adobecloud/problem/limit/namelength",
    LIMIT_OPERATION_COUNT = "http://ns.adobe.com/adobecloud/problem/limit/operationcount",
    LIMIT_PATH_SEGMENT_COUNT = "http://ns.adobe.com/adobecloud/problem/limit/pathsegmentcount",
    LIMIT_RESOURCE_COUNT = "http://ns.adobe.com/adobecloud/problem/limit/resourcecount",
    LIMIT_RESOURCE_SIZE = "http://ns.adobe.com/adobecloud/problem/limit/resourcesize",
    /**
     * When user have zero storage assigned to them
     */
    NOT_ENTITLED = "http://ns.adobe.com/adobecloud/problem/notentitled",
    OPERATION_FAILED = "http://ns.adobe.com/adobecloud/problem/operationfailed",
    OPERATION_TARGET_CONFLICT = "http://ns.adobe.com/adobecloud/problem/operation/targetconflict",
    QUOTA_EXCEEDED = "http://ns.adobe.com/adobecloud/problem/quotaexceeded",
    REPOSITORY_NOT_FOUND = "http://ns.adobe.com/adobecloud/problem/repositorynotfound",
    RESOURCE_BLOCKED = "http://ns.adobe.com/adobecloud/problem/resourceblocked",
    RESOURCE_NOT_ALLOWED = "http://ns.adobe.com/adobecloud/problem/resourcenotallowed",
    RESOURCE_NOT_FOUND = "http://ns.adobe.com/adobecloud/problem/resourcenotfound",
    RESOURCE_NOT_READY = "http://ns.adobe.com/adobecloud/problem/resourcenotready",
    RESPONSE_TOO_LARGE = "http://ns.adobe.com/adobecloud/problem/responsetoolarge",
    USER_BLOCKED = "http://ns.adobe.com/adobecloud/problem/userblocked",
    VERSION_NOT_FOUND = "http://ns.adobe.com/adobecloud/problem/versionnotfound",
    PARTIAL_ASSET = "http://ns.adobe.com/adobecloud/problem/partialasset"
}

/** @internal */
export declare const _responseToError: (response: AdobeResponse) => AdobeDCXError<unknown> | undefined;

export declare function unexpectedResponse(message: string, error?: unknown, response?: AdobeResponse): AdobeDCXError;

export { }
