/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
import { AdobeResponse, AdobeDCXError as IAdobeDCXError } from '@dcx/common-types';
import { ProblemTypes } from './problems';

//******************************************************************************
// Error Codes
//******************************************************************************

export const ErrorCodes = {
    /**
     * This error indicates that the asset has changed regions and the existing links are no longer valid.
     * New links are sent along with a server generated response for asset moved errors.
     * Generally speaking, dcx-js will attempt to handle this error and update the links and retry the requested
     * operation using the new asset location. If this error is received by a client, dcx-js handling is likely
     * not yet implemented or does not have a viable resolution without further user input.
     */
    ASSET_MOVED: 'ASSET_MOVED',
    /**
     * Parsing JSON data has failed.
     */
    INVALID_JSON: 'INVALID_JSON',

    /** Trying to modify an immutable object or property.
     */
    READ_ONLY: 'READ_ONLY',
    /** The parameters passed to a function are not as expected.
     * @constant {String}
     */
    INVALID_PARAMS: '',
    /**
     * Links are invalid, used with leaf functions when they don't receive valid links.
     * Links are either missing or do not contain the required relation.
     * @constant {string}
     */
    INVALID_LINKS: 'INVALID_LINKS',
    /**
     * A conditional check failed for the request
     * @constant {String}
     */
    PRECONDITION_FAILED: 'PRECONDITION_FAILED',
    /** Data is invalid. Usually this means that a document read from disk or from an http request is bad.
     * @constant {String}
     */
    INVALID_DATA: 'INVALID_DATA',
    /** Uniqueness constraint violated.
     * @constant {String}
     */
    DUPLICATE_VALUE: 'DUPLICATE_VALUE',
    /** Trying to invoke functionality that requires that AdobeDCX being initialized with the xhrBaseBranchSupport option.
     * @constant {String}
     */
    NO_BASE_BRANCH_DATA: 'NO_BASE_BRANCH_DATA',
    /** An object is not in the expected state.
     * @constant {String}
     */
    INVALID_STATE: 'INVALID_STATE',
    /** Invalid operation on a deleted composite.
     * @constant {String}
     */
    DELETED_COMPOSITE: 'DELETED_COMPOSITE',
    /** Composite did not pass integrity check. Incomplete composite.
     * @constant {String}
     */
    INCOMPLETE_COMPOSITE: 'INCOMPLETE_COMPOSITE',
    /** Unexpected Response
     * @constant {String}
     */
    UNEXPECTED_RESPONSE: 'UNEXPECTED_RESPONSE',
    /** Network error
     * @constant {String}
     */
    NETWORK_ERROR: 'NETWORK_ERROR',
    /** Component(s) download failure -- see error.failedComponents for more details.
     * @constant {String}
     */
    COMPONENT_DOWNLOAD_ERROR: 'COMPONENT_DOWNLOAD_ERROR',
    /** Component(s) upload failure -- see error.failedComponents for more details.
     * @constant {String}
     */
    COMPONENT_UPLOAD_ERROR: 'COMPONENT_UPLOAD_ERROR',
    /** Component file could not be removed since it was modified -- see error.componentId for the id of the component.
     * @constant {String}
     */
    COMPONENT_MODIFIED_ERROR: 'COMPONENT_MODIFIED_ERROR',
    /** Update Conflict
     * @constant {String}
     */
    UPDATE_CONFLICT: 'UPDATE_CONFLICT',
    /** No composite. Possibly deleted.
     * @constant {String}
     */
    NO_COMPOSITE: 'NO_COMPOSITE',
    /** Respouce already Exists.
     * @constant {String}
     */
    ALREADY_EXISTS: 'ALREADY_EXISTS',
    /** HTTP session is in inactive state.
     * @constant {String}
     */
    SERVICE_IS_INACTIVE: 'SERVICE_IS_INACTIVE',
    /** Exceeds quota
     * @constant {String}
     */
    EXCEEDS_QUOTA: 'EXCEEDS_QUOTA',
    /** Unimplemented server request
     * @constant {String}
     */
    NOT_IMPLEMENTED: 'NOT_IMPLEMENTED',
    /** Retryable server error
     * @constant {String}
     */
    RETRYABLE_SERVER_ERROR: 'RETRYABLE_SERVER_ERROR',
    /** Timed-out request
     * @constant {String}
     */
    TIMED_OUT: 'TIMED_OUT',
    /** Unexpected failure, usually a problem acquiring a local resource such as a file
     * @constant {String}
     */
    UNEXPECTED: 'UNEXPECTED',
    /** Input stream terminated abnormaly
     * @constant {String}
     */
    TERMINATED_INPUTSTREAM: 'TERMINATED_INPUTSTREAM',
    /** Trying to access an asset from a different endpoint
     * @constant {String}
     */
    WRONG_ENDPOINT: 'WRONG_ENDPOINT',
    /** A file operation failed because the disk is full
     * @constant {String}
     */
    OUT_OF_SPACE: 'ENOSPC',
    // We use the POSIX error code returned by the file system so that we do not have to wrap/translate.
    // If you change this make sure to add the proper wrapping/translation.

    /** A composite could not be created because a regular file already exists at the specified path
     * @constant {String}
     */
    FILE_EXISTS_IN_CLOUD: 'FILE_EXISTS_IN_CLOUD',
    /** A resource (manifest, component or rendition) could not be found.
     * @constant {String}
     */
    ASSET_NOT_FOUND: 'ASSET_NOT_FOUND',
    /** A composite could not be found.
     * @constant {String}
     */
    COMPOSITE_NOT_FOUND: 'COMPOSITE_NOT_FOUND',
    /**
     * Generic not found error.
     * @constant {string}
     */
    NOT_FOUND: 'NOT_FOUND',
    /**
     * Unauthorized (401)
     * @constant {string}
     */
    UNAUTHORIZED: 'UNAUTHORIZED',
    /**
     * Forbidden (403)
     * @constant {string}
     */
    FORBIDDEN: 'FORBIDDEN',
    /**
     * Method Not Allowed (HTTP Status Code 405)
     */
    METHOD_NOT_ALLOWED: 'METHOD_NOT_ALLOWED',
    /**
     * Method Not Allowed (HTTP Status Code 406)
     */
    NOT_ACCEPTABLE: 'NOT_ACCEPTABLE',
    /**
     * Bandwidth Limit Exceeded (HTTP Status Code 509)
     */
    BANDWIDTH_LIMIT_EXCEEDED: 'BANDWIDTH_LIMIT_EXCEEDED',
    /**
     * Request aborted
     */
    ABORTED: 'ABORTED',
    /**
     * Request was redirected over set maximum redirects
     */
    TOO_MANY_REDIRECTS: 'TOO_MANY_REDIRECTS',
    /**
     * Request containing Authorization header redirected to HTTP
     */
    INSECURE_REDIRECT: 'INSECURE_REDIRECT',
} as const;

const noRewrap = {
    [ErrorCodes.SERVICE_IS_INACTIVE]: true,
    [ErrorCodes.ABORTED]: true,
    [ErrorCodes.INSECURE_REDIRECT]: true,
    [ErrorCodes.TOO_MANY_REDIRECTS]: true,
    [ErrorCodes.NOT_IMPLEMENTED]: true,
    [ErrorCodes.EXCEEDS_QUOTA]: true,
    [ErrorCodes.RETRYABLE_SERVER_ERROR]: true,
    [ErrorCodes.TIMED_OUT]: true,
    [ErrorCodes.TERMINATED_INPUTSTREAM]: true,
    [ErrorCodes.WRONG_ENDPOINT]: true,
    [ErrorCodes.OUT_OF_SPACE]: true,
    [ErrorCodes.INVALID_PARAMS]: true,
    [ErrorCodes.INVALID_STATE]: true,
} as const;

/**
 * @class
 * @classdesc `AdobeDCXError` is a subclass of `Error` which defines the DCX specific errors.
 * @augments Error
 * @hideconstructor
 * @param {String} code            The error code string.
 * @param {Str}    message         A string describing the error.
 * @param {Error}  underlyingError Optional underlying error.
 */
export class AdobeDCXError<T = unknown> extends Error implements IAdobeDCXError<T> {
    readonly name = 'AdobeDCXError';
    private _message: string | undefined;
    private _additionalData: T = {} as T;
    private _response?: AdobeResponse;
    private _underlyingError?: Error | AdobeDCXError;

    public stack: string | undefined;

    static readonly ABORTED = ErrorCodes.ABORTED;
    static readonly INSECURE_REDIRECT = ErrorCodes.INSECURE_REDIRECT;
    static readonly TOO_MANY_REDIRECTS = ErrorCodes.TOO_MANY_REDIRECTS;
    static readonly INVALID_JSON = ErrorCodes.INVALID_JSON;
    static readonly READ_ONLY = ErrorCodes.READ_ONLY;
    static readonly INVALID_PARAMS = ErrorCodes.INVALID_PARAMS;
    static readonly INVALID_DATA = ErrorCodes.INVALID_DATA;
    static readonly DUPLICATE_VALUE = ErrorCodes.DUPLICATE_VALUE;
    static readonly NO_BASE_BRANCH_DATA = ErrorCodes.NO_BASE_BRANCH_DATA;
    static readonly INVALID_STATE = ErrorCodes.INVALID_STATE;
    static readonly DELETED_COMPOSITE = ErrorCodes.DELETED_COMPOSITE;
    static readonly INCOMPLETE_COMPOSITE = ErrorCodes.INCOMPLETE_COMPOSITE;
    static readonly UNEXPECTED_RESPONSE = ErrorCodes.UNEXPECTED_RESPONSE;
    static readonly NETWORK_ERROR = ErrorCodes.NETWORK_ERROR;
    static readonly COMPONENT_DOWNLOAD_ERROR = ErrorCodes.COMPONENT_DOWNLOAD_ERROR;
    static readonly COMPONENT_UPLOAD_ERROR = ErrorCodes.COMPONENT_UPLOAD_ERROR;
    static readonly COMPONENT_MODIFIED_ERROR = ErrorCodes.COMPONENT_MODIFIED_ERROR;
    static readonly UPDATE_CONFLICT = ErrorCodes.UPDATE_CONFLICT;
    static readonly NO_COMPOSITE = ErrorCodes.NO_COMPOSITE;
    static readonly ALREADY_EXISTS = ErrorCodes.ALREADY_EXISTS;
    static readonly SERVICE_IS_INACTIVE = ErrorCodes.SERVICE_IS_INACTIVE;
    static readonly EXCEEDS_QUOTA = ErrorCodes.EXCEEDS_QUOTA;
    static readonly NOT_IMPLEMENTED = ErrorCodes.NOT_IMPLEMENTED;
    static readonly RETRYABLE_SERVER_ERROR = ErrorCodes.RETRYABLE_SERVER_ERROR;
    static readonly TIMED_OUT = ErrorCodes.TIMED_OUT;
    static readonly UNEXPECTED = ErrorCodes.UNEXPECTED;
    static readonly TERMINATED_INPUTSTREAM = ErrorCodes.TERMINATED_INPUTSTREAM;
    static readonly WRONG_ENDPOINT = ErrorCodes.WRONG_ENDPOINT;
    static readonly OUT_OF_SPACE = ErrorCodes.OUT_OF_SPACE;
    static readonly FILE_EXISTS_IN_CLOUD = ErrorCodes.FILE_EXISTS_IN_CLOUD;
    static readonly ASSET_NOT_FOUND = ErrorCodes.ASSET_NOT_FOUND;
    static readonly COMPOSITE_NOT_FOUND = ErrorCodes.COMPOSITE_NOT_FOUND;
    static readonly NOT_FOUND = ErrorCodes.NOT_FOUND;
    static readonly UNAUTHORIZED = ErrorCodes.UNAUTHORIZED;
    static readonly FORBIDDEN = ErrorCodes.FORBIDDEN;
    static readonly PRECONDITION_FAILED = ErrorCodes.PRECONDITION_FAILED;

    constructor(
        readonly code: string,
        message?: string,
        underlyingError?: unknown,
        response?: AdobeResponse,
        additionalData?: T,
    ) {
        super();
        // try to make sure the error is usable.
        if (
            response?.headers?.['content-type'] === 'application/problem+json' &&
            response.response &&
            typeof response.response === 'object' &&
            typeof response.response.slice === 'function'
        ) {
            try {
                const json = JSON.parse(new TextDecoder('utf-8').decode(response.response));
                response.response = json;
            } catch (parseError) {
                const originalBody = response.response;
                response.response = {
                    originalBody,
                    message: 'Failed to parse JSON problem type response body.',
                    parseError,
                };
            }
        }
        if (underlyingError instanceof Error) {
            this._underlyingError = underlyingError;
        }
        this._response = response || (isAdobeDCXError(underlyingError) ? underlyingError.response : undefined);
        this._additionalData = additionalData as T;

        this._message = message;
        this.message = `${typeof code === 'string' && code !== '' ? '[' + code + '] ' : ''}` + (this._message || '');

        Object.setPrototypeOf(this, AdobeDCXError.prototype);

        if (Error.captureStackTrace) {
            Error.captureStackTrace(this, AdobeDCXError);
        } else {
            try {
                const tmp = new Error();
                tmp.name = this.name;
                /* istanbul ignore else */
                if (tmp.stack) {
                    // remove the stack frame created above
                    const parts = tmp.stack.split('\n');
                    /* istanbul ignore else */
                    if (parts.length > 0) {
                        parts.splice(1, 1);
                    }
                    this.stack = parts.join('\n');
                }
            } catch (_) {
                //noop
            }
        }
    }
    public get response() {
        return this._response;
    }

    public get problemType() {
        if (this._response?.headers['content-type'] !== 'application/problem+json') {
            return;
        }
        return this._response.response.type;
    }
    public get underlyingError() {
        return this._underlyingError;
    }
    public get additionalData(): T {
        return this._additionalData;
    }
    /**
     * @internal
     */
    public set additionalData(val: T) {
        this._additionalData = val;
    }

    /**
     * For backwards compatibility.
     * Allow some properties to be accessible through their old property keys.
     *
     * Don't include in documentation or type declarations.
     *
     * @internal
     * @private
     */
    /* istanbul ignore next */
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    public get failedComponents(): any {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        return (this as any)._additionalData.failedComponents;
    }

    /** @private */
    static wrapError(code: string, message: string, error?: unknown, response?: AdobeResponse): AdobeDCXError {
        // Don't re-wrap certain error codes
        if (error && (noRewrap as Record<string, boolean>)[(error as AdobeDCXError).code]) {
            return error as AdobeDCXError;
        }

        if (response && typeof response === 'object') {
            const status = response.statusCode;
            const isQuotaExceededErrorResponse =
                status === 403 &&
                // CCStorage returns a code in the message, 403.1 indicates quota exceeded by the user
                (response.response?.message?.match(/code=(\d+.\d+)/)?.[1] === '403.1' ||
                    // RAPI returns the problem type alongside the 403 status code
                    response.response?.type === ProblemTypes.QUOTA_EXCEEDED);

            // If status code is a 5xx error, return suitable error code
            /* istanbul ignore else  */
            if ((status >= 500 && status < 600) || isQuotaExceededErrorResponse) {
                if (status === 501) {
                    code = ErrorCodes.NOT_IMPLEMENTED;
                    message = 'Unimplemented request';
                } else if (status === 507 || isQuotaExceededErrorResponse) {
                    code = ErrorCodes.EXCEEDS_QUOTA;
                    message = 'Quota exceeded';
                } else {
                    code = ErrorCodes.RETRYABLE_SERVER_ERROR;
                    message = 'Server error';
                }
            } else if (
                error instanceof AdobeDCXError &&
                code === error.code &&
                error.code === this.UNEXPECTED_RESPONSE
            ) {
                return error as AdobeDCXError;
            }
        }
        return new AdobeDCXError(code, message, error, response);
    }

    toString(): string {
        return `${this.name}: ${this.message}`;
    }

    public static networkError(message: string, error?: unknown, response?: AdobeResponse) {
        return AdobeDCXError.wrapError(ErrorCodes.NETWORK_ERROR, message, error, response);
    }

    public static unexpectedResponse(message: string, error?: unknown, response?: AdobeResponse) {
        return AdobeDCXError.wrapError(ErrorCodes.UNEXPECTED_RESPONSE, message, error, response);
    }
}

export function networkError(message: string, error?: unknown, response?: AdobeResponse): AdobeDCXError {
    return AdobeDCXError.networkError(message, error, response);
}

export function unexpectedResponse(message: string, error?: unknown, response?: AdobeResponse): AdobeDCXError {
    return AdobeDCXError.unexpectedResponse(message, error, response);
}

export function isAdobeDCXError<T = unknown>(p: unknown): p is AdobeDCXError<T> {
    if (!p || typeof p !== 'object') {
        return false;
    }

    return (p as Record<string, string>).name === 'AdobeDCXError';
}

/**
 *  This utility function assists the upper stream in extracting error messages from the server response.
 *  A server response with an error may be presented in two formats: as a property of the response object, or nested within an array in the response
 *  Please refer to https://git.corp.adobe.com/pages/caf/api-spec/chapters/responses/service_responses.html#Error-Responses for the specific format of the error.
 * @param response
 * @internal
 */
export function _handleErrorResponsePayload(response: AdobeResponse) {
    const error = Array.isArray(response.response)
        ? response.response.reduce((err, res) => err || res.error, undefined)
        : response.response.error;
    // early exit -- no error just keep moving
    if (!error) {
        return response;
    }
    const maybeError = _defaultStatusValidator(error.status, response);
    if (maybeError instanceof AdobeDCXError) {
        throw maybeError;
    }
    throw new AdobeDCXError(
        error.type || AdobeDCXError.UNEXPECTED_RESPONSE,
        error.title || 'Unexpected Error',
        error,
        response,
    );
}
/** @internal */
export const HTTP_STATUS_ERROR_MAP = new Map([
    [400, { code: ErrorCodes.UNEXPECTED_RESPONSE, message: 'Bad request' }],
    [401, { code: ErrorCodes.UNAUTHORIZED, message: 'Unauthorized' }],
    [403, { code: ErrorCodes.FORBIDDEN, message: 'Forbidden' }],
    [404, { code: ErrorCodes.NOT_FOUND, message: 'Not found' }],
    [
        405,
        {
            code: ErrorCodes.METHOD_NOT_ALLOWED,
            message: 'The user is authorized to act on this resource, but cannot use the specified method.',
        },
    ],
    [
        406,
        {
            code: ErrorCodes.NOT_ACCEPTABLE,
            message:
                'Unable to obtain resource in a content type matching the Accept header or rendition type parameter.',
        },
    ],
    [409, { code: ErrorCodes.ALREADY_EXISTS, message: 'Already exists' }],
    [412, { code: ErrorCodes.PRECONDITION_FAILED, message: 'Precondition failed' }],
    [501, { code: ErrorCodes.NOT_IMPLEMENTED, message: 'Not implemented' }],
    [507, { code: ErrorCodes.EXCEEDS_QUOTA, message: 'Exceeds quota' }],
    [509, { code: ErrorCodes.BANDWIDTH_LIMIT_EXCEEDED, message: 'Bandwidth limit exceeded' }],
]);

const PROBLEM_TYPE_ERROR_MAP = new Map(
    Object.entries({
        [ProblemTypes.ASSET_MOVED]: {
            code: ErrorCodes.ASSET_MOVED,
            message: 'Asset moved to a different region while operation was in progress',
        },
        [ProblemTypes.COMPOSITE_INTEGRITY]: {
            code: ErrorCodes.INCOMPLETE_COMPOSITE,
            message: 'Incomplete composite. invoke missingComponentsFromError with this error for more information.',
        },
        [ProblemTypes.PARTIAL_ASSET]: {
            code: ErrorCodes.NO_COMPOSITE,
            message: 'Asset is partially created. No Manifest found',
        },
    }),
);

/** @internal */
export const _responseToError = (response: AdobeResponse): AdobeDCXError<unknown> | undefined => {
    const errorInfo =
        PROBLEM_TYPE_ERROR_MAP.get(response.response?.type ?? '') || HTTP_STATUS_ERROR_MAP.get(response.statusCode);
    return errorInfo
        ? new AdobeDCXError(errorInfo.code, errorInfo.message, undefined, response, response.response)
        : undefined;
};
/** @internal */
export const _defaultStatusValidator = (
    statusCode?: number,
    response?: AdobeResponse,
): boolean | AdobeDCXError<unknown> => {
    if (!statusCode || !response) {
        return new AdobeDCXError(
            ErrorCodes.NETWORK_ERROR,
            'Invalid or missing status code or response',
            undefined,
            response,
        );
    }

    // unless explicitly marked an error, 200-299 is valid
    if (statusCode < 300 && statusCode > 199) {
        return true;
    }

    return _responseToError(response) || false;
};
