# @dcx/http

HTTP related classes and methods. Notably includes the AdobeHTTPService interface implementation. Instances of an [HTTPService](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_http.httpservice.html) are required by any method or class that uses HTTP.                                   

<a name="authentication"></a>
## Authentication

Whether you are dealing with dcx specific workflows or just wanting to work directly with the Repository API you will need to create an HTTP Service instance. The [@dcx/http](/packages/http) module provides the [AdobeHTTPService](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_http.httpservice.html) class used by a storage session or the HLA classes in the assets module to make requests. It also exposes a `createHTTPService` factory method to aid in creating an [AdobeHTTPService](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_http.httpservice.html) instance.

#### Creating an HTTP Service

A factory method call `createHTTPService` can be used to instantiate an HTTP Service.
```js
const { createHTTPService } = require('@dcx/http');
```

How you instantiate the HTTP service depends on how you want to handle authentication.

#### Token-based Authentication

If your application authenticates directly with IMS you need to provide the service with an
authentication token:
```js
const { createHTTPService } = require('@dcx/dcx-js');

const service = createHTTPService(function (service) {
    // get a new token asynchronously
    myGetToken(function (token) {
        service.setAuthToken(token);
    });
});
service.setApiKey('MyApiKey');
service.setAdditionalHeaders( {'foo': 'bar'} );
```
The constructor for the [AdobeHTTPService](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_http.httpservice.html) takes a function as its only (optional) argument which
gets called whenever it needs a new authentication token. The call to `setAuthToken()` does not have
to happen immediately, the service will pause until it gets a new token.

Use `setApiKey()` to set the API key for your client and `setAdditionalHeaders()` to set any
additional required headers. Finally, you pass the new service instance along with the base URL of
the service into the constructor for a new `AdobeRepoAPISession`.

You will want to hold on to the session instance since you are going to need it later.

#### Cookie-based Authentication

In browser-based environments it is generally safer to hide the credentials/token from the JavaScript
code by using HTTPS-only cookies.

```js
const service = createHTTPService(function (service) {
    // renew the cookie asynchronously
    myRenewCookie(function () {
        // resume the service once we have a new cookie
        service.resume();
    });
});
service.crossOriginCredentials = true;
service.setAdditionalHeaders( {'foo': 'bar'} );
const session = createRepoAPISession(service, 'https://my.proxy.adobe.com');
```

The constructor for the [AdobeHTTPService](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_http.httpservice.html) takes a function as its only (optional) argument which
gets called whenever a request fails with a 401 response. Your code should then get/renew the cookie
and call `service.resume()` when that is done. The call to `resume()` does not have
to happen immediately, the service will pause until you call `resume()`.

Set `service.crossOriginCredentials` in order to ensure that the cookie gets used when making
cross origin requests. Use `setAdditionalHeaders()` to set any additional required headers.
Finally, you pass the new service instance along with the base URL of the service into the constructor
for a new `AdobeRepoAPISession`.

You will want to hold on to the session instance since you are going to need it later.

#### No Authentication

If you don't need to authenticate (e.g. your app is only using publicly shared composites) you can
simply not provide a callback to `createHTTPService`:

```js
const service = createHTTPService();
const session = createRepoAPISession(service, 'https://platform-cs-stage.adobe.io');
```

You will want to hold on to the session instance since you are going to need it later.

### Configuration
The HTTPService is highly configurable. It can be configured during construction of the Service, through setters on it and on individual `invoke()` calls though the options argument.

### fetch
By default the HTTPService uses XMLHttpRequest for all requests except those with a "stream" responseType. If you would prefer to have the HTTPService use the 
fetch API instead, you can pass in the `preferFetch` option configured to `true`.
```js
const service = createHTTPService(authenticationCallback, {perferFetch: true});
```
When preferFetch is true, the global `fetch` API will be used for requests instead of XMLHttpRequest.

### Feature flags
There are some feature flags which can be set on the HTTPService directly. These feature flags are expected to be temporary in nature, and should eventually be ignored as features and their integration solidify into stable APIs. When constructing an HTTPService, feature flags can be passed directly in the HTTPOptions parameter. After construction they can be configured directly on the httpService.featureFlags property.
#### Current Feature Flags
| Feature Flag | Type | Description | Default |
| ------------ | ---- | ----------- | ------- |
| useLinksAPI | boolean | Whether or not to use the Links API (ResolveLite) when attempting to fetch links for a given asset | false |

### Retry Options
Requests are retried by default when a 429 or 500-series (except 501/507) status code is received. These can be set by passing an options object to the `HTTPService#setRetryOptions` API.

| Option | Type | Description | Default |
| ---- | ---- | ---- | ---- |
| retryCodes | `(number | RegExp)[]` | An array of numbers or regular expressions to check the status code against to determine whether to retry. If the list contains the status code or one of the regular expressions matches, the request will be retried. | `[/^(?!^501$|^507$)^(5\d{2})$|429|423$/]` (423, 429, 5XX except 501 & 507)
| timeoutAfter | `number` | A total amount of time to wait in milliseconds for all combined request waits. Does not factor in the duration of the request, only the duration of the waits between retries. | `72000`
| increase | `BackoffIncreaser`, `(requestNumber: number, lastWait: number, initialWait: number) => number` | A function that returns the next duration in milliseconds to wait before retrying a request. | Exponential backoff
| preferRetryAfterHeader | `boolean` | Whether to attempt to use the failure response's `Retry-After` header. Uses either Date [directive](https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Retry-After#directives) or seconds directive, depending on header. If no retry-after header is found, will default to using the increaser function. | `true`
| initialWait | `number` | Number of milliseconds to wait before attempting the first retry. | `2000`
| maxWait | `number` | Number of milliseconds at which to stop increasing the wait duration. If the increaser function is customized, this will not be honored and it is up to clients to cap their increaser function. | `32000`
| disableRetry | `boolean` | Disable all retries entirely | `false`
| preScheduleCallback | `ScheduleCallback`, `(snapshot?: BackoffSnapshot<T>) => void | Promise<void>` | Called when the request is about to be scheduled for some duration in the future. | `undefined`
| postScheduleCallback | `ScheduleCallback` | Called immediately after the request has been scheduled for retry. | `undefined`
| preCallback | `PrePostBackoffCallback`, `(xhr?: AdobeXhr<T>, snapshot?: BackoffSnapshot<T>) => void | Promise<void>` | Called immediately before the retry request is sent. | `undefined`
| postCallback | `PrePostBackoffCallback` | Called immediately on response of the retry request. | `undefined`
| pollHeader | `string` | Response header to check for a URL to poll, when a specific response code is encountered. Used only if polling is enabled for the response code encountered. | `undefined`
| pollMethod | `HTTPMethod` | Method to invoke during polling | `GET`
| pollCodes | `(number | RegExp)[]` | Similar to retryCodes, but for initiating polling on receipt of a specific response code. | `[]` 

> Note: The options `pollHeader`, `pollMethod`, and `pollCodes` may be overridden by DCX-JS methods which invoke the service to handle internal async polling.