import { AdditionalNodeOptions } from '@dcx/common-types';
import { AdobeAuthProvider } from '@dcx/common-types';
import { AdobeDCXError } from '@dcx/common-types';
import { AdobeHTTPService } from '@dcx/common-types';
import { AdobeHTTPServiceOptions } from '@dcx/common-types';
import AdobePromise from '@dcx/promise';
import { AdobeResponse } from '@dcx/common-types';
import { AdobeResponseType } from '@dcx/common-types';
import { AdobeXhr } from '@dcx/common-types';
import { AdobeXhrNode } from '@dcx/common-types';
import { AuthChangeHandler } from '@dcx/common-types';
import { AuthData } from '@dcx/common-types';
import { AuthenticationCallback } from '@dcx/common-types';
import { BackoffOptions } from '@dcx/common-types';
import { BodyType } from '@dcx/common-types';
import { DetachHandler } from '@dcx/common-types';
import { HTTPMethod } from '@dcx/common-types';
import { InvokeOptions } from '@dcx/common-types';
import { PersistentRefreshCallback } from '@dcx/common-types';
import { PostRequestHook } from '@dcx/common-types';
import { PreRequestHook } from '@dcx/common-types';
import { ProgressListener } from '@dcx/common-types';
import { RequestDescriptor } from '@dcx/common-types';
import { ResponseCallback } from '@dcx/common-types';
import { ResponseTypeMap } from '@dcx/common-types';
import { SinonFakeXMLHttpRequestStatic } from 'sinon';
import { XhrOptions } from '@dcx/common-types';
import { XhrResponse } from '@dcx/common-types';

export { AdobeHTTPService }

export { AdobeHTTPServiceOptions }

/**
 * The {@link AdobeNetworkHTTPService} class. Gives access to the DCX network service.
 * @readonly
 * @type {AdobeNetworkHTTPService}
 */
export declare const AdobeNetworkHTTPService: {
    new (authHandler?: AuthenticationCallback | AdobeAuthProvider, options?: AdobeHTTPServiceOptions): AdobeHTTPService;
};

export { AdobeResponse }

export { AdobeResponseType }

export { AuthenticationCallback }

export declare class AuthProvider implements AdobeAuthProvider {
    private _authToken?;
    private _apiKey?;
    private _pendingAuth;
    private _hasBaseRefreshCb;
    private _authListeners;
    private _persistentListeners;
    private _refreshResolve?;
    private _refreshPromise?;
    private _authenticationAllowList;
    private _authTokenScheme;
    constructor(_authToken?: string | undefined, _apiKey?: string | undefined, persistentRefreshCb?: PersistentRefreshCallback);
    /**
     * Used for redirects.
     *
     * Sets a list of hostnames that forwarding auth token & API key is allowed.
     *
     * By default, will strip auth token from any non-https host
     * as well as any host that is not identical to the initial hostname.
     */
    get authenticationAllowList(): string[];
    set authenticationAllowList(val: string[]);
    /**
     * Returns true when the AuthProvider is initialized without any persistent refresh callback
     * Assumes there's no way for the authentication to come from later handlers.
     * If this isn't the case, clients can set this manually after registering their
     * handler that will provide refreshing.
     */
    get isNoAuthMode(): boolean;
    set isNoAuthMode(val: boolean);
    get apiKey(): string | undefined;
    get authToken(): string | undefined;
    get authTokenScheme(): string | undefined;
    set authTokenScheme(val: string | undefined);
    /**
     * Sets a new auth token and emits the {@link AuthEvent} `updated`.
     *
     * @param {string} authToken
     */
    setAuthToken(authToken: string | undefined): void;
    /**
     * Set API Key.
     *
     * @param {string} apiKey - apiKey to set
     */
    setApiKey(apiKey: string): void;
    /**
     * Explicitly mark provider as no longer pending authentication.
     * Can be used instead of Promise returns.
     */
    resume(): void;
    /**
     * Get pending state.
     * Returns true when the provider is waiting for authentication to complete.
     */
    get pendingAuth(): boolean;
    /**
     * Register a change handler.
     * Possible event types: {@link AuthEvent}
     * Handlers may return promises, but unlike the refreshAuthCallback
     * constructor parameter, the promises are assumed to return void
     * and resolution values are awaited, but not used.
     * @param {AuthChangeHandler} handler
     * @returns {DetachHandler}
     */
    onChange(handler: AuthChangeHandler, persistent?: boolean): DetachHandler;
    /**
     * Remove all non-persistent listeners.
     * If passed `true`, will clear persistent listeners, too.
     *
     * @param {boolean} [clearPersistent = false] - Clear persistent listeners.
     */
    clearListeners(clearPersistent?: boolean): void;
    /**
     * @internal
     */
    get refreshPromise(): Promise<AuthData> | undefined;
    /**
     * First set pending state according to the event.
     * Then call auth change listeners.
     *
     * If a listener returns a promise, collect it and resolve all promises together.
     *
     * If the event is `updated`, resolve any waiting promise AFTER the listeners.
     * This means that the refreshAuth promise will resolve AFTER all registered
     * listeners, which is good since HTTPService will reactivate itself in a listener.
     *
     * @param {AuthEvent} event
     */
    private _authChanged;
    private _resolveRefresh;
    /**
     * Emit the {@link AuthEvent} `unauthenticated` event,
     * wait for all handlers to complete,
     * then return the auth data.
     *
     * @returns {Promise<AuthData>}
     */
    refreshAuth(): Promise<AuthData>;
    /**
     * Get current authentication data as object.
     *
     * @returns {AuthData}
     */
    getAuthData(): AuthData;
    /**
     * Get current auth data as a promise.
     */
    getAuth(): Promise<AuthData>;
    /**
     * Check if URL has domain that belongs to allow list.
     * @param url
     */
    isAuthorizedURL(url: string): boolean;
    /**
     * Clear auth data, reset to pending authentication.
     */
    logout(): void;
    /**
     * Applies Authorization and X-Api-Key headers if the url is part of the allowed
     * @param headers
     */
    applyAuthHeaders(url: string, headers: Record<string, string>): Record<string, string>;
}

/**
 * Constructs and returns a network service instance.
 * @public
 * @example
 * ```js
 * // using AuthProvider
 * const refreshCb = async (service, provider) => {
 *   const t = await getToken();
 *   service.setAuthToken(t);
 * }
 * const ap = new AuthProvider('token', 'apikey', refreshCb);
 * const service = createHTTPService(ap);
 * service.setAdditionalHeaders( {'foo': 'bar'} );
 * ```
 *
 * @example
 * ```js
 * // using callback
 * const service = createHTTPService((service) => {
 *      myGetToken((token) => {
 *          service.setAuthToken(token);
 *      });
 * });
 * service.setApiKey('MyApiKey');
 * service.setAdditionalHeaders( {'foo': 'bar'} );
 * ```
 *
 * @param  {AuthenticationCallback | AdobeAuthProvider} [authHandler]
 *                                          - Either a function or AuthProvider instance.
 *                                            Called when the HTTPService receives a 401 status code.
 *                                            The callback should handle setting a new auth token, which
 *                                            then resumes the service, or explicitly call `resume()`.
 *                                            If not provided, will ignore authentication.
 * @param { AdobeHTTPServiceOptions }   [options] - Configuration options
 */
export declare const createHTTPService: (authHandler?: AuthenticationCallback | AdobeAuthProvider, options?: AdobeHTTPServiceOptions) => AdobeHTTPService;

/**
 * @class
 * @classdesc A network service instance.
 * <p>The constructor for `AdobeNetworkHTTPService` is private. Refer to {@link AdobeDCX} to learn how
 * to create instances of `AdobeNetworkHTTPService`.
 * @param {AuthenticationCallback | AuthProvider} [authHandler] Called when authentication fails. This callback is only called
 * once per expired/invalid token/cookie. The service will queue up any further requests. <p>The client must obtain and set a new
 * authentication token or cookie and then call resume()</p> <p>If not provided then the service will not try to manage
 * authentication.</p> <p>Signature: authCallback(httpService)</p>
 */
export declare class HTTPService implements AdobeHTTPService {
    readonly name = "AdobeHTTPService";
    private _requestQueue;
    private _requestsOutstanding;
    private _authProvider;
    private _isActive;
    private _waitingForAuthentication;
    private _forcedXhr?;
    private _preferFetch;
    private _fetch?;
    private _handlesRedirects;
    private _maxOutstanding;
    private _server?;
    private _withCredentials;
    private _timeout;
    private _additionalHeaders?;
    private _additionalNodeOptions;
    private _retryOptions;
    private _beforeHook?;
    private _afterHook?;
    private _isStatusValid?;
    private _requestIdPrefix?;
    private _serviceGuid;
    private _reqNum;
    featureFlags: any;
    /**
     * @param {AuthProvider | AuthenticationCallback} authHandler - AuthProvider or callback.
     *      When using a callback, use {@link AdobeHTTPServiceOptions} `useAuthProvider` option set
     *      to true to receive the auth provider instead of the service.
     */
    constructor(authHandler?: AuthenticationCallback | AdobeAuthProvider, options?: AdobeHTTPServiceOptions);
    /**
     * An inactive service will error out with a AdobeDCXError.SERVICE_IS_INACTIVE error
     * whenever a request is being made. Clients can set isActive to false when the user
     * has explicitly logged out.
     * @memberof AdobeNetworkHTTPService#
     * @default true
     * @type {Boolean}
     */
    get isActive(): boolean;
    set isActive(value: boolean);
    /**
     * If true then XHR instances will have the withCredentials property set which means that user
     * credentials (stored in a cookie) will be used in cross origin requests.
     * @memberof AdobeNetworkHTTPService#
     * @type {Boolean}
     */
    get crossOriginCredentials(): boolean;
    set crossOriginCredentials(value: boolean);
    /**
     * The maximum number of outstanding requests over this HTTP Service instance.
     * The default value is 5, to stay below browser connection limits, but the limit
     * can be increased if you have special circumstances that warrant it.
     * Remember that there is no use in exceeding the bandwith capacity of your client
     * or of the host, and that in error cases it's better to have half of your
     * uploads complete than all of your uploads half complete.
     * @memberof AdobeNetworkHTTPService#
     */
    get maxOutstanding(): number;
    set maxOutstanding(value: number);
    /**
     * Whether the platform handles redirects natively.
     * Returns true for browser and false for Node,
     * although Node builds may also be configured to handle redirects.
     */
    get handlesRedirects(): boolean;
    get server(): string | undefined;
    set server(val: string | undefined);
    /** force use of xhr (generally for testing)
     * @internal
     */
    _forceXhr(xhr: SinonFakeXMLHttpRequestStatic | undefined, xhrDoesNotHandleRedirects?: boolean): void;
    /** force use of a specific fetch API (generally for testing)
     * @internal
     */
    _useFetchApi(fetchApi: typeof fetch): void;
    /**
     * Sets additional headers.
     * @param {Object} additionalHeaders An object containing key-value pairs for each additional header to supply.
     */
    setAdditionalHeaders(additionalHeaders: Record<string, string>): void;
    /**
     * Sets a function that denotes whether a Promise should resolve or reject given a status
     * By default, this is function that always returns true meaning any status code will resolve.
     * @param {(status?: number, response?: AdobeResponse) => boolean} fn
     */
    setValidateStatus(fn: (status?: number, xhr?: AdobeResponse) => boolean): void;
    /**
     * Sets additional options.
     * @param {Object} additionalOptions An object containing additional options to
     *                                   supply with each http request (in Node only).
     */
    setAdditionalNodeOptions(aOpts: AdditionalNodeOptions): void;
    /**
     * Set options for retrying failed requests
     * @param {BackoffOptions<any>} retryOptions
     */
    setRetryOptions(retryOptions: BackoffOptions<never>): void;
    /**
     * Used for redirects.
     *
     * Sets a list of hostnames that forwarding auth token & API key is allowed.
     *
     * By default, will strip auth token from any non-https host
     * as well as any host that is not identical to the initial hostname.
     */
    set authenticationAllowList(value: string[]);
    get authenticationAllowList(): string[];
    get authProvider(): AdobeAuthProvider;
    /**
     * Sets the API key.
     * @param {String} apiKey The apikey to supply with each request.
     */
    setApiKey(apiKey: string): void;
    /**
     * Sets the timeout value for requests.
     * This is the per-request timeout, not related to backoff timeouts.
     * To set the backoff timeout, provide a value in the `retryOptions`
     * property of the invoke options.
     * @param {Integer} timeout timeout value in ms or undefined if no timeout
     */
    setTimeout(timeout: number): void;
    /**
     * Sets the authentication token and resumes the service.
     * @param {String} token The authentication token, if not provided, logs out
     */
    setAuthToken(token?: string): void;
    private _onAuthChange;
    /**
     * Call this when you do not use an auth token for authentication and you need to let the
     * service know that you have renewed the authentication cookie so that the service can
     * resume to make requests.
     */
    resume(): void;
    /**
     * Sets callbacks before and after invocation hook.
     * (For testing)
     * @private
     * @param before a hook to call before each http request is issued. Signature: before(req).
     * @param after  a hook to be called after each http request is initiated: Signature: after(req, xhr).
     */
    setRequestHooks(before?: PreRequestHook, after?: PostRequestHook): void;
    invoke<T extends AdobeResponseType>(method: HTTPMethod, href: string, headers?: Record<string, string>, body?: BodyType, options?: InvokeOptions<T>): AdobePromise<AdobeResponse<T>, AdobeDCXError, RequestDescriptor>;
    /**
     * Create Request and add to the outstandingRequest map.
     * @param requestDesc
     */
    private _makeRequest;
    /**
     * Initiate _checkQueueLoop asynchronously
     */
    private _checkQueue;
    /**
     * Iterate through queue until one of:
     * 1. maxOutstanding is reached
     * 2. authentication expires
     * 3. queue is empty
     *
     * Make the request, then notify queuer that the request has been sent.
     */
    private _checkQueueLoop;
    /**
     * Get the request Promise.
     * The request is always queued, which returns a promise that resolves when the request is
     * sent and added to the outstandingRequest map.
     * @param {RequestDescriptor} requestDesc - request descriptor
     * @returns {AdobePromise<[AdobeResponse<T>, T], AdobeDCXError, RequestDescriptor>}
     */
    private _getRequestPromise;
    /** @private */
    private abort;
    /** @internal */
    abortAllWithToken(token: unknown): void;
}

export { RequestDescriptor }

/**
 * Type alias to {@link InvokeOptions}
 */
export declare type RequestOptions<T extends AdobeResponseType = AdobeResponseType> = InvokeOptions<T>;

export { ResponseCallback }

export declare class Xhr<T extends AdobeResponseType> implements AdobeXhr<T> {
    private _autoParsedResponse;
    private _autoParseJson;
    private _bytesReported;
    private _errorCode;
    private _estimatedTotalBytes?;
    private _isFetchRequest;
    private _fetch?;
    private _fetchAbort;
    private _fetchResponse?;
    private _parsedResponseHeaders?;
    private _fetchBodyAsResponseType?;
    private _postCallback?;
    private _preCallback?;
    private _preferFetch;
    private _promise;
    private _resolve;
    private _response?;
    private _sent;
    private _timeout;
    private _timeoutTimeout;
    private _underlyingError?;
    private _xhr;
    href: string;
    method: HTTPMethod;
    headers: Record<string, string | string[]>;
    body?: BodyType;
    responseType: AdobeResponseType;
    private _progressListeners;
    constructor(opts?: XhrOptions<T>);
    get xhr(): XMLHttpRequest | AdobeXhrNode;
    private _parseFetchResponse;
    private _shouldAutoParseResponse;
    private _fetchWithTimeout;
    private _finalize;
    private _validateResponseType;
    send(url: string, method: HTTPMethod, body?: BodyType, headers?: Record<string, string | string[]>, responseType?: T, options?: XhrOptions<T>): Promise<AdobeXhr<T>>;
    abort(): void;
    getResponseHeader(header: string): string | undefined;
    getAllResponseHeaders(): Record<string, string>;
    isError(): boolean;
    isAborted(): boolean;
    isTimedOut(): boolean;
    isSent(): boolean;
    getErrorCode(): number | string;
    getStatus(): number;
    getResponse(): XhrResponse<T>;
    toJSON(): {
        statusCode: number;
        headers: Record<string, string>;
        responseType: T;
        response: ResponseTypeMap[T];
        message: string;
    };
    getResponseDataAsJSON(): Promise<any>;
    getResponseData(): ResponseTypeMap[T];
    onProgress(handler: ProgressListener): () => void;
    private _notifyProgressListeners;
}

export declare const XhrErrorCodes: {
    NO_ERROR: string;
    ABORTED: "ABORTED";
    NETWORK: "NETWORK_ERROR";
    TIMEOUT: "TIMED_OUT";
    TOO_MANY_REDIRECTS: "TOO_MANY_REDIRECTS";
    INSECURE_REDIRECT: "INSECURE_REDIRECT";
};

export { }
