/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    AdobeAuthProvider,
    AuthChangeHandler,
    AuthData,
    AuthEvent,
    DetachHandler,
    PersistentRefreshCallback,
} from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import { getDomainFromURL, pruneUndefined, validateParams } from '@dcx/util';

const dbg = newDebug('dcx:http:auth');

export class AuthProvider implements AdobeAuthProvider {
    private _pendingAuth = false;
    private _hasBaseRefreshCb = false;
    private _authListeners: (AuthChangeHandler | undefined)[] = [];
    private _persistentListeners: number[] = [];
    private _refreshResolve?: (authData: AuthData) => void;
    private _refreshPromise?: Promise<AuthData>;
    private _authenticationAllowList: string[] = ['adobe.com', 'adobe.io', 'adobelogin.com', 'fotolia.net'];
    private _authTokenScheme: string | undefined = 'Bearer';

    constructor(
        private _authToken?: string,
        private _apiKey?: string,
        persistentRefreshCb?: PersistentRefreshCallback,
    ) {
        // TODO: decode token and setTimeout for token expiration

        validateParams(
            ['authToken', _authToken, 'string', true],
            ['apiKey', _apiKey, 'string', true],
            ['refreshCb', persistentRefreshCb, 'function', true],
        );

        if (persistentRefreshCb) {
            this._hasBaseRefreshCb = true;
            this.onChange((event, provider) => {
                if (event === 'unauthenticated') {
                    persistentRefreshCb.call(null, provider);
                }
            }, true);
        }

        if (!_authToken || !_apiKey) {
            dbg('init unauthenticated');
            this._pendingAuth = true;

            // Allow constructor to register hooks
            // by emitting unauth event on next tick.
            // Also, it's possible clients call resume() in same
            // tick as constructor, to signify use of cookies.
            setTimeout(() => {
                dbg('after tick', this._pendingAuth);
                if (this._pendingAuth) {
                    this.refreshAuth();
                }
            });
        }
    }

    /**
     * Used for redirects.
     *
     * Sets a list of hostnames that forwarding auth token & API key is allowed.
     *
     * By default, will strip auth token from any non-https host
     * as well as any host that is not identical to the initial hostname.
     */
    public get authenticationAllowList(): string[] {
        return this._authenticationAllowList;
    }
    public set authenticationAllowList(val: string[]) {
        if (!Array.isArray(val)) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting an array.');
        }
        this._authenticationAllowList = val;
    }

    /**
     * Returns true when the AuthProvider is initialized without any persistent refresh callback
     * Assumes there's no way for the authentication to come from later handlers.
     * If this isn't the case, clients can set this manually after registering their
     * handler that will provide refreshing.
     */
    public get isNoAuthMode(): boolean {
        return !this._hasBaseRefreshCb;
    }
    public set isNoAuthMode(val: boolean) {
        this._hasBaseRefreshCb = !val;
    }

    public get apiKey(): string | undefined {
        return this._apiKey;
    }
    public get authToken(): string | undefined {
        return this._authToken;
    }

    public get authTokenScheme(): string | undefined {
        return this._authTokenScheme;
    }
    public set authTokenScheme(val: string | undefined) {
        this._authTokenScheme = val;
    }

    /**
     * Sets a new auth token and emits the {@link AuthEvent} `updated`.
     *
     * @param {string} authToken
     */
    public setAuthToken(authToken: string | undefined): void {
        dbg('setAuthToken');

        this._authToken = authToken;
        this._pendingAuth = false;
        this._authChanged('updated');
    }

    /**
     * Set API Key.
     *
     * @param {string} apiKey - apiKey to set
     */
    public setApiKey(apiKey: string): void {
        this._apiKey = apiKey;
    }

    /**
     * Explicitly mark provider as no longer pending authentication.
     * Can be used instead of Promise returns.
     */
    public resume() {
        dbg('resume()');
        this._pendingAuth = false;
        this._authChanged('updated');
    }

    /**
     * Get pending state.
     * Returns true when the provider is waiting for authentication to complete.
     */
    public get pendingAuth() {
        return this._pendingAuth;
    }

    /**
     * Register a change handler.
     * Possible event types: {@link AuthEvent}
     * Handlers may return promises, but unlike the refreshAuthCallback
     * constructor parameter, the promises are assumed to return void
     * and resolution values are awaited, but not used.
     * @param {AuthChangeHandler} handler
     * @returns {DetachHandler}
     */
    public onChange(handler: AuthChangeHandler, persistent = false): DetachHandler {
        dbg('onChange, persistent:', persistent);

        const index = this._authListeners.push(handler) - 1;
        if (persistent) {
            this._persistentListeners.push(index);
        }
        return () => {
            try {
                if (persistent) {
                    this._persistentListeners = this._persistentListeners.filter((e) => e !== index);
                }
                delete this._authListeners[index];
            } catch (_) {
                // already deleted
            }
        };
    }

    /**
     * Remove all non-persistent listeners.
     * If passed `true`, will clear persistent listeners, too.
     *
     * @param {boolean} [clearPersistent = false] - Clear persistent listeners.
     */
    public clearListeners(clearPersistent = false) {
        dbg('clearListeners, persistent:', clearPersistent);

        if (clearPersistent === true) {
            this._authListeners = [];
            this._persistentListeners = [];
            return;
        }
        this._authListeners = this._authListeners.map((val, ind) => {
            if (!this._persistentListeners.includes(ind)) {
                return undefined;
            }
            return val;
        });
    }

    /**
     * @internal
     */
    get refreshPromise(): Promise<AuthData> | undefined {
        return this._refreshPromise;
    }

    /**
     * First set pending state according to the event.
     * Then call auth change listeners.
     *
     * If a listener returns a promise, collect it and resolve all promises together.
     *
     * If the event is `updated`, resolve any waiting promise AFTER the listeners.
     * This means that the refreshAuth promise will resolve AFTER all registered
     * listeners, which is good since HTTPService will reactivate itself in a listener.
     *
     * @param {AuthEvent} event
     */
    private async _authChanged(event: AuthEvent) {
        dbg('authChanged', event);

        if (event === 'unauthenticated') {
            this._pendingAuth = true;
        } else {
            this._pendingAuth = false;
        }

        queueMicrotask(async () => {
            const toWait: Promise<void>[] = [];
            this._authListeners.map((handler) => {
                if (typeof handler === 'function') {
                    const res = handler.call(null, event, this);
                    if (res && typeof res === 'object' && (res as Promise<void>).then) {
                        toWait.push(res);
                    }
                }
            });

            await Promise.all(toWait);

            if (event === 'updated') {
                // resolve the waiting promise if exists
                this._resolveRefresh();
            }
        });
    }

    private _resolveRefresh() {
        dbg('_resolveRefresh');

        if (this._refreshResolve) {
            this._refreshResolve(this.getAuthData());
        }
        this._refreshResolve = undefined;
        this._refreshPromise = undefined;
    }

    /**
     * Emit the {@link AuthEvent} `unauthenticated` event,
     * wait for all handlers to complete,
     * then return the auth data.
     *
     * @returns {Promise<AuthData>}
     */
    public refreshAuth(): Promise<AuthData> {
        dbg('refreshAuth');

        if (this._refreshPromise) {
            // only refresh one time
            return this._refreshPromise;
        }

        this._refreshPromise = new Promise((resolve) => {
            this._refreshResolve = resolve;
        });

        this._authChanged('unauthenticated');

        return this._refreshPromise;
    }

    /**
     * Get current authentication data as object.
     *
     * @returns {AuthData}
     */
    public getAuthData(): AuthData {
        return {
            authToken: this._authToken,
            apiKey: this._apiKey,
        };
    }

    /**
     * Get current auth data as a promise.
     */
    async getAuth(): Promise<AuthData> {
        return Promise.resolve(this.getAuthData());
    }

    /**
     * Check if URL has domain that belongs to allow list.
     * @param url
     */
    public isAuthorizedURL(url: string): boolean {
        const domain = getDomainFromURL(url);
        return this._authenticationAllowList.includes(domain);
    }

    /**
     * Clear auth data, reset to pending authentication.
     */
    public logout() {
        dbg('logout');

        this._apiKey = undefined;
        this._authToken = undefined;
        if (this._pendingAuth === false) {
            this._pendingAuth = true;
            this._authChanged('unauthenticated');
        }
    }

    /**
     * Applies Authorization and X-Api-Key headers if the url is part of the allowed
     * @param headers
     */
    public applyAuthHeaders(url: string, headers: Record<string, string>): Record<string, string> {
        const authHeaders: Record<string, string | undefined> = {
            'x-api-key': undefined,
            authorization: undefined,
        };

        if (this.isAuthorizedURL(url)) {
            if (headers['x-api-key'] !== null && this.apiKey) {
                authHeaders['x-api-key'] = this.apiKey;
            }
            if (headers['authorization'] !== null && this.authToken) {
                authHeaders['authorization'] =
                    (this.authTokenScheme ? `${this.authTokenScheme} ` : '') + this.authToken;
            }
        }

        headers = pruneUndefined({
            ...headers,
            ...authHeaders,
        });

        return headers;
    }
}
