/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    AdditionalNodeOptions,
    AdobeAuthProvider,
    AdobeDCXError,
    AdobeHTTPService,
    AdobeHTTPServiceOptions,
    AdobeRequest,
    AdobeResponse,
    AdobeResponseType,
    AuthenticationCallback,
    AuthEvent,
    BackoffOptions,
    BodyType,
    HTTPMethod,
    InvokeOptions,
    PostRequestHook,
    PreRequestHook,
    RequestDescriptor,
    ResponseCallback,
} from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import {
    endPointOf,
    generateUuid,
    isAnyFunction,
    isNode,
    isObject,
    merge,
    now,
    pruneUndefined,
} from '@dcx/util';
import { SinonFakeXMLHttpRequestStatic } from 'sinon';
import { AuthProvider } from './AuthProvider';
import { DEFAULT_TIMEOUT } from './defaults';
import { AdobeRequestMaker } from './Request';
import { RequestMap } from './RequestMap';
import { QueuedRequest, RequestQueue } from './RequestQueue';

const dbg = newDebug('dcx:http:service');

// 5 Minutes = The maximum time we allow a request to be delayed using the noSoonerThen property
const MAXDELAY = 5 * 60 * 1000;

// maximum concurrent requests
const DEFAULT_OUTSTANDING = 5;

/**
 * @class
 * @classdesc A network service instance.
 * <p>The constructor for `AdobeNetworkHTTPService` is private. Refer to {@link AdobeDCX} to learn how
 * to create instances of `AdobeNetworkHTTPService`.
 * @param {AuthenticationCallback | AuthProvider} [authHandler] Called when authentication fails. This callback is only called
 * once per expired/invalid token/cookie. The service will queue up any further requests. <p>The client must obtain and set a new
 * authentication token or cookie and then call resume()</p> <p>If not provided then the service will not try to manage
 * authentication.</p> <p>Signature: authCallback(httpService)</p>
 */
export class HTTPService implements AdobeHTTPService {
    public readonly name = 'AdobeHTTPService';

    private _requestQueue: RequestQueue = new RequestQueue();
    private _requestsOutstanding: RequestMap = new RequestMap();
    private _authProvider: AuthProvider = undefined as unknown as AuthProvider;

    private _isActive = true;
    private _waitingForAuthentication!: boolean;
    private _forcedXhr?: SinonFakeXMLHttpRequestStatic | undefined;
    private _preferFetch = false;
    private _fetch?: typeof fetch;
    private _handlesRedirects = true;
    private _maxOutstanding: number;

    private _server?: string;

    private _withCredentials = false;

    private _timeout: number;

    private _additionalHeaders?: { [key: string]: string };

    private _additionalNodeOptions: AdditionalNodeOptions = {};
    private _retryOptions: BackoffOptions<never> = {};

    private _beforeHook?: PreRequestHook;
    private _afterHook?: PostRequestHook;

    private _isStatusValid?: (status?: number, response?: AdobeResponse) => boolean | AdobeDCXError;

    private _requestIdPrefix?: string;

    private _serviceGuid: string = generateUuid();

    private _reqNum = 0;

    public featureFlags: any = {};

    /**
     * @param {AuthProvider | AuthenticationCallback} authHandler - AuthProvider or callback.
     *      When using a callback, use {@link AdobeHTTPServiceOptions} `useAuthProvider` option set
     *      to true to receive the auth provider instead of the service.
     */
    constructor(authHandler?: AuthenticationCallback | AdobeAuthProvider, options: AdobeHTTPServiceOptions = {}) {
        if (authHandler instanceof AuthProvider || (isObject(authHandler) && isAnyFunction(authHandler.onChange))) {
            this._authProvider = authHandler as AuthProvider;
        } else if (isAnyFunction(authHandler)) {
            // For backwards compatibility, wrap callbacks with service if the useAuthProvider option isn't set.
            if (!options.useAuthProvider) {
                this._authProvider = new AuthProvider(undefined, undefined, () => authHandler.call(null, this));
                this._waitingForAuthentication = true;
            } else {
                this._authProvider = new AuthProvider(undefined, undefined, authHandler as AuthenticationCallback);
                this._waitingForAuthentication = true;
            }
        }

        if (!this._authProvider) {
            // if still null, means running in unauthenticated mode
            // resume immediately and don't register hooks to stop/resume
            this._authProvider = new AuthProvider();
            this._authProvider.resume();
        } else {
            // register auth hooks to activate/deactivate service
            this._authProvider.onChange(this._onAuthChange.bind(this));
        }

        this._maxOutstanding = options.maxOutstanding || DEFAULT_OUTSTANDING;
        this._withCredentials = options.crossOriginCredentials || false;
        this._timeout = options.timeout == null ? DEFAULT_TIMEOUT : options.timeout;
        this._preferFetch = options.preferFetch === true;
        this._requestIdPrefix = options.requestIdPrefix;

        if (options.server) {
            this.server = options.server;
        }
    }

    //******************************************************************************
    // Getters/setters for properties
    //******************************************************************************

    /**
     * An inactive service will error out with a AdobeDCXError.SERVICE_IS_INACTIVE error
     * whenever a request is being made. Clients can set isActive to false when the user
     * has explicitly logged out.
     * @memberof AdobeNetworkHTTPService#
     * @default true
     * @type {Boolean}
     */
    get isActive() {
        return this._isActive;
    }
    set isActive(value) {
        const wasActive = this._isActive;
        if (wasActive !== value) {
            this._isActive = value;
            if (!value) {
                this._authProvider.logout();
                this._checkQueue();
            } else {
                this._checkQueue();
            }
        }
    }

    /**
     * If true then XHR instances will have the withCredentials property set which means that user
     * credentials (stored in a cookie) will be used in cross origin requests.
     * @memberof AdobeNetworkHTTPService#
     * @type {Boolean}
     */
    get crossOriginCredentials() {
        return this._withCredentials;
    }
    set crossOriginCredentials(value) {
        this._withCredentials = value;
    }

    /**
     * The maximum number of outstanding requests over this HTTP Service instance.
     * The default value is 5, to stay below browser connection limits, but the limit
     * can be increased if you have special circumstances that warrant it.
     * Remember that there is no use in exceeding the bandwith capacity of your client
     * or of the host, and that in error cases it's better to have half of your
     * uploads complete than all of your uploads half complete.
     * @memberof AdobeNetworkHTTPService#
     */
    get maxOutstanding() {
        return this._maxOutstanding;
    }
    set maxOutstanding(value) {
        this._maxOutstanding = value;
        // increasing the value might allow queued requests to be sent now.
        this._checkQueue();
    }

    /**
     * Whether the platform handles redirects natively.
     * Returns true for browser and false for Node,
     * although Node builds may also be configured to handle redirects.
     */
    get handlesRedirects() {
        return this._handlesRedirects;
    }

    get server(): string | undefined {
        return this._server;
    }
    set server(val: string | undefined) {
        this._server = val && val.endsWith('/') ? val.substr(0, val.length - 1) : val;
    }

    /** force use of xhr (generally for testing)
     * @internal
     */
    public _forceXhr(xhr: SinonFakeXMLHttpRequestStatic | undefined, xhrDoesNotHandleRedirects = false) {
        this._forcedXhr = xhr;
        this._handlesRedirects = !xhrDoesNotHandleRedirects;
    }

    /** force use of a specific fetch API (generally for testing)
     * @internal
     */
    public _useFetchApi(fetchApi: typeof fetch) {
        this._fetch = fetchApi;
    }

    /**
     * Sets additional headers.
     * @param {Object} additionalHeaders An object containing key-value pairs for each additional header to supply.
     */
    public setAdditionalHeaders(additionalHeaders: Record<string, string>) {
        this._additionalHeaders = additionalHeaders || {};
    }

    /**
     * Sets a function that denotes whether a Promise should resolve or reject given a status
     * By default, this is function that always returns true meaning any status code will resolve.
     * @param {(status?: number, response?: AdobeResponse) => boolean} fn
     */
    public setValidateStatus(fn: (status?: number, xhr?: AdobeResponse) => boolean) {
        this._isStatusValid = fn;
    }

    /**
     * Sets additional options.
     * @param {Object} additionalOptions An object containing additional options to
     *                                   supply with each http request (in Node only).
     */
    public setAdditionalNodeOptions(aOpts: AdditionalNodeOptions) {
        this._additionalNodeOptions = aOpts;
        if (isNode() && aOpts && aOpts.maxRedirects === 0) {
            this._handlesRedirects = false;
        }
    }

    /**
     * Set options for retrying failed requests
     * @param {BackoffOptions<any>} retryOptions
     */
    public setRetryOptions(retryOptions: BackoffOptions<never>) {
        this._retryOptions = retryOptions;
    }

    /**
     * Used for redirects.
     *
     * Sets a list of hostnames that forwarding auth token & API key is allowed.
     *
     * By default, will strip auth token from any non-https host
     * as well as any host that is not identical to the initial hostname.
     */
    public set authenticationAllowList(value: string[]) {
        this._authProvider.authenticationAllowList = value;
    }
    public get authenticationAllowList(): string[] {
        return this._authProvider.authenticationAllowList;
    }

    public get authProvider(): AdobeAuthProvider {
        return this._authProvider;
    }

    /**
     * Sets the API key.
     * @param {String} apiKey The apikey to supply with each request.
     */
    public setApiKey(apiKey: string) {
        this._authProvider.setApiKey(apiKey);
    }

    /**
     * Sets the timeout value for requests.
     * This is the per-request timeout, not related to backoff timeouts.
     * To set the backoff timeout, provide a value in the `retryOptions`
     * property of the invoke options.
     * @param {Integer} timeout timeout value in ms or undefined if no timeout
     */
    public setTimeout(timeout: number) {
        this._timeout = timeout;
    }

    /**
     * Sets the authentication token and resumes the service.
     * @param {String} token The authentication token, if not provided, logs out
     */
    public setAuthToken(token?: string) {
        if (!token) {
            this._authProvider.logout();
        } else {
            this._authProvider.setAuthToken(token);
        }
    }

    private _onAuthChange(event?: AuthEvent, provider?: AdobeAuthProvider) {
        dbg('_oAC', event);
        const before = this._waitingForAuthentication;
        if (event === 'unauthenticated') {
            this._waitingForAuthentication = true;
        } else {
            this._waitingForAuthentication = false;
            if (before !== false) {
                this._checkQueue();
                // provider.resume();
            }
        }
    }

    /**
     * Call this when you do not use an auth token for authentication and you need to let the
     * service know that you have renewed the authentication cookie so that the service can
     * resume to make requests.
     */
    public resume() {
        this._waitingForAuthentication = false;
        this._authProvider.resume();
        this._checkQueue();
    }

    /**
     * Sets callbacks before and after invocation hook.
     * (For testing)
     * @private
     * @param before a hook to call before each http request is issued. Signature: before(req).
     * @param after  a hook to be called after each http request is initiated: Signature: after(req, xhr).
     */
    public setRequestHooks(before?: PreRequestHook, after?: PostRequestHook) {
        this._beforeHook = before;
        this._afterHook = after;
    }

    invoke<T extends AdobeResponseType>(
        method: HTTPMethod,
        href: string,
        headers?: Record<string, string>,
        body?: BodyType,
        options?: InvokeOptions<T>,
    ): AdobePromise<AdobeResponse<T>, AdobeDCXError, RequestDescriptor>;
    /**
     * Issue a request.
     * By default, the request is a GET.
     * The body param specifies the data, if any.
     * The generic T denotes the responseType for type completion/checking.
     *
     * Note: use ONE OF callback or promise, using both may lead to undefined resolution.
     *
     * @example
     * ```js
     * // with callback
     * service.invoke<SomeInterface>('GET', 'https://server.url/path', {}, null,
     *  { responseType: 'json' },
     *  (err, xhr, data) => {
     *    // do stuff
     *    // data is typeof === 'object' if response was valid JSON
     *    // data is Type `SomeInterface`
     *  });
     * ```
     *
     * * @example
     * ```js
     * // with promise, async/await
     * try {
     *   const resp = await service.invoke<SomeInterface>(
     *      'GET',
     *      'https://server.url/path',
     *      {},
     *      null,
     *      { responseType: 'json' });
     *   // resp.response is typeof === 'object'
     *   // resp.response is Type `SomeInterface`
     * } catch(e) {
     *   // handle network error
     * }
     * ```
     * @param  {String}            method      - HTTP method
     * @param  {String}            href        - The URL; may be relative or absolute.
     *                                          If relative, the `server` property will be prepended (if exists)
     *                                          If absolute (contains endpoint & protocol), will be sent as-is
     * @param  {Record<string, string | string[]>} headers
     *                                         - The request headers
     * @param  {BodyType}          [body]      - The request body (Buffer/ArrayBufferView/Blob: data)
     * @param  {InvokeOptions}     [options]   - Object capturing additional request options.
     * @param  {ResponseCallback}  callback    - Called when the response arrives. Signature: callback(error, xhr)
     * @return {AdobePromise<[AdobeResponse<T>, T], AdobeDCXError, RequestDescriptor>}  -
     *                                          An AdobePromise with extended props of a RequestDescriptor.
     *                                          Resolves to a tuple of [AdobeResponse, Data] where data is the type of
     *                                          the `responseType` option. For type completion, use the generic `T` with
     *                                          the expected responseType.
     */
    public invoke<T extends AdobeResponseType = AdobeResponseType>(
        method: HTTPMethod = 'GET',
        href: string,
        headers: Record<string, string> = {},
        body?: BodyType,
        options: InvokeOptions<T> = {},
        callback?: ResponseCallback<T>,
    ): AdobePromise<AdobeResponse<T>, AdobeDCXError, RequestDescriptor> {
        dbg('invoke', method, href, options);

        options = options || {};

        let autoParseJson = options.autoParseJson || false;
        let responseType = options.responseType;

        if (responseType == null || responseType === 'void') {
            autoParseJson = options.autoParseJson != null ? options.autoParseJson : true;
            responseType = options.responseType = 'text' as T;
        }

        // set the default buffer based on platform
        if (responseType === 'defaultbuffer') {
            if (isNode()) {
                responseType = options.responseType = 'buffer' as T;
            } else {
                responseType = options.responseType = 'arraybuffer' as T;
            }
        }

        if (responseType === 'buffer') {
            /* istanbul ignore if */
            if (typeof Buffer !== 'function') {
                throw new DCXError(DCXError.INVALID_PARAMS, 'No Buffer class');
            }
        } else if (responseType === 'blob') {
            /* istanbul ignore if */
            if (typeof Blob !== 'function') {
                throw new DCXError(DCXError.INVALID_PARAMS, 'No Blob class');
            }
        } else if (
            responseType &&
            responseType !== 'text' &&
            responseType !== 'json' &&
            responseType !== 'arraybuffer' &&
            responseType !== 'stream'
        ) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Unsupported response type: ' + responseType);
        }

        // get endpoint, if relative and baseUrl is set, use it
        if (!endPointOf(href) && this.server) {
            href = `${this.server}/${href.startsWith('/') ? href.substr(1, href.length) : href}`;
        }
        dbg('invoke href', href);

        const additionalHeaders = merge({}, headers, this._additionalHeaders);
        dbg('invoke headers', headers);

        additionalHeaders['x-request-id'] = [
            this._requestIdPrefix,
            this._serviceGuid,
            additionalHeaders['x-request-id'],
            `${this._reqNum++}`,
        ]
            .filter((isDefined) => isDefined)
            .join('.');

        options.additionalNodeOptions = Object.assign(
            {},
            this._additionalNodeOptions || {},
            options.additionalNodeOptions || {},
        );

        options.isStatusValid = options.isStatusValid || this._isStatusValid;

        options.retryOptions = Object.assign({}, this._retryOptions || {}, options.retryOptions || {});

        // Create the request descriptor
        let requestDesc: RequestDescriptor = {
            method: method,
            href: href,
            headers: additionalHeaders,
            token: undefined,
            body: body,
            options: options,
            progress: options.reuseRequestDesc?.progress,
            autoParseJson,
        } as RequestDescriptor;

        // Check to see whether we are asked to reuse a previous request descriptor
        // If provided with a previous promise, use it's props as the descriptor
        let reuse = options.reuseRequestDesc;
        if (reuse && reuse instanceof AdobePromise && 'props' in reuse) {
            reuse = (reuse as AdobePromise<never, never, RequestDescriptor>).props;
        }

        if (reuse) {
            if (
                this._requestQueue.exists(reuse as RequestDescriptor) ||
                (reuse.id && this._requestsOutstanding.get(reuse.id) != null)
            ) {
                // in the future, may want to throw an error if the request is still queued
                // if the request is still outstanding, the
                dbg('requestDesc still in use');
            }

            requestDesc = merge(reuse, requestDesc);
        }

        // Ensure each request descriptor has an ID before creating the promise.
        // When reusing a request descriptor, keep the original ID.
        requestDesc.id = requestDesc.id || generateUuid();

        // For complete backwards compatibility:
        // When a callback is set, and maxRedirects is not
        // explicitly set, default to 0.
        // Same with retry/backoff.
        if (callback) {
            if ('maxRedirects' in options.additionalNodeOptions) {
                options.additionalNodeOptions.maxRedirects = 0;
            }

            if (!options.retryOptions || Object.keys(options.retryOptions).length === 0) {
                options.retryOptions = { disableRetry: true };
            }
        }

        const promise = this._getRequestPromise<T>(requestDesc);

        // after each resolution, check the queue
        promise.getPromise().then(this._checkQueue.bind(this)).catch(this._checkQueue.bind(this));

        if (isAnyFunction(callback)) {
            dbg('invoke - cb');
            // FUTURE: 6.0 deprecation warning

            promise
                .then((response) => {
                    dbg('invoke - cb - resolve ', response.statusCode);
                    try {
                        callback(undefined, response, response.response);
                    } catch (e) {
                        console.error('[dcx:http] error in success callback', e, (e as any).stack);
                    }
                })
                .catch((err) => {
                    dbg('invoke - cb - reject: ', err);

                    try {
                        callback(err, err.response);
                    } catch (e) {
                        console.error('[dcx:http] error in failure callback', e, (e as any).stack);
                    }
                });
        }

        this._checkQueue();

        return promise;
    }

    /**
     * Create Request and add to the outstandingRequest map.
     * @param requestDesc
     */
    private _makeRequest<T extends AdobeResponseType>(requestDesc: RequestDescriptor): AdobeRequest<T> {
        dbg('_makeRequest(): ', requestDesc.id);
        const opts = requestDesc.options || {};

        const req = AdobeRequestMaker.makeRequest<T>(
            pruneUndefined({
                url: requestDesc.href,
                autoParseJson: requestDesc.autoParseJson,
                descriptor: requestDesc,
                ...requestDesc,
                timeout: opts.timeout || this._timeout,
                authProvider: this._authProvider,
                forceXhr: this._forcedXhr,
                fetch: this._fetch,
                responseType: opts.responseType,
                preCallback: this._beforeHook,
                postCallback: this._afterHook,
                isStatusValid: opts.isStatusValid,
                additionalNodeOptions: opts.additionalNodeOptions,
                retryOptions: opts.retryOptions,
                isExternalRequest: opts.isExternalRequest,
                preferFetch: this._preferFetch,
            }),
        );
        this._requestsOutstanding.addRequest(requestDesc.id, req);

        // Set the start time
        requestDesc.startTime = new Date().valueOf();

        return req;
    }

    /**
     * Initiate _checkQueueLoop asynchronously
     */
    private _checkQueue() {
        queueMicrotask(this._checkQueueLoop.bind(this));
    }

    /**
     * Iterate through queue until one of:
     * 1. maxOutstanding is reached
     * 2. authentication expires
     * 3. queue is empty
     *
     * Make the request, then notify queuer that the request has been sent.
     */
    private _checkQueueLoop() {
        dbg(
            '_checkQueueLoop()',
            this._waitingForAuthentication,
            this.isActive,
            this._requestsOutstanding.length,
            '<?',
            this.maxOutstanding,
        );

        if (!this._isActive) {
            dbg('_cQL inactive');

            const err = new DCXError(DCXError.SERVICE_IS_INACTIVE, 'Network request in inactive state');
            // Cancel outstanding requests
            this._requestsOutstanding.clear(err);

            // Cancel queued requests
            this._requestQueue.clear(err);
        }

        // Iterate over the request queue repeatedly until we have reached the maximum number of requests, exhausted
        // the request queue of executeable request or lost our auth token.
        let qReq: QueuedRequest | undefined | true = true;
        while (qReq && !this._waitingForAuthentication && this._requestsOutstanding.length < this.maxOutstanding) {
            qReq = this._requestQueue.pop();
            if (qReq == null) {
                break;
            }

            const req = this._makeRequest(qReq.descriptor as RequestDescriptor);

            // notify sent
            qReq.notifySent(req);
        }
        dbg('_cQL done');
    }

    /**
     * Get the request Promise.
     * The request is always queued, which returns a promise that resolves when the request is
     * sent and added to the outstandingRequest map.
     * @param {RequestDescriptor} requestDesc - request descriptor
     * @returns {AdobePromise<[AdobeResponse<T>, T], AdobeDCXError, RequestDescriptor>}
     */
    private _getRequestPromise<T extends AdobeResponseType>(
        requestDesc: RequestDescriptor,
    ): AdobePromise<AdobeResponse<T>, AdobeDCXError, RequestDescriptor> {
        dbg('_getRequestPromise()');

        return new AdobePromise<AdobeResponse<T>, AdobeDCXError, RequestDescriptor>((resolve, reject, onCancel) => {
            if (!this._isActive) {
                dbg('_gRP inactive');
                return reject(new DCXError(DCXError.SERVICE_IS_INACTIVE, 'Network request in inactive state'));
            }

            // Register cancel handler to remove request from queue,
            // until the request is issued, then cancel it.
            onCancel(() => {
                dbg('_gRP cancel 1', requestDesc.id);
                this._requestQueue.remove(requestDesc);
            });

            // Add to queue which will return a promise
            // that resolves when the request has been popped.
            // If there's a time to wait, the queue will initiate a checkQueue loop
            // when the request's timeout is reached and it is moved to the queue.
            let wait = requestDesc.noSoonerThen || null;
            if (wait) {
                wait = wait - now();
                wait = wait < 0 ? 0 : wait > MAXDELAY ? MAXDELAY : wait;
            }
            delete requestDesc.noSoonerThen;

            return this._requestQueue
                .push(requestDesc, wait, this._checkQueue.bind(this))
                .then((req) => {
                    dbg('_gRP sent', requestDesc.id);

                    if (req.canceled) {
                        dbg('_gRP reject 1: ', requestDesc.id);
                        return reject(new DCXError(DCXError.ABORTED, 'Request aborted', req.error));
                    }

                    onCancel((reason) => {
                        dbg('_gRP cancel 2', requestDesc.id, reason);
                        req.cancel(new DCXError(DCXError.ABORTED, 'Request aborted', reason as Error));
                    });

                    return req
                        .getPromise()
                        .then((resp) => {
                            dbg('_gRP resolve 1', requestDesc.id, resp);

                            return resolve(resp);
                        })
                        .catch((err) => {
                            dbg('_gRP reject 2: ', requestDesc.id, err);
                            return reject(err);
                        });
                })
                .catch(reject);
        }, requestDesc);
    }

    /** @private */
    private abort(requestDesc: AdobePromise<any, any, RequestDescriptor> | RequestDescriptor) {
        if (requestDesc && (requestDesc as AdobePromise).cancel) {
            (requestDesc as AdobePromise).cancel();
        } else if (this._requestQueue.exists(requestDesc)) {
            this._requestQueue.remove(requestDesc);
        } else {
            this._requestsOutstanding.removeById(requestDesc.id);
        }
    }

    /** @internal */
    public abortAllWithToken(token: unknown) {
        dbg('abortAllWithToken()');
        this._requestsOutstanding.removeAllWithToken(token);
        this._requestQueue.removeAllWithToken(token);
    }
}

export default HTTPService;
