/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc..
 **************************************************************************/

// import { AdobeDCXError } from '@dcx/error';
import { AdobeDCXError, AdobeResponseType, AdobeXhr, BackoffSnapshot, BodyType, HTTPMethod } from '@dcx/common-types';
import { RequestEventType, RequestEvents, RequestOptions } from './RequestEvents';
import { backoff } from './backoff';

/**
 * Base network request events
 * All requests have these hooks
 */
export type NetworkRequestEvents = 'success' | 'error' | 'cancel' | 'progress';
export const NetworkRequestEvents = {
    success: 'success',
    error: 'error',
    cancel: 'cancel',
    progress: 'progress',
};

/**
 * Non-pausable network request
 * GET, HEAD, OPTIONS
 * and PUT, POST when body size < min chunk size
 */
export interface AdobeNetworkRequest<T extends AdobeResponseType> {
    getPromise(): Promise<AdobeXhr<T>>;
    cancel(err?: AdobeDCXError): void;
    getSnapshot(): BackoffSnapshot<T>;
    onProgress(handler): ReturnType<typeof backoff>['onProgress'];
}

export class NetworkRequest<T extends AdobeResponseType> implements AdobeNetworkRequest<T> {
    private _promise: Promise<AdobeXhr<T>>;
    private _cancel: (err?: AdobeDCXError) => void;
    private _getSnapshot: () => BackoffSnapshot<T>;
    onProgress;

    constructor(
        url: string,
        method: HTTPMethod,
        body: BodyType | undefined,
        headers: Record<string, string>,
        responseType: AdobeResponseType = 'text',
        notify?: <K extends RequestEventType>(event: K, data: RequestEvents[K]) => void,
        authCallback?: (url: string, headers: Record<string, string>) => Promise<Record<string, string>>,
        options: Partial<RequestOptions<T>> = {},
    ) {
        const { descriptor } = options;
        delete options.descriptor;

        const { cancel, getPromise, onProgress, getSnapshot } = backoff<T>(
            url,
            method,
            body,
            headers,
            options.timeout,
            {
                ...options,
                responseType,
                authCallback,
                ...options.retryOptions,
                descriptor,
                forceXhr: options.forceXhr,
                autoParseJson: options.autoParseJson,
            },
        );
        this.onProgress = onProgress;
        if (typeof notify === 'function') {
            onProgress((sentOrReceived, total) => notify('progress', { total, sentOrReceived }));
        }

        this._cancel = cancel;
        this._promise = getPromise();
        this._getSnapshot = getSnapshot;
    }

    getSnapshot(): BackoffSnapshot<T> {
        return this._getSnapshot();
    }

    getPromise(): Promise<AdobeXhr<T>> {
        return this._promise;
    }

    cancel(err?: AdobeDCXError) {
        this._cancel(err);
    }
}
