/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc..
 **************************************************************************/

import {
    AdobeAuthProvider,
    AdobeDCXError,
    AdobeRequest,
    AdobeResponse,
    AdobeResponseType,
    RequestDescriptor,
} from '@dcx/common-types';
import { DCXError, _defaultStatusValidator } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import { normalizeHeaders } from '@dcx/util';
import { AdobeNetworkRequest, NetworkRequest } from './NetworkRequest';
import { RequestEventHandler, RequestEventType, RequestEvents, RequestOptions } from './RequestEvents';
import { XhrErrorCodes } from './Xhr';

const dbg = newDebug('dcx:http:req');

/**
 * Handles creating the request or job
 *
 * @note
 * Jobs are currently not used, UploadJob logic is instead being implemented as BlockUpload in the assets module.
 *
 */
export class Request<T extends AdobeResponseType = any> implements AdobeRequest<T> {
    private static _internalOnlyHeaders = ['x-request-id', 'x-api-key', 'authorization'];
    private _id: string;
    private _networkRequest: AdobeNetworkRequest<T>;
    private _pausable = false;
    private _promise: Promise<AdobeResponse<T>>;
    private _isStatusValid: (status?: number, response?: AdobeResponse) => boolean | AdobeDCXError;
    private _listeners: Record<RequestEventType, RequestEventHandler<keyof RequestEvents>[]> = {
        progress: [],
        cancel: [],
    };
    private _isExternalRequest: boolean | undefined;

    private _authProvider: AdobeAuthProvider;
    private _descriptor: RequestDescriptor;

    constructor(opts: RequestOptions<T>) {
        this._isStatusValid = opts.isStatusValid || _defaultStatusValidator;
        this._isExternalRequest = opts.isExternalRequest;
        this._authProvider = opts.authProvider;
        this._id = opts.id;
        this._descriptor = opts.descriptor;

        // register first listener if it exists
        /* istanbul ignore next */
        if (opts.descriptor && opts.descriptor.progress) {
            const progress = opts.descriptor.progress;
            this.on('progress', ({ sentOrReceived, total }) => {
                progress.call(undefined, sentOrReceived, total);
            });
        }

        // apply auth headers, lowercase header keys, remove undefined/null entries
        const headers = this._authProvider.applyAuthHeaders(opts.url, normalizeHeaders(opts.headers || {}));

        // If a request is declared as an external request -- always remove internal only headers
        if (this._isExternalRequest) {
            Request._internalOnlyHeaders.forEach((header) => delete headers[header]);
        }

        // otherwise, use a regular and unpausable network request
        this._networkRequest = new NetworkRequest<T>(
            opts.url,
            opts.method,
            opts.body,
            headers,
            opts.responseType,
            opts.descriptor?.progress ? this._emit.bind(this) : undefined,
            this._getAuthCb(),
            opts,
        );

        this._promise = this._networkRequest
            .getPromise()
            .then((val) => {
                // check status code
                // allow option to override what is considered an error status
                const errCode = val.getErrorCode();
                const validOrError = errCode || this._isStatusValid(val.getStatus(), val.getResponse());

                if (errCode || validOrError !== true) {
                    // no test for network failure
                    /* istanbul ignore else */
                    if (errCode === XhrErrorCodes.ABORTED) {
                        throw new DCXError(DCXError.ABORTED, 'Aborted');
                    } else if (errCode === XhrErrorCodes.NETWORK) {
                        throw new DCXError(DCXError.NETWORK_ERROR, 'Network error', undefined, val.getResponse());
                    } else if (errCode === XhrErrorCodes.TIMEOUT) {
                        throw new DCXError(DCXError.TIMED_OUT, 'Timeout', undefined, val.getResponse());
                    }

                    // allow caller to set their own errors
                    if (validOrError instanceof DCXError || (validOrError as unknown) instanceof Error) {
                        // prefer AdobeDCXError._message if available, since it doesn't contain the [CODE] prefix
                        throw new DCXError(
                            (validOrError as DCXError).code || DCXError.UNEXPECTED_RESPONSE,
                            (validOrError as unknown as { _message: string })._message ||
                                (validOrError as AdobeDCXError).message,
                            (validOrError as AdobeDCXError).underlyingError,
                            val.getResponse(),
                        );
                    }

                    throw new DCXError(
                        DCXError.UNEXPECTED_RESPONSE,
                        'Unexpected response',
                        undefined,
                        val.getResponse(),
                    );
                }

                const requests = this._networkRequest.getSnapshot().requests;

                dbg('resolve', opts.id);

                return {
                    ...val.getResponse(),
                    xhr: requests[requests.length - 1],
                } as AdobeResponse<T>;
            })
            .catch((err) => {
                dbg('reject', opts.id);
                throw err;
            });
    }

    public get id(): string {
        return this._id;
    }

    public get descriptor(): RequestDescriptor {
        return this._descriptor;
    }

    private _getAuthCb() {
        if (!this._authProvider.isNoAuthMode) {
            return this._authCb.bind(this);
        }
        return;
    }

    private _authCb(url: string, headers: Record<string, string>): Promise<Record<string, string>> {
        dbg('_authCb()');
        if (this._authProvider.isAuthorizedURL(url)) {
            return this._authProvider.refreshAuth().then(() => this._authProvider.applyAuthHeaders(url, headers));
        }
        return Promise.reject(
            new DCXError(DCXError.UNAUTHORIZED, 'URL is not part of authenticationAllowList.', undefined, undefined, {
                url,
            }),
        );
    }

    private _emit<K extends RequestEventType>(event: K, data: RequestEvents[K]) {
        this._listeners[event].map((handler) => handler.call(null, data));
    }

    getPromise(): Promise<AdobeResponse<T>> {
        return this._promise;
    }

    cancel(err?: AdobeDCXError) {
        return this._networkRequest.cancel(err);
    }

    on<K extends RequestEventType>(event: K, handler: RequestEventHandler<K>) {
        if (event === 'progress' && this._listeners[event].length === 0) {
            this._networkRequest?.onProgress((sentOrReceived, total) => this._emit(event, { total, sentOrReceived }));
        }
        this._listeners[event].push(handler as RequestEventHandler<RequestEventType>);
    }
}

export const AdobeRequestMaker = {
    makeRequest: <T extends AdobeResponseType = AdobeResponseType>(opts: RequestOptions<T>): AdobeRequest<T> => {
        return new Request<T>(opts);
    },
};
