/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc..
 **************************************************************************/

import { AdobeDCXError, AdobeRequest, AdobeResponseType, RequestDescriptor } from '@dcx/common-types';
import { newDebug } from '@dcx/logger';

const dbg = newDebug('dcx:http:q');

export type NotifySentFn<T = AdobeRequest<any>> = (req: T) => void;
export type NotifyCancelFn = (err?: AdobeDCXError) => void;

export const RD_INDEX = 0;
export const SENT_INDEX = 1;
export const REJ_INDEX = 2;

const DEFAULT_IS_PRIORITY = (req: RequestDescriptor): boolean => {
    return req.method.toLowerCase() === 'head';
};

export interface QueuedRequest<T = AdobeRequest<any>> {
    descriptor: RequestDescriptor;
    notifySent: NotifySentFn<T>;
    wait?: number;
}

interface InternalQueuedRequest<T = AdobeRequest<any>> extends QueuedRequest<T> {
    notifyCanceled: NotifyCancelFn;
    notifyReady?: () => void;
    readyTimeout?: any;
}

interface CanceledRequest {
    canceled?: true;
    error?: AdobeDCXError;
}

/**
 * RequestQueue
 * Queues requests in preferential order for HEAD requests.
 *
 * Head end pointer points to the first index that is not a HEAD request, for example:
 * [HEAD][HEAD][HEAD][HEAD][POST][GET][PUT]
 * _headEndPtr = 4
 */
export class RequestQueue {
    private _queue: InternalQueuedRequest<AdobeRequest<any>>[] = [];
    private _later: Record<string, InternalQueuedRequest<AdobeRequest<any>>> = {};
    private _headEndPtr = 0;
    private _isPriority = DEFAULT_IS_PRIORITY;
    private _usePriority = false;

    /**
     * Add request to queue.
     * Returns promise that resolves when:
     * 1. The request is sent and the sender calls QueuedRequest.notifySent with the sent request
     * 2. The request is removed from the queue before sending, in which case `canceled` will be `true`
     *
     * @param request
     */
    public async push<K extends AdobeResponseType = any>(
        request: RequestDescriptor,
        wait?: number | null,
        onReady?: (req?: QueuedRequest<AdobeRequest<K>>) => void,
    ): Promise<AdobeRequest<K> & CanceledRequest> {
        let resolve;
        const promise = new Promise<AdobeRequest<K>>((resolve_) => {
            resolve = resolve_;
        });

        if (typeof wait !== 'number' || wait <= 0) {
            // add to the queue immediately
            const toAdd = {
                descriptor: request,
                notifySent: (req: AdobeRequest<K>) => resolve(req),
                notifyCanceled: this._notifyCanceled(resolve),
            };
            this._push(toAdd);
            return promise;
        }

        // if there's a wait, set timeout for when it's ready to add to the queue
        const { id } = request;
        const timer = setTimeout(() => {
            this._ready(id);
        }, wait);

        // make reject unregister the timer
        const notifySent = (req: AdobeRequest<K>) => resolve(req);
        this._later[id] = {
            readyTimeout: timer,
            wait,
            descriptor: request,
            notifySent,
            notifyCanceled: this._notifyCanceled(resolve),
            notifyReady: () => {
                if (onReady) {
                    onReady.call(null, { wait, descriptor: request, notifySent });
                }
            },
        };

        return promise;
    }

    private _notifyCanceled(resolveFn: (canceled: CanceledRequest) => void) {
        return (err?: AdobeDCXError) => {
            if (!err) {
                return resolveFn({ canceled: true });
            }
            resolveFn({ canceled: true, error: err });
        };
    }

    private _push<K extends AdobeResponseType>(qReq: InternalQueuedRequest<AdobeRequest<K>>) {
        if (this._usePriority && this._isPriority(qReq.descriptor)) {
            this._queue.splice(this._headEndPtr++, 0, qReq);
        } else {
            this._queue.push(qReq);
        }
    }

    // public async asyncPush(method: HTTPMethod, request: RequestDescriptor) {

    // }

    public remove(req: RequestDescriptor) {
        if (req.id in this._later) {
            dbg('remove from later', req.id);

            return this._remove(req.id);
        }

        const index = this._indexOf(req);
        dbg('remove from q', index);

        if (index >= 0) {
            return this._remove(index);
        }
    }

    private _remove(reqIdOrIndex: string | number, err?: AdobeDCXError) {
        dbg('_remove', reqIdOrIndex);

        if (typeof reqIdOrIndex === 'string') {
            const entry = this._later[reqIdOrIndex];
            entry.notifyCanceled.call(null, err);
            entry.readyTimeout && clearTimeout(entry.readyTimeout);
            delete this._later[reqIdOrIndex];
            return;
        }

        const toRemove = this._queue[reqIdOrIndex];
        toRemove.notifyCanceled.call(null, err);
        this._queue.splice(reqIdOrIndex, 1);
        if (reqIdOrIndex < this._headEndPtr) {
            this._headEndPtr--;
        }
    }

    /**
     * Returns the index of a queued request.
     * Does not take into account the later-queued requests.
     * @param req
     */
    private _indexOf(req: RequestDescriptor): number {
        const hasMethod = !!req.method;
        const isHead = hasMethod && this._usePriority && this._isPriority(req);
        const len = isHead ? this._headEndPtr : this._queue.length;
        const start = isHead || !hasMethod ? 0 : this._headEndPtr;

        for (let i = start; i < start + len; i++) {
            if (req.id === this._queue[i].descriptor.id) {
                return i;
            }
        }
        return -1;
    }

    public exists(req: RequestDescriptor): boolean {
        if (req.id in this._later) {
            return true;
        }
        if (this._indexOf(req) >= 0) {
            return true;
        }
        return false;
    }

    /**
     * Move request from later to queue
     * @param reqId
     */
    private _ready(reqId: string) {
        const req = this._later[reqId];
        const notify = req.notifyReady;

        delete this._later[reqId];
        delete req.notifyReady;
        delete req.readyTimeout;
        delete req.wait;

        this._push(req);

        typeof notify === 'function' && notify.call(null);
    }

    pop<K extends AdobeResponseType = any>(): QueuedRequest<AdobeRequest<K>> | undefined {
        const popped = this._queue.shift();
        if (this._headEndPtr > 0) {
            this._headEndPtr--;
        }
        return popped;
    }

    public get length(): number {
        dbg('length: ', this._queue.length, Object.keys(this._later).length);
        return this._queue.length + Object.keys(this._later).length;
    }

    clear(err?: AdobeDCXError) {
        const len = this._queue.length - 1;
        for (let i = len; i >= 0; i--) {
            this._remove(i, err);
        }
        this._queue = [];

        const keys = Object.keys(this._later);
        for (const i in keys) {
            const id = keys[i];
            this._remove(id, err);
        }
        this._later = {};
    }

    removeAllWithToken(token: unknown) {
        const len = this._queue.length - 1;
        for (let i = len; i >= 0; i--) {
            if (this._queue[i].descriptor.token === token) {
                this._remove(i);
            }
        }

        const keys = Object.keys(this._later);
        for (const i in keys) {
            const id = keys[i];
            if (this._later[id].descriptor.token === token) {
                this._remove(id);
            }
        }
    }
}
