/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc..
 **************************************************************************/

import {
    AdobeResponseType,
    AdobeXhr,
    AdobeXhrNode,
    BodyType,
    HTTPMethod,
    PrePostCallback,
    ProgressListener,
    ResponseTypeMap,
    XhrOptions,
    XhrResponse,
} from '@dcx/common-types';
import { AdobeDCXError, DCXError, ErrorCodes as DCXErrorCodes } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import {
    arrayBufferToString,
    isFunction,
    isJsonContentType,
    isNode,
    normalizeHeaders,
    objectFromEntries,
    parseHeaders,
} from '@dcx/util';
import { XhrNode } from './XhrNode';
import { _getGlobalFetch } from './_getGlobalFetch';
import { DEFAULT_TIMEOUT } from './defaults';

const dbg = newDebug('dcx:http:xhr');

// shim XHR
let XMLHttpRequest_: typeof XMLHttpRequest;
/* istanbul ignore next */
if (process.env.APP_ENV !== 'browser' && isNode()) {
    // eslint-disable-next-line @typescript-eslint/no-var-requires
    XMLHttpRequest_ = require('./XhrNode').XhrNode;
} else {
    XMLHttpRequest_ = typeof window !== 'undefined' ? window.XMLHttpRequest : XMLHttpRequest;
    if (XMLHttpRequest_ == null) {
        throw new AdobeDCXError(AdobeDCXError.INVALID_STATE, 'XMLHttpRequest module not found.');
    }
}

export const XhrErrorCodes = {
    NO_ERROR: '',
    ABORTED: DCXErrorCodes.ABORTED,
    NETWORK: DCXErrorCodes.NETWORK_ERROR,
    TIMEOUT: DCXErrorCodes.TIMED_OUT,
    TOO_MANY_REDIRECTS: DCXErrorCodes.TOO_MANY_REDIRECTS,
    INSECURE_REDIRECT: DCXErrorCodes.INSECURE_REDIRECT,
};

export class Xhr<T extends AdobeResponseType> implements AdobeXhr<T> {
    private _autoParsedResponse: unknown;
    private _autoParseJson = false;
    private _bytesReported = 0;
    private _errorCode: number | string = XhrErrorCodes.NO_ERROR;
    private _estimatedTotalBytes?: number;
    private _isFetchRequest = false;
    private _fetch?: typeof fetch;
    // eslint-disable-next-line @typescript-eslint/no-empty-function
    private _fetchAbort: () => void = () => {};
    private _fetchResponse?: Response;
    private _parsedResponseHeaders?: Record<string, string>;
    private _fetchBodyAsResponseType?: ResponseTypeMap[T];
    private _postCallback?: PrePostCallback<T>;
    private _preCallback?: PrePostCallback<T>;
    private _preferFetch = false;
    private _promise: Promise<AdobeXhr<T>>;
    private _resolve!: (result: AdobeXhr<T>) => void;
    private _response?: XhrResponse<T>;
    private _sent = false;
    private _timeout: number;
    private _timeoutTimeout;
    private _underlyingError?: AdobeDCXError | ProgressEvent<XMLHttpRequestEventTarget>;
    private _xhr: XMLHttpRequest;

    public href!: string;
    public method!: HTTPMethod;
    // To-do: breaking change to tighten type to `Record<string,string>`
    public headers: Record<string, string | string[]> = {};
    public body?: BodyType;
    public responseType: AdobeResponseType = 'text';

    private _progressListeners: ProgressListener[] = [];

    constructor(opts: XhrOptions<T> = {}) {
        const { forceXhr, preCallback, postCallback, timeout, preferFetch } = opts;

        this._preCallback = preCallback;
        this._postCallback = postCallback;

        this._timeout = timeout == null ? DEFAULT_TIMEOUT : timeout;

        this._xhr = forceXhr ? new forceXhr() : new XMLHttpRequest_();
        this._xhr.timeout = this._timeout;
        this._preferFetch = preferFetch === true;
        this._fetch = opts.fetch ? opts.fetch : _getGlobalFetch();
        this._parseFetchResponse = this._parseFetchResponse.bind(this);
        this.onProgress = this.onProgress.bind(this);

        this._autoParseJson = opts.autoParseJson != null ? opts.autoParseJson : true;

        // node specific options
        /* istanbul ignore next */
        if (opts.additionalNodeOptions && (this._xhr as unknown as XhrNode).setNodeOptions) {
            (this._xhr as unknown as XhrNode).setNodeOptions(opts.additionalNodeOptions);
        }

        // if (opts.maxRedirects != null) {
        //     ((this._xhr as unknown) as XhrNode).maxRedirects = opts.maxRedirects;
        // }

        this._promise = new Promise<AdobeXhr<T>>((resolve) => {
            this._resolve = resolve;

            this._xhr.addEventListener('abort', () => {
                dbg('aborted', this._errorCode, this._timeout);

                // retain predefined error code
                // other errors may use abort to halt the request
                this._errorCode = this._errorCode || XhrErrorCodes.ABORTED;
                this._finalize();
            });

            this._xhr.addEventListener('error', (err) => {
                dbg('err', this._errorCode, err, this._xhr.status, this._timeout);
                this._underlyingError = err;

                const code = err ? (err as unknown as { code: string }).code : undefined;
                switch (code) {
                    case 'ERR_FR_TOO_MANY_REDIRECTS':
                        this._errorCode = XhrErrorCodes.TOO_MANY_REDIRECTS;
                        break;
                    case AdobeDCXError.INSECURE_REDIRECT:
                        this._errorCode = XhrErrorCodes.INSECURE_REDIRECT;
                        break;
                    case XhrErrorCodes.TIMEOUT as unknown as string:
                        this._errorCode = XhrErrorCodes.TIMEOUT;
                        break;
                    default:
                        this._errorCode = XhrErrorCodes.NETWORK;
                        break;
                }
                this._finalize();
            });
            this._xhr.addEventListener('load', () => {
                dbg('load');
                // send final progress event on POST/PUT/PATCH
                if (this._estimatedTotalBytes && this._estimatedTotalBytes > this._bytesReported) {
                    this._notifyProgressListeners(this._estimatedTotalBytes, this._estimatedTotalBytes, false);
                }
                this._finalize();
            });
            this._xhr.addEventListener('timeout', () => {
                dbg('timeout', this._timeout);
                this._errorCode = XhrErrorCodes.TIMEOUT;
                this._finalize();
            });
        });
    }

    public get xhr(): XMLHttpRequest | AdobeXhrNode {
        return this._xhr;
    }
    private async _parseFetchResponse(response: Response) {
        if (response.status === 204) {
            // no content -- do not attempt to parse
            return response;
        }
        const isChunked = response.headers.get('transfer-encoding') === 'chunked';
        if (isChunked || parseInt(response.headers.get('content-length') || '0') > 0) {
            switch (this.responseType) {
                case 'json':
                    if (isJsonContentType(response.headers.get('content-type') || '')) {
                        this._fetchBodyAsResponseType = (await response.json()) as ResponseTypeMap[T];
                    }
                    break;
                case 'arraybuffer':
                    this._fetchBodyAsResponseType = (await response.arrayBuffer()) as ResponseTypeMap[T];
                    break;
                case 'blob':
                    this._fetchBodyAsResponseType = (await response.blob()) as ResponseTypeMap[T];
                    break;
                case 'text':
                    this._fetchBodyAsResponseType = (await response.text()) as ResponseTypeMap[T];
                    break;
                case 'void':
                    break;
                case 'buffer':
                case 'defaultbuffer':
                    this._fetchBodyAsResponseType = (await response
                        .arrayBuffer()
                        .then((buffer) => new Uint8Array(buffer))) as ResponseTypeMap[T];
            }
        }
        // transfer-encoding "chunked" responses generally will not include a content-length
        // the stream also does not require any pre-parsing, so it is directly applied to
        if (this.responseType === 'stream') {
            this._fetchBodyAsResponseType = response.body as ResponseTypeMap[T];
        }

        return response;
    }
    private _shouldAutoParseResponse(): boolean {
        const shouldParse =
            this._errorCode === XhrErrorCodes.NO_ERROR &&
            this._sent &&
            !this._isFetchRequest &&
            this._xhr.responseType === 'text' &&
            this._autoParseJson &&
            typeof this._xhr.response === 'string' &&
            this._xhr.response.length < 10 * 10 * 1024 &&
            isJsonContentType(this.getResponseHeader('content-type') as string);

        dbg('_shouldAutoParseResponse()', shouldParse);

        return shouldParse;
    }

    /* istanbul ignore next */
    private _fetchWithTimeout(
        resource: string,
        options: RequestInit & { timeout?: number } = {},
    ): ReturnType<typeof fetch> {
        if (typeof this._fetch !== 'function') {
            throw new DCXError(DCXError.UNEXPECTED, 'fetch method not found but was invoked');
        }

        const { timeout, ...fetchOptions } = options;
        this._isFetchRequest = true;

        const createRequestTimeoutManager = (errorReceiver: (error: AdobeDCXError) => void) => () => {
            this._errorCode = this._errorCode || XhrErrorCodes.TIMEOUT;
            errorReceiver(new DCXError(DCXError.TIMED_OUT, 'request aborted due to timeout'));
            this._finalize();
        };

        if (typeof AbortController !== 'function') {
            return new Promise<Response>(async (resolve, reject) => {
                this._timeoutTimeout = setTimeout(createRequestTimeoutManager(reject), timeout);
                const response = await this._fetch!(resource, fetchOptions);
                clearTimeout(this._timeoutTimeout);
                return this._parseFetchResponse(response).then(resolve);
            }).finally(() => {
                clearTimeout(this._timeoutTimeout);
            });
        }

        const controller = new AbortController();

        this._timeoutTimeout = setTimeout(createRequestTimeoutManager(controller.abort.bind(controller)), timeout);

        this._fetchAbort = () => {
            this._errorCode = this._errorCode || XhrErrorCodes.ABORTED;
            controller.abort();
            this._finalize();
        };

        return this._fetch(resource, { signal: controller.signal, ...fetchOptions })
            .then(this._parseFetchResponse)
            .finally(() => {
                clearTimeout(this._timeoutTimeout);
            });
    }

    private _finalize() {
        dbg('_finalize', this._xhr.status, this._errorCode);
        // If it's not an error, and the request was sent
        // and if responseType is set to `text` (default)
        // and responseType isn't explicitly set
        // then check headers for content-type.

        // If it's application/json or application/{string}+json
        // and the string data isn't extremely large
        // then try to JSON parse it.
        if (this._shouldAutoParseResponse()) {
            try {
                const parsed = JSON.parse(this._xhr.response);
                this._autoParsedResponse = parsed;
                this._xhr.responseType = 'json';
            } catch (_) {
                // noop
            }
        }

        if (this._postCallback) {
            this._postCallback(this);
        }
        if (this._timeoutTimeout) {
            clearTimeout(this._timeoutTimeout);
        }
        this._progressListeners = [];
        this._resolve(this);
    }
    private _validateResponseType(responseType: AdobeResponseType): AdobeResponseType {
        if (responseType === 'buffer') {
            if (typeof Buffer !== 'function') {
                throw new AdobeDCXError(AdobeDCXError.INVALID_PARAMS, 'No Buffer class');
            }
        } else if (responseType === 'blob') {
            if (typeof Blob !== 'function') {
                throw new AdobeDCXError(AdobeDCXError.INVALID_PARAMS, 'No Blob class');
            }
        } else if (
            responseType !== 'text' &&
            responseType !== 'json' &&
            responseType !== 'arraybuffer' &&
            responseType !== 'stream'
        ) {
            throw new AdobeDCXError(AdobeDCXError.INVALID_PARAMS, 'Unsupported response type');
        }
        return responseType.toLowerCase() as AdobeResponseType;
    }

    send(
        url: string,
        method: HTTPMethod,
        body?: BodyType,
        headers: Record<string, string | string[]> = {},
        responseType: T = 'text' as T,
        options: XhrOptions<T> = {},
    ): Promise<AdobeXhr<T>> {
        dbg('send');

        if (this._sent) {
            throw new Error('Xhr already sent');
        }

        this.href = url;
        this.method = method.toUpperCase() as HTTPMethod;
        this.body = body;

        // try to determine the total size upfront
        if (this.body) {
            this._estimatedTotalBytes =
                (this.body as Buffer).byteLength || (this.body as string).length || (this.body as Blob).size;
        } else {
            this._estimatedTotalBytes = Number.MAX_SAFE_INTEGER;
        }

        const progressCallback = (ev) => {
            dbg(`progress ${ev.loaded}/${ev.total}`);

            // In case of XMLHttpRequest progress event's ev.loaded indicates the amount of work already performed by the underlying process. Ref: https://developer.mozilla.org/en-US/docs/Web/API/XMLHttpRequest/progress_event
            // where as in case of XhrNode progress event's ev.loaded indicates only the current chunk size
            if (process.env.APP_ENV !== 'browser' && isNode()) {
                this._bytesReported += ev.loaded;
            } else {
                this._bytesReported = ev.loaded;
            }

            if (ev.lengthComputable) {
                // total computable
                this._estimatedTotalBytes = ev.total;
                this._notifyProgressListeners(this._bytesReported, this._estimatedTotalBytes ?? Infinity, false);
            } else {
                if (this._estimatedTotalBytes && this._estimatedTotalBytes > this._bytesReported) {
                    // total is probably undefined, but definitely holds no value
                    this._notifyProgressListeners(this._bytesReported, this._estimatedTotalBytes || ev.total, true);
                }
            }
        };

        if (['POST', 'PUT', 'PATCH'].includes(this.method) && this._xhr.upload) {
            this._xhr.upload.onprogress = progressCallback;
        } else {
            this._xhr.addEventListener('progress', progressCallback);
        }

        this._timeout = options.timeout || this._timeout || DEFAULT_TIMEOUT;
        dbg('setting timeout', this._timeout);
        this._xhr.timeout = this._timeout;

        if (responseType) {
            responseType = this._validateResponseType(responseType) as T;
            dbg('responseType: ', responseType);
            this.responseType = (
                responseType === 'buffer'
                    ? 'arraybuffer'
                    : responseType === 'stream'
                    ? 'stream'
                    : responseType === 'void'
                    ? 'text'
                    : responseType
            ) as AdobeResponseType;
        }
        const normalizedHeaders = normalizeHeaders(headers);
        this.headers = normalizedHeaders;
        if (this.href.startsWith('http:') && this.headers['authorization'] !== null) {
            // don't allow tokens over HTTP
            throw new AdobeDCXError(AdobeDCXError.INVALID_PARAMS, 'Must not send auth token over unsecured connection');
        }

        if (this._preCallback) {
            this._preCallback(this);
        }
        if ((this._preferFetch || responseType === 'stream') && typeof this._fetch === 'function') {
            this._xhr.responseType = this.responseType as XMLHttpRequestResponseType;
            this._promise = new Promise<AdobeXhr<T>>((resolve) => {
                this._resolve = resolve;
                this._fetchWithTimeout(this.href, {
                    body: ['GET', 'HEAD'].includes(this.method.toUpperCase()) ? undefined : body,
                    credentials: options.withCredentials ? 'include' : undefined,
                    headers: normalizedHeaders,
                    method: this.method,
                    timeout: this._timeout,
                }).then((res) => {
                    this._fetchResponse = res;
                    this._finalize();
                });
                this._sent = true;
            });
            return this._promise;
        }

        this._xhr.open(this.method, this.href, true);
        this._xhr.responseType = this.responseType as XMLHttpRequestResponseType;

        for (const [key, value] of Object.entries(normalizedHeaders)) {
            this._xhr.setRequestHeader(key, value);
        }

        // apply options
        if (options.withCredentials != null) {
            this._xhr.withCredentials = options.withCredentials;
        }

        if (body != null) {
            this._xhr.send(body);
        } else {
            this._xhr.send();
        }

        this._sent = true;

        return this._promise;
    }

    abort() {
        dbg('abort()');

        if (!this._sent) {
            throw new Error('Cannot abort before sending.');
        }
        if (this._isFetchRequest) {
            this._fetchAbort();
        } else {
            this._xhr.abort();
        }
    }

    getResponseHeader(header: string): string | undefined {
        if (!this._sent) {
            throw new Error('Cannot getResponseHeader before sending.');
        }

        const key = header.toLowerCase();
        const hdrs = this.getAllResponseHeaders();

        if (!(key in hdrs)) {
            return undefined;
        }
        return hdrs[key];
    }

    getAllResponseHeaders(): Record<string, string> {
        if (!this._sent) {
            throw new Error('Cannot getAllResponseHeaders before sending.');
        }
        if (this._parsedResponseHeaders) {
            return this._parsedResponseHeaders;
        }

        const headers = this._isFetchRequest
            ? objectFromEntries(this._fetchResponse?.headers.entries() ?? [])
            : this._xhr.getAllResponseHeaders();

        if (typeof headers === 'string') {
            this._parsedResponseHeaders = parseHeaders(headers);
        } else {
            this._parsedResponseHeaders = normalizeHeaders(headers);
        }
        return this._parsedResponseHeaders;
    }

    isError(): boolean {
        if (!this._sent) {
            throw new Error('Cannot check isError before sending.');
        }
        return this._errorCode !== XhrErrorCodes.NO_ERROR;
    }

    isAborted(): boolean {
        if (!this._sent) {
            throw new Error('Cannot check isAborted before sending.');
        }
        return this._errorCode === XhrErrorCodes.ABORTED;
    }

    isTimedOut(): boolean {
        if (!this._sent) {
            throw new Error('Cannot check isTimedOut before sending.');
        }
        return this._errorCode === XhrErrorCodes.TIMEOUT;
    }

    isSent(): boolean {
        return this._sent;
    }

    getErrorCode(): number | string {
        return this._errorCode;
    }

    getStatus() {
        if (!this._sent) {
            throw new Error('Cannot getStatus before sending.');
        }
        if (this._fetchResponse) {
            return this._fetchResponse.status;
        }
        return this._xhr.status;
    }

    getResponse(): XhrResponse<T> {
        if (!this._sent) {
            throw new Error('Cannot getResponse before sending.');
        }

        if (this._response) {
            return this._response;
        }

        this._response = {
            statusCode: this.getStatus(),
            headers: this.getAllResponseHeaders(),
            responseType: this._autoParsedResponse ? ('json' as T) : (this.responseType as T),
            response: this.getResponseData(),
            message: this._isFetchRequest ? this._fetchResponse?.statusText || '' : this._xhr.statusText,
            xhr: this,
        };
        if (this._autoParsedResponse) {
            this._response.originalResponseData = this._xhr.response;
        }
        return this._response;
    }

    toJSON() {
        return {
            statusCode: this.getStatus(),
            headers: this.getAllResponseHeaders(),
            responseType: this._autoParsedResponse ? ('json' as T) : (this.responseType as T),
            response: this.getResponseData(),
            message: this._xhr.statusText,
        };
    }

    async getResponseDataAsJSON(): Promise<any> {
        try {
            if (this._fetchResponse) {
                return await this._fetchResponse.json();
            }

            // response type not explicitly set, response was already JSON
            if (this._autoParsedResponse) {
                return this._autoParsedResponse;
            }

            if (this._xhr.responseType === 'json') {
                if (typeof this._xhr.response === 'string') {
                    return JSON.parse(this._xhr.response);
                }
                if (
                    this.xhr.response === null &&
                    ['application/problem+json', 'application/json'].includes(
                        this.xhr.getResponseHeader('content-type')!,
                    )
                ) {
                    throw new DCXError(DCXError.UNEXPECTED, 'Unexpected response type');
                }
                return this.xhr.response;
            }

            let resp = this._xhr.response;

            if (this._xhr.responseType === 'text' && this._xhr.responseText !== null) {
                resp = this._xhr.responseText;
            } else if (this._xhr.responseType === 'arraybuffer') {
                resp = arrayBufferToString(this._xhr.response);
            } else if (this._xhr.responseType === 'blob' && (resp instanceof Blob || isFunction(resp.text))) {
                return JSON.parse(await resp.text());
            } else if (this.responseType === 'stream') {
                await new Promise((resolve, reject) => {
                    resp = '';
                    if (typeof this.xhr.response.on === 'function') {
                        this.xhr.response.on('data', (data) => {
                            resp += data;
                        });
                        this.xhr.response.on('end', resolve);
                        this.xhr.response.on('error', reject);
                        return;
                        /* istanbul ignore next */
                    } else if (typeof this.xhr.response === 'string') {
                        return resolve((resp = this.xhr.response));
                    }
                    /* istanbul ignore next */
                    throw new DCXError(DCXError.UNEXPECTED, 'Unexpected response type');
                });
            } else {
                /* istanbul ignore next */
                resp = this._xhr.responseText ? this._xhr.responseText : resp;
            }
            if (typeof resp === 'string') {
                return JSON.parse(resp);
            }
            return resp;
        } catch (error) {
            throw new DCXError(DCXError.INVALID_JSON, 'Could not parse response as JSON', error, this.toJSON());
        }
    }

    getResponseData(): ResponseTypeMap[T] {
        if (!this._sent) {
            throw new Error('Cannot getResponseData before sending.');
        }

        if (this._isFetchRequest && this._fetchBodyAsResponseType) {
            return this._fetchBodyAsResponseType;
        }

        return this._autoParsedResponse || this._xhr.response;
    }

    onProgress(handler: ProgressListener): () => void {
        const id = this._progressListeners.push(handler) - 1;
        return () => {
            try {
                delete this._progressListeners[id];
            } catch (_) {
                // already deleted
            }
        };
    }

    private _notifyProgressListeners(sor: number, t: number, indeterminate: boolean | undefined) {
        this._progressListeners.map((h) => {
            return h && h.call && h.call(null, sor, t, indeterminate);
        });
    }
}
