/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc..
 **************************************************************************/

import { AdditionalNodeOptions, AdobeDCXError, AdobeResponseType, HTTPMethod } from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import { consumeStream, isFunction, isNodeReadableStream, isReadableStream, normalizeHeaders } from '@dcx/util';
import { FollowOptions, http as httpFollow, https as httpsFollow } from 'follow-redirects';
import * as http from 'http';
import { ClientRequest } from 'http';
import * as https from 'https';
import { Readable } from 'stream';
import { UrlWithStringQuery, parse } from 'url';
import { brotliDecompressSync, createUnzip } from 'zlib';
import { DEFAULT_TIMEOUT } from './defaults';

const dbg = newDebug('dcx:http:xhrnode');

type XhrInternalEvents = 'progress' | 'load' | 'error' | 'abort';
interface XhrNodeInternal {
    addEventListener<T extends XhrInternalEvents>(
        type: T,
        listener: (this: XMLHttpRequest, ev: ProgressEvent<XMLHttpRequestEventTarget>) => never,
        options?: boolean | AddEventListenerOptions,
    ): void;
}

interface XhrNodeResponseInternal {
    status: number;
    statusText: string;
    headers: Record<string, string | string[]>;
    config: NodeXhrRequestConfig;
    request: ClientRequest;
    data?: any;
}

interface CancelToken {
    promise: Promise<Error>;
    cancel: (err?: Error) => void;
}

interface NodeXhrRequestConfig {
    decompress?: boolean;
    responseType: AdobeResponseType;
    data?: any;
    onProgress?: any;
    cancelToken?: CancelToken;
    maxContentLength?: number;
    responseEncoding?: BufferEncoding;
}

function stripBOM(content: string) {
    if (content.charCodeAt(0) === 0xfeff) {
        content = content.slice(1);
    }
    return content;
}

export class XhrNode implements XhrNodeInternal {
    private _promise!: Promise<XhrNodeResponseInternal>;
    private _method!: HTTPMethod;
    private _url!: string;
    private _requestHeaders: Record<string, number | string | string[] | undefined> = {};
    private _responseHeaders: Record<string, string | string[]> = {};
    private _listeners: Record<string, any[]> = {};
    private _response: any;
    private _status?: number;
    private _statusText?: string;
    private _bytesReported = 0;
    private _totalBytes = 0;
    private _authenticationAllowList: string[] = [];
    private _additionalOptions: https.RequestOptions & { authenticationAllowList?: string[] } = {};
    private _parsedUrl?: UrlWithStringQuery;
    private _cancelToken: CancelToken;

    readyState?: number;
    public get response() {
        return this._response;
    }
    public get status() {
        return this._status;
    }
    public get statusText() {
        return this._statusText;
    }

    private _responseText?: string;
    public get responseText(): string {
        return this._responseText || this._response;
    }
    public set responseText(val: string) {
        this._responseText = val;
    }

    responseEncoding?: BufferEncoding;
    responseType?: AdobeResponseType;
    responseURL?: string;
    responseXML?: Document;
    timeout?: number;
    upload?: XMLHttpRequestUpload;
    withCredentials?: boolean;
    maxRedirects?: number;

    constructor() {
        let resolve;
        const p = new Promise<Error>((resolve_) => {
            resolve = resolve_;
        });
        this._cancelToken = {
            cancel: resolve,
            promise: p,
        };
    }

    public get requestHeaders() {
        return this._requestHeaders;
    }

    public setNodeOptions(opts: AdditionalNodeOptions = {}) {
        this._additionalOptions = opts;
    }

    abort(): void {
        this._cancelToken.cancel();
        this._notifyListeners('abort', this);
    }

    getAllResponseHeaders(): Record<string, string | string[]> {
        return this._responseHeaders;
    }
    getResponseHeader(name: string): string | string[] | undefined {
        return this._responseHeaders[name];
    }

    private _isHostnameOnAllowList(host: string | undefined | null): boolean {
        dbg('_isDomainOnAllowList()');

        // relative href
        if (host == null) {
            return true;
        }

        const hostLength = host.length;

        const count = this._authenticationAllowList.length;
        for (let i = 0; i < count; i++) {
            const domain = this._authenticationAllowList[i];
            const domainLength = domain.length;
            let hostDomain;
            // The host name must end with the domain:
            if (hostLength > domainLength) {
                hostDomain = host.slice(-domainLength);
            } else {
                hostDomain = host;
            }

            if (hostDomain === domain) {
                dbg('_iDOAL true');
                return true;
            }
        }

        dbg('_iDOAL false');
        return false;
    }

    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    open(method: string, url: string, async?: boolean): void {
        this._method = method as HTTPMethod;
        this._url = url;
        this._parsedUrl = parse(this._url);

        // start with the original hostname as allowed for auth redirects\
        if (this._parsedUrl && this._parsedUrl.hostname) {
            this._authenticationAllowList.push(this._parsedUrl.hostname);
        }
    }

    send(
        body?:
            | string
            | Document
            | Blob
            | ArrayBufferView
            | ArrayBuffer
            | FormData
            | URLSearchParams
            | ReadableStream<Uint8Array>,
    ): void {
        // before send, apply authentication allow list
        if (this._additionalOptions.authenticationAllowList) {
            this._authenticationAllowList = this._additionalOptions.authenticationAllowList;
        }

        const options: https.RequestOptions & FollowOptions<https.RequestOptions> = {
            method: this._method || 'GET',
            headers: { ...this._requestHeaders },
            maxRedirects: this.maxRedirects || 5,
            maxBodyLength: Infinity,
        };

        if (this.timeout) {
            options.timeout = this.timeout;
        }

        Object.assign(options, this._parsedUrl, this._additionalOptions);

        const config: NodeXhrRequestConfig = {
            responseType: this.responseType || 'text',
            data: body,
            responseEncoding: this.responseEncoding || 'utf8',
        };

        this._promise = this._sendRequest(options, config);

        this._status = 0;

        this._promise
            .then((res) => {
                this._response = res.data;
                this._status = res.status;
                this._statusText = res.statusText;
                this._responseHeaders = normalizeHeaders(res.headers);
                this._notifyListeners('load', res);
            })
            .catch((err) => {
                this._notifyListeners('error', err);
            });
    }

    /**
     * Transform the data collected into the expected responseType,
     * then resolve the promise.
     *
     * If the requested responseType is `stream`, this function is skipped entirely.
     *
     * If `json` is requested, and parsing fails, return the data as a string.
     *
     * @param resolve
     * @param reject
     * @param config
     * @param res
     */
    private async _settleResponse(
        resolve: (result: XhrNodeResponseInternal) => void,
        reject: (err: AdobeDCXError) => void,
        config: NodeXhrRequestConfig,
        res: XhrNodeResponseInternal,
    ) {
        const { responseType } = config;
        // for error status codes, force text into responseText property
        if (res.status > 299 && res.status < 600 && responseType !== 'text') {
            this.responseText = res.data.toString(config.responseEncoding);
            if (!config.responseEncoding || config.responseEncoding === 'utf8') {
                /* istanbul ignore next */
                this.responseText = this.responseText ? stripBOM(this.responseText) : '';
            }
        }

        dbg('_settleResponse as', responseType);
        if (res.status !== 204 && config.decompress !== false && res.headers['content-encoding'] === 'br') {
            // performing this decompression as part of the incoming stream had produced
            // Incorrect header check errors coming from zlib. Collecting all of the data
            // and processing it in sync after the request has complete resolves that issue.
            res.data = brotliDecompressSync(res.data);
        }

        try {
            switch (responseType) {
                case 'buffer':
                case 'arraybuffer':
                    return resolve(res);
                case 'blob':
                    res.data = new Blob([res.data], { type: res.headers['content-type'] as string });
                    return resolve(res);
                case 'text':
                case 'json':
                    res.data = res.data.toString(config.responseEncoding);
                    if (!config.responseEncoding || config.responseEncoding === 'utf8') {
                        res.data = stripBOM(res.data);
                    }
                    break;
            }
            if (responseType === 'json') {
                try {
                    res.data = JSON.parse(res.data);
                } catch (_) {
                    // ignore parse errors, return as string
                }
            }
            return resolve(res);
        } catch (err) {
            reject(new DCXError(DCXError.INVALID_DATA, 'Could not convert to requested responseType'));
        }
    }

    /**
     * Ensure tokens are not sent over http.
     * Reject the promise if that would happen.
     *
     * @param {Function} reject
     * @param {*} options
     * @param {Record<string, string>} headers
     */
    private _secureAuth(
        reject: (err: AdobeDCXError) => void,
        options: https.RequestOptions & FollowOptions<https.RequestOptions>,
        responseHeaders: Record<string, string | number | string[] | undefined> = {},
    ) {
        dbg('_secureAuth() opts', options, responseHeaders);

        const allowInsecure = (this as any)._allowAuthTokenToBeSentInsecurely_DONT_USE_THIS_IN_PRODUCTION;
        const isHttps = options.hostname == null || (options.protocol && options.protocol.indexOf('https:') === 0);
        const hostnameAllowed = this._isHostnameOnAllowList(options.hostname);
        const headers = options.headers || {};

        if (((isHttps && hostnameAllowed) || allowInsecure) && 'authorization' in this._requestHeaders) {
            // set auth token if it was stripped by follow-redirects
            headers.authorization = this._requestHeaders.authorization as string;
        }

        if (((isHttps && hostnameAllowed) || allowInsecure) && 'x-api-key' in this._requestHeaders) {
            // set auth token if it was stripped by follow-redirects
            headers['x-api-key'] = this._requestHeaders['x-api-key'];
        }

        if (!allowInsecure) {
            if (!hostnameAllowed && headers.authorization != null) {
                // strip auth header
                dbg('strip auth header');
                delete headers.authorization;
            }

            if (!isHttps && headers.authorization != null) {
                return reject(new DCXError(DCXError.INSECURE_REDIRECT));
            }
        }

        options.headers = headers;
    }

    /**
     * Perform https request.
     * Use follow-redirects if maxRedirects is specified.
     * Returns promise that resolves when request ends,
     * or rejects on network errors.
     *
     * @param httpsOpts
     * @param config
     */
    private async _sendRequest(
        httpsOpts: https.RequestOptions & FollowOptions<https.RequestOptions>,
        config: NodeXhrRequestConfig,
    ): Promise<XhrNodeResponseInternal> {
        let resolve;
        let reject;
        const promise = new Promise<XhrNodeResponseInternal>((resolve_, reject_) => {
            resolve = resolve_;
            reject = reject_;
        });

        let transport: typeof httpsFollow | typeof httpFollow;
        if (httpsOpts.maxRedirects && httpsOpts.maxRedirects > 0) {
            transport = httpsOpts.protocol === 'http:' ? httpFollow : httpsFollow;
        } else {
            transport = httpsOpts.protocol === 'http:' ? (http as typeof httpFollow) : (https as typeof httpsFollow);
        }

        let { data } = config;
        if (data == null) {
            // nothing
        } else if (isReadableStream(data)) {
            // nothing
        } else if (data instanceof Buffer) {
            // nothing
        } else if (data instanceof ArrayBuffer) {
            data = Buffer.from(new Uint8Array(data));
        } else if (data instanceof Uint8Array) {
            data = Buffer.from(data);
        } else if (typeof data === 'string') {
            data = Buffer.from(data, 'utf8');
        } else {
            return reject(new DCXError(DCXError.INVALID_PARAMS, 'Invalid data type'));
        }

        // ensure tokens aren't sent across http
        this._secureAuth(reject, httpsOpts);

        httpsOpts.headers = httpsOpts.headers || {};

        // if a content length was specified in header, use it
        // otherwise try to get length from data
        const contentLength = httpsOpts.headers['content-length'] || data ? data.length || data.size : undefined;
        if (contentLength != null) {
            httpsOpts.headers['content-length'] = contentLength;
        }

        // For progress events:
        // try to get the total bytes from headers or body size for data-sending requests
        // for GET requests, check the content-type header of the response later
        if ((httpsOpts.method as string).toLowerCase() !== 'get') {
            this._totalBytes = parseInt(contentLength);
        }

        httpsOpts.beforeRedirect =
            httpsOpts.beforeRedirect ||
            ((options, { headers }) => {
                // Before each redirect, check that the redirect is to https
                // and that the hostname is on the allowed list,
                // if not, strip auth token header if it exists.
                this._secureAuth(reject, options, headers);

                // Then update request headers to the new headers,
                // these are returned by Xhr#getResponse().
                this._requestHeaders = options && options.headers ? options.headers : this._requestHeaders;
                httpsOpts.headers = this._requestHeaders;
                dbg('after secureAuth before redirect', httpsOpts.headers);
            });

        httpsOpts.headers['accept-encoding'] = 'deflate, gzip, br';
        dbg('httpsOpts: ', httpsOpts);

        const req = transport
            .request(httpsOpts, (res) => {
                if ((req as any).aborted) {
                    return;
                }

                if ((httpsOpts.method as string).toLowerCase() === 'get') {
                    this._totalBytes = res.headers['content-length'] ? parseInt(res.headers['content-length']) : 0;
                }

                // uncompress the response body transparently if required
                let stream: Readable = res;

                // return the last request in case of redirects
                const lastRequest = (res as any).req || req;
                // if no content, is HEAD request or decompress disabled we should not decompress
                if (res.statusCode !== 204 && lastRequest.method !== 'HEAD' && config.decompress !== false) {
                    switch (res.headers['content-encoding']) {
                        // gzip and deflate can be handled in a pipeline
                        // supposedly brotli can also be processed this way, however in
                        // experimentation an incorrect header check was observed while using brotli decompress inline.
                        // For this reason, brotli decompression is occurring in the _settleResponse method.
                        case 'gzip':
                        case 'deflate':
                            // add the unzipper to the body stream processing pipeline
                            stream = res.pipe(createUnzip());
                            break;
                    }
                }

                const response: XhrNodeResponseInternal = {
                    status: res.statusCode as number,
                    statusText: res.statusMessage as string,
                    headers: res.headers as Record<string, string>,
                    config: config,
                    request: lastRequest,
                };

                if (response.status === 401) {
                    consumeStream(stream);
                    return resolve(response);
                }

                // When a server responds with a short payload to a request with a large payload that has not finished uploading,
                // node client throws an EPIPE error on the request object
                if (response.status === 413) {
                    lastRequest.abort();
                    return resolve(response);
                }

                if ((config.responseType as string) === 'stream') {
                    response.data = stream;
                    return resolve(response);
                }

                const responseChunks: Buffer[] = [];
                let streamedBytes = 0;
                stream.on('data', (chunk) => {
                    const length = chunk && chunk.length ? chunk.length : 0;
                    streamedBytes += length;

                    // make sure the content length is not over the maxContentLength if specified
                    if (
                        config.maxContentLength &&
                        config.maxContentLength > -1 &&
                        streamedBytes > config.maxContentLength
                    ) {
                        stream.destroy();
                        return reject(new Error('maxContentLength exceeded.'));
                    }

                    responseChunks.push(chunk);
                    if ((httpsOpts.method as string).toLowerCase() === 'get') {
                        this._reportProgress(length);
                    }
                });

                stream.on(
                    'error',
                    /* istanbul ignore next (difficult to simulate) */ (err) => {
                        if ((req as any).aborted) {
                            return;
                        }
                        if (
                            (err as AdobeDCXError).code === 'ECONNRESET' ||
                            (err as AdobeDCXError).code === 'ECONNREFUSED'
                        ) {
                            return reject(new DCXError(DCXError.NETWORK_ERROR, 'node.js network error', err));
                        }
                        return reject(err);
                    },
                );

                stream.on('end', () => {
                    const responseData: Buffer = Buffer.concat(responseChunks);
                    const unreported = responseData.length - this._bytesReported;
                    if (unreported > 0) {
                        if ((httpsOpts.method as string).toLowerCase() === 'get') {
                            this._reportProgress(unreported);
                        }
                    }

                    response.data = responseData;
                    return this._settleResponse(resolve, reject, config, response);
                });
            })
            .setTimeout(httpsOpts.timeout ?? DEFAULT_TIMEOUT, () => {
                req.destroy(new DCXError(DCXError.TIMED_OUT, 'Request timed out'));
            });

        // Handle errors
        req.on('error', (err) => {
            dbg('onerror', err);
            if ((req as any).aborted && (err as AdobeDCXError).code !== 'ERR_FR_TOO_MANY_REDIRECTS') {
                return;
            }

            if ((err as AdobeDCXError).code === 'ECONNRESET' || (err as AdobeDCXError).code === 'ECONNREFUSED') {
                return reject(new DCXError(DCXError.NETWORK_ERROR, 'node.js network error', err));
            }
            reject(err);
        });

        // Handle cancelation
        this._cancelToken.promise.then(() => {
            dbg('canceled');
            if ((req as any).aborted) {
                return;
            }

            req.destroy(new DCXError(DCXError.ABORTED, 'Request aborted'));
        });

        // handle progress events on writing data streams
        const doWrite = req.write;
        req.write = (
            chunk,
            encoding: BufferEncoding | undefined | ((error: Error) => void),
            cb?: (error: Error | undefined | null) => void,
        ) => {
            dbg('write to stream');
            if (!cb && isFunction(encoding)) {
                cb = encoding as (error: Error | undefined | null) => void;
                encoding = undefined;
            } else if (!cb) {
                // eslint-disable-next-line @typescript-eslint/no-empty-function
                cb = () => {};
            }
            const length = chunk && chunk.length;

            return doWrite.call(req, chunk, encoding as BufferEncoding, (err) => {
                if ((httpsOpts.method as string).toLowerCase() !== 'get') {
                    this._reportProgress.call(this, length);
                }
                cb && cb.call(req, err);
            });
        };

        // Send the request
        if (isNodeReadableStream(data)) {
            data.on('error', (err) => {
                reject(err);
            }).pipe(req);
        } else {
            req.end(data);
        }

        return promise;
    }

    private _reportProgress(chunkSize: number) {
        this._bytesReported += chunkSize;
        this._notifyListeners('progress', {
            lengthComputable: true,
            loaded: chunkSize,
            total: this._totalBytes,
        });
    }

    setRequestHeader(name: string, value: string): void {
        this._requestHeaders[name] = value;
    }

    addEventListener<
        K extends 'readystatechange' | 'abort' | 'error' | 'load' | 'loadend' | 'loadstart' | 'progress' | 'timeout',
    >(type: K, listener: (ev: XMLHttpRequestEventMap[K]) => any): void {
        this._listeners[type] = this._listeners[type] || [];
        this._listeners[type].push(listener);
    }

    private _notifyListeners(event: string, eventData: any) {
        (this._listeners[event] || []).map((h) => {
            return h && h.call(null, eventData);
        });
    }
}
