/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc..
 **************************************************************************/

import {
    AdobeResponseType,
    AdobeXhr,
    Backoff,
    BackoffOptions,
    BackoffSnapshot,
    BodyType,
    HTTPMethod,
    PrePostBackoffCallback,
} from '@dcx/common-types';
import { AdobeDCXError, DCXError, ErrorCodes as DCXErrorCodes } from '@dcx/error';
import { log, newDebug } from '@dcx/logger';
import { DEFAULT_RETRY_CODES, checkRetriable, isReadableStream, now } from '@dcx/util';
import { Xhr, XhrErrorCodes } from './Xhr';

// const log = getGlobalLogger().log;

const dbg = newDebug('dcx:http:backoff');

/**
 * Call a handle function after some timeout
 * On failure, retry after a wait time
 * Increase the wait time by some function
 *
 * Options: {@link BackoffOptions}
 *
 * @param {string} url
 * @param {HTTPMethod} method
 * @param {BodyType} body
 * @param {Record<string, string>} headers
 * @param {BackoffOptions} options
 *
 * @returns {Backoff}
 */
export function backoff<T extends AdobeResponseType>(
    url: string,
    method: HTTPMethod,
    body?: BodyType,
    headers: Record<string, string> = {},
    perRequestTimeout?: number,
    options: BackoffOptions<T> = {},
    isPolling = false,
): Backoff<T> {
    const {
        disableRetry = false,
        retryNetworkError = true,
        responseType = 'text',
        authCallback = null,
        progressListeners = [],
        initialWait = 2000, //ms
        maxWait = 32000,
        preCallback,
        postCallback,
        preScheduleCallback,
        postScheduleCallback,
        preferRetryAfterHeader = true,
        pollCodes = [],
        pollHeader,
        pollMethod = 'get',
    } = options;

    let {
        retryCodes = [],
        timeoutAfter = 72000, //ms
    } = options;

    if (isPolling) {
        retryCodes = [...pollCodes, ...retryCodes];
    } else if (disableRetry || isReadableStream(body)) {
        retryCodes = [];
    } else {
        retryCodes = options.retryCodes || DEFAULT_RETRY_CODES;
    }

    dbg('retry codes', retryCodes);

    const increase =
        options.increase ||
        ((c: number, l: number, i: number) => {
            if (c === 1) {
                return i;
            }
            return l * l > maxWait ? maxWait : l * l;
        });

    let wait = 0; // current duration to wait
    let count = 0; // request count
    let pending = false; // request in flight
    const backoffStart = now();
    let backoffStop: number | undefined = undefined;
    let waitStart = now();
    let totalWait = 0;
    let canceled = false;
    let timedOut = false;
    let resolve: (value: AdobeXhr<T> | PromiseLike<AdobeXhr<T>>) => void;
    let reject: (reason: AdobeDCXError) => void;
    let xhr: AdobeXhr<T>;
    const requests: AdobeXhr<T>[] = [];
    let reqTimeout; // setTimeout id for backoff

    function cancel() {
        dbg('cancel()');
        canceled = true;

        if (xhr != null) {
            xhr.abort();
        }

        if (!pending) {
            dbg('abort');
            // request is in timeout period, cancel it and reject with error
            clearTimeout(reqTimeout);
            resolve({ getErrorCode: () => XhrErrorCodes.ABORTED } as any);
        }
    }

    function onProgress(h: (sor: number, t: number) => void) {
        if (progressListeners.includes(h)) {
            // This progress listener has already been registered.
            return;
        }

        // TODO: only show forward progress, even on retries
        progressListeners.push(h);

        if (xhr != null) {
            return xhr.onProgress && xhr.onProgress(h);
        }
    }

    function getSnapshot(): BackoffSnapshot<T> {
        dbg('getSnapshot()', pending, wait, now(), waitStart, totalWait);
        const waitingFor = pending || backoffStop != null ? 0 : wait - (now() - waitStart);

        let totalWaited = totalWait;
        totalWaited += pending ? now() - waitStart : 0;

        const duration = (backoffStop || now()) - backoffStart;

        return {
            count,
            canceled,
            timedOut,
            requests,
            duration,
            totalWaited,
            requestPending: pending,
            waitingFor,
        };
    }

    function increaseWait(): number {
        const newWait = increase(count, wait, initialWait);
        return Math.min(newWait, maxWait);
    }

    function wrapPrePost(cb?: PrePostBackoffCallback<T>): undefined | ((xhr?: AdobeXhr<T>) => void) {
        if (!cb) {
            return undefined;
        }

        return (xhr_?: AdobeXhr<T>) => {
            return cb(xhr_, getSnapshot());
        };
    }

    function getRetryAfterWait(result) {
        const retryAfter = result.getResponseHeader('retry-after');
        if (preferRetryAfterHeader && retryAfter) {
            if (isNaN(retryAfter as unknown as number)) {
                // using date directive
                const nextWait = Date.parse(retryAfter) - Date.now();
                dbg('nextWait from retry-after', nextWait);
                if (nextWait < 0) {
                    return increaseWait();
                }
                return nextWait;
            }
            // using seconds directive
            return parseInt(result.getResponseHeader('retry-after')) * 1000;
        }
    }

    async function doRequest(timeout: number = wait) {
        if (totalWait >= timeoutAfter) {
            dbg('timed out', totalWait, timeoutAfter);
            timedOut = true;
            return resolve(xhr);
        }

        waitStart = now();

        if (preScheduleCallback) {
            await preScheduleCallback(getSnapshot());
        }

        dbg('retry in ', timeout);
        reqTimeout = setTimeout(() => {
            dbg('retry start');
            log(`Request: ${method.toUpperCase()} ${url} ${headers?.['x-request-id'] ?? ''}`);

            pending = true;
            totalWait += timeout;

            xhr = new Xhr({
                ...options,
                timeout: perRequestTimeout,
                preCallback: wrapPrePost(preCallback),
                postCallback: wrapPrePost(postCallback),
            });
            requests.push(xhr);
            // xhr = xhr_;
            count++;

            for (const listener of progressListeners) {
                xhr.onProgress(listener);
            }

            xhr.send(url, method, body, headers, responseType).then(async (result: AdobeXhr<T>) => {
                log(
                    `Response: ${method.toUpperCase()} ${url} ${
                        result.headers?.['x-request-id']
                    } ${result.getStatus()}`,
                );
                pending = false;
                if (
                    // not error
                    !result.isError() &&
                    // non-retriable (retry may be 200-series code)
                    !checkRetriable(result.getStatus(), retryCodes) &&
                    // not auth issue, or no ability to fix the auth issue
                    (result.getStatus() !== 401 || authCallback == null) &&
                    // not pollable
                    (typeof pollHeader !== 'string' ||
                        pollCodes == null ||
                        !checkRetriable(result.getStatus(), pollCodes))
                ) {
                    backoffStop = now();
                    return resolve(result);
                }

                if (result.isAborted() || canceled) {
                    // reject and don't continue
                    backoffStop = now();
                    canceled = true;
                    return resolve(result);
                }

                // if not polling, should poll, and is a poll-able status code
                // adjust retry codes to include poll codes, reset timeouts
                if (
                    !isPolling &&
                    typeof pollHeader === 'string' &&
                    pollCodes != null &&
                    checkRetriable(result.getStatus(), pollCodes)
                ) {
                    const pollHeaderValue = result.getResponseHeader(pollHeader.toLowerCase());
                    if (pollHeaderValue) {
                        isPolling = true;
                        url = pollHeaderValue;
                        method = pollMethod;
                        body = undefined;
                        // add poll codes to retry codes
                        retryCodes = [...retryCodes, ...pollCodes];
                        totalWait = 0;
                        // bump up timeout, polling might take a while
                        timeoutAfter = timeoutAfter * 3;

                        const retryAfter = result.getResponseHeader('retry-after');
                        if (preferRetryAfterHeader && retryAfter) {
                            const newWait = getRetryAfterWait(result);
                            if (newWait != null) {
                                wait = newWait;
                                return doRequest(wait);
                            }
                        }

                        return doRequest(0);
                    }
                }

                // check if it's a retriable error
                if (result.getStatus() === 401) {
                    // update token if possible
                    if (authCallback) {
                        try {
                            headers = await authCallback(url, headers);
                        } catch (e) {
                            return reject(new DCXError(DCXError.UNAUTHORIZED, 'Authentication Failed', result));
                        }
                        totalWait += now() - waitStart;
                        return doRequest(0);
                    }

                    backoffStop = now();
                    return reject(new DCXError(DCXError.UNAUTHORIZED, 'Authentication Failed', result));
                } else if (
                    checkRetriable(result.getStatus(), retryCodes) ||
                    (retryNetworkError && result.getErrorCode() === DCXErrorCodes.NETWORK_ERROR)
                ) {
                    // if the response has a retry-after header, use it
                    const retryAfter = result.getResponseHeader('retry-after');
                    if (preferRetryAfterHeader && retryAfter) {
                        const newWait = getRetryAfterWait(result);
                        if (newWait != null) {
                            wait = newWait;
                            return doRequest(wait);
                        }
                    }

                    // bump timeouts
                    wait = increaseWait();
                    return doRequest(wait);
                }
                // reject it
                backoffStop = now();
                return resolve(result);
            });
        }, timeout);

        if (postScheduleCallback) {
            await postScheduleCallback(getSnapshot());
        }
    }

    const promise = new Promise<AdobeXhr<T>>((resolve_, reject_) => {
        resolve = resolve_;
        reject = reject_;
        doRequest(0);
    });

    return {
        getPromise: () => promise,
        cancel,
        onProgress,
        getSnapshot,
    };
}
