import { AdobeDCXBranch } from '@dcx/common-types';
import { AdobeDCXComposite } from '@dcx/common-types';
import { AdobeDCXError } from '@dcx/common-types';
import { AdobeResponse } from '@dcx/common-types';
import { EventEmitter } from '@dcx/util';

export declare class AdobeDCXLogger extends EventEmitter<AnalyticsEventHandlers> {
    private static _instance;
    private _logCallback;
    private _logLevel;
    private _prevDebugTime;
    private _debugFormatter;
    private _debugNamespaces;
    private _debugSkips;
    suppressDeprecationWarnings: boolean;
    private constructor();
    /**
     * Current active debug namespace regexes
     * Set with setDebugNamespaces()
     */
    get debugNamespaces(): RegExp[];
    /**
     * Currently active skipped namespace regexes
     * Set with setDebugNamespaces() using a `-` prefix
     */
    get debugSkips(): RegExp[];
    /**
     * Set a log level.
     *
     * @example
     * ```js
     * logger.logLevel = LogLevel.Warn;
     * ```
     */
    set logLevel(lvl: LogLevel);
    get logLevel(): LogLevel;
    /**
     * Level 5 verbosity
     */
    static LEVEL_DEBUG: LogLevel;
    /**
     * Level 3 verbosity
     */
    static LEVEL_LOG: LogLevel;
    /**
     * Level 3 verbosity
     */
    static LEVEL_WARN: LogLevel;
    /**
     * Level 2 verbosity
     */
    static LEVEL_ERROR: LogLevel;
    /**
     * Level 1 verbosity
     * Can also disable deprecation warnings through AdobeDCXLogger#suppressDeprecationWarnings
     */
    static LEVEL_DEPRECATED: LogLevel;
    /**
     *
     * @param event
     * @param handler
     * @returns
     */
    on<E extends keyof AnalyticsEventHandlers>(event: E | AnalyticsEvent.All, handler: AnalyticsEventHandlers[E]): number;
    /**
     * Get singleton instance of logger
     * @returns
     */
    static getInstance(): AdobeDCXLogger;
    /**
     * Return new logger instance
     * @param {Function} lcb Log callback
     */
    static newLogger(lcb?: LogCallback): AdobeDCXLogger;
    /**
     * Set a log callback
     *
     * @public
     * Setting a log callback will skip any other callbacks registered.
     */
    set logCallback(val: LogCallback | undefined);
    get logCallback(): LogCallback | undefined;
    private _initNamespaces;
    /**
     * Log a message as some level
     *
     * @param messageLvl        Message log level.
     * @param debugNamespace    Namespace of debug, if message is debug level.
     * @param messages          Array of items to log.
     */
    private _log;
    /**
     * Report a log message. This gets reported to a DCX client if they provided a logCallback function
     * @param {string|StringLike} message Message to log.
     */
    log(...messages: (string | StringLike)[]): void;
    /**
     * Log warn message
     * @param {string|StringLike} message
     */
    warn(...messages: (string | StringLike)[]): void;
    /**
     * Log error message
     * @param {string|StringLike} message
     */
    error(...messages: (string | StringLike)[]): void;
    /**
     * Log a deprecation message
     * Can be supressed individually
     * Logged as a warning
     * @param {string|StringLike} message
     */
    deprecated(...messages: (string | StringLike)[]): void;
    /**
     * Returns true if the given mode namespace is enabled, false otherwise.
     *
     * @param namespace
     * @return {Boolean}
     */
    private _debugEnabled;
    /**
     * Override default debug formatter.
     * @param formatter
     */
    setDebugFormatter(formatter: DebugFormatter): void;
    /**
     * Enable debug namespaces.
     * Overwrites existing settings, is not additive.
     * This is set at instantiation from a value pulled from
     * the dcx window namespace (on browser) or from the environment
     * variable process.env.DCX_DEBUG (on Node).
     *
     *
     * @example
     * ```js
     * // enable all except http
     * logger.setDebugNamespaces("dcx:*,-dcx:http:*")
     * ```
     *
     * @param namespaces
     */
    setDebugNamespaces(namespaces?: string): void;
    /**
     * Get namespace debug function
     * @param {string|StringLike} message
     */
    Debug(namespace: string): (...msgs: any[]) => void;
}

export declare type AnalyticsComponentEvent = Partial<AnalyticsComponentSuccessEvent> | Partial<AnalyticsComponentFailureEvent>;

declare interface AnalyticsComponentFailureEvent extends AnalyticsComponentSuccessEvent, AnalyticsErrorEvent {
    'dcx.block_transferred': boolean;
    'content.type': string;
    'content.size': number;
    'dcx.component_rel'?: string;
}

/**
 * ========================================
 * ========= Component Events =============
 * ========================================
 */
declare interface AnalyticsComponentSuccessEvent extends AnalyticsEventBase {
    'content.mimetype': string;
}

export declare type AnalyticsCompositeEvent = Partial<AnalyticsCompositeSuccessEvent> | Partial<AnalyticsCompositeFailureEvent>;

declare interface AnalyticsCompositeFailureEvent extends AnalyticsCompositeSuccessEvent, AnalyticsErrorEvent {
    'dcx.root_error_desc': string;
}

/**
 * ========================================
 * ========= Composite Events =============
 * ========================================
 */
declare interface AnalyticsCompositeSuccessEvent extends AnalyticsEventBase {
    'dcx.num_total_components': number;
    'custom.content.xmp_derivation_type': number;
}

declare interface AnalyticsErrorEvent {
    'event.error_code': string;
    'event.error_desc': string;
    'dcx.underlying_error_code'?: string;
    'dcx.underlying_error_desc'?: string;
}

/**
 * Emitted event types valid for registration.
 */
export declare enum AnalyticsEvent {
    PushComposite = "analyticsPush",
    CreateComposite = "analyticsCreate",
    PullComposite = "analyticsPull",
    PullCompositeVersion = "analyticsPullVersion",
    UploadComponent = "analyticsUpload",
    DownloadComponent = "analyticsDownload",
    All = "*"
}

declare interface AnalyticsEventBase {
    /**
     * Library name
     * See {@link https://git.corp.adobe.com/DMA/dcx-js/pull/575#discussion_r3873327}
     * and {@link https://git.corp.adobe.com/DMA/dcx-js/pull/575#discussion_r3876987}
     */
    'env.com.name': 'dcx-js';
    /**
     * API type (cc-storage, css, ..)
     * See {@link https://git.corp.adobe.com/DMA/dcx-js/pull/575#discussion_r3876987}
     */
    'env.svc.name': string;
    /**
     * dcx-js version
     */
    'env.com.version': string;
    /**
     * x-request-id from the request
     * This may be the last request or the failed request
     * See {@link https://git.corp.adobe.com/DMA/dcx-js/pull/575#discussion_r3875392}
     */
    'event.request_id': string;
    /**
     * Cloud asset ID
     */
    'event.cloud_id': string;
    /**
     * Cloud repository ID
     */
    'custom.content.repository_id': string;
    /**
     * Type
     * (success or failure)
     */
    'event.type': AnalyticsEventType;
    /**
     * Not set
     * See {@link https://git.corp.adobe.com/DMA/dcx-js/pull/575#discussion_r3873385}
     * and {@link https://git.corp.adobe.com/DMA/dcx-js/pull/575#discussion_r3876968}
     */
    /**
     * Category
     * (only CompositeXfer)
     */
    'event.category': AnalyticsEventCategory;
    /**
     * Subcategory
     * (push, pull, etc.)
     */
    'event.subcategory': AnalyticsEventSubCategory;
}

export declare enum AnalyticsEventCategory {
    CompositeXfer = 0
}

declare type AnalyticsEventHandlers = {
    [AnalyticsEvent.PushComposite]: (event: AnalyticsCompositeEvent) => void;
    [AnalyticsEvent.CreateComposite]: (event: AnalyticsCompositeEvent) => void;
    [AnalyticsEvent.PullComposite]: (event: AnalyticsCompositeEvent) => void;
    [AnalyticsEvent.PullCompositeVersion]: (event: AnalyticsCompositeEvent) => void;
    [AnalyticsEvent.UploadComponent]: (event: AnalyticsComponentEvent) => void;
    [AnalyticsEvent.DownloadComponent]: (event: AnalyticsComponentEvent) => void;
};

export declare enum AnalyticsEventSubCategory {
    Push = 0,
    MinPull = 1,
    VersionPull = 2,
    Upload = 3,
    Download = 4,
    Create = 5,
    Unknown = 6
}

export declare enum AnalyticsEventType {
    Success = 0,
    Error = 1
}

declare type ComponentAnalyticsKeys = Exclude<AnalyticsEvent, AnalyticsEvent.All | AnalyticsEvent.CreateComposite | AnalyticsEvent.PullComposite | AnalyticsEvent.PullCompositeVersion | AnalyticsEvent.PushComposite>;

declare type CompositeAnalyticsKeys = Exclude<AnalyticsEvent, AnalyticsEvent.All | AnalyticsEvent.DownloadComponent | AnalyticsEvent.UploadComponent>;

export declare type DebugFormatter = (namespace: string, time: number, prevTime: number, messages: any[]) => string;

declare const _default: AdobeDCXLogger;
export default _default;

export declare const emitAnalyticsEvent: <T extends [type: (import("./src/analytics").AnalyticsEvent.PushComposite | import("./src/analytics").AnalyticsEvent.CreateComposite | import("./src/analytics").AnalyticsEvent.PullComposite | import("./src/analytics").AnalyticsEvent.PullCompositeVersion) | (import("./src/analytics").AnalyticsEvent.UploadComponent | import("./src/analytics").AnalyticsEvent.DownloadComponent), data: {
    composite: AdobeDCXComposite;
    branch?: AdobeDCXBranch | undefined;
    response: AdobeResponse<any>;
    error?: AdobeDCXError<unknown> | undefined;
    component?: {
        relationship?: string | undefined;
        type?: string | undefined;
        length?: number | undefined;
    } | undefined;
    derivationType?: number | undefined;
    isBlockTransferred?: boolean | undefined;
}, service?: string | undefined]>(key: T[0], data: T[1]) => void;

export declare const getGlobalLogger: () => AdobeDCXLogger;

export declare const getGlobalOrLocalLogger: () => AdobeDCXLogger;

/**
 * Log any number of messages
 *
 * @param {(string | StringLike)[]} messages - set of messages to log
 */
export declare const log: (...messages: (string | StringLike)[]) => void;

/**
 * Can be set during construction or via `logger.logCallback = logCallback`.
 * @callback LogCallback@callback LogCallback
 *   @param {String} message The message to be logged.
 */
export declare type LogCallback = (...messages: any[]) => void;

export declare enum LogLevel {
    Deprecated = 0,
    Error = 1,
    Warn = 2,
    Log = 3,
    Debug = 4
}

/**
 * Make an analytics event.
 * @param type
 * @param data
 * @param service
 */
export declare function makeAnalyticsEvent(type: ComponentAnalyticsKeys, data: {
    composite: AdobeDCXComposite;
    response: AdobeResponse;
    error?: AdobeDCXError | undefined;
    component: {
        relationship?: string;
        type?: string;
        length?: number;
    };
    isBlockTransferred?: boolean;
}, service?: string): AnalyticsComponentEvent;

export declare function makeAnalyticsEvent(type: CompositeAnalyticsKeys, data: {
    composite: AdobeDCXComposite;
    branch: AdobeDCXBranch;
    response: AdobeResponse;
    error?: AdobeDCXError;
    derivationType: number;
}, service?: string): AnalyticsCompositeEvent;

export declare function makeAnalyticsEvent(type: CompositeAnalyticsKeys | ComponentAnalyticsKeys, data: {
    composite: AdobeDCXComposite;
    branch?: AdobeDCXBranch;
    response: AdobeResponse;
    error?: AdobeDCXError | undefined;
    component?: {
        relationship?: string;
        type?: string;
        length?: number;
    };
    derivationType?: number;
    isBlockTransferred?: boolean;
}, service?: string): AnalyticsComponentEvent | AnalyticsCompositeEvent;

/**
 * Get Debug instance
 *
 * @param {string} name - namespace of debugger
 */
export declare const newDebug: (namespace: string) => (...args: any[]) => void;

/**
 * Sets the logCallback function
 * Backward compatible, intercepts logs at or above specified log level
 *
 * @param {LogCallback} logCallback called when DCX wants to log something to the client.
 */
export declare const setLogCallback: (logCallback?: LogCallback) => void;

export declare interface StringLike {
    toString(): string;
}

export { }
