/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { AdobeDCXLogger, LogCallback, StringLike } from './src/AdobeDCXLogger';
import { AnalyticsEventHandlers, makeAnalyticsEvent } from './src/analytics';
export { AdobeDCXLogger, DebugFormatter, LogCallback, LogLevel, StringLike } from './src/AdobeDCXLogger';
export {
    AnalyticsComponentEvent,
    AnalyticsCompositeEvent,
    AnalyticsEvent,
    AnalyticsEventCategory,
    AnalyticsEventSubCategory,
    AnalyticsEventType,
    makeAnalyticsEvent,
} from './src/analytics';

export const getGlobalOrLocalLogger = () => {
    const gLogger = getGlobalLogger();
    if ((gLogger as any)._logCallback !== undefined) {
        return gLogger;
    }
    return AdobeDCXLogger.getInstance();
};

export const emitAnalyticsEvent = <T extends Parameters<typeof makeAnalyticsEvent>>(key: T[0], data: T[1]) => {
    const logger = getGlobalOrLocalLogger();
    const ev = makeAnalyticsEvent(key, data);
    try {
        logger.emit(key as keyof AnalyticsEventHandlers, [ev]);
    } catch (e) {
        const message = e instanceof Error ? e.message : 'Unknown Error';
        logger.error(`Error in analytics hook: `, message);
    }
};

export const getGlobalLogger = (): AdobeDCXLogger => {
    if (
        typeof globalThis === 'object' &&
        typeof globalThis.dcxjs === 'object' &&
        globalThis.dcxjs.logger &&
        globalThis.dcxjs.logger.getInstance
    ) {
        return globalThis.dcxjs.logger.getInstance();
    }

    // console.warn('[@dcx/logger] Not registered to namespace. Returning no-op logger.');

    // Logger not registered, likely using a module without importing @dcx/core.
    // Give a stubbed version, TODO: show a console warning. Right now commented out
    // since it blocks up test output.
    // eslint-disable-next-line @typescript-eslint/no-empty-function
    const noop = () => {};
    return {
        log: noop,
        warn: noop,
        error: noop,
        deprecated: noop,
        debug: noop,
        newLogger: getGlobalLogger,
    } as unknown as AdobeDCXLogger;
};

/**
 * Sets the logCallback function
 * Backward compatible, intercepts logs at or above specified log level
 *
 * @param {LogCallback} logCallback called when DCX wants to log something to the client.
 */
export const setLogCallback = (logCallback?: LogCallback): void => {
    getGlobalOrLocalLogger().logCallback = logCallback;
};

/**
 * Log any number of messages
 *
 * @param {(string | StringLike)[]} messages - set of messages to log
 */
export const log = (...messages: (string | StringLike)[]): void => {
    const inst = getGlobalOrLocalLogger();
    messages.forEach(inst.log.bind(inst));
};

/**
 * Get Debug instance
 *
 * @param {string} name - namespace of debugger
 */
export const newDebug = (namespace: string): ((...args: any[]) => void) => {
    const inst = getGlobalOrLocalLogger();
    return inst.Debug(namespace);
};

export default getGlobalOrLocalLogger();
