/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { EventEmitter, isNode, isObject, now } from '@dcx/util';
import { AnalyticsEvent, AnalyticsEventHandlers } from './analytics';

export interface StringLike {
    toString(): string;
}

export enum LogLevel {
    Deprecated = 0,
    Error = 1,
    Warn = 2,
    Log = 3,
    Debug = 4,
}
const LogLevelStr: Record<LogLevel, string> = {
    [LogLevel.Deprecated]: 'error',
    [LogLevel.Error]: 'error',
    [LogLevel.Log]: 'log',
    [LogLevel.Warn]: 'warn',
    [LogLevel.Debug]: 'debug',
};

export type DebugFormatter = (namespace: string, time: number, prevTime: number, messages: any[]) => string;

/**
 * Can be set during construction or via `logger.logCallback = logCallback`.
 * @callback LogCallback
 *   @param {String} message The message to be logged.
 */
export type LogCallback = (...messages: any[]) => void;

export class AdobeDCXLogger extends EventEmitter<AnalyticsEventHandlers> {
    private static _instance: AdobeDCXLogger | undefined;
    private _logCallback: LogCallback | undefined;
    private _logLevel: LogLevel = LogLevel.Warn;
    private _prevDebugTime: number = now();
    private _debugFormatter: DebugFormatter = (namespace: string, time: number, prevTime: number, messages: any[]) => {
        return `[${namespace} (+${((time - prevTime) * 1000).toFixed(0)})] ${messages
            .map((m) => (typeof m === 'string' ? m : JSON.stringify(m)))
            .join(' ')}`;
    };
    private _debugNamespaces: RegExp[] = [];
    private _debugSkips: RegExp[] = [];

    public suppressDeprecationWarnings = false;

    private constructor(lcb?: LogCallback) {
        super([
            AnalyticsEvent.CreateComposite,
            AnalyticsEvent.UploadComponent,
            AnalyticsEvent.PushComposite,
            AnalyticsEvent.PullComposite,
            AnalyticsEvent.PullCompositeVersion,
            AnalyticsEvent.DownloadComponent,
        ]);
        if (lcb) {
            this._logCallback = lcb;
        }
        this._initNamespaces();
    }

    /**
     * Current active debug namespace regexes
     * Set with setDebugNamespaces()
     */
    public get debugNamespaces() {
        return this._debugNamespaces;
    }

    /**
     * Currently active skipped namespace regexes
     * Set with setDebugNamespaces() using a `-` prefix
     */
    public get debugSkips() {
        return this._debugSkips;
    }

    /**
     * Set a log level.
     *
     * @example
     * ```js
     * logger.logLevel = LogLevel.Warn;
     * ```
     */
    public set logLevel(lvl: LogLevel) {
        if (!Object.values(LogLevel).includes(lvl)) {
            throw new Error(`Invalid LogLevel, must be one of: ${Object.values(LogLevel).join(', ')}.`);
        }
        this._logLevel = lvl;
    }
    public get logLevel() {
        return this._logLevel;
    }

    /**
     * Level 5 verbosity
     */
    public static LEVEL_DEBUG: LogLevel = LogLevel.Debug;
    /**
     * Level 3 verbosity
     */
    public static LEVEL_LOG: LogLevel = LogLevel.Log;
    /**
     * Level 3 verbosity
     */
    public static LEVEL_WARN: LogLevel = LogLevel.Warn;
    /**
     * Level 2 verbosity
     */
    public static LEVEL_ERROR: LogLevel = LogLevel.Error;
    /**
     * Level 1 verbosity
     * Can also disable deprecation warnings through AdobeDCXLogger#suppressDeprecationWarnings
     */
    public static LEVEL_DEPRECATED: LogLevel = LogLevel.Deprecated;

    /**
     *
     * @param event
     * @param handler
     * @returns
     */
    public on<E extends keyof AnalyticsEventHandlers>(
        event: E | AnalyticsEvent.All,
        handler: AnalyticsEventHandlers[E],
    ): number {
        if (event !== AnalyticsEvent.All) {
            return super.on(event, handler);
        }

        // register handler to all hooks
        const hooks = Object.values(AnalyticsEvent);
        let last;
        for (let i = 0, len = hooks.length; i < len; i++) {
            const e = hooks[i];
            last = super.on(e as keyof AnalyticsEventHandlers, handler);
        }
        return last;
    }

    /**
     * Get singleton instance of logger
     * @returns
     */
    public static getInstance(): AdobeDCXLogger {
        if (AdobeDCXLogger._instance == null) {
            AdobeDCXLogger._instance = new AdobeDCXLogger();
        }
        return AdobeDCXLogger._instance;
    }

    /**
     * Return new logger instance
     * @param {Function} lcb Log callback
     */
    public static newLogger(lcb?: LogCallback) {
        return new AdobeDCXLogger(lcb);
    }

    /**
     * Set a log callback
     *
     * @public
     * Setting a log callback will skip any other callbacks registered.
     */
    public set logCallback(val: LogCallback | undefined) {
        this._logCallback = val;
    }
    public get logCallback() {
        return this._logCallback;
    }

    /* istanbul ignore next */
    private _initNamespaces() {
        if (isNode()) {
            this.setDebugNamespaces(process.env.DCX_DEBUG || '');
        } else if (isObject(globalThis) && isObject(globalThis.dcxjs)) {
            this.setDebugNamespaces(globalThis.dcxjs.debug || '');
        }

        if (this._debugNamespaces.length > 0) {
            this._logLevel = LogLevel.Debug;
        }
    }

    /**
     * Log a message as some level
     *
     * @param messageLvl        Message log level.
     * @param debugNamespace    Namespace of debug, if message is debug level.
     * @param messages          Array of items to log.
     */
    private _log(messageLvl: LogLevel, debugNamespace: string | undefined, messages: any[]) {
        try {
            if (messageLvl === AdobeDCXLogger.LEVEL_DEPRECATED) {
                if (!this.suppressDeprecationWarnings && this._logLevel >= messageLvl) {
                    console.warn(...messages);
                }
            } else if (typeof this._logCallback === 'function' && this._logLevel >= messageLvl) {
                this._logCallback.call(undefined, ...messages);
            } else if (this._logLevel >= messageLvl) {
                if (messageLvl === LogLevel.Debug) {
                    // if debug namespace is enabled, log debug message
                    const enabled = this._debugEnabled(debugNamespace as string);
                    if (enabled) {
                        const prevTime = this._prevDebugTime;
                        this._prevDebugTime = now();
                        // Fallback to console.log if debug doesn't exist
                        /* istanbul ignore next */
                        const dbg = console.debug || console.log;
                        dbg(this._debugFormatter(debugNamespace as string, this._prevDebugTime, prevTime, messages));
                    }
                } else {
                    // @TODO: Track down where these messages are being treated like an array callback
                    // The last two items are an index (0) and the array of messages itself.
                    console[LogLevelStr[messageLvl]](...messages.slice(0, -2));
                }
            }
        } catch (e) {
            // ignore
        }
    }

    /**
     * Report a log message. This gets reported to a DCX client if they provided a logCallback function
     * @param {string|StringLike} message Message to log.
     */
    public log(...messages: (string | StringLike)[]) {
        this._log(LogLevel.Log, undefined, messages);
    }

    /**
     * Log warn message
     * @param {string|StringLike} message
     */
    public warn(...messages: (string | StringLike)[]) {
        this._log(LogLevel.Warn, undefined, messages);
    }

    /**
     * Log error message
     * @param {string|StringLike} message
     */
    public error(...messages: (string | StringLike)[]) {
        this._log(LogLevel.Error, undefined, messages);
    }

    /**
     * Log a deprecation message
     * Can be supressed individually
     * Logged as a warning
     * @param {string|StringLike} message
     */
    public deprecated(...messages: (string | StringLike)[]) {
        this._log(LogLevel.Deprecated, undefined, messages);
    }

    /**
     * Returns true if the given mode namespace is enabled, false otherwise.
     *
     * @param namespace
     * @return {Boolean}
     */
    private _debugEnabled(namespace: string) {
        if (namespace[namespace.length - 1] === '*') {
            return true;
        }

        let i;
        let len;
        for (i = 0, len = this._debugSkips.length; i < len; i++) {
            if (this._debugSkips[i].test(namespace)) {
                return false;
            }
        }

        for (i = 0, len = this._debugNamespaces.length; i < len; i++) {
            if (this._debugNamespaces[i].test(namespace)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Override default debug formatter.
     * @param formatter
     */
    public setDebugFormatter(formatter: DebugFormatter) {
        this._debugFormatter = formatter;
    }

    /**
     * Enable debug namespaces.
     * Overwrites existing settings, is not additive.
     * This is set at instantiation from a value pulled from
     * the dcx window namespace (on browser) or from the environment
     * variable process.env.DCX_DEBUG (on Node).
     *
     *
     * @example
     * ```js
     * // enable all except http
     * logger.setDebugNamespaces("dcx:*,-dcx:http:*")
     * ```
     *
     * @param namespaces
     */
    public setDebugNamespaces(namespaces?: string) {
        this._debugNamespaces = [];
        this._debugSkips = [];

        let i;
        const split = (typeof namespaces === 'string' ? namespaces : '').split(/[\s,]+/);
        const len = split.length;

        for (i = 0; i < len; i++) {
            if (!split[i]) {
                // ignore empty strings
                continue;
            }

            namespaces = split[i].replace(/\*/g, '.*?');

            if (namespaces[0] === '-') {
                this._debugSkips.push(new RegExp('^' + namespaces.substr(1) + '$'));
            } else {
                this._debugNamespaces.push(new RegExp('^' + namespaces + '$'));
            }
        }
    }

    /**
     * Get namespace debug function
     * @param {string|StringLike} message
     */
    public Debug(namespace: string) {
        return (...msgs: any[]) => {
            this._log(LogLevel.Debug, namespace, msgs);
        };
    }
}
