/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { AdobeDCXBranch, AdobeDCXComposite, AdobeDCXError, AdobeResponse } from '@dcx/common-types';
import { isObject, isUndefined, validateObject, validateParams } from '@dcx/util';

export const VERSION: string = process.env.DCX_VERSION as string;
export const COM_NAME = 'dcx-js';
export const SVC_NAME = 'css';

export enum AnalyticsEventType {
    Success,
    Error,
}

export enum AnalyticsEventCategory {
    CompositeXfer,
    // Not applicable to dcx-js
    // StandAloneXfer,
    // Sync,
    // GarbageCollection,
    // ConflictResolution,
}

export enum AnalyticsEventSubCategory {
    Push,
    MinPull, // called "pull" in dcx-js, but same as min pull elsewhere
    VersionPull,
    Upload,
    Download,
    Create,
    Unknown,
    // Not applicable to dcx-js
    // Pull,
    // InSync
}

interface AnalyticsEventBase {
    /**
     * Library name
     * See {@link https://git.corp.adobe.com/DMA/dcx-js/pull/575#discussion_r3873327}
     * and {@link https://git.corp.adobe.com/DMA/dcx-js/pull/575#discussion_r3876987}
     */
    'env.com.name': 'dcx-js';
    /**
     * API type (cc-storage, css, ..)
     * See {@link https://git.corp.adobe.com/DMA/dcx-js/pull/575#discussion_r3876987}
     */
    'env.svc.name': string;

    /**
     * dcx-js version
     */
    'env.com.version': string;

    /**
     * x-request-id from the request
     * This may be the last request or the failed request
     * See {@link https://git.corp.adobe.com/DMA/dcx-js/pull/575#discussion_r3875392}
     */
    'event.request_id': string;

    /**
     * Cloud asset ID
     */
    'event.cloud_id': string;

    /**
     * Cloud repository ID
     */
    'custom.content.repository_id': string;

    /**
     * Type
     * (success or failure)
     */
    'event.type': AnalyticsEventType;

    /**
     * Not set
     * See {@link https://git.corp.adobe.com/DMA/dcx-js/pull/575#discussion_r3873385}
     * and {@link https://git.corp.adobe.com/DMA/dcx-js/pull/575#discussion_r3876968}
     */
    // 'content.category': AnalyticsEventCategory;

    /**
     * Category
     * (only CompositeXfer)
     */
    'event.category': AnalyticsEventCategory;

    /**
     * Subcategory
     * (push, pull, etc.)
     */
    'event.subcategory': AnalyticsEventSubCategory;

    /**
     * Not applicable to dcx-js
     */
    // Local ID, does not exist in dcx-js
    // 'content.id': string;
    // Storage area, unknown to dcx-js, possibly known to clients
    // 'custom.content.storage_area': string;
}

interface AnalyticsErrorEvent {
    // Error code
    'event.error_code': string;

    // Error message
    'event.error_desc': string;

    // Underlying error code, if any
    'dcx.underlying_error_code'?: string;

    // Underlying error message, if any
    'dcx.underlying_error_desc'?: string;
}

/**
 * ========================================
 * ========= Composite Events =============
 * ========================================
 */
interface AnalyticsCompositeSuccessEvent extends AnalyticsEventBase {
    'dcx.num_total_components': number;
    'custom.content.xmp_derivation_type': number;

    /**
     * The following fields are not applicable to dcx-js
     *
     * During push, no components are transferred in the online operational model
     * so no info is provided for these fields. The exception being copy of
     * components when pushing a composite of DerivationType=COPY on the first push,
     * but that is S2S copy and, again, doesn't use block transfer. It would be possible
     * to provide numbers for component copies, but that is currently omitted.
     *
     * See {@link https://git.corp.adobe.com/DMA/dcx-js/pull/575#discussion_r3875499}
     */
    // 'dcx.num_modified_components': number;
    // 'dcx.num_xferred_block': number;
    // 'dcx.regular_xferred_cumulative_size': number;
    // 'dcx.num_xferred_regular': number;
    // 'dcx.block_xferred_cumulative_size': number;
}

interface AnalyticsCompositeFailureEvent extends AnalyticsCompositeSuccessEvent, AnalyticsErrorEvent {
    'dcx.root_error_desc': string;
}

// TODO: Remove these partials when event format is absolutely defined
export type AnalyticsCompositeEvent = Partial<AnalyticsCompositeSuccessEvent> | Partial<AnalyticsCompositeFailureEvent>;

/**
 * ========================================
 * ========= Component Events =============
 * ========================================
 */
interface AnalyticsComponentSuccessEvent extends AnalyticsEventBase {
    'content.mimetype': string;
}
interface AnalyticsComponentFailureEvent extends AnalyticsComponentSuccessEvent, AnalyticsErrorEvent {
    // Whether block tranfer was used
    'dcx.block_transferred': boolean;

    // Media type of component
    'content.type': string;

    // Content size of component
    'content.size': number;

    // Relation of component
    'dcx.component_rel'?: string;
}
// TODO: Remove these partials when event format is absolutely defined
export type AnalyticsComponentEvent = Partial<AnalyticsComponentSuccessEvent> | Partial<AnalyticsComponentFailureEvent>;

/**
 * Emitted event types valid for registration.
 */
export enum AnalyticsEvent {
    PushComposite = 'analyticsPush',
    CreateComposite = 'analyticsCreate',
    PullComposite = 'analyticsPull',
    PullCompositeVersion = 'analyticsPullVersion',
    UploadComponent = 'analyticsUpload',
    DownloadComponent = 'analyticsDownload',
    All = '*',
}
type CompositeAnalyticsKeys = Exclude<
    AnalyticsEvent,
    AnalyticsEvent.All | AnalyticsEvent.DownloadComponent | AnalyticsEvent.UploadComponent
>;
// | AnalyticsEventKey.CreateComposite
// | AnalyticsEventKey.PullComposite
// | AnalyticsEventKey.PullCompositeVersion
// | AnalyticsEventKey.PushComposite;

type ComponentAnalyticsKeys = Exclude<
    AnalyticsEvent,
    | AnalyticsEvent.All
    | AnalyticsEvent.CreateComposite
    | AnalyticsEvent.PullComposite
    | AnalyticsEvent.PullCompositeVersion
    | AnalyticsEvent.PushComposite
>;

export type AnalyticsEventHandlers = {
    [AnalyticsEvent.PushComposite]: (event: AnalyticsCompositeEvent) => void;
    [AnalyticsEvent.CreateComposite]: (event: AnalyticsCompositeEvent) => void;
    [AnalyticsEvent.PullComposite]: (event: AnalyticsCompositeEvent) => void;
    [AnalyticsEvent.PullCompositeVersion]: (event: AnalyticsCompositeEvent) => void;
    [AnalyticsEvent.UploadComponent]: (event: AnalyticsComponentEvent) => void;
    [AnalyticsEvent.DownloadComponent]: (event: AnalyticsComponentEvent) => void;
};

// These overloads are ugly, but due to the way Typescript
// determines Parameter types, there needs to be one extra
// overload that should be the implementation signature.
// The third overload here would usually not be needed,
// but is used by the emitAnalyticsEvent definition in index.ts.
/**
 * Make an analytics event.
 * @param type
 * @param data
 * @param service
 */
export function makeAnalyticsEvent(
    type: ComponentAnalyticsKeys,
    data: {
        composite: AdobeDCXComposite;
        response: AdobeResponse;
        error?: AdobeDCXError | undefined;
        component: { relationship?: string; type?: string; length?: number };
        isBlockTransferred?: boolean;
    },
    service?: string,
): AnalyticsComponentEvent;
export function makeAnalyticsEvent(
    type: CompositeAnalyticsKeys,
    data: {
        composite: AdobeDCXComposite;
        branch: AdobeDCXBranch;
        response: AdobeResponse;
        error?: AdobeDCXError;
        derivationType: number;
    },
    service?: string,
): AnalyticsCompositeEvent;
export function makeAnalyticsEvent(
    type: CompositeAnalyticsKeys | ComponentAnalyticsKeys,
    data: {
        composite: AdobeDCXComposite;
        branch?: AdobeDCXBranch;
        response: AdobeResponse;
        error?: AdobeDCXError | undefined;
        component?: { relationship?: string; type?: string; length?: number };
        derivationType?: number;
        isBlockTransferred?: boolean;
    },
    service?: string,
): AnalyticsComponentEvent | AnalyticsCompositeEvent;
export function makeAnalyticsEvent(
    type: CompositeAnalyticsKeys | ComponentAnalyticsKeys,
    data: {
        composite: AdobeDCXComposite;
        branch?: AdobeDCXBranch;
        response: AdobeResponse;
        error?: AdobeDCXError | undefined;
        component?: { relationship?: string; type?: string; length?: number };
        derivationType?: number;
        isBlockTransferred?: boolean;
    },
    service: string = SVC_NAME,
): AnalyticsComponentEvent | AnalyticsCompositeEvent {
    validateParams(['data', data, 'object']);
    validateObject(data, 'data', ['composite', 'object'], ['response', 'object'], ['error', 'object', true]);

    const { component, error, composite, branch, derivationType, isBlockTransferred } = data;
    // Use the failed response if there is one
    const response = data.error?.response || data.response;
    let category: AnalyticsEventSubCategory;
    let isCompositeEvent = false;

    switch (type) {
        case AnalyticsEvent.CreateComposite:
            isCompositeEvent = true;
            category = AnalyticsEventSubCategory.Create;
            break;
        case AnalyticsEvent.PushComposite:
            isCompositeEvent = true;
            category = AnalyticsEventSubCategory.Push;
            break;
        case AnalyticsEvent.PullComposite:
            isCompositeEvent = true;
            category = AnalyticsEventSubCategory.MinPull;
            break;
        case AnalyticsEvent.PullCompositeVersion:
            isCompositeEvent = true;
            category = AnalyticsEventSubCategory.VersionPull;
            break;
        case AnalyticsEvent.DownloadComponent:
        case AnalyticsEvent.UploadComponent:
            validateParams(['component', component, 'object']);
            validateObject(component!, 'data.component', ['type', 'string']);
            category =
                type === AnalyticsEvent.DownloadComponent
                    ? AnalyticsEventSubCategory.Download
                    : AnalyticsEventSubCategory.Upload;
            break;
        default:
            category = AnalyticsEventSubCategory.Unknown;
    }

    if (isCompositeEvent) {
        validateParams(['branch', branch, 'object', true]);
    }

    const event: AnalyticsEventBase | AnalyticsErrorEvent = {
        'env.com.name': COM_NAME,
        'env.svc.name': service,
        'env.com.version': VERSION,
        'event.request_id': response.headers['x-request-id'],
        'event.cloud_id': composite.assetId as string,
        'custom.content.repository_id': composite.repositoryId as string,
        'event.type': AnalyticsEventType.Success,
        'event.category': AnalyticsEventCategory.CompositeXfer,
        'event.subcategory': category,
    };

    // Error base data, if undefined these are filtered out before return
    /* istanbul ignore next */
    const underlyingErrorCode = (error?.underlyingError as AdobeDCXError)?.code;
    /* istanbul ignore next */
    const underlyingErrorMsg = error?.underlyingError?.message as string;
    if (error) {
        // AnalyticsErrorEvent
        event['event.error_code'] = error.code;
        event['event.error_desc'] = error.message;
        event['dcx.underlying_error_code'] = underlyingErrorCode;
        event['dcx.underlying_error_desc'] = underlyingErrorMsg;
        event['event.type'] = AnalyticsEventType.Error;
    }

    if (category === AnalyticsEventSubCategory.Unknown) {
        // AnalyticsEvent | AnalyticsErrorEvent
        return _filterEvent(event);
    }

    if (isCompositeEvent) {
        // AnalyticsCompositeEvent
        event['dcx.num_total_components'] = branch?.allComponents().length;
        event['custom.content.xmp_derivation_type'] = derivationType;
        if (error) {
            event['dcx.root_error_desc'] = underlyingErrorMsg;
        }
        return _filterEvent(event);
    }

    // Reaching here means component event
    // AnalyticsComponentEvent
    event['content.mimetype'] = component!.type;
    if (error) {
        // AnalyticsComponentFailureEvent
        event['dcx.block_transferred'] = isBlockTransferred != null ? isBlockTransferred : false; // todo
        event['content.type'] = component!.type;
        event['content.size'] = component!.length as number;
        event['dcx.component_rel'] = component!.relationship;
    }
    return _filterEvent(event);
}

/**
 * Remove non-primitives and undefined/null entries.
 * @param event
 * @returns
 */
function _filterEvent(event: AnalyticsEventBase): AnalyticsEventBase {
    const out: AnalyticsEventBase = {} as AnalyticsEventBase;
    for (const k in event) {
        const v = event[k];
        if (!isUndefined(v) && !Array.isArray(v) && !isObject(v)) {
            out[k] = v;
        }
    }
    return out;
}
