/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
import { EventEmitter } from '@dcx/util';
import { AnalyticsEvent, AnalyticsEventHandlers } from './analytics';
export interface StringLike {
    toString(): string;
}
export declare enum LogLevel {
    Deprecated = 0,
    Error = 1,
    Warn = 2,
    Log = 3,
    Debug = 4
}
export type DebugFormatter = (namespace: string, time: number, prevTime: number, messages: any[]) => string;
/**
 * Can be set during construction or via `logger.logCallback = logCallback`.
 * @callback LogCallback
 *   @param {String} message The message to be logged.
 */
export type LogCallback = (...messages: any[]) => void;
export declare class AdobeDCXLogger extends EventEmitter<AnalyticsEventHandlers> {
    private static _instance;
    private _logCallback;
    private _logLevel;
    private _prevDebugTime;
    private _debugFormatter;
    private _debugNamespaces;
    private _debugSkips;
    suppressDeprecationWarnings: boolean;
    private constructor();
    /**
     * Current active debug namespace regexes
     * Set with setDebugNamespaces()
     */
    get debugNamespaces(): RegExp[];
    /**
     * Currently active skipped namespace regexes
     * Set with setDebugNamespaces() using a `-` prefix
     */
    get debugSkips(): RegExp[];
    /**
     * Set a log level.
     *
     * @example
     * ```js
     * logger.logLevel = LogLevel.Warn;
     * ```
     */
    set logLevel(lvl: LogLevel);
    get logLevel(): LogLevel;
    /**
     * Level 5 verbosity
     */
    static LEVEL_DEBUG: LogLevel;
    /**
     * Level 3 verbosity
     */
    static LEVEL_LOG: LogLevel;
    /**
     * Level 3 verbosity
     */
    static LEVEL_WARN: LogLevel;
    /**
     * Level 2 verbosity
     */
    static LEVEL_ERROR: LogLevel;
    /**
     * Level 1 verbosity
     * Can also disable deprecation warnings through AdobeDCXLogger#suppressDeprecationWarnings
     */
    static LEVEL_DEPRECATED: LogLevel;
    /**
     *
     * @param event
     * @param handler
     * @returns
     */
    on<E extends keyof AnalyticsEventHandlers>(event: E | AnalyticsEvent.All, handler: AnalyticsEventHandlers[E]): number;
    /**
     * Get singleton instance of logger
     * @returns
     */
    static getInstance(): AdobeDCXLogger;
    /**
     * Return new logger instance
     * @param {Function} lcb Log callback
     */
    static newLogger(lcb?: LogCallback): AdobeDCXLogger;
    /**
     * Set a log callback
     *
     * @public
     * Setting a log callback will skip any other callbacks registered.
     */
    set logCallback(val: LogCallback | undefined);
    get logCallback(): LogCallback | undefined;
    private _initNamespaces;
    /**
     * Log a message as some level
     *
     * @param messageLvl        Message log level.
     * @param debugNamespace    Namespace of debug, if message is debug level.
     * @param messages          Array of items to log.
     */
    private _log;
    /**
     * Report a log message. This gets reported to a DCX client if they provided a logCallback function
     * @param {string|StringLike} message Message to log.
     */
    log(...messages: (string | StringLike)[]): void;
    /**
     * Log warn message
     * @param {string|StringLike} message
     */
    warn(...messages: (string | StringLike)[]): void;
    /**
     * Log error message
     * @param {string|StringLike} message
     */
    error(...messages: (string | StringLike)[]): void;
    /**
     * Log a deprecation message
     * Can be supressed individually
     * Logged as a warning
     * @param {string|StringLike} message
     */
    deprecated(...messages: (string | StringLike)[]): void;
    /**
     * Returns true if the given mode namespace is enabled, false otherwise.
     *
     * @param namespace
     * @return {Boolean}
     */
    private _debugEnabled;
    /**
     * Override default debug formatter.
     * @param formatter
     */
    setDebugFormatter(formatter: DebugFormatter): void;
    /**
     * Enable debug namespaces.
     * Overwrites existing settings, is not additive.
     * This is set at instantiation from a value pulled from
     * the dcx window namespace (on browser) or from the environment
     * variable process.env.DCX_DEBUG (on Node).
     *
     *
     * @example
     * ```js
     * // enable all except http
     * logger.setDebugNamespaces("dcx:*,-dcx:http:*")
     * ```
     *
     * @param namespaces
     */
    setDebugNamespaces(namespaces?: string): void;
    /**
     * Get namespace debug function
     * @param {string|StringLike} message
     */
    Debug(namespace: string): (...msgs: any[]) => void;
}
//# sourceMappingURL=AdobeDCXLogger.d.ts.map