/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

// import Debug from 'debug';

// const dbg = Debug('dcx:promise');
import { AdobeDCXError, AnyObject, IAdobePromise } from '@dcx/common-types';
import { isObject } from '@dcx/util';

// eslint-disable-next-line @typescript-eslint/no-explicit-any
export type PartialRecord<K extends keyof any, T> = {
    [P in K]?: T;
};

type onResolveFn<P = unknown, R = unknown> = (result: P) => R | PromiseLike<R>;
type onRejectFn<P = unknown, R = unknown> = (reason: P) => R | PromiseLike<R>;
type onCancelFn<R = unknown> = (reason?: R) => void;

type HandlerKey = 'resolve' | 'reject' | 'cancel';
type HandlerType<T> = onResolveFn<T> | onRejectFn | onCancelFn;

const isFunction = (val: unknown) => {
    return toString.call(val) === '[object Function]';
};

interface SettledRejectedPromise<E> {
    status: 'rejected';
    reason: E;
}
interface SettledFulfilledPromise<T> {
    status: 'fulfilled';
    value: T;
}
export type SettledPromise<T = any, E = any> = SettledRejectedPromise<E> | SettledFulfilledPromise<T>;

class _AdobePromise<T = any, E = any, X extends AnyObject = AnyObject> {
    /** @internal */
    private _promise: Promise<any> = null as unknown as Promise<any>;
    /** @internal */
    private _props: X = {} as X;
    /** @internal */
    private _registeredProps: string[] = [];
    // private _handlers: PartialRecord<HandlerKey, HandlerType<T>[]> = {
    //     cancel: []
    // };
    /** @internal */
    private _handlers: Record<'cancel', onCancelFn[]> = {
        cancel: [],
    };
    /** @internal */
    private _done = false;
    /** @internal */
    private _canceled = false;
    /** @internal */
    private _cancelReason: unknown;
    /** @internal */
    private _internalKeys: string[] = [];

    constructor(
        method: (
            this: AdobePromise<T, E, X>,
            resolve: (result?: T) => void,
            reject: (reason?: E) => void,
            onCancel: (handler: onCancelFn) => void,
        ) => unknown,
        source?: X,
    ) {
        this._internalKeys = [
            ...Object.keys(this),
            ...Object.keys(Object.getOwnPropertyDescriptors(Object.getPrototypeOf(this))),
            '_cancelReason',
        ];

        if (source && typeof source === 'object') {
            // assign properties
            this._setProps(source);
        }

        this._promise = new Promise<T>((resolve_, reject_) => {
            const resolve = (val: any) => {
                if (!this._done) {
                    this._done = true;
                    resolve_(val);
                }
            };
            const reject = (err: any) => {
                if (!this._done) {
                    this._done = true;
                    reject_(err);
                }
            };

            return method.call(
                this as unknown as AdobePromise<T, E, X>,
                resolve,
                reject,
                this._registerCancelHandler.bind(this),
            );
        });

        return new Proxy(this, {
            set: function (target: _AdobePromise<T, E, X>, key: string, value) {
                if (!target._internalKeys.includes(key)) {
                    // doesn't exist in internals, set on props and add to registered props
                    if (!target._registeredProps.includes(key)) {
                        target._registeredProps.push(key);
                    }

                    (target.props as Record<string, unknown>)[key] = value;
                } else if (['_promise', '_canceled', '_cancelReason', '_props', '_registeredProps'].includes(key)) {
                    // allow certain internals to be overwritten
                    (target as unknown as Record<string, unknown>)[key] = value;
                } else {
                    throw new Error('Cannot overwrite internal AdobePromise property.');
                }
                return true;
            },
            get: function (target: _AdobePromise<T, E, X>, key: string) {
                if (typeof key !== 'symbol' && !target._internalKeys.includes(key)) {
                    return (target.props as Record<string, unknown>)[key];
                }
                return (target as unknown as Record<string, unknown>)[key];
            },
        });
    }

    readonly name: string = 'AdobePromise';
    get [Symbol.toStringTag]() {
        return this.name;
    }

    /**
     * Return immediately rejected Promise with AdobePromise type completion.
     * @param {E2 = any} reason - Rejection value
     */
    static reject<E2 = any, X2 extends AnyObject = AnyObject>(
        reason?: E2 | PromiseLike<E2>,
        source?: X2,
    ): _AdobePromise<any, E2, X2> & X2 {
        return new AdobePromise<any, E2, X2>((_, reject) => {
            return Promise.reject(reason).catch((e) => {
                reject && reject(e);
            });
        }, source) as _AdobePromise<any, E2, X2> & X2;
    }

    /**
     * Return immediately resolved Promise with AdobePromise type completion.
     * @param {T2 = any} val - Resolution value
     */
    static resolve<T2 = any, E2 = any, X2 extends AnyObject = AnyObject>(
        val?: T2 | PromiseLike<T2>,
        source?: X2,
    ): _AdobePromise<T2, E2, X2> & X2 {
        return new AdobePromise<T2, E2, X2>((resolve) => {
            if (isObject(val) && isFunction(val.then)) {
                (val as unknown as Promise<any>).then((res) => {
                    resolve(res);
                });
            } else {
                resolve(val as any);
            }
            // return Promise.resolve(val).then((res) => {
            //     resolve(res);
            // });
        }, source) as _AdobePromise<T2, E2, X2> & X2;
    }

    public static allSettled<T2 = any, E2 = any, X2 extends AnyObject = AnyObject>(
        promises: (AdobePromise<T2, E2> | Promise<T2>)[],
        source?: X2,
    ): _AdobePromise<SettledPromise<T2, E2>[], E2, X2> & X2 {
        if (promises.length === 0) {
            return AdobePromise.resolve([], source) as unknown as _AdobePromise<SettledPromise<T2, E2>[], E2, X2> & X2;
        }

        return new AdobePromise<SettledPromise<T2, E2>[], E2, X2>((resolve) => {
            const collector: SettledPromise<T2, E2>[] = [];
            promises.map((p, i) => {
                (p as AdobePromise<T2, E2>)
                    .then((value) => {
                        collector[i] = {
                            status: 'fulfilled',
                            value,
                        };
                    })
                    .catch((reason) => {
                        collector[i] = {
                            status: 'rejected',
                            reason,
                        };
                    })
                    .then(() => {
                        // check if outer promise should resolve
                        if (collector.filter((ap) => !!ap).length === promises.length) {
                            resolve(collector);
                        }
                    });
            });
        }, source) as _AdobePromise<SettledPromise<T2, E2>[], E2, X2> & X2;
    }

    public get canceled(): boolean {
        return this._canceled;
    }

    /**
     * Get internal promise
     */
    getPromise<T2 = T>(): Promise<T2> {
        return this._promise;
    }

    /** @internal */
    private _resolveOrReject(resolve: (result?: any) => void, reject?: (reason?: any) => void) {
        return this._promise
            .then((res) => {
                if (this._canceled) {
                    return reject && reject(this._cancelReason);
                }

                return resolve(res);
            })
            .catch((err) => {
                return reject && reject(this._cancelReason || err);
            });
    }

    then<T2 = T, E2 = E, X2 = X>(
        onResolve: onResolveFn<T, T2> | null | undefined,
        onReject?: onRejectFn<E2> | null | undefined,
    ): _AdobePromise<T2, E2, X & X2> & (X & X2) {
        if (onResolve == null && onReject == null) {
            return this as unknown as _AdobePromise<T2, E2, X & X2> & (X & X2);
        }

        const p = new AdobePromise<T2, E2, X & X2>((resolve, reject, onCancel) => {
            onCancel &&
                onCancel((reason) => {
                    // attach later promise cancels to cancel entire chain
                    this.cancel.call(this, reason);

                    // attach cancel to reject
                    reject && reject(reason as any);
                });

            return this._resolveOrReject.call(this, resolve, reject);
        }, this._props as X & X2) as unknown as _AdobePromise<T2, E2, X & X2> & (X & X2);

        p._promise = p
            .getPromise()
            .then((val) => {
                const res = onResolve && onResolve(val as unknown as T);

                if (res instanceof AdobePromise) {
                    const resProm = res as _AdobePromise<any, any, Record<string, unknown>>;

                    // if then returns an AdobePromise
                    // attach cancel to cancel it as well
                    this.onCancel((reason) => resProm.cancel(reason));
                    // allow later promises to override initial props
                    // but keep the object the same
                    for (const key in this.props) {
                        if (resProm.props[key] == null) {
                            resProm.props[key] = this.props[key];
                        }
                    }

                    resProm._setProps(resProm.props);
                    this._setProps(resProm.props);
                    p._setProps(resProm.props);
                }
                return res;
            })
            .catch(onReject);

        return p;
    }

    parallel<T2 = T, E2 = E>(onResolve: onResolveFn<T, T2>, onReject?: onRejectFn<E2, any>): Promise<T2> {
        return this._promise.then(onResolve, onReject);
    }

    catch<E2 = E, R = void>(onReject: onRejectFn<E2, R>): _AdobePromise<T, E2, X> & X {
        this._promise = this._promise.catch<R>(onReject);
        return this as unknown as _AdobePromise<T, E2, X> & X;
    }

    finally(fn: () => void): _AdobePromise<T, E, X> & X {
        this._promise = this._promise.finally(fn);
        return this as unknown as _AdobePromise<T, E, X> & X;
    }

    /**
     * Call cancel handlers, mark promise as canceled.
     * @param {unknown} reason
     */
    cancel(reason?: unknown): void {
        if (!this._canceled) {
            this._canceled = true;
            this._cancelReason = reason;

            this._callHandlers('cancel', reason || new Error('Aborted'));
        }

        // this._reject(reason);
    }

    /**
     * Alias to AdobePromise#cancel
     * @param {unknown} reason
     */
    abort(reason?: unknown): void {
        this.cancel(reason);
    }

    /**
     * Register a cancel handler
     * @param handler
     */
    onCancel(handler: (reason?: unknown) => void): void {
        if (isFunction(handler)) {
            this._registerCancelHandler(handler);
        }
    }

    /**
     * Get additional props registered to AdobePromise
     */
    get props(): X {
        return this._props;
    }

    /**
     * Set props object and getters/setters.
     * If props is a class instance, proxy getters and setters.
     * Doesn't overwrite existing properties unless the property
     * was already defined from a previous call to setProps.
     *
     * @internal
     *
     * @param {any} source
     */
    private _setProps<X2 extends AnyObject = X>(source: X2) {
        if ((this as unknown as _AdobePromise<T, E, X2> & X2)._props === source) {
            return;
        }

        (this as unknown as _AdobePromise<T, E, X2> & X2)._props = source;

        const newProps: string[] = [];

        const ownDescriptors = Object.getOwnPropertyDescriptors(AdobePromise);
        const descriptors = Object.getOwnPropertyDescriptors(Object.getPrototypeOf(source));
        if ((descriptors.constructor as any).value !== Object) {
            // class, proxy getters and setters
            for (const key in descriptors) {
                if (key in ownDescriptors || (key in this && !this._registeredProps.includes(key))) {
                    continue;
                }
                newProps.push(key);
                const descriptor = descriptors[key];
                const newDescriptor: PropertyDescriptor = Object.assign({}, descriptor);
                delete newDescriptor.get;
                delete newDescriptor.set;

                if (descriptor.get || descriptor.set) {
                    delete newDescriptor.value;
                    delete newDescriptor.writable;

                    if (descriptor.get) {
                        newDescriptor.get = descriptor.get.bind(source);
                    }
                    if (descriptor.set) {
                        newDescriptor.set = descriptor.set.bind(source);
                    }
                }
                Object.defineProperty(this, key, newDescriptor);
            }
        }

        for (const key of Object.keys(source)) {
            newProps.push(key);

            if (key in ownDescriptors || (key in this && !this._registeredProps.includes(key))) {
                continue;
            }

            Object.defineProperty(this, key, {
                get: () => {
                    return (this._props as Record<string, unknown>)[key];
                },
                set: (v: unknown) => {
                    (this._props as Record<string, unknown>)[key] = v;
                },
                configurable: true,
            });
        }

        this._registeredProps = newProps;

        return this as unknown as _AdobePromise<T, E, X2> & X2;
    }

    /** @internal */
    private _registerCancelHandler(handler: (reason?: unknown) => void) {
        this._handlers.cancel.push(handler);
    }

    /** @internal */
    private _callHandlers(key: 'cancel', args: any) {
        // remove handlers futher down the chain
        this._handlers[key].map((fn) => fn && fn(args));
    }

    /** @internal */
    private _destroy() {
        this._handlers.cancel = [];
    }
}

export const AdobePromise: {
    new <T = any, E = AdobeDCXError, X = unknown>(
        method: (
            resolve: (result?: T) => void,
            reject: (reason?: E) => void,
            onCancel: (handler: onCancelFn) => void,
        ) => unknown,
        source?: X,
    ): IAdobePromise<T, E, X>;
    reject: <E2 = any, X2 extends AnyObject = AnyObject>(
        reason?: E2 | PromiseLike<E2>,
        source?: X2,
    ) => IAdobePromise<any, E2, X2>;
    resolve: <T2 = any, E2 = any, X2 extends AnyObject = AnyObject>(
        val?: T2 | PromiseLike<T2> | undefined | Promise<T2>,
        source?: X2,
    ) => IAdobePromise<T2, E2, X2>;
    allSettled: <T2 = any, E2 = any, X2 extends AnyObject = AnyObject>(
        promises: (AdobePromise<T2, E2> | Promise<T2>)[],
        source?: X2,
    ) => IAdobePromise<SettledPromise<T2, E2>[], E2, X2>;
} = _AdobePromise as any;

export type AdobePromise<T = any, E = any, X = unknown> = IAdobePromise<T, E, X>;
