# @dcx/repo-api-session

The Repository API Session module provides a session for working with the repository API. The [session class](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html) provides a single point of entry to the underlying functionality in the Content Storage Service (CSS). The repository API session class aims to be at feature parity with (and syntactically similar to) cpp/java implementations.   

## Getting Started

The repository session requires an instance of the [AdobeHTTPService](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_http.httpservice.html) class in order to make requests. Refer to [Authentication](/README.md#authentication) for more details on creating an HTTP Service.

```js
const { createHTTPService } = require('@dcx/http');

const service = createHTTPService(function (service) {
    // get a new token asynchronously
    myGetToken(function (token) {
        service.setAuthToken(token);
    });
});
service.setApiKey('MyApiKey');
service.setAdditionalHeaders( {'foo': 'bar'} );
```

### Creating a Session
Session's requires an http service instance along with the endpoint of the desired server.

```js
const { createRepoAPISession } = require('@dcx/repo-api-session');

const session = createRepoAPISession(service, 'https://platform-cs-stage-va7.adobe.io');
```

### Links Cache

The repository session by default keep a cache of all links returned in responses headers as well as a cache of the Index Document Links including the resolveByPath and resolveById links. The cache uses a least-recently used cache eviction strategy and holds references to a limited number of values. Each time a value is accessed, it is moved to the head of a queue. When a value is added to a full cache, the value at the end of that queue is evicted. By default the cache is configured to hold 100,000 objects but that can be configured by creating your own cache and calling `setLinksCache` on the session.

```js
const { RepositoryLinksCache } = require('@dcx/assets');

const MAX_LINKS_CACHED = 500;
const cache = new RepositoryLinksCache(MAX_LINKS_CACHED);

session.setLinksCache(cache);
```

If a client application has cache the links for the Index Document or an Asset on their own and want to save a HEAD call they can update the cache manually.

```js
const myAsset = {...};
//Cache the links for an asset
session.updateCachedAssetLinks(myAsset);

//Set the Index Document Links (LinkSet)
session.updateCachedIndexLinks(indexLinks);
```

The cache can be reset by calling

```js
session.clearLinksCache();
```

### Block Download Threshold

The instance property `AdobeRepoAPISession#blockDownloadThreshold` allows clients control over the threshold at which block download is used for certain methods. For example, `AdobeRepoAPISession#getCompositeComponentUrlForDownload()` will return a component link if the `componentSize` argument is below this threshold, otherwise it will make a request to the Block Download link and return a presigned direct-download URL.

## Links

By default all methods on the repository session will check if the link required to perform the operation exist on the asset first before making the request. If the required link is not found the session with perform a head operation on the asset in order to get the links. The asset itself is then hydrated with it's links along with the cache.

### Get an assets links

[getLinksForAsset(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getlinksforasset) retrieves an assets links.

```js
const asset = {
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg',
    assetId: 'urn:aaid:sc:US:c85a408d-4291-4c96-9038-7e9d36b7c87c'
};
const linkSet = await session.getLinksForAsset(asset);
```

### Get the href for a link relation type

[getLinkHrefForAsset(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getlinkhrefforasset) retrieves the link href for a particular relType

```js
const { LinkRelation } = require("@dcx/assets");
const asset = {
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg',
    assetId: 'urn:aaid:sc:US:6fdc6b8b-43fe-40e5-a307-86fad4440050'
};
const href = await session.getLinkHrefForAsset(asset, LinkRelation.ACL_POLICY);
```

## Head Operations

### Generic HEAD operation

[headHTTPResource(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#headhttpresource) can be used to perform a generic HTTP HEAD operation on an href.

```js
const { LinkUtils } = require("@dcx/util");
const { headers: sessionResponseHeaders } = await session.headHTTPResource(
    'https://platform-cs-stage-va6.adobe.io/content/storage/id/urn:aaid:sc:US:280bd296-4bb8-47e3-af1e-6f0e6ca5450b'
);
const sessionHeaders = LinkUtils.parse(sessionResponseHeaders.link);
```

### Head a primary resourse

[headPrimaryResource(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#headprimaryresource) can be used to issues an HTTP HEAD request on an asset's primary resource.

```js
const { LinkUtils } = require("@dcx/util");
const asset = {
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg',
    assetId: 'urn:aaid:sc:US:c85a408d-4291-4c96-9038-7e9d36b7c87c'
};
const { headers:responseHeaders } = await session.headPrimaryResource(asset);
const headers = LinkUtils.parse(responseHeaders.link);
```

## Index Repository

The Index Repository is the first Content Platform service that users contact. From this Repository, users can discover and manage all other Repositories to which they have access. As the starting point for accessing the Platform, it is available at a well-known URL (below). Each user, however, will see a filtered view of the Index Repository based on the access rights and privileges associated with the account on which the user is operating.

### Get Index Repository Links

[getIndexLinks(...)](https://dcx-migrate.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getindexlinks) performs a HEAD operation against the Index Repository and returns the resolve and operations links.

```js
const sessionIndexLinks = await session.getIndexLinks();
```

### Get Index Repository

[getIndexRepository()](https://dcx-migrate.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getindexrepository) can be used to fetch the Index Repository Document.

## Index Document

The Index Document is one of the three Assets contained in the Index Repository. In it, you will find the following:

* Discoverable Regions
* Assigned Directories
* Resolve and Operations Links

#### Get Index Document

[getIndexDocument()](https://dcx-migrate.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getindexdocument) can be used to fetch the Index Document.

```js
const res = await session.getIndexDocument();
```

## Working with Composites

### Head composite manifest

[headCompositeManifest(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#headcompositemanifest) will perform a HEAD operation on a composite asset.

```js
const { LinkUtils } = require("@dcx/util");
const asset = {
    repositoryId: "09EC14AC5E84F4630A494030@AdobeOrg",
    assetId: "urn:aaid:sc:US:e595e736-2a07-42a5-9321-0c0e4f4d274b",
};

const { headers } = await session.headCompositeManifest(asset);
const headers = LinkUtils.parse(responseHeaders.link);
```

### Get Composite Manifest

[getCompositeManifest(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getcompositemanifest) retrieves the manifest resource for a specified composite.

```js
const compositeAsset = {
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg',
    assetId: 'urn:aaid:sc:US:280bd296-4bb8-47e3-af1e-6f0e6ca5450b'
};

const { 
    manifestData,
    manifestEtag,
    response
}  = await session.getCompositeManifest<any>(compositeAsset);
```

### Get composite component
[getCompositeComponent(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getcompositecomponent) returns the specified version of a composite component.

```js
const compositeAsset = {
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg',
    assetId: 'urn:aaid:sc:US:280bd296-4bb8-47e3-af1e-6f0e6ca5450b'
};


const componentId = '43a912b8-79e0-4976-9c0e-689dd026e465';
const componentVersion = '3';
const responseType = 'json';
const result = await session.getCompositeComponent(
    compositeAsset,
    componentId,
    componentVersion,
    responseType
);
```

### Get URL for downloading components
The URL that is suitable for downloading a component depends on the size of the component. A component above a server limit will respond with a 400 and "Response too large" problem type, if using the component link directly. In this case, clients must use a presigned direct download URL in conjunction with optional [range requests](https://developer.mozilla.org/en-US/docs/Web/HTTP/Range_requests).

> Note: The presigned URL will not be an Adobe domain, and clients must not send authentication headers to these endpoints. The AdobeHTTPService handles stripping of these headers through the `authenticationAllowList` property.

[getCompositeComponentUrlForDownload(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getcompositecomponenturlfordownload) handles fetching the URL that is suitable for the specific component and size, according to the `componentSize` argument.

```ts
const { isPresignedUrl, url, response } = await session.getCompositeComponentUrlForDownload(
    compositeAsset,
    componentId,
    componentSize
);

// if isPresignedUrl === true, response === undefined
// if isPresignedUrl, client must not send auth headers to url
```


### Put composite component
[putCompositeComponent(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#putcompositecomponent) create or update a composite component.

Clients have two options for uploading components. 

#### Component Direct upload
Components can be directly uploaded by performing a PUT operation on the component resource on the composite asset. This method should only be used if the component size is less then `maxSingleTransferSize` which can be retrievd from the '/rel/block/init' link.

```js
import { getLinkProperty } from '@dcx/util';
const maxSingleTransferSize = getLinkProperty(
    asset.links,
    LinkRelation.BLOCK_UPLOAD_INIT,
    BlockTransferProperties.MAX_SINGLE_TRANSFER_SIZE
);
```    

To initiate a direct upload to the component resource a string or buffer containing the component data should be passed to the putCompositeComponent method. While the size parameter is optional it is recommended for clients to provide this value if it is known.

If clients pass a buffer or string that is larger then `maxSingleTransferSize` then dcx-js will automatically handle slicing the buffer and performing a block upload of the component.

```js
const { generateUuid } = require('@dcx/util');
const asset = {
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg',
    assetId: 'urn:aaid:sc:US:280bd296-4bb8-47e3-af1e-6f0e6ca5450b'
};

const componentId = generateUuid();
const contentType = 'application/json';
const isNew = true;
const size = 63340016;
const componentMD5 = 'Dq6vQLzdarl88fvv5vgKcA==';

const putComponentResponse = await session.putCompositeComponent(
    asset,
    componentId,
    JSON.stringify({ foo: 'bar', ... }),
    contentType,
    isNew,
    size,
    componentMD5
);
```

#### Component Block Upload 
If the size of the component is larger then `maxSingleTransferSize` then clients should use block upload instead of direct upload to transfer the data.

To initiate a block upload of a component clients should provide a GetSliceCallback instead of a string or buffer to the putCompositeComponent method. The GetSliceCallback function should accept a start byte where the buffer slice should start and an end byte where the buffer slice should end (not inclusive). The callback should return a Promise containing the buffer slice or a buffer of length 0 indicating the end of the buffer. 

When performing a block upload a size estimate is required in order to determine the number of requests required for the block update. If the size estimate is less then the actual size of the file dcx-js will handle extending the block transfer document in order to transfer all the bytes of the component. In order to reduce network calls clients should try and provide the correct size of the component being uploaded.

```js
const { generateUuid } = require('@dcx/util');
const asset = {
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg',
    assetId: 'urn:aaid:sc:US:280bd296-4bb8-47e3-af1e-6f0e6ca5450b'
};
const buffer = ...;
const exampleGetSliceCallback = (startBuf, endBuf): Promise<any> => {
    const slice = buffer.slice(startBuf, endBuf);
    return new Promise((resolve) => {
        resolve(slice);
    });
};

const componentId = generateUuid();
const contentType = 'video/quicktime';
const isNew = true;
const size = 63340016;
const componentMD5 = 'Ad6vHEzfjel88ddv5vgKcA==';

const putComponentResponse = await session.putCompositeComponent(
    asset,
    componentId,
    exampleGetSliceCallback,
    contentType,
    isNew,
    size,
    componentMD5
);
```

### Get rendition of a composite component

[getRendition(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getrendition) returns a rendition of a composite component.

```js
const compositeAsset = {
    repositoryId: "09EC14AC5E84F4630A494030@AdobeOrg",
    assetId: "urn:aaid:sc:US:280bd296-4bb8-47e3-af1e-6f0e6ca5450b",
};

const renditionOpts = {
    size: 10,
    revision: 0,
    component_id: "044b795b-2c20-4e0b-8735-24e69bafda74"
}

const rendition = await session.getRendition(
    compositeAsset,
    renditionOpts,
    "text"
);
```

## Access Control

### Get effective privileges

[getEffectivePrivileges(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#geteffectiveprivileges) retrieves the effective privileges for the resources of an asset

```js
const asset = {
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg',
    assetId: 'urn:aaid:sc:US:6fdc6b8b-43fe-40e5-a307-86fad4440050'
};
const { result = await session.getEffectivePrivileges(asset);
```

### getACLPolicy

[getACLPolicy(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-badges/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getaclpolicy) retrieves the ACL resource for the asset.


```js
const asset = {
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg',
    assetId: 'urn:aaid:sc:US:1381ca75-b83c-47cc-b5c6-442aee9b9374'
};

const { result } = await session.getACLPolicy(asset);
```

### ACL Privilege Check

[checkACLPrivilege(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#checkaclprivilege) checks whether the current user has the requested Privilege on the specified Resource of an Asset.

The representation of the ACL Check Resource is not defined by the Repository API Specification. Therefore, users should ignore any representation returned in the response body and just inspect the status code.

Below are the possible status codes and what they signify in relation to the checked Privilege.

* 404 Not Found: Neither ACK nor READ is granted on the target Resource, and the Access Control on the target Resource cannot be checked.
* 204 No Content: The checked Privilege is granted.
* 403 Forbidden: The checked Privilege is not granted.

```js
const unHydratedAsset = {
    repositoryId: "09EC14AC5E84F4630A494030@AdobeOrg",
    assetId: "urn:aaid:sc:US:1381ca75-b83c-47cc-b5c6-442aee9b9374",
};

const { response } = await session.checkACLPrivilege(
    unHydratedAsset,
    "write",
    LinkRelation.REPOSITORY
);
```

### Patch ACL Policy

[patchACLPolicy(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-badges/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#patchaclpolicy) issues a PATCH operation against an assets ACL Policy resource.

```js
const unHydratedAsset = {
    repositoryId: "09EC14AC5E84F4630A494030@AdobeOrg",
    assetId: "urn:aaid:sc:US:1381ca75-b83c-47cc-b5c6-442aee9b9374",
};

const patchPolicy = [{
    op: 'add',
    path: '/repo:acl/-',
    value: {
        'repo:principal': {
            'xdm:provider': {
                '@id': 'https://ims-na1.adobelogin.com//'
            },
            '@id': '67c566904aa96e029920faae@AdobeID',
            '@type': 'https://ns.adobe.com/adobecloudplatform/ims/user'
        },
        'repo:modifier': 'grant',
        'repo:privileges': ['ack', 'read', 'write', 'delete'],
        'repo:relations': ['http://ns.adobe.com/adobecloud/rel/primary'],
        'repo:inheritance': 'deep'
    }
}]

const policy = await session.patchACLPolicy(unHydratedAsset, patchPolicy);
```

### Delete ACL Policy

[deleteACLPolicy(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-badges/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#deleteaclpolicy) issues a DELETE operation against an assets ACL Policy removing all ACEs.

```js
const unHydratedAsset = {
    repositoryId: "09EC14AC5E84F4630A494030@AdobeOrg",
    assetId: "urn:aaid:sc:US:1381ca75-b83c-47cc-b5c6-442aee9b9374",
};

const policy = await session.deleteACLPolicy(unHydratedAsset);
```

### Get Application Metadata

[getAppMetadata(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getappmetadata) returns the application metadata that lives on the asset.

```js
const asset = {
    repositoryId: "09EC14AC5E84F4630A494030@AdobeOrg",
    assetId: "urn:aaid:sc:US:c85a408d-4291-4c96-9038-7e9d36b7c87c"
};

const { result } = await session.getAppMetadata(asset);
```

## Directories

### Get a directory listing

[getDirectory(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getdirectory) retrieves the repo metadata of the directory. The response includes a paged object for iterating the children in the directory.

This API returns a PagedResource. See [Working with PageResources](/packages/assets) for more info.

```js
const asset = {
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg',
    assetId: 'urn:aaid:sc:US:3326dc94-aaa4-5597-a299-d5787b3ac532'
};
const pageOpts = { limit: 3 };
const response = await session.getDirectory(asset, pageOpts);

const nextPage = response.paged.getNextPage();
```

### Get directory by URL

[getDirectoryByURL(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getdirectorybyurl) retrieves the repo metadata of the directory identified by absolute URL.

The URL may identify the root directory of a repository, in which case the directory resource will contain the links for the repository.

```js
const { result } = await session.getDirectoryByURL(
    `https://server.url/content/storage/id/urn:aaid:sc:US:3326dc94-aaa4-5597-a299-d5787b3ac532`
);
```

### Discoverable Assets

The [Discoverable Assets Document](https://developers.corp.adobe.com/index-repository-api/docs/discoverable-assets-document.md) lists the set of discoverable assets for the requesting user.

This API returns a PagedResource. See [Working with PageResources](/packages/assets) for more info.

#### Get discoverable assets

[getDiscoverableAssets(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getdiscoverableassets) retrieves a paged discoverable assets response.

```js
const { paged } = await session.getDiscoverableAssets();
```

#### Get discoverable assets with page options
```js
//PAGED
const pageOptsNew = { limit: 1 };
const { paged } = await session.getDiscoverableAssets(
    pageOptsNew
);
```

#### Get next page
```js
const { paged } = await limitedPage.getNextPage();
```

### Discoverable Repositories

The [Discoverable Repositories Document](https://developers.corp.adobe.com/index-repository-api/docs/discoverable-repositories-document.md) is a Repository that contains, for the requesting user, at least one Discoverable Asset. The Discoverable Repositories Document enumerates the list of Discoverable Repositories and embeds the Repository Resource for each Repository.

[getDiscoverableRepos(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getdiscoverablerepos) retrieves a paged discoverable repositories response.

This API returns a PagedResource. See [Working with PageResources](/packages/assets) for more info.

#### Get discoverable repositories
```js
const { paged } = await session.getDiscoverableRepos();
```

#### Get discoverable repositories with page options
```js
//PAGED
const pageOptsNew = { limit: 1 };
const { paged } = await session.getDiscoverableRepos(
    pageOptsNew
);
```

#### Get next page
```js
const { paged } = await limitedPage.getNextPage();
```

## Assets

### Get Repository Metadata for asset

[getRepoMetadata(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getrepometadata) retrieves the repository metadata resource for a specified asset.

```js
const asset = {
    repositoryId: "09EC14AC5E84F4630A494030@AdobeOrg",
    assetId: "urn:aaid:sc:US:86bbcbb3-2f77-4cfd-9632-6de0fd3481f4",
};

const { result } = await session.getRepoMetadata(asset);
```

### Get the Primary Resource

[getPrimaryResource(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getprimaryresource) download an asset's primary resource.

```js
const directoryAsset = {
    repositoryId: "09EC14AC5E84F4630A494030@AdobeOrg",
    assetId: "urn:aaid:sc:US:3326dc94-aaa4-5597-a299-d5787b3ac532",
};
const { response } = await session.getPrimaryResource(
    directoryAsset,
    "json"
);
```

### Get a Rendition

[getRendition(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getrendition) returns a rendition of given asset matching given options.

```js
const unHydratedAsset = {
    repositoryId: "09EC14AC5E84F4630A494030@AdobeOrg",
    assetId: "urn:aaid:sc:US:86bbcbb3-2f77-4cfd-9632-6de0fd3481f4",
};

const renditionOpts = {
    size: 1,
    revision: 0,
    type: "image/jpg",
}

const { result } = await session.getRendition(
    unHydratedAsset,
    renditionOpts,
    "arraybuffer"
);
```

### Get Embedded Metadata

[getEmbeddedMetadata(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#getembeddedmetadata) retrieves the embedded metadata of a jpg asset.

```js
const jpgAsset = {
    repositoryId: "09EC14AC5E84F4630A494030@AdobeOrg",
    assetId: "urn:aaid:sc:US:dc5f83c0-d5e3-4dd5-9b93-265878473100",
};

const { result } = await session.getEmbeddedMetadata(jpgAsset);
```

### Get an asset version

```js
const asset: AdobeAsset = {
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg',
    assetId: 'urn:aaid:sc:US:280bd296-4bb8-47e3-af1e-6f0e6ca5450b'
};
const response = await session.getVersionResource(asset, '1');
```

### Patch an asset version

A version can be patched for purposes of marking it as a milestone version.

```js
const asset = {
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg',
    assetId: 'urn:aaid:sc:US:86bbcbb3-2f77-4cfd-9632-6de0fd3481f4'
};

const addPatchDoc = [
    {
        op: 'add',
        path: '/children/version=0/milestone',
        value: {
            label: 'My First Milestone',
            description: 'Label description'
        }
    }
];

const result = await session.patchVersions(asset, JSON.stringify(addPatchDoc), ':1611932908062');
``` 

## Bulk Operations

### Bulk requests

[performBulkRequest(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#performbulkrequest) executes a bulk request that allows clients to combine multiple read or write operations into a single HTTP request. There are [restrictions](https://developers.corp.adobe.com/storage-api/docs/reference/bulk-requests.md#restrictions) around what can and cannot be included in a bulk request as well as what order the requests should be placed in. dcx-js will guard against too many sub-requests being included in the bulk request or mixing operations types (READ and WRITE). The rest of the restrictions is up to clients to follow.

#### Example Bulk READ
```js
const demoAsset = {
    repositoryId: "09EC14AC5E84F4630A494030@AdobeOrg",
    assetId: "urn:aaid:sc:US:280bd296-4bb8-47e3-af1e-6f0e6ca5450b",
};

const req1 = {
    method: "GET",
    href: getLinkHref(
    demoAsset[Properties.LINKS],
    LinkRelation.EFFECTIVE_PRIVILAGES
    ),
    headers: { accept: "application/json" },
};

const req2 = {
    method: "GET",
    href: getLinkHref(
    demoAsset[Properties.LINKS],
    LinkRelation.ACL_POLICY
    ),
};

const req3 = {
    method: "GET",
    href: getLinkHref(
    demoAsset[Properties.LINKS],
    LinkRelation.EMBEDDED_METADATA,
    "id"
    ),
    headers: { accept: "application/json" },
};

const { result } = await session.performBulkRequest(demoAsset, [req1, req2, req3]);
```

#### Example Bulk WRITE

```js
const demoAsset = {
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg',
    assetId: 'urn:aaid:sc:US:ce704760-f31b-4d9c-96b8-70e19398dc8c'
};

const aclPatch = JSON.stringify({
    'repo:acl': [
        {
            'repo:principal': {
                'xdm:provider': {
                    '@id': 'https://ims-na1.adobelogin.com/'
                },
                '@id': '67c566904aa96e029920faae@AdobeID',
                '@type': 'https://ns.adobe.com/adobecloudplatform/ims/user'
            },
            'repo:modifier': 'grant',
            'repo:privileges': ['ack', 'read', 'write', 'delete'],
            'repo:relations': ['http://ns.adobe.com/adobecloud/rel/primary'],
            'repo:inheritance': 'deep'
        }
    ]
});

const req1:BulkRequestDescriptor = {
    method: 'PATCH',
    href: getLinkHref(composite.links, LinkRelation.ACL_POLICY),
    body: aclPatch,
    headers: {
        'content-type': 'application/vnd.adobecloud.accesscontrolpolicy+json',
        'content-length': getDataLength(aclPatch)
    }
};

const xmpPatch = JSON.stringify([
    {
        op: 'add',
        path: '/xmp:CreatorTool',
        value: '13.76'
    }
]);

const req2:BulkRequestDescriptor = {
    method: 'PATCH',
    href: getLinkHref(composite.links, LinkRelation.EMBEDDED_METADATA),
    body: xmpPatch,
    headers: {
        'content-type': 'application/json-patch+json',
        'content-length': getDataLength(xmpPatch),
        'if-match': '*'
    }
};

const res = await session.performBulkRequest(demoAsset, [req1, req2]);
```

## Operations

### [Move](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#moveasset)

```ts
const sourceAsset = { 
    repositoryId: 'repoId', 
    assetId: 'urn:aaid:sc:US:12345678-abcd-1234-abcd-123456789abc'
}

const { result } = await session.moveAsset(
    sourceAsset, 
    { path: 'some/destination', repositoryId: 'repoId' }, createIntermediates, 
    overwriteExisting
);
// result is AdobeAsset data object
```

### [Copy](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#copyasset)

```ts
const { result } = await session.copyAsset(
    sourceAsset, 
    { path: 'some/destination', repositoryId: 'repoId' }, createIntermediates, 
    overwriteExisting
);
// result is AdobeAsset data object
```

### [Package](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#packageassets)

```ts
const { result } = await session.package(
    [sourceAsset], 
    { path: '/some/package.zip', repositoryId: 'repoId' }, createIntermediates, 
    overwriteExisting
);
// on success, result.success === true
```

### [Discard](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#discardasset)

```ts
const { result } = await session.discardAsset(sourceAsset, recursive);
// on success, result.success === true
```

### [Restore](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#restoreasset)

```ts
const { result } = await session.restoreAsset(sourceAsset);
// on success, result.success === true
```

### [Delete](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html#deleteasset)

```ts
const { result } = await asset.deleteAsset(sourceAsset, recursive);
// on success, result.success === true
```

See the [docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html) for a list of all methods on the AdobeRepoAPISession class.