/// <reference types="node" />

import { ACPAccessControlList } from '@dcx/common-types';
import { ACPRepository } from '@dcx/common-types';
import { AdobeAsset } from '@dcx/assets';
import { AdobeAssetEmbedded } from '@dcx/common-types';
import { AdobeBlockUpload } from '@dcx/common-types';
import { AdobeDCXBranch } from '@dcx/common-types';
import { AdobeDCXComponent } from '@dcx/common-types';
import { AdobeDCXComposite } from '@dcx/common-types';
import { AdobeDCXError } from '@dcx/common-types';
import { AdobeDCXError as AdobeDCXError_2 } from '@dcx/error';
import { AdobeDirectory } from '@dcx/assets';
import { AdobeGetDirectoryDataResult } from '@dcx/assets';
import { AdobeGetPageResult } from '@dcx/assets';
import { AdobeHTTPService } from '@dcx/common-types';
import { AdobeManifestResult } from '@dcx/assets';
import { AdobeOperationResult } from '@dcx/assets';
import AdobePromise from '@dcx/promise';
import { AdobeRepoMetadata } from '@dcx/common-types';
import { AdobeRepository } from '@dcx/common-types';
import { AdobeRepoUploadResult } from '@dcx/common-types';
import { AdobeResponse } from '@dcx/common-types';
import { AdobeResponseType } from '@dcx/common-types';
import { AdobeSessionBase } from '@dcx/common-types';
import { AdobeUploadRecord } from '@dcx/common-types';
import { AdobeUploadResults } from '@dcx/common-types';
import { AdobeVersion } from '@dcx/assets';
import { AdobeVersionPatchOperation } from '@dcx/assets';
import { AdobeVersionSet } from '@dcx/assets';
import { AnyOperationDocument } from '@dcx/assets';
import { AssetWithRepoAndPathOrId } from '@dcx/common-types';
import { BulkRequestDescriptor } from '@dcx/assets';
import { ComponentRequestByPath } from '@dcx/assets';
import { ComponentUploadInfo } from '@dcx/common-types';
import { CopyResourceDesignator } from '@dcx/common-types';
import { CopyResourcesOperationResult } from '@dcx/assets';
import { EffectivePrivileges } from '@dcx/common-types';
import { EmbeddableResource } from '@dcx/common-types';
import { EmbeddedMetadataRepresentation } from '@dcx/assets';
import { GetSliceCallback } from '@dcx/common-types';
import { IndexDocument } from '@dcx/common-types';
import { IndexRepository } from '@dcx/assets';
import { JSONPatchDocument } from '@dcx/common-types';
import { Link } from '@dcx/common-types';
import { LinkMode } from '@dcx/common-types';
import { LinkProvider } from '@dcx/util';
import { LinkRelationKey } from '@dcx/common-types';
import { LinkSet } from '@dcx/common-types';
import { ManifestAndComponentsByPathResponse } from '@dcx/assets';
import { PageOptions } from '@dcx/assets';
import { PageResource } from '@dcx/assets';
import { PathOrIdAssetDesignator } from '@dcx/assets';
import { Privilege } from '@dcx/common-types';
import { RenditionOptions } from '@dcx/common-types';
import { RepoDownloadStreamableReturn } from '@dcx/common-types';
import { RepoMetaPatch } from '@dcx/common-types';
import { RepoResponse } from '@dcx/assets';
import { RepoResponseResult } from '@dcx/assets';
import { RepoResponseSuccessResult } from '@dcx/assets';
import { RepositoryLinksCache } from '@dcx/assets';
import { RequestDescriptor } from '@dcx/common-types';
import { RequireAtLeastOne } from '@dcx/common-types';
import { ResolvableResourceRelations } from '@dcx/common-types';
import { ResourceDesignator } from '@dcx/common-types';
import { ServiceConfig } from '@dcx/assets';
import { SliceableData } from '@dcx/common-types';
import { UploadProgressCallback } from '@dcx/common-types';
import { UploadRecipe } from '@dcx/common-types';

export declare interface AdobeRepoAPISession {
    singleTransferMaxDownloadSize: number;
    /**
     * Creates a new asset at the specified cloud location.
     * @param parentDir             Asset object that identifies the asset's repository
     *                                  and base directory, to which the path is relative
     * @param relPath               The relative path of the asset to be created wrt
     *                                  directory represented by assetWithPath.
     * @param createIntermediates   If true create all the intermediates directories
     *                                  else expect all the directories in relPath to be present.
     * @param contentType           The content type of the primary resource
     * @param resourceDesignator    Indicates a resource to include in the create response.
     * @param additionalHeaders     Headers to add to request
     */
    createAsset(parentDir: AdobeAsset, relPath: string, createIntermediates: boolean, contentType: string, resourceDesignator?: ResourceDesignator, additionalHeaders?: Record<string, string>, primaryResource?: SliceableData | GetSliceCallback, primaryResourceSize?: number, repoMetaPatch?: RepoMetaPatch): AdobePromise<RepoResponseResult<AdobeAsset>, AdobeDCXError, RequestDescriptor>;
    /**
     * Create a composite.
     *
     * @param parentDir         Directory to create the composite in.
     *                              Must contain links, [id, repositoryId] or [path, repositoryId]
     * @param relPath           The path on the server at which to create the composite relative to parentDir.
     * @param contentType       The content type of the new composite. This must end with "+dcx".
     * @param respondWith       A resource designator to request the server to send in the response.
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     */
    createComposite(parentDir: AdobeAsset, relPath: string, contentType: string, respondWith?: ResourceDesignator, additionalHeaders?: Record<string, string>, snapshot?: SliceableData | GetSliceCallback, snapshotSize?: number, repoMetaPatch?: RepoMetaPatch): AdobePromise<RepoResponseResult<AdobeAsset>, AdobeDCXError, RequestDescriptor>;
    /**
     * Copies resources from source to target asset using the COPY_RESOURCES operation.
     * See {@link https://git.corp.adobe.com/pages/caf/api-spec/chapters/operations/copying_resources.html}
     *
     *
     * @param {PathOrIdAssetDesignator} sourceAsset                     The source asset
     * @param {PathOrIdAssetDesignator} targetAsset                     The destination asset
     * @param {CopyResourceDesignator[]} resources                      An array of resource designators to be copied from the source to the target asset.
     * @param {Record<string, string>} additionalHeaders                Additional headers to apply to HTTP Requests
     *
     * @returns {AdobePromise<RepoResponseResult<AdobeAsset, 'json'>, AdobeDCXError>}
     */
    copyResources(sourceAsset: PathOrIdAssetDesignator, targetAsset: PathOrIdAssetDesignator, resources: CopyResourceDesignator[], intermediates?: boolean, manifestPatch?: JSONPatchDocument, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<CopyResourcesOperationResult, 'json'>, AdobeDCXError>;
    /**
     * Issues HTTP HEAD request for an asset's primary resource
     * @param asset                 Asset object that identifies the asset
     * @param additionalHeaders           Any additional heads to include in the call.
     */
    headPrimaryResource(asset: AdobeAsset, additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'void'>>;
    /**
     * Generic HTTP HEAD operation that may be used on secondary resources
     * and link hrefs that might not yet have dedicated methods on this object
     * @param baseUrl               The baseUrl of the resource to HEAD
     * @param href                  The href of the resource to HEAD
     * @param additionalHeaders           Any additional heads to include in the call.
     */
    headHTTPResource(href: string, additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse, AdobeDCXError, RequestDescriptor>;
    /**
     * Retrieves the index links (resolve by id/path and ops)
     * @param pageOpts              Optional paging and embedding parameters
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    getIndexLinks(pageOpts?: Omit<PageOptions, 'embed'>, additionalHeaders?: Record<string, string>): AdobePromise<LinkSet>;
    /**
     * Retrieves the Index Repository.
     *
     * The Index Repository is the first Content Platform service that users contact.
     * From this Repository, users can discover and manage all other Repositories to which they have access.
     * As the starting point for accessing the Platform, it is available at a well-known URL.
     * Each user, however, will see a filtered view of the Index Repository based on the access rights and privileges
     * associated with the account on which the user is operating.
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    getIndexRepository(additionalHeaders?: Record<string, string>): AdobePromise<IndexRepository>;
    /**
     * Retrieves the index document containing the users Discoverable Regions,
     * Assigned Directories & Resolve and Operation Links.
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    getIndexDocument(additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<IndexDocument>>;
    /**
     * Retrieves a Discoverable Assets from the user's index directory.
     *
     * See {@link https://adobe.ly/34YVSJC | Optimizations to Index Format}
     *
     * @param pageOpts              Optional paging and embedding parameters
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     * @note
     * pageOpts only accepts comma seperated reltypes.
     */
    getDiscoverableAssets<T extends EmbeddableResource = never>(pageOpts?: PageOptions<T>, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponse<AdobeGetPageResult<AdobeAsset>>>;
    /**
     * Retrieves a Discoverable Repositories
     * (https://adobe.ly/34WDvVI)
     * from the user's index directory
     * @param pageOpts              Optional paging and embedding parameters
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    getDiscoverableRepos(pageOpts?: Omit<PageOptions, 'embed'>, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponse<AdobeGetPageResult<AdobeRepository>>>;
    /**
     * Retrieves the repo metadata of the directory.
     * @param asset                 Asset object that identifies the asset
     * @param pageOpts              Paging options for request
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     * @note
     * see {@link https://developers.corp.adobe.com/storage-api/docs/reference/parent-directory.md}
     * dcx-js does not format pageOpts, so the caller of getDirectory needs to format pageOpts to resource
     * designator style according to the examples given in the link above. An array of mix of reltypes and resource
     * designator format will be rejected by server with 400.
     */
    getDirectory<T extends EmbeddableResource = never>(dirAsset: AdobeAsset, pageOpts?: PageOptions<T>, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponse<AdobeGetPageResult<AdobeAssetEmbedded<T>>>, AdobeDCXError>;
    /**
     * Retrieves the repo metadata of the directory.
     * @param asset                 Asset object that identifies the asset
     * @param format                Response format
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    getEmbeddedMetadata<T = any>(asset: AdobeAsset, format?: string, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<T>>;
    /**
     * Update entire embedded/XMP metadata resource.
     *
     * @note
     * Replaces existing XMP resource.
     *
     * @note
     * Currently the service requires an etag, but spec states unguarded updates should be possible.
     * see {@link https://jira.corp.adobe.com/browse/SYSE-7940}
     * and {@link https://jira.corp.adobe.com/browse/SYSE-5943}
     *
     * @param svc               HTTPService to use
     * @param asset             Asset whose embedded metadata should be updated
     * @param data              New embedded metadata
     * @param etag              ETag of metadata resource to update.
     * @param format            Defines the representation of the body, either XML or JSON.
     *                              If using XML, clients must pass the data as a string.
     *                              If using XML with TypeScript, clients must specify the generic as `string`.
     *                              Defaults to json.
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     */
    putEmbeddedMetadata<T = any>(asset: AdobeAsset, data: T, etag?: string /** TODO: Revisit optional etag */, format?: EmbeddedMetadataRepresentation, additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'void'>>;
    /**
     * Update embedded/XMP metadata using JSON Patch document.
     *
     * @see
     * https://tools.ietf.org/html/rfc6902#page-6
     *
     * @note
     * Currently the service requires an etag, but spec states unguarded updates should be possible.
     * see {@link https://jira.corp.adobe.com/browse/SYSE-7940}
     * and {@link https://jira.corp.adobe.com/browse/SYSE-5943}
     *
     * @param {AdobeHTTPService}                    svc       HTTPService to use
     * @param {AdobeAsset}                          asset     Asset to update, must be composite
     * @param {PatchDocumentOperation[] | string}   data      Data to use as PATCH body
     * @param {string}                              [etag]    ETag of the embedded metadata resource to be updated
     * @param {Record<string, string>}              additionalHeaders     Additional headers to attach to HTTP Requests
     */
    patchEmbeddedMetadata(asset: AdobeAsset, data: JSONPatchDocument | string, etag?: string /** TODO: Revisit optional etag */, additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'void'>>;
    /**
     * Perform block download of specified byte range on an asset.
     * To get the last N bytes of the file, use `endByte` set to a negative number.
     * If neither startByte or endByte are specified, will perform direct download of the entire asset.
     *
     * @example
     * ```js
     * // fetch entire asset
     * const res = await session.blockDownloadAsset(asset, undefined, undefined);
     *
     * // fetch last 10 bytes
     * const res = await session.blockDownloadAsset(asset, undefined, -10);
     *
     * // fetch first 10 bytes
     * const res = await session.blockDownloadAsset(asset, 0, 10);
     * ```
     *
     * @param assetOrPresignedUrl   Asset to fetch or presigned URL to use.
     * @param startByte             First byte to fetch
     * @param endByte               Last byte to fetch
     * @param resource              Resource to fetch
     * @param componentId           Specific component ID to fetch
     * @param version               Version of resource to fetch
     * @param responseType          Type to transform response body into,
     *                                  defaults to Buffer (Node) or ArrayBuffer (browser)
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    blockDownloadAsset<T extends AdobeResponseType = 'defaultbuffer'>(assetOrPresignedUrl: AdobeAsset | string, startByte?: number, endByte?: number, resource?: LinkRelationKey, componentId?: string, version?: string, responseType?: T, additionalHeaders?: Record<string, string>): RepoDownloadStreamableReturn<T>;
    /**
     * Retrieves the repo metadata of the directory identified by absolute URL.
     *
     * The URL may identify the root directory of a repository,
     * in which case the directory resource will contain the links
     * for the repository.
     *
     * @param url                   The Directory Url
     * @param pageOpts              Paging options for request
     */
    getDirectoryByURL(url: string): AdobePromise<RepoResponse<AdobeGetDirectoryDataResult>, AdobeDCXError>;
    /**
     * Retrieves an assets links
     * @param asset                 Asset object that identifies the asset
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    getLinksForAsset(asset: AdobeAsset, additionalHeaders?: Record<string, string>): AdobePromise<LinkSet>;
    /**
     * Call resolveByID or resolveByPath on an asset.
     * If resource is specified (and valid) the additional resource will be included in the response object.
     *
     * @note
     * This method sets cached links, but does not return cached links.
     * It will always make an API call to resolveBy*.
     *
     * @param asset                 The asset to resolve, must be resolvable
     * @param mode                  Whether to return path- or id-based links
     * @param resource              Additional resource to include in the response
     * @param responseType          Optional parsing of additional resource, defaults to 'json'
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    resolveAsset<T extends AdobeResponseType = 'json'>(asset: AssetWithRepoAndPathOrId, mode?: 'id' | 'path', resource?: ResourceDesignator, responseType?: T, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<AdobeAsset>, AdobeDCXError>;
    /**
     * Fetch links for asset if they don't exist on the asset or in the cache.
     * Promise will reject with error if the fetched links don't contain linksToPopulate.
     *
     * @param asset
     * @param linksToPopulate
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    fetchLinksIfMissing(asset: AdobeAsset, linksToPopulate: LinkRelationKey[], additionalHeaders?: Record<string, string>): AdobePromise<LinkSet, AdobeDCXError, RequestDescriptor>;
    /**
     * Issues GET on a resource link if it exists in either the asset object
     * or in the links cache. If neither exists, use the resolve API to fetch
     * the resource in a single request.
     *
     * @note The list of relations that can be fetched using the resolve API will likely
     * change over time. `ResolvableResourceRelations` will have to be updated with
     * each new valid resource.
     *
     * @param asset             The asset to fetch the resource for.
     * @param resource          resource to fetch
     * @param responseType      Response type to return resource body as
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     */
    useLinkOrResolveResource<T extends AdobeResponseType = AdobeResponseType>(asset: AdobeAsset, resource: ResolvableResourceRelations, responseType?: T, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<AdobeAsset>, AdobeDCXError>;
    /**
     * Retrieves a repositories links
     * @param repoId                The id of the repository
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    getLinksForRepo(repoId: string, additionalHeaders?: Record<string, string>): AdobePromise<LinkSet>;
    /**
     * Retrieves the repository metadata resource for a specified asset
     * @param asset                 Asset object that identifies the asset
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    getRepoMetadata(asset: AdobeAsset, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<AdobeRepoMetadata>, AdobeDCXError>;
    /**
     * @hidden
     * Retrieves the repository metadata for the base directory of the specified asset
     * @param asset                 Asset object that identifies the asset
     */
    getBaseDirectoryMetadata(asset: AdobeAsset): AdobePromise<AdobeRepoMetadata>;
    /**
     * Downloads an asset's primary resource
     * @param asset             Asset object that identifies the asset
     * @param responseType      Type to tranform the body
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     */
    getPrimaryResource<T extends AdobeResponseType = 'defaultbuffer'>(asset: AdobeAsset, responseType?: T, additionalHeaders?: Record<string, string>): RepoDownloadStreamableReturn<T>;
    /**
     * Update the asset's primary resource.
     *
     * @note Implicitly creates a new version of the asset.
     *
     * @param dataOrSliceCallback   Data to use for update, or callback to provide blocks for block upload.
     *                              If a callback is used, block upload will be used, regardless of size of the asset.
     * @param contentType           The content type of the resource to upload.
     * @param size                  The size of the resource in bytes.
     * @param etag                  The etag of the existing resource.
     * @param md5                   The MD5 hash of the resource, used for block upload finalize.
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    updatePrimaryResource(asset: AdobeAsset, dataOrSliceCallback: SliceableData | GetSliceCallback, contentType: string, size?: number, etag?: string, md5?: string, additionalHeaders?: Record<string, string>): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, {
        blockUpload?: AdobeBlockUpload;
    }>;
    /**
     * Retrieves a paged list of versions for an asset
     * @param asset                 Asset object that identifies the asset.
     * @param pageOpts              Optional paging and embedding parameters
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    getVersions(asset: AdobeAsset, pageOpts?: Omit<PageOptions, 'embed'>, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponse<AdobeGetPageResult<AdobeVersion>>>;
    /**
     * Retrieves a specific version of an asset.
     * @param asset                 Asset object that identifies the asset.
     * @param version               Resource version
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    getVersionResource(asset: AdobeAsset, version: string, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<AdobeVersion>>;
    /**
     * Get rendition of given asset matching given options.
     * @param asset                 Asset object that identifies the asset
     * @param renditionOpts         Rendition preferences. Either longest side should be
     *                              given Or both width and height in options.
     * @param responseType          Type to transform response body into.
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    getRendition<T extends AdobeResponseType = 'json'>(asset: AdobeAsset, renditionOpts?: RenditionOptions, responseType?: AdobeResponseType, linkProvider?: LinkProvider<RenditionOptions>, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<T>>;
    /**
     * Get the repository resource for an asset
     * @param asset                 Asset object that identifies the asset
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    getRepositoryResource(asset: AdobeAsset, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<ACPRepository>, AdobeDCXError>;
    /**
     * Retrieves the effective privileges for the various resources of an asset
     * @param asset                 Asset object that identifies the asset
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    getEffectivePrivileges(asset: AdobeAsset, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<EffectivePrivileges>, AdobeDCXError>;
    /**
     * Retrieves the link href for a particular relType
     * @param asset                 Asset object that identifies the asset
     * @param relType               The LinkRelation type
     * @param linkMode              An optional link mode (id or path)
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    getLinkHrefForAsset(asset: AdobeAsset, relType: LinkRelationKey, linkMode?: LinkMode, additionalHeaders?: Record<string, string>): AdobePromise<string>;
    /**
     * Executes a bulk request that allows clients to combine multiple read or write operations into a single HTTP request.
     *
     * Bulk Requests have the limitations below
     * * Bulk requests must pertain to the Resources of a single Asset.
     * * User agents can either read multiple Resources (with HEAD and GET requests) or write multiple Resources
     *   (with POST, PUT, PATCH, and DELETE requests), but cannot mix these.
     * * Bulk requests may be issued using both ID-based and path-based Links, but these may not be mixed in
     *   a single request. The addressing mode used in each sub-request must match that of the bulk request.
     *
     * NOTE: Currently only supports bulk READ Operations
     * @param asset              The asset to work against
     * @param requests           A list of BulkRequestDescriptor's to be included as part of the bulk request
     * @param linkMode           The link mode used by the requests
     * @param additionalHeaders  Additional headers to apply to HTTP Requests
     */
    performBulkRequest(asset: AdobeAsset, requests: BulkRequestDescriptor[], linkMode?: LinkMode, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<AdobeResponse[]>, AdobeDCXError>;
    /**
     * Performs a HEAD operation on a composite asset
     * @param asset                 Asset object that identifies the asset
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    headCompositeManifest(asset: AdobeAsset, additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'void'>>;
    /**
     * Retrieves the URL pointing to a composite's manifest.
     *
     * @param asset
     *                          Asset object that identifies the composite asset
     * @param version           If this is valid, then the request will be made against the specified version
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     */
    getCompositeManifestUrl(asset: AdobeAsset | AdobeDCXComposite, version?: string, additionalHeaders?: Record<string, string>): AdobePromise<string, AdobeDCXError>;
    /**
     * Retrieves the manifest resource for a specified composite.
     *
     * @param asset             Asset object that identifies the composite asset
     * @param version           If this is valid, then the request will be made against the specified version
     * @param etag              If specific then its value will be passed with the If-Match header of the
     *                              manifest request and not bulk request. Note that this
     *                              etag refers to the manifest resource.
     *                              The etag property on the asset argument is ignored.
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     */
    getCompositeManifest<T = any>(asset: AdobeAsset | AdobeDCXComposite, version?: string, etag?: string, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponse<AdobeManifestResult<T>>, AdobeDCXError, RequestDescriptor>;
    /**
     * @param asset AdobeAsset or AdobeDCXComposite from which to pull the manifest and components.
     * @param components An array of ComponentRequestsByPath.
     * @example
     *  ```
     *  const components = [
     *    {
     *      component_path: "/model/database.json",
     *      skipblockDownload: true
     *    },
     *    {
     *      "/model/database2.json",
     *      responseType: "json" // default value "defaultbuffer"
     *    }
     *  ]
     *  ```
     * @param version Version of the composite manifest to fetch (components all come from this version of the manifest)
     * @param etag Etag to use within the if-match header of the request
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     * @example
     * ```
     * {
     *   manifest: {
     *     data?: ManifestData, // JSON parsed manifest data
     *     error?: DCXError, // present if there was an error fetching the manifest
     *     response: AdobeResponse // response for the manifest sub-request (or secondary follow-up manifest request)
     *   },
     *   components: {
     *     "/model/database.json": { // component paths are the keys
     *       component_path: "/model/database.json",
     *       data?: ResponseTypeMap[ResponseType], // parsed into the requested responseType
     *       error?: DCXError, // present if there was an error fetching the manifest
     *       response: AdobeResponse // response for component sub-request (or follow-up request)
     *     },
     *     "/model/database2.json": { // component paths are the keys
     *       component_path: "/model/database3.json",
     *       responseType: "json",
     *       data?: ResponseTypeMap[ResponseType], // parsed into the requested responseType
     *       error?: DCXError, // present if there was an error fetching the manifest
     *       response: AdobeResponse // response for component sub-request (or follow-up request)
     *     }
     *   },
     *   responses: AdobeResponse[] // array of Bulk request responses (1 response per bulk request issued)
     * }
     * ```
     */
    getManifestAndComponentsByPath<C extends ComponentRequestByPath[]>(asset: AdobeAsset | AdobeDCXComposite, components: C, version?: string, etag?: string, additionalHeaders?: Record<string, string>): AdobePromise<ManifestAndComponentsByPathResponse<C>, AdobeDCXError>;
    /**
     * Retrieves the URL pointing to a component's primary resource.
     * When using the URL for uploading a component (PUT), the revision is not required.
     * Otherwise, component revision is a required parameter.
     * @param asset             Asset object that identifies the composite asset.
     * @param componentId       The id of the component to retrieve
     * @param componentRevision          Component revision to fetch
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     */
    getCompositeComponentUrl(asset: AdobeAsset | AdobeDCXComposite, componentId: string, componentRevision?: string, additionalHeaders?: Record<string, string>): AdobePromise<string, AdobeDCXError>;
    /**
     * Retrieves the URL pointing to a component's primary resource.
     * When using an AdobeDCXComponent instance as the first parameter,
     * the 2nd and 3rd parameters are ignored. Values on the
     * AdobeDCXComponent are directly used instead.
     *
     * @param asset             AdobeDCXComponent
     * @param ignored1          Ignored. (string) component id in other overloads
     * @param ignored2           Ignored. (string) component revision in other overloads
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     */
    getCompositeComponentUrl(component: AdobeDCXComponent, ignored1: never, ignored2: never, additionalHeaders?: Record<string, string>): AdobePromise<string, AdobeDCXError>;
    getCompositeComponentUrl(asset: AdobeAsset | AdobeDCXComponent | AdobeDCXComposite, componentId?: string, componentRevision?: string, additionalHeaders?: Record<string, string>): AdobePromise<string, AdobeDCXError>;
    /**
     * Get a URL suitable for downloading a component.
     * This may return either a presigned URL or the resolved component link, depending on size.
     *
     * @note
     * If a presigned URL is returned, `isPresignedUrl` will be true and
     * the resolved object will have a response. Otherwise, response will be null.
     *
     * @example Response for single download URL
     * { response: null, result: { isPresignedUrl: false, href: string }}
     *
     * @example Response for presigned URL
     * { response: AdobeResponse, result: { isPresignedUrl: true, href: string }}
     *
     * @param asset
     * @param componentId
     * @param componentSize
     * @param version
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    getCompositeComponentUrlForDownload(asset: AdobeAsset | AdobeDCXComposite, componentId: string, componentSize: number, componentRevision: string, additionalHeaders?: Record<string, string>): AdobePromise<{
        url: string;
        isPresignedUrl: boolean;
        response: AdobeResponse<'json'> | undefined;
    }, AdobeDCXError>;
    /**
     * Get a URL suitable for downloading a component.
     * This may return either a presigned URL or the resolved component link, depending on size.
     *
     * In this implementation overload, the 2nd, 3rd, and 4th parameters
     * (component id, component size, component revision) are read directly
     * from the AdobeDCXComponent instance provided as the 1st parameter.
     *
     * @note
     * If a presigned URL is returned, `isPresignedUrl` will be true and
     * the resolved object will have a response. Otherwise, response will be null.
     *
     * @example Response for single download URL
     * { response: null, result: { isPresignedUrl: false, href: string }}
     *
     * @example Response for presigned URL
     * { response: AdobeResponse, result: { isPresignedUrl: true, href: string }}
     *
     * @param component
     * @param ignored1              Ignored. (string) component id in other implementation overloads
     * @param ignored2              Ignored. (number) component size in other implementation overloads
     * @param ignored3              I gnored. (string) component revision in other implementation overloads
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    getCompositeComponentUrlForDownload(asset: AdobeDCXComponent, ignored1: never, ignored2: never, ignored3: never, additionalHeaders?: Record<string, string>): AdobePromise<{
        url: string;
        isPresignedUrl: boolean;
        response: AdobeResponse<'json'> | undefined;
    }, AdobeDCXError>;
    getCompositeComponentUrlForDownload(asset: AdobeAsset | AdobeDCXComponent | AdobeDCXComposite, componentId?: string, componentSize?: number, componentRevision?: string, additionalHeaders?: Record<string, string>): AdobePromise<{
        url: string;
        isPresignedUrl: boolean;
        response: AdobeResponse<'json'> | undefined;
    }, AdobeDCXError>;
    /**
     * Retrieves the component (by well-known path) from the composite
     * @see {@link https://wiki.corp.adobe.com/display/CA/Proposal%3A+Download+Components+by+DCX+Path Proposal: Download DCX Components by Path}
     *
     * @param asset             Asset object that identifies the composite asset.
     * @param componentPath       The path of the component to retrieve (must be well-known by the service)
     * @param responseType      Type to tranform the body
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     * @param skipBlockDownload skip block download & return presigned url instead
     * @example
     * ```
     * const response = await session.getCompositeComponentByPath(
     *   asset,
     *   '/path/to/component',
     *   'json',
     *   { 'x-foo': 'bar' },
     * );
     * ```
     *
     */
    getCompositeComponentByPath<T extends AdobeResponseType = 'defaultbuffer'>(asset: AdobeAsset | AdobeDCXComposite, componentPath: string, responseType?: T, additionalHeaders?: Record<string, string>, skipBlockDownload?: boolean): RepoDownloadStreamableReturn<T>;
    /**
     * Retrieves the component (byId) from the asset composite
     * @param asset             Asset object that identifies the composite asset.
     * @param componentId       The id of the component to retrieve
     * @param componentRevision          Component revision to fetch
     * @param responseType      Type to tranform the body
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     */
    getCompositeComponent<T extends AdobeResponseType = 'defaultbuffer'>(asset: AdobeAsset | AdobeDCXComposite, componentId: string, componentRevision: string, responseType?: T, additionalHeaders?: Record<string, string>, componentSize?: number): RepoDownloadStreamableReturn<T>;
    /**
     * Retrieves the component.
     * In this implementation overload, the 2nd and 3rd parameters are ignored.
     * Their respective values are obrtained from the first parameter.
     *
     * @param component         AdobeDCXComponent.
     * @param ignored1          Ignored. (string) component id in other overloads
     * @param ignored2          Ignored. (string) component revision in other overloads
     * @param responseType      Type to tranform the body
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     */
    getCompositeComponent<T extends AdobeResponseType = 'defaultbuffer'>(component: AdobeDCXComponent, ignored1: never, ignored2: never, responseType?: T, additionalHeaders?: Record<string, string>): RepoDownloadStreamableReturn<T>;
    getCompositeComponent<T extends AdobeResponseType = 'defaultbuffer'>(asset: AdobeAsset | AdobeDCXComponent | AdobeDCXComposite, componentId?: string, componentRevision?: string, responseType?: T, additionalHeaders?: Record<string, string>, componentSize?: number): RepoDownloadStreamableReturn<T>;
    /**
     * Checks whether the current user has the requested Privilege on the specified Resource of an Asset.
     * @param asset                 Asset object that identifies the composite asset.
     * @param privilege             The Privilege to be checked. Legal values are read, write, delete, and ack.
     * @param relation              The LinkRelation type of the Resource whose Privilege will be checked.
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    checkACLPrivilege(asset: AdobeAsset, privilege: Omit<Privilege, 'none'>, relation: LinkRelationKey, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<boolean>, AdobeDCXError>;
    /**
     * Retrieves the ACL resource for the asset
     * @param asset                 Asset object that identifies the composite asset.
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    getACLPolicy(asset: AdobeAsset, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<ACPAccessControlList>, AdobeDCXError>;
    /**
     * Retrieves the URLs pointing to the components of a composite.
     * @param asset             Asset object that identifies the composite asset.
     * @param uploads           An array of ComponentUploadInfo objects.
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     * @returns                 An array of RepoResponseResult objects, one for each component.
     */
    getCompositeComponentsUrlsForUpload(composite: AdobeAsset, uploads: ComponentUploadInfo[], additionalHeaders?: Record<string, string>): AdobePromise<UploadRecipe[], AdobeDCXError>;
    /**
     * Patch the ACL policy for this asset using a JSON Patch Document or stringified representation.
     * @param svc HTTP Service or ServiceConfig
     * @param asset Asset to issue the ACL PATCH on
     * @param policy A JSON Patch Document in string on JSON format representing the patch operations to perform on the ACL
     * @param etag Optional etag for the ACL policy. If supplied, the patch will only be performed if the remote policy's etag matches the one provided.
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     * @see {@link https://acpcs.corp.adobe.com/apis/?choose=aclfrontservice#operation/jsonPatchPolicy patch documentation}
     */
    patchACLPolicy(asset: AdobeAsset, policy: JSONPatchDocument | string, etag?: string, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<ACPAccessControlList>, AdobeDCXError>;
    /**
     * Issues a DELETE request against an asset's ACL Policy removing all ACEs.
     *
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     */
    deleteACLPolicy(asset: AdobeAsset, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<'json'>, AdobeDCXError>;
    /**
     * Updates an existing DiscoverableAssets object by requesting an additional "page"
     * of data according to the specified page options
     * @internal
     * @param assets                DiscoverableAssets object returned from a previous call to
     *                               getDiscoverableAssets or refreshDiscoverableAssets.
     *                               Will be updated with the next page's content.
     * @param pageOpts              Optional paging and embedding parameters
     */
    /**
     * Updates an existing DiscoverableRepos object by requesting
     * an additional "page" of data according to the specified page options
     * @internal
     * @param repos                 DiscoverableAssets object returned from a previous call to
     *                               getDiscoverableAssets or refreshDiscoverableAssets.
     *                               Will be updated with the next page's content.
     * @param pageOpts              Optional paging and embedding parameters
     */
    /**
     * Updates an existing VersionSet object by requesting
     * an additional "page" of data according to the specified page options
     * @internal
     * @param resource
     * @param pageOpts
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    refreshVersionsWithPage(resource: AdobeVersionSet, pageOpts: Omit<PageOptions, 'embed'>, additionalHeaders?: Record<string, string>): AdobePromise<PageResource<AdobeVersionSet>>;
    /**
     * Updates an existing Directory object by requesting an additional "page" of
     * data according to the specified page options
     * @internal
     * @param resource
     * @param pageOpts
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    refreshDirectoryWithPage(resource: AdobeDirectory, pageOpts?: PageOptions, additionalHeaders?: Record<string, string>): AdobePromise<PageResource<AdobeAsset>>;
    /**
     * Updates a composite's manifest
     * @param asset                     Asset or branch whose manifest should be updated.
     * @param manifest                  The new manifest contents as an object or string.
     * @param overwrite                 If truthy, `etag` is ignored and "*" is used as the If-Match header of the manifest request.
     *                                      If falsey, `etag` is sent as the If-Match header value.
     *                                      Does not affect the bulk request headers.
     * @param validationLevel           For validation of the manifest object passed. Defaults to 1.
     * @param etag                      If specified, will be passed with the If-Match header of the manifest request.
     *                                      Note that this etag refers to the manifest resource. The etag property on the asset argument is ignored.
     *                                      Does not affect the bulk request headers.
     * @param additionalHeaders         Additional headers to attach to HTTP Requests
     */
    updateCompositeManifest<T = Record<string, unknown>>(asset: AdobeAsset | AdobeDCXBranch, manifest: T, overwrite: boolean, validationLevel?: number, etag?: string, additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'json'>, AdobeDCXError, RequestDescriptor>;
    /**
     *  Uploads a composite component. If the size of the data is less then repo:maxSingleTransferSize data can be directly uploaded.
     * @param asset                 Asset object that identifies the composite asset.
     * @param componentId           The id of the component
     * @param data                  A buffer containing the data to be uploaded
     * @param contentType           Content type of the component
     * @param maybeIsNew            Is this component assumed to be new?
     * @param size                  The size of the component being uploaded
     * @param md5                   The md5 hash of the component
     * @param progressCb            A progress callback to be invoked with upload progress updates
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    putCompositeComponent(asset: AdobeAsset | AdobeDCXComposite, componentId: string, data: Buffer | ArrayBuffer | Blob | string, contentType: string, maybeIsNew?: boolean, size?: number, md5?: string, progressCb?: UploadProgressCallback, additionalHeaders?: Record<string, string>): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, RequestDescriptor>;
    /**
     * Uploads a composite component. If the size of the data is greater then repo:maxSingleTransferSize a GetSliceCallback should be provided instead of the data.
     * @param asset                 Asset object that identifies the composite asset.
     * @param componentId           The id of the component
     * @param getSliceCallback      A callback function to return a slice of data to upload
     *                               The GetSliceCallback function accepts a start byte where the buffer slice should start and an end byte where
     *                               the buffer slice should end (not inclusive). The callback should return a Promise containing the buffer slice or a buffer
     *                               of length 0 indicating the end of the buffer.
     * @param contentType           Content type of the component
     * @param maybeIsNew            (Optional) Is this component assumed to be new?
     * @param size                  (Optional) The size of the component being uploaded
     * @param md5                   (Optional) The md5 hash of the component
     * @param progressCb            A progress callback to be invoked with upload progress updates
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    putCompositeComponent(asset: AdobeAsset | AdobeDCXComposite, componentId: string, getSliceCallback: GetSliceCallback, contentType: string, maybeIsNew?: boolean, size?: number, md5?: string, progressCb?: UploadProgressCallback, additionalHeaders?: Record<string, string>): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, {
        blockUpload?: AdobeBlockUpload;
    } & RequestDescriptor>;
    /**
     * Puts a composite component
     * @remarks Service can override the block size client specified @see {@link https://git.corp.adobe.com/caf/api-spec/blob/gh-pages/schemas/repository/transfer-document.schema.json#L45}
     * @param asset                 Asset object that identifies the composite asset.
     * @param componentId           The id of the component
     * @param dataOrSliceCallback   Either the component data or a GetSliceCallback.
     *                               If the size of the data is greater then repo:maxSingleTransferSize a GetSliceCallback should be provided instead of the data.
     *                               The GetSliceCallback function accepts a start byte where the buffer slice should start and an end byte where
     *                               the buffer slice should end (not inclusive). The callback should return a Promise containing the buffer slice or a buffer
     *                               of length 0 indicating the end of the buffer.
     * @param contentType           Content type of the component
     * @param maybeIsNew            (Optional) Is this component assumed to be new?
     * @param size                  (Optional) The size of the component being uploaded
     * @param md5                   (Optional) The md5 hash of the component
     * @param progressCb            A progress callback to be invoked with upload progress updates
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     * @param blockSize             Desired block size for the upload in bytes
     */
    putCompositeComponent(asset: AdobeAsset | AdobeDCXComposite, componentId: string, dataOrSliceCallback: SliceableData | GetSliceCallback, contentType: string, maybeIsNew?: boolean, size?: number, md5?: string, progressCb?: UploadProgressCallback, additionalHeaders?: Record<string, string>, blockSize?: number): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, {
        blockUpload?: AdobeBlockUpload;
    } & RequestDescriptor>;
    /**
     * Puts a composite component
     * @param asset                 Asset object that identifies the composite asset.
     * @param componentId           The id of the component
     * @param dataOrSliceCallback   Either the component data or a GetSliceCallback.
     *                               If the size of the data is greater then repo:maxSingleTransferSize a GetSliceCallback should be provided instead of the data.
     *                               The GetSliceCallback function accepts a start byte where the buffer slice should start and an end byte where
     *                               the buffer slice should end (not inclusive). The callback should return a Promise containing the buffer slice or a buffer
     *                               of length 0 indicating the end of the buffer.
     * @param contentType           Content type of the component
     * @param maybeIsNew            (Optional) Is this component assumed to be new?
     * @param size                  (Optional) The size of the component being uploaded
     * @param md5                   (Optional) The md5 hash of the component
     * @param progressCb            (Optional) callback for updating upload progress
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    putCompositeComponent(asset: AdobeAsset | AdobeDCXComposite, componentId: string, dataOrSliceCallback: SliceableData | GetSliceCallback, contentType: string, maybeIsNew?: boolean, size?: number, md5?: string, progressCb?: UploadProgressCallback, additionalHeaders?: Record<string, string>): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, {
        blockUpload?: AdobeBlockUpload;
    } & RequestDescriptor>;
    /**
     * Updates the cached links for the specified repository
     *
     * Clients that already have the links for a given repository can call this function to avoid
     *  additional HTTP requests from the session that would otherwise be required to fetch the links
     *  for the specified repository on demand.
     *
     * @param repoId                The repository id for which the links are valid
     * @param links                 The links
     */
    updateCachedRepoLinks(repoId: string, links: LinkSet): void;
    /**
     * Caches the asset links for the specified asset
     *
     * Clients that already have the links for a given asset can call this function to
     *  avoid additional HTTP requests from the session that would otherwise be required to fetch
     *  the links for the specified asset on demand.
     *
     * @param asset                 An Asset object containing an assetId and links properties
     */
    updateCachedAssetLinks(asset: AdobeAsset): void;
    /**
     * Caches the index links for the current user
     *
     * Clients that already have the links for a given asset can call this function to
     *  avoid additional HTTP requests from the session that would otherwise be required to fetch
     *  the links for the specified asset on demand.
     *
     * @param links                 The links
     */
    updateCachedIndexLinks(links: LinkSet): void;
    /**
     * Get links cache instance
     */
    getLinksCache(): RepositoryLinksCache;
    /**
     * Set links cache
     * @param cache
     */
    setLinksCache(cache: RepositoryLinksCache): any;
    /**
     * Clear all entries in links cache
     */
    clearLinksCache(): void;
    /**
     * Performs a HEAD request on the application metadata resource.
     * @param asset             The asset
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     */
    headAppMetadata(asset: AdobeAsset, additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'void'>, AdobeDCXError>;
    /**
     * Returns the application metadata associated with the asset.
     *
     * @note
     * Browser clients may prefer *not* using the etag parameter, instead relying on
     * the browser's HTTP cache to add the header, unless you are managing a cache yourself.
     *
     * @param asset             Asset whose app metadata to fetch.
     * @param etag              If specified, will be passed as the If-None-Match header value.
     *                              Note: this etag refers to the application metadata resource.
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     */
    getAppMetadata<T = any>(asset: AdobeAsset, etag?: string, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<T> & {
        etag: string;
    }, AdobeDCXError>;
    /**
     * Update an asset's application metadata resource.
     *
     * @note Replaces the existing resource.
     *
     * @param asset             Asset object that identifies the asset
     * @param metadata          New application metadata JSON object
     * @param etag              If specified, will be passed as the If-Match header value.
     *                              Note: this etag refers to the application metadata resource.
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     */
    putAppMetadata<T = string | Record<string, unknown>>(asset: AdobeAsset, metadata: T, etag?: string, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<{
        etag: string;
    }>, AdobeDCXError>;
    /**
     * Update application metadata associated with an asset.
     *
     * @param asset                 The asset whose metadata to update
     * @param patchDoc              JSON Patch Document to apply.
     *                              {@link https://git.corp.adobe.com/pages/caf/api-spec/#patch|RFC 6902}
     * @param etag                  ETag of the application metadata resource.
     *                              Used as the If-Match header value.
     *                              For unconditional update, use wildcard "*".
     *
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    patchAppMetadata<T = string | JSONPatchDocument>(asset: AdobeAsset, patchDoc: T, etag: string, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<{
        etag: string;
    }>, AdobeDCXError>;
    /**
     * Updates an asset's versions resource
     * @param {AdobeAsset}                                      asset       Asset object that identifies the asset
     * @param {string | AdobeVersionPatchOperation[]}           patchDoc    A JSON Patch Document (see {@link https://git.corp.adobe.com/pages/caf/api-spec/#patch|RFC6902})
     * @param {string}                                          etag        If specified then its value will be passed with the If-Match header.
     *                                                                      Version resource will only be updated if eTag matches what is on the server.
     *                                                                      Note that this etag refers to the Versions resource. The etag property on the asset argument is ignored.
     * @param {Record<string, string>}                          additionalHeaders Additional headers to be applied to HTTP requests
     */
    patchVersions(asset: AdobeAsset, patchDoc: string | AdobeVersionPatchOperation[], etag?: string, additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse>;
    /**
     * ******************************************************************************
     * Operations
     * ******************************************************************************
     */
    /**
     * Copy an asset.
     *
     * @param sourceAsset           Asset to copy. Must contain assetId or path & repositoryId.
     * @param targetAsset           Destination to copy into. Must contain assetId or path & repositoryId.
     * @param createIntermediates   Whether to create intermediate directories.
     * @param overwriteExisting     Whether to overwrite existing target asset.
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     * @param manifestPatch         A JSON Patch Document to be applied to the target asset's manifest.
     */
    copyAsset(sourceAsset: PathOrIdAssetDesignator, targetAsset: PathOrIdAssetDesignator, createIntermediates: boolean, overwriteExisting: boolean, additionalHeaders?: Record<string, string>, manifestPatch?: JSONPatchDocument): AdobePromise<RepoResponseResult<AdobeAsset, 'json'>, AdobeDCXError>;
    /**
     * Move an asset.
     *
     * @param {AdobeAsset} sourceAsset - Asset to move. Must contain assetId or path & repositoryId.
     * @param {AdobeAsset} targetAsset - Destination to move to. Must contain assetId or path & repositoryId.
     * @param {boolean} createIntermediates - Whether to create intermediate directories.
     * @param {boolean} overwriteExisting - Whether to overwrite existing target asset.
     * @param {Record<string, string>} additionalHeaders - Additional headers to apply to HTTP Requests
     */
    moveAsset(sourceAsset: PathOrIdAssetDesignator, targetAsset: PathOrIdAssetDesignator, createIntermediates: boolean, overwriteExisting: boolean, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<AdobeAsset, 'json'>, AdobeDCXError>;
    /**
     * Delete an asset.
     *
     * @note
     * If the asset is a directory, recursive is required.
     *
     * @note
     * This is irreversible, use discard to be able to restore an asset.
     *
     * @param {AdobeAsset} assetToDelete                 - Asset to delete. Must contain assetId or path & repositoryId.
     * @param {string}  [etag]                           - Etag of asset to delete, if not provided will delete unconditionally.
     * @param {boolean} [recursive]                      - Whether the delete should happen recursively. Required to delete directories.
     * @param {Record<string, string>} additionalHeaders - Additional headers to apply to HTTP Requests
     */
    deleteAsset(assetToDelete: PathOrIdAssetDesignator | AdobeDCXComposite, etag?: string, recursive?: boolean, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError>;
    /**
     * Discard an asset, can be reversed with restore.
     *
     * @param {AdobeAsset}  assetToDiscard      - Asset to discard. Must contain assetId or path & repositoryId.
     * @param {string}      [etag]              - Etag of asset to discard, if not provided will discard unconditionally.
     * @param {boolean}     [recursive=false]   - Whether the discard should happen recursively. Required to discard directories.
     * @param {Record<string, string>} [additionalHeaders] - Additional headers to apply to HTTP Requests
     */
    discardAsset(assetToDiscard: PathOrIdAssetDesignator | AdobeDCXComposite, etag?: string, recursive?: boolean, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError>;
    /**
     * Restore an asset, can only be done on discarded assets.
     *
     * @param {AdobeAsset} assetToRestore - Asset to restore. Must contain assetId & repositoryId.
     * @param {Record<string, string>} additionalHeaders - Additional headers to apply to HTTP Requests
     */
    restoreAsset(assetToRestore: RequireAtLeastOne<AdobeAsset, 'assetId'> | AdobeDCXComposite, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<AdobeAsset, 'json'>, AdobeDCXError>;
    /**
     * Package one or more assets into a compressed zip.
     *
     * @param {AdobeAsset | AdobeAsset[]} sources - Assets to package. Each must contain assetId/path & repositoryId.
     * @param {AdobeAsset} destination - Destination asset for package. Must contain assetId/path & repositoryId.
     * @param {boolean} createIntermediates - Whether to create intermediate directories if missing.
     * @param {boolean} overwriteExisting - Whether to overwrite existing destination asset.
     * @param {Record<string, string>} additionalHeaders - Additional headers to apply to HTTP Requests.
     */
    packageAssets(sources: PathOrIdAssetDesignator | PathOrIdAssetDesignator[], destination: PathOrIdAssetDesignator, createIntermediates: boolean, overwriteExisting: boolean, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError>;
    /**
     * Execute a POST on the operations endpoint using the operation document as body.
     *
     * @param operationDocument Document describing the operation(s).
     * @param additionalHeaders Additional headers for the request
     */
    performOperation(operationDocument: AnyOperationDocument | AnyOperationDocument[], additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'json'>, AdobeDCXError>;
    /**
     * Execute a POST on the operations endpoint using the operation document as body.
     *
     * @note
     * `performBatchOperation` differs from `performOperation` by parsing the response body
     * into an array, translating any errors with problem types into DCX error instances.
     *
     * @param operationDocument Document describing the operation(s).
     * @param additionalHeaders Additional headers to apply to HTTP Requests
     */
    performBatchOperation(operationDocument: AnyOperationDocument | AnyOperationDocument[] | string, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<AdobeOperationResult[], 'json'>, AdobeDCXError, RequestDescriptor>;
    /**
     * ******************************************************************************
     * Public conversion APIs
     * ******************************************************************************
     */
    /**
     * Convert an object conforming to the AdobeRepoUploadResult interface
     * into an instance of UploadResults.
     *
     * @param repoUploadResults
     * @param compositeId
     */
    uploadResultsFromAdobeRepoUploadResult(repoUploadResults: AdobeRepoUploadResult, compositeId?: string): AdobePromise<AdobeUploadResults, AdobeDCXError>;
}

export declare class AdobeRepoAPISession implements AdobeRepoAPISession, AdobeSessionBase {
    private _endPoint;
    protected _linksCache: RepositoryLinksCache;
    protected _service: AdobeHTTPService;
    protected _authenticationAllowList: string[];
    protected _blockUploadThreshold: number;
    constructor(httpService: AdobeHTTPService, server: string, linksCache?: RepositoryLinksCache);
    get serviceConfig(): Required<ServiceConfig>;
    get blockUploadThreshold(): number;
    set blockUploadThreshold(threshold: number);
    get blockDownloadThreshold(): number;
    set blockDownloadThreshold(threshold: number);
    /**
     * ******************************************************************************
     * Private Methods
     * ******************************************************************************
     */
    /**
     * Since the getComponent* APIs accept an AdobeDCXComponent as the first parameter,
     * and also accepts the id/revision as the 2nd or 3rd parameters,
     * this is a helper method to resolve the revision && id in the most resilient way possible.
     * It prioritizes its lookup in the following order:
     * 1. If an AdobeDCXComponent was provided, use the ID and revision directly from it, disregard other parameters
     * 2. Return provided ID/Revision (2nd/3rd parameters) OR optionally just the id if revision is considered optional (Uploads)
     * 3. If AdobeDCXComposite is provided as first parameter, Use component id (2nd parameter) to look up local component revision
     * 4. Attempt to read revision from remote manifest based on first parameter (AdobeAsset or AdobeDCXComposite) and component id
     * @param asset
     * @param id
     * @param revision
     * @param additionalHeaders (used when reading the manifest)
     * @param isRevisionRequired Determines whether or not to throw an error if the revision could not be resolved.  If `true` this method will throw
     * DCXError.INVALID_PARAMS if the revision could not be determined.
     */
    private _resolveComponentIdAndRevision;
    /**
     * Returns true if the asset contains an owner that contains a repo/asset id or core
     * @param asset
     */
    private _isDCXComponentLike;
    /**
     * Get an immediately resolving promise with source object.
     *
     * @param source
     */
    private _withSourcePromise;
    /**
     * Attempt to convert any incoming asset type into AdobeAsset
     * Use duck-typing on properties, if they exist, use them.
     *
     * @param {AdobeAsset | AdobeDCXComponent | AdobeDCXComposite | AdobeDCXBranch} convertable
     */
    private _getAsAdobeAsset;
    /**
     * ******************************************************************************
     * ISessionBase implementation
     * ******************************************************************************
     */
    /**
     * A list of domain names that should get sent the auth credentials of the underlying service.
     * @memberof AdobeSessionBase#
     * @type Array
     */
    get authenticationAllowList(): string[];
    set authenticationAllowList(value: string[]);
    /**
     * A _very_ naive URL resolver whose function is to prepend the a base to a relative URL
     * or to simply return the href directly if it's absolute.
     * Nothing is done about . or .. elements.
     *
     * @internal
     */
    _resolveUrl(href: string): string;
    /**
     * Register links with the session.
     * If provided with assetId or repositoryId, those will be used in the cache.
     * If not, repoId will be set to `faux-repo-id` and a random assetId will be used.
     *
     * Note: Only pass repo/asset IDs if they are real and unique, otherwise the cache may be corrupted.
     *
     * @param {Record<string,Link>} links - Links to register
     * @param {string} [repositoryId = 'faux-repo-id'] - Repository ID associated with the links
     * @param {string} [assetId] - Asset ID associated with the links
     * @returns {*}
     *      @property {string} assetId
     *      @property {string} repositoryId
     */
    registerLinks(links: Record<string, Link>, repositoryId?: string, assetId?: string): {
        assetId: string;
        repositoryId: string;
    };
}

/**
 * Constructs and returns a repository session instance.
 * @public
 * @example
 * ```js
 * // create instance of AdobeHTTPService
 * ...
 * var session = createRepoAPISession(service, 'https://platform-cs-stage-va7.adobe.io');
 * ```
 * @param   {AdobeHTTPService} httpService The HTTP service.
 * @param   {String}                  endpoint    The URL for the storage service.
 */
export declare const createRepoAPISession: (httpService: AdobeHTTPService, endpoint: string, linksCache?: RepositoryLinksCache) => AdobeRepoAPISession;

/**
 * Create upload record for use in UploadResults.
 *
 * @param componentId
 * @param etag
 * @param version
 * @param md5
 * @param length
 * @param type
 */
export declare function createUploadRecord(componentId: string, etag: string, version: string, md5: string, length: number, type: string): AdobeUploadRecord;

/**
 * Create UploadResults instance with empty records.
 *
 * @param compositeId       - The composite's UUID
 * @param compositeAssetId  - The composites cloud URN
 * @param repositoryId      - The repository ID
 */
export declare function createUploadResults(compositeId: string | undefined, compositeAssetId: string, repositoryId: string | undefined): AdobeUploadResults;

/**
 * @internal - Used to indicate the hash value format for the component descriptor.
 */
export declare const CURRENT_COMPONENT_DESCRIPTOR_HASH_TYPE = "md5";

/**
 * @internal - Used to indicate the serialization format for the serialized component descriptor.
 */
export declare const CURRENT_COMPONENT_DESCRIPTOR_VERSION = "0";

/**
 * Create instance of UploadResults given an object
 * conforming to the AdobeRepoUploadResult interface.
 *
 * @param repoUploadResults object to convert
 */
export declare function uploadResultsFromAdobeRepoUploadResult(session: AdobeRepoAPISession, repoUploadResults: AdobeRepoUploadResult, compositeId?: string): AdobePromise<AdobeUploadResults, AdobeDCXError_2>;

/**
 * Create UploadResults from serialized component descriptor string
 *
 * @param descriptor - Serialized component descriptor string
 *
 * Example component descriptor
 * {
 versionId: string,
 type: string,
 cloudAssetId: string,
 componentId: string,
 componentRevisionId: string,
 repositoryId: string,
 hashValue?: string,
 hashType?: string,
 etag?: string,
 size?: integer
 }
 */
export declare function uploadResultsFromComponentDescriptor(descriptor: string): AdobeUploadResults;

export { }
