/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { assertLinksContainAny, LinkRelation } from '@dcx/assets';
import {
    AdobeRepoUploadResult,
    AdobeUploadRecord,
    AdobeUploadResults,
    DeserializedComponentDescriptor,
} from '@dcx/common-types';
import DCXError, { AdobeDCXError } from '@dcx/error';
import AdobePromise from '@dcx/promise';
import { getLinkHrefTemplated, validateObject, validateParams } from '@dcx/util';
import type { AdobeRepoAPISession } from '../AdobeRepoAPISession';
import { CURRENT_COMPONENT_DESCRIPTOR_VERSION, UploadRecord, UploadResults } from './internal';
export { CURRENT_COMPONENT_DESCRIPTOR_HASH_TYPE, CURRENT_COMPONENT_DESCRIPTOR_VERSION } from './internal';
/**
 * Create UploadResults instance with empty records.
 *
 * @param compositeId       - The composite's UUID
 * @param compositeAssetId  - The composites cloud URN
 * @param repositoryId      - The repository ID
 */
export function createUploadResults(
    compositeId: string | undefined,
    compositeAssetId: string,
    repositoryId: string | undefined,
): AdobeUploadResults {
    validateParams(
        ['compositeId', compositeId, 'string', true],
        ['compositeAssetId', compositeAssetId, 'string'],
        ['repositoryId', repositoryId, ['string', 'undefined']],
    );
    return new UploadResults(compositeId, compositeAssetId, repositoryId);
}

/**
 * Create upload record for use in UploadResults.
 *
 * @param componentId
 * @param etag
 * @param version
 * @param md5
 * @param length
 * @param type
 */
export function createUploadRecord(
    componentId: string,
    etag: string,
    version: string,
    md5: string,
    length: number,
    type: string,
): AdobeUploadRecord {
    validateParams(
        ['componentId', componentId, 'string'],
        ['etag', etag, 'string'],
        ['version', version, 'string'],
        ['md5', md5, 'string'],
        ['length', length, 'number'],
        ['type', type, 'string'],
    );
    return new UploadRecord(componentId, etag, version, md5, length, type);
}

/**
 * Create UploadResults from serialized component descriptor string
 *
 * @param descriptor - Serialized component descriptor string
 * 
 * Example component descriptor
 * {
    versionId: string,
    type: string,
    cloudAssetId: string,
    componentId: string,
    componentRevisionId: string,
    repositoryId: string,
    hashValue?: string,
    hashType?: string,
    etag?: string,
    size?: integer
}
 */
export function uploadResultsFromComponentDescriptor(descriptor: string): AdobeUploadResults {
    validateParams(['decriptor', descriptor, 'string']);

    let parsed: Required<DeserializedComponentDescriptor>;
    try {
        parsed = JSON.parse(descriptor);
    } catch (e) {
        throw new DCXError(DCXError.INVALID_JSON, 'Invalid component descriptor', e);
    }

    switch (parsed.versionId) {
        case CURRENT_COMPONENT_DESCRIPTOR_VERSION:
        case undefined: // attempt to use current serialization version if not defined
            // validate repository independendently
            // other params are validated in factory function
            if (!parsed.repositoryId) {
                throw new DCXError(DCXError.INVALID_DATA, 'Component descriptor missing repositoryId');
            }

            try {
                const results = createUploadResults(
                    parsed.compositeId,
                    parsed.cloudAssetId,
                    parsed.repositoryId,
                ) as UploadResults;

                const record = createUploadRecord(
                    parsed.componentId,
                    parsed.etag,
                    parsed.componentRevisionId,
                    parsed.hashValue,
                    parsed.size,
                    parsed.type,
                );
                results.addUploadRecord(parsed.componentId, record);

                return results;
            } catch (e) {
                throw new DCXError(DCXError.INVALID_DATA, 'Invalid component descriptor', e);
            }
        default:
            // Unrecognized, invalid, or not implemented serialization versionId.
            throw new DCXError(
                DCXError.INVALID_DATA,
                'ComponentDescriptor serialization versionId not valid or not yet handled.',
            );
    }
}

/**
 * Create instance of UploadResults given an object
 * conforming to the AdobeRepoUploadResult interface.
 *
 * @param repoUploadResults object to convert
 */
export function uploadResultsFromAdobeRepoUploadResult(
    session: AdobeRepoAPISession,
    repoUploadResults: AdobeRepoUploadResult,
    compositeId?: string,
): AdobePromise<AdobeUploadResults, AdobeDCXError> {
    validateParams(['repoUploadResults', repoUploadResults, 'object'], ['compositeId', compositeId, 'string', true]);
    validateObject(repoUploadResults, 'repoUploadResults', ['result', 'object']);

    // FUTURE: Remove this flexibility with deprecation of AdobeRepoUploadResult#compositeAsset
    const asset = repoUploadResults.asset || (repoUploadResults as any).compositeAsset;

    const { result: repoResult } = repoUploadResults;
    if ((!asset.assetId || !asset.repositoryId) && !asset.links && !repoResult.links) {
        throw new DCXError(
            DCXError.INVALID_PARAMS,
            'AdobeRepoUploadResult#asset object missing repositoryId or assetId, and links',
        );
    }

    // create the record first, it may throw synchronously
    const record = createUploadRecord(
        repoResult.id as string,
        repoResult.etag as string,
        repoResult.revision as string,
        repoResult.md5 as string,
        repoResult.length as number,
        repoResult.type as string,
    );

    let p: AdobePromise<AdobeUploadResults>;
    if (asset.assetId && asset.repositoryId) {
        p = AdobePromise.resolve(createUploadResults(compositeId, asset.assetId, asset.repositoryId));
    } else {
        // Only check that session exists when it's needed.
        validateParams(['session', session, 'object']);
        const links = asset.links || repoResult.links;
        const link = assertLinksContainAny(links, [
            LinkRelation.PRIMARY,
            LinkRelation.ID,
            LinkRelation.PATH,
            LinkRelation.COMPONENT,
        ]);
        // The template args are not used when the link doesn't contain the template.
        const href = getLinkHrefTemplated(links, link, { component_id: 'manifest' });

        p = session.headHTTPResource(href).then((res) => {
            const repositoryId = res.headers['repository-id'];
            const assetId = res.headers['asset-id'];
            if (!repositoryId || !assetId) {
                throw new DCXError(DCXError.INVALID_DATA, 'Fetched data missing repositoryId or assetId');
            }

            return createUploadResults(compositeId, assetId, repositoryId);
        });
    }

    return p.then((results) => {
        results.addUploadRecord(repoResult.id as string, record);
        return results;
    });
}
