/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    AdobeDCXError,
    AdobeResponse,
    AdobeResponseType,
    AdobeUploadRecord,
    AdobeUploadResults,
    DeserializedComponentDescriptor,
    InternalFunction,
    RepoDownloadStreamableReturn,
} from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import AdobePromise from '@dcx/promise';
import { pruneUndefined, validateObject, validateParams } from '@dcx/util';
import { AdobeRepoAPISession } from '../AdobeRepoAPISession';

/**
 * @internal - Used to indicate the serialization format for the serialized component descriptor.
 */
export const CURRENT_COMPONENT_DESCRIPTOR_VERSION = '0';
/**
 * @internal - Used to indicate the hash value format for the component descriptor.
 */
export const CURRENT_COMPONENT_DESCRIPTOR_HASH_TYPE = 'md5';

/**
 * @internal
 */
export class UploadRecord implements AdobeUploadRecord {
    constructor(
        public id: string,
        public etag: string,
        public version: string,
        public md5: string,
        public length: number,
        public type: string,
    ) {}
}

/**
 * @internal
 */
export class UploadResults implements AdobeUploadResults {
    public records: Record<string, AdobeUploadRecord> = {};

    /**
     * @internal
     */
    constructor(
        public compositeId: string | undefined,
        public compositeAssetId: string,
        public repositoryId?: string,
    ) {}

    public addUploadRecord(componentId: string, record: AdobeUploadRecord) {
        validateParams(['componentId', componentId, 'string'], ['record', record, 'object']);
        this.records[componentId] = record;
    }

    public getComponentDescriptor(componentId: string): string {
        const record = this._checkRAPIComponentParams(componentId);

        return JSON.stringify(
            _getComponentDescriptor(this.compositeAssetId, this.compositeId, this.repositoryId, record),
        );
    }

    public getComponentURL(
        session: AdobeRepoAPISession,
        componentId: string,
    ): AdobePromise<
        { isPresignedUrl: boolean; url: string; response: AdobeResponse<'json'> | undefined },
        AdobeDCXError
    > {
        const record = this._checkRAPIComponentParams(componentId);

        return session.getCompositeComponentUrlForDownload(
            { repositoryId: this.repositoryId, assetId: this.compositeAssetId },
            componentId,
            record.length,
            record.version,
        );
    }

    public getComponent<T extends AdobeResponseType = 'defaultbuffer'>(
        session: AdobeRepoAPISession,
        componentId: string,
        responseType?: T,
    ): RepoDownloadStreamableReturn<T> {
        const record = this._checkRAPIComponentParams(componentId);

        return session.getCompositeComponent<T>(
            { repositoryId: this.repositoryId, assetId: this.compositeAssetId },
            componentId,
            record.version,
            responseType,
        );
    }

    private _checkRAPIComponentParams(componentId: string): UploadRecord {
        if (!this.repositoryId) {
            throw new DCXError(DCXError.INVALID_STATE, 'Repository ID must be defined.', undefined, undefined, {
                componentId,
            });
        }

        const record = this.records[componentId];
        if (!record) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'UploadRecord does not exist', undefined, undefined, {
                componentId,
            });
        }

        return record;
    }
}

/**
 * Get object representation of component descriptor.
 *
 * @param cloudAssetId
 * @param repositoryId
 * @param uploadRecord
 */
function _getComponentDescriptor(
    cloudAssetId: string,
    compositeId: string | undefined,
    repositoryId: string | undefined,
    uploadRecord: AdobeUploadRecord,
): DeserializedComponentDescriptor {
    try {
        validateObject(
            uploadRecord,
            'UploadRecord',
            ['id', 'string'],
            ['version', 'string'],
            ['length', 'number'],
            ['etag', 'string'],
            ['type', 'string'],
        );
    } catch (e) {
        // throws INVALID_PARAMS, switch it to INVALID_STATE with underlyingError
        throw new DCXError(DCXError.INVALID_STATE, 'Invalid record data', e);
    }

    const descriptor: DeserializedComponentDescriptor = {
        versionId: CURRENT_COMPONENT_DESCRIPTOR_VERSION, // when format changes, this must be incremented
        componentId: uploadRecord.id,
        cloudAssetId,
        compositeId,
        repositoryId: repositoryId,
        componentRevisionId: uploadRecord.version,
        type: uploadRecord.type,
        cloudExpiration: undefined, // gets pruned
        size: uploadRecord.length,
        etag: uploadRecord.etag,
        hashType: CURRENT_COMPONENT_DESCRIPTOR_HASH_TYPE, // may change in later versions
        hashValue: uploadRecord.md5, // but for now we lock to the md5 property
    };

    return pruneUndefined(descriptor);
}

/**
 * Internals for testing
 *
 * @private
 * @internal
 */
export interface InternalTypes {
    _getComponentDescriptor: InternalFunction<typeof _getComponentDescriptor>;
}

let _internals: InternalTypes = undefined as unknown as InternalTypes;
/* istanbul ignore next */
if (process.env.NODE_ENV === 'development') {
    _internals = {
        _getComponentDescriptor,
    };
}

/**
 * @private
 * @internal
 **/
export const internals = _internals;
