/// <reference types="node" />

import { AdobeAssetWithLinks } from '@dcx/common-types';
import { BasicLink } from '@dcx/common-types';
import { JSONPatchDocument } from '@dcx/common-types';
import { LinkMode } from '@dcx/common-types';
import { LinkSet } from '@dcx/common-types';
import { PlatformReadableStream } from '@dcx/common-types';
import type { Readable } from 'stream';
import { RenditionLink } from '@dcx/common-types';
import { RenditionOptions } from '@dcx/common-types';
import { RequireAtLeastOne } from '@dcx/common-types';

declare type AnyLinkSetInput = LinkSet | UnderscoreLinksObject | LinksObject | AdobeAssetWithLinks | undefined;

/**
 * Appends path elements, adding a slash between non-null components if they do not already start or end in a slash.
 * NOTE: appending a '/' will ensure the path ends in a slash.
 * @private
 * @returns {the} appended path.
 */
export declare const appendPathElements: (...objs: (string | undefined)[]) => string;

/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
/// <reference types="node" />
/**
 * Utility functions for converting between different data types
 */
export declare function arrayBufferToString(buff: ArrayBuffer | Buffer, numOfBytes?: number): string;

/**
 * Assert a generic condition.
 * If it returns false, throw an INVALID_PARAMS error with the provided message.
 * If it returns anything else, return undefined.
 *
 * @throws {AdobeDCXError}
 *
 * @param {Function} condition - function that returns a boolean and takes no arguments
 * @param {string} message
 */
export declare const assert: (condition: () => boolean, message: string) => void;

declare type AsyncFunction<OUT = unknown, IN = unknown> = Function & ((...args: IN[]) => Promise<OUT>);

/**
 * Converts a ReadableStream to an Async Iterable using a Generator
 * This functionality will hopefulyl be built directly into the WHATWG ReadableStreams in the future
 * Some tracking information can be found in the referenced link.
 * @see {@link https://github.com/whatwg/streams/issues/778#issuecomment-461341033 | WHATWG Stream Async Iterator Issue}
 * @param stream the stream to read from
 */
export declare function asyncIterableFromStream(stream: any): AsyncGenerator<any, void, unknown>;

declare type BasicLinkSelector<T> = (link: BasicLink[], options?: T) => BasicLink;

export declare function checkRetriable(status?: number, retryCodes?: RegExp | number | (RegExp | number)[]): boolean;

/**
 * Converts an array of items into an array of subarrays. Each subarray should have a maximum length of chunkSize
 * Ordering of subarrays is not guaranteed to match the ordering of the original array.
 * ex: `chunk([1,2,3,4,5], 2) // [[5],[3,4],[1,2]]`
 * @param array
 * @param chunkSize
 * @returns
 */
export declare function chunkArray<T extends any[]>(array: T, chunkSize: number): any;

/**
 * Combines objects, mutating them in place to proxy to each object.
 * Assigns to the left, so later objects in the argument list may overwrite existing properties.
 * Does not combine deeply, properties that are objects themselves will not be combined.
 *
 * TODO: Also assign getters/setters in the case of class instances.
 *
 * @param {unknown} sources - objects to combine
 */
export declare function combineInPlace(...sources: unknown[]): void;

export declare function concatUint8Arrays(array1: Uint8Array, array2: Uint8Array): Uint8Array;

/**
 * Consume and discard the data from a stream.
 * @private
 * @param {Object}   stream   the stream to drain.
 * @param {Function} callback if not undefined, called when stream is consumed
 * @param {Function} data     if not undefined, called with received data
 */
export declare const consumeStream: (stream: any, callback?: any, data?: any) => void;

/**
 * Produce a 'application/json-patch+json'-type patch to get from one object to another.
 * This does not alter `input` or `output` unless they have a property getter with
 * side-effects (which is not a good idea anyway).
 *
 * `diff` is called on each pair of comparable non-primitive nodes in the
 * `input`/`output` object trees, producing nested patches.
 *
 * Return `undefined` to fall back to default behaviour.
 *
 * Returns list of operations to perform on `input` to produce `output`.
 *
 * See {@link https://tools.ietf.org/html/rfc6902 | RFC6902}
 *
 * @param   input   - The "before patch" object
 * @param   output  - The "after patch" object
 * @param   [diff]  - An optional callback for each non-primitive node comparison
 *                    Useful if, for example, you are patching classes.
 */
export declare function createJSONPatch(input: any, output: any, diff?: JSONPatchVoidableDiff): JSONPatchDocument;

/**
 * Create an instance of PatchDocumentBuilder
 */
export declare function createPatchDocumentBuilder(initialDocument?: JSONPatchDocument): PatchDocumentBuilder;

export declare function dataToBuffer(data: Buffer | Blob | ArrayBuffer | string): undefined | Buffer | Blob | ArrayBuffer;

/**
 * Creates and returns a deep copy of the object utilizing JSON.
 * @private
 * @param   {Object}   obj
 * @returns {Object}
 */
export declare const deepCopy: <T extends object>(obj: T) => T;

/**
 * Utilities related to retrying requests
 * @internal
 */
export declare const DEFAULT_RETRY_CODES: RegExp[];

export declare const endPointOf: (href: string) => string | undefined;

/**
 * Ensure that the href begins with a slash if it is a relative href
 * @private
 * @param   {String} href the href
 * @returns {String}    the href
 */
export declare const ensureRelativeHrefStartsWithSlash: (href: string) => string;

/**
 * Escapes path characters and normalizes unicode
 *
 * @param {string} path - path to escape
 */
export declare const escapeURLPath: (path: string) => string;

/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
/**
 * EventEmitter polyfill for browser environments
 * @internal
 */
export declare class EventEmitter<T extends Record<string, (...args: any[]) => unknown>> {
    private _handlers;
    constructor(eventNames: (keyof T)[]);
    on<E extends keyof T>(event: E, handler: T[E]): number;
    emit<E extends keyof T>(event: E, args: Parameters<T[E]>): void;
    protected removeHandler<E extends keyof T>(event: E, index: number): void;
    protected removeAllHandlers<E extends keyof T = never>(event?: E): void;
}

/**
 * Returns an expansion of URI template using the supplied values
 * @param {String} template URI template string
 * @param {Object} values   Map from name to values. Values are either strings or arrays or objects.
 *                          A string is a simple value; an array is treated as a sequence of values interpreted
 *                          as strings and an an object is treated as a set of (string, string) pairs.
 * @returns {String} The expanded URI string
 */
export declare const expandURITemplate: (template: string, values?: Record<string, (string | number) | (string | number)[] | Record<string, string | number>>) => string;

/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
/**
 * Validation Utilities
 * @internal
 */
export declare type ExpectType = 'function' | 'function[]' | 'boolean' | 'boolean[]' | 'number' | 'number[]' | 'string' | 'string[]' | 'integer' | 'integer[]' | 'object' | 'object[]' | 'null' | 'undefined' | 'nullish' | '+number' | '-number' | 'enum' | 'array';

/**
 * Creates and returns a flat copy of the object passed in. This means that the returned object
 * has the same keys and for each key exactly the same value as the original. This can be used
 * to create copies of lookup tables.
 * @private
 * @param   {Object} obj
 * @returns {Object}
 */
export declare const flatCopy: <T extends Record<string, unknown> = any>(obj: T) => T;

/**
 * Generate UUID
 *
 * @return {string} UUIDv4
 */
export declare const generateUuid: () => string;

/**
 * Get domain from URL, does not include subdomains.
 *
 * URL may or may not contain port number, protocol, path or query parameters
 *
 * @param url
 */
export declare const getDomainFromURL: (url: string) => string;

/**
 * Returns the first matching group from input string
 * @param {String} responseData response body
 * @param {string | RegExp} regexp   Regex pattern
 * @returns {String} pick the first matched group otherwise return empty string
 */
export declare const getFirstRegexpCapture: (responseData: string, regexp: string | RegExp) => string;

/**
 * Generic link parser to get the href property of a link relationship
 * @param links             A LinkSet provided in any of the formats in which they may be stored on a dcx-js class
 * @param relationship      The relationship key
 * @param linkMode          If the relationship contains an array of links, allows specifying the exact mode to return the property for.
 * @return                  The href if the relationship and href exists on the provided links. Else undefined
 */
export declare function getLinkHref(links: AnyLinkSetInput, relationship: string, linkMode?: LinkMode): string;

/**
 * Generic link parser to get the result href from a templated link with values.
 *
 * @param links             A LinkSet provided in any of the formats in which they may be stored on a dcx-js class
 * @param relationship      The relationship key
 * @param values            The key/values to bind to the template. If any value is undefined, it is stripped from the values object.
 * @param linkMode          If the relationship contains an array of links, allows specifying the exact mode to return the property for.
 * @return                  The expanded href template if the relationship exists and the link is marked as templated=true. Else undefined
 */
export declare function getLinkHrefTemplated(links: AnyLinkSetInput, relationship: string, values: Record<string, (string | number | undefined) | (string | number)[] | Record<string, string | number | undefined>>, linkMode?: LinkMode): string;

/**
 * Generic link parser to get a specific property value from a link relationship
 * @param links             A LinkSet provided in any of the formats in which they may be stored on a dcx-js class
 * @param relationship      The relationship key
 * @param property          The property to pull from the relationship
 * @param linkMode          If the relationship contains an array of links, allows specifying the exact mode to return the property for.
 * @return                  The property if the relationship and property exist on the provided links. Else undefined
 */
export declare function getLinkProperty(links: AnyLinkSetInput, relationship: string, property: string, linkMode?: LinkMode): string | undefined;

/**
 * Get Performance (or Performance-like) module.
 * High precision if available, fall back to Date.
 * Only method guaranteed available is Performance#now().
 */
export declare const getPerformance: () => {
    now(): number;
} & Partial<Performance>;

export declare const getStartIdxOfMimeType: (type: string) => number;

export declare const getSuffixIdxOfMimeType: (type: string, suffix: string) => number;

export declare const hexVal: (ch: string) => number;

declare interface InvalidPropertyIssue {
    property: string;
    issue: string;
}

export declare const isAnyFunction: <OUT = unknown, IN = unknown>(val: unknown) => val is Function | AsyncFunction<OUT, IN>;

export declare const isArray: <T = unknown>(p: unknown) => p is T[];

export declare const isArrayBuffer: (p: unknown) => p is unknown[];

export declare const isArrayBufferView: (p: unknown) => p is unknown[];

/**
 * This will narrow down to any function which returns a promise.
 * It can additionally accept parameter types to allow for more specific
 * narrowing of the value under test.
 *
 * i.e.
 * ```typescript
 * if (isAsyncFunction<number>(maybeFn)) {
 *   const valPlus5 = 5 + (await maybeFn());
 * }
 *```

 * Note that the actual return value is not directly tested.
 * The generics cannot be used for further discrimination/matching.
 * As an example, the following code will ALWAYS result in the first branch being taken.
 *
 * ```typescript
 * const booleanAsyncFn = async () => false;
 *
 * if (isAsyncFunction<number>(booleanAsyncFn)) {
 *   // This branch will be executed as return types are not evaluated against generic type parameters.
 * } else if (isAsyncFunction<boolean>(booleanAsyncFn)) {
 *   // does not get executed -- return values are not inspected
 * }
 * ```
 *
 * @deprecated in v7 - removed in v8 (use isAnyFunction instead)
 */
export declare const isAsyncFunction: <OUT = unknown, IN = unknown>(val: unknown) => val is AsyncFunction<OUT, IN>;

/**
 * Detect Browser Environment
 *
 * @return {boolean}
 */
export declare const isBrowser: () => boolean;

/**
 * Detect if SDK is built for browser
 *
 * @return {boolean}
 */
export declare const isBrowserSDK: () => boolean;

export declare const isBuffer: (p: unknown) => p is Buffer;

export declare const isDefined: <T extends unknown>(o: T | null | undefined) => o is T;

/**
 * @deprecated in v7 - removed in v8 (use isAnyFunction instead)
 */
export declare const isFunction: <OUT = unknown, IN = unknown>(val: unknown) => val is Function | AsyncFunction<OUT, IN>;

export declare const isHexChar: (ch: string) => boolean;

/**
 * Returns true if string matches:
 * `application/json`
 * `application/<anything>+json`
 * `<anything>;application/json;<anything>`
 * `<anything>;application/<anything>+json;<anything>`
 *
 * @param str
 */
export declare const isJsonContentType: (str: string) => boolean;

/**
 * Detect Node.js Environment
 * ref: https://github.com/iliakan/detect-node/
 *
 * @return {boolean}
 */
export declare const isNode: () => boolean;

export declare const isNodeReadableStream: (val: unknown) => val is Readable;

export declare const isObject: (val: unknown) => val is Record<string, unknown>;

export declare const isPctEncoding: (str: string) => boolean;

export declare const isPromise: (p: unknown) => p is Promise<unknown>;

/**
 * Determine if a value is a Stream
 *
 * @param {Object} val The value to test
 * @returns {boolean} True if value is a Stream, otherwise false
 */
export declare const isReadableStream: (val: unknown) => val is PlatformReadableStream;

export declare const isUndefined: <T extends unknown>(o: T | null | undefined) => o is null | undefined;

export declare const isValidAbsolutePath: (absPath: string) => boolean;

/**
 * Check if string matches a Shared Cloud asset's URN
 *
 * Matches a string with the format:
 * `urn:aaid:{two chars of a-Z}:{to chars of a-Z}:{valid URNv1-5}`
 * ie. `urn:aaid:sc:US:48a91bdd-d956-5ac1-b45e-c1ab1b6bd9f7`
 * OR `urn:aaid:xy:EU:48a91bdd-d956-5ac1-b45e-c1ab1b6bd9f7`
 *
 * An alternate form, matching only the prefix urn:aaid:sc:{2 char region}:
 * `/^urn:aaid:sc:[a-zA-Z]{2}:[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i`
 *
 * @param {string} str
 * @deprecated in v7 - removed in v8
 */
export declare const isValidAdobeURN: (str: string) => boolean;

/**
 * Returns true if the given path is a valid path for a component or node.
 *
 * A path is valid if all of its components (derived by splitting it with the forward
 * slash / as a separator) fulfill these criteria:
 * - it must be 1 to 255 characters long
 * - it must not end with a . (dot)
 * - it must not contain any of the following characters
 * o U+0022 " QUOTATION MARK
 * o U+002A * ASTERISK
 * o U+007F | vertical bar
 * o U+003A : COLON
 * o U+003C < LESS-THAN SIGN
 * o U+003E > GREATER-THAN SIGN
 * o U+003F ? QUESTION MARK
 * o U+005C \ REVERSE SOLIDUS
 * o The C0 controls, U+0000 through U+001F and U+007F
 * o it does not end with a . (period) or ' ' (space)
 * - it does not equal:
 * o manifest
 * o mimetype
 * o .dcx*
 * o dcx*
 * - it does not equal to any of the following names with or without a file extension:
 * o CON
 * o PRN
 * o AUX
 * o NUL
 * o COM[1-9]
 * o LPT[1-9]
 * @see [schema definition](https://git.corp.adobe.com/caf/xcm/blob/master/schemas/dcx/manifest.schema.json#L295)
 * @param   {String}  path
 * @returns {Boolean}
 */
export declare const isValidPath: (path: string, pathIsAbsolute?: boolean) => boolean;

export declare const isVoid: (p: unknown) => p is void;

export declare const isWebWorker: () => boolean;

export declare const isWHATWGReadableStream: (val: unknown) => val is ReadableStream<any>;

/**
 * JSON Pointer representation
 */
export declare class JSONPatchPointer {
    tokens: string[];
    constructor(tokens?: string[]);
    /**
     * `path` *must* be a properly escaped string.
     */
    static fromJSON(path: string): JSONPatchPointer;
    toString(): string;
    /**
     * Returns an object with 'parent', 'key', and 'value' properties.
     * In the special case that this Pointer's path == "",
     * this object will be {parent: null, key: '', value: object}.
     * Otherwise, parent and key will have the property such that parent[key] == value.
     */
    evaluate(object: any): JSONPatchPointerEvaluation;
    get(object: any): any;
    set(object: any, value: any): void;
    push(token: string): void;
    /**
     * `token` should be a String. It'll be coerced to one anyway.
     * immutable (shallowly)
     */
    add(token: string): JSONPatchPointer;
}

declare interface JSONPatchPointerEvaluation {
    parent: any;
    key: string;
    value: any;
}

/**
 * Callback to allow clients to provide a partial `diff()` function.
 * If this callback returns void, we fall back to the built-in `diffAny()`.
 */
export declare type JSONPatchVoidableDiff = (input: any, output: any, ptr: JSONPatchPointer) => JSONPatchDocument | void;

export declare class Link {
    /** @type {Array} URI references */
    refs: Record<string, any>[];
    /**
     * Link
     * @constructor
     * @param {String} [value]
     * @returns {Link}
     */
    constructor(value?: string);
    /**
     * Get refs with given relation type
     * @param {String} value
     * @returns {Array<Object>}
     */
    rel(value: any): Record<string, unknown>[];
    /**
     * Get refs where given attribute has a given value
     * @param {String} attr
     * @param {String} value
     * @returns {Array<Object>}
     */
    get(attr: string, value: string): Record<string, any>[];
    set(link: Record<string, any>): Link;
    has(attr: string, value: string): boolean;
    parse(pValue: string, pOffset?: number): this;
    toString(): string;
}

declare type LinkExpander<T> = (link: string, values?: T) => string;

export declare interface LinkProvider<T extends Record<string, any>> {
    selector?: T extends RequireAtLeastOne<Exclude<RenditionOptions, keyof BasicLink>, keyof RenditionOptions> ? RenditionLinkSelector : BasicLinkSelector<T>;
    expander?: LinkExpander<T>;
}

declare type LinksObject = {
    links: LinkSet;
};

export declare const LinkUtils: typeof _LinkUtils;

declare namespace _LinkUtils {
    export {
        getLinkProperty,
        getLinkHref,
        getLinkHrefTemplated,
        Link,
        parse
    }
}

/**
 * Merges the objects in the argument list, modifying and returning the first.
 * To merge into a new object, pass {} as the first parameter.
 * Only arguments (except the first) that are dictionaries are merged, so passing
 * 'undefined' as a parameter has no effect.
 * @param   {Object} res The object to merge.
 * @returns {Object} The merged object.
 */
export declare const merge: (...objs: unknown[]) => any;

/**
 * Merge objects recursively from right to left
 * @param {object} target
 * @param {object} sources
 * @returns {*}
 */
export declare const mergeDeep: (target: any, ...sources: any[]) => any;

/**
 * Does nothing, returns nothing
 * @private
 */
export declare const noOp: () => void;

/**
 * Turns header keys to lowercase and removes undefined value entries.
 *
 * @param headers - incoming headers to be normalized
 */
export declare const normalizeHeaders: (headers: Record<string, string | string[]>) => Record<string, string>;

/**
 * Get now as milliseconds.
 */
export declare const now: () => number;

export declare function objectFromEntries<T = any>(entries: IterableIterator<[string, T]> | ReturnType<typeof Object.entries>): Record<string, T>;

/**
 * Does a deep compare of two objects.
 * @private
 * @param   {Object}  obj1               One object.
 * @param   {Object}  obj2               The other object.
 * @param   {Object}  propertiesToIgnore Optional. Object with properties that should not be compared.
 * @returns {Boolean}
 */
export declare const objectsEqual: (obj1: Object, obj2: Object, propertiesToIgnore?: Record<string, boolean>) => boolean;

export declare const parse: (value: string, offset?: number) => Link;

/**
 * Parses the http headers in allHeadersAsText as per http://www.w3.org/Protocols/rfc2616/rfc2616-sec4.html and returns
 * a dictionary with the lower-cased header fields as keys and the header values as values.
 * @private
 * @param   {String} allHeadersAsText
 * @returns {Object}
 */
export declare const parseHeaders: (allHeadersAsText: string) => Record<string, string>;

/**
 * Extract the path (query and fragment) from a URL.
 * Should potentially be moved to dcx-js and made internal
 * @public
 * @param   {String} url the url
 * @returns {Object}    { scheme, authority, path, query, fragment }
 */
export declare const parseURI: (url: string) => {
    scheme: string;
    authority: string;
    path: string;
    query: string;
    fragment: string;
};

declare class PatchDocumentBuilder {
    /**
     * The operations included in the current document.
     */
    private _operations;
    /**
     * Get the document operations as an array of objects
     */
    get operations(): JSONPatchDocument;
    constructor(initialDocument?: JSONPatchDocument);
    /**
     * Get the document as a string
     */
    getDocument(): string;
    /**
     * Add "add" operation to document.
     *
     * @param {string}  path        Path to add value to.
     * @param {unknown} value       Value to add.
     */
    add(path: string, value: unknown): PatchDocumentBuilder;
    /**
     * Add a value to a list.
     * Handles adding the index to the end of the path.
     *
     * @param {string} path                     Path to the array to add to.
     * @param {unknown} value                   Value to add to the list.
     * @param {number | 'end' | '-'} index      If a number is provided, used as the index at which to add the value.
     *                                          Otherwise adds the value to the end of the list.
     */
    addToList(path: string, value: unknown, index: number | 'end' | '-'): PatchDocumentBuilder;
    /**
     * Add "remove" operation to document.
     *
     * @param {string}  path        Path to property to remove.
     */
    remove(path: string): PatchDocumentBuilder;
    /**
     * Add "replace" operation to document.
     * @param {string}  path        Path to property to replace.
     * @param {unknown} value       New value to place at the path.
     */
    replace(path: string, value: unknown): PatchDocumentBuilder;
    /**
     * Add "move" operation to document.
     *
     * @param {string}  source          Path to property to move.
     * @param {string}  destination     Path to move the property to.
     */
    move(source: string, destination: string): PatchDocumentBuilder;
    /**
     * Add "copy" operation to document.
     *
     * @param {string}  source          Path to property to copy.
     * @param {string}  destination     Path to copy the property to.
     */
    copy(source: string, destination: string): PatchDocumentBuilder;
}

export declare function provideLink<T extends Record<string, any>>(links: BasicLink | BasicLink[], linkProvider?: LinkProvider<T>, opts?: T): any;

export declare function provideLink(links: RenditionLink | RenditionLink[], linkProvider?: LinkProvider<RenditionOptions>, opts?: RenditionOptions): any;

/**
 * Prune undefined entries from an object.
 * Returns new object, does not mutate existing object.
 * Does not recurse, only prunes first level.
 *
 * @param {Object} obj - object to prune
 */
export declare function pruneUndefined<T = Record<string, unknown>>(obj: unknown): T;

/**
 * Take out leading slash for an absolute path
 *
 * @param {string} path - path to be processed
 */
export declare const removeLeadingSlashForPath: (path: string) => string;

declare type RenditionLinkSelector = (links: RenditionLink[], options?: RenditionOptions) => RenditionLink;

/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
/**
 * Object (POJO) utilities
 */
export declare function safeGet<T extends object, K extends keyof T>(obj: T, key: K): T[K] | undefined;

export declare const SDKType: () => string;

export declare function stringToBuffer(data: string): Buffer | Uint8Array;

/**
 * Returns a time stamp in the form of the number of milliseconds since 1 January 1970 00:00:00 UTC.
 * @param   {Boolean} roundDownToFullSeconds Whether to limit the resolutio to full seconds. Can be
 *                                         useful if you want to use the timestamp to compare it
 *                                         with the modification time stamps of files, which are
 *                                         also limited to full seconds.
 * @returns {Number}  The number of milliseconds since 1 January 1970 00:00:00 UTC.
 */
export declare const timeStamp: (roundDownToFullSeconds?: boolean) => number;

/** @internal */
export declare const _toUTF8Array: (str: string) => number[];

declare type UnderscoreLinksObject = {
    _links: LinkSet;
};

/**
 * Perform simple validation on a JSON Patch document.
 *
 * This only validates that the individual operations are valid,
 * it does not validate that more complex ordering will fail, even
 * if that is statically known.
 *
 * @param doc
 * @returns {true | InvalidPropertyIssue[]}
 */
export declare function validateJSONPatchDocument(doc: unknown): true | InvalidPropertyIssue[];

/**
 * Validate individual JSON Patch operation.
 *
 * @param op
 * @returns {boolean}
 */
export declare function validateJSONPatchOperation(op: any): true | InvalidPropertyIssue[];

/**
 * Validate multiple object properties from an object.
 * If optional not set, defaults to `false`.
 * If expectedType is set to `enum`, only will validate that the value matches one of `possibleVals`.
 *
 * @note
 * Do not specify generic types, except possibly the first one.
 * They should be inferred from use.
 *
 * @note
 * This function asserts and sets properties as required on validated object.
 * IF OPTIONAL, THE VALIDATED PROPERTY SHOULD STILL BE TREATED AS OPTIONAL/POSSIBLY UNDEFINED.
 *
 * @param {unknown} opts options object to validate
 * @param {[
     *  paramName: string,
     *  expectedType: ExpectType,
     *  optional?: boolean,
     *  possibleVals?: unknown[]
     *  ][]} expects expected type tuple
 *
 * @example
 * validateObject(obj, 'objectName', ['prop1', 'string', false], ['prop2', 'number', true])
 */
export declare function validateObject<T extends Object, U extends Extract<keyof T, string>, V extends T = T & Pick<Required<T>, U>>(obj: T, objectName?: string, ...expects: [U, ExpectType | ExpectType[], boolean?, unknown[]?][]): asserts obj is V;

/**
 * Validate a single parameter, throw error if not pass
 * @param {string} paramName - The parameter name to validate, used in the error
 * @param {unknown} paramVal - The parameter value itself, this is validated
 * @param {ExpectType} expectType - The expected type, if set to enum will ONLY check possibleVals array
 * @param {boolean} [optional = false] - If true, null/undefined is a valid value
 * @param {unknown[]} [possibleVals = []] - Array of concrete values the param must match
 */
export declare const validateParam: (paramName: string, paramVal: unknown, expectType: ExpectType | ExpectType[], optional?: boolean, possibleVals?: unknown[]) => boolean;

/**
 * Validate multiple parameter types specified as tuples
 *
 * @note
 * If optional not set, defaults to `false`.
 *
 * @note
 * If expectedType is set to `enum`, only will validate that the value matches one of `possibleVals`.
 *
 * @param {[paramName: string,
     paramVal: unknown,
     expectedType: ExpectType,
     optional?: boolean,
     possibleVals?: unknown[]][]} expects expected type and value tuple
 *
 * @example
 * ```js
 * validateParams(['paramName', paramVal, 'type', isOptional])
 * ```
 *
 * @example
 * ```
 * // Multiple params to validate:
 * validateParams(['param', param, 'string'], ['param2', param2, 'number', true])
 * ```
 *
 * @example
 * ```
 * // Multiple valid types:
 * validateParams(['stringOrNumber', stringOrNumber, ['string', 'number'], isOptional])
 * ```
 *
 * @example
 * ```
 * // Enum of valid types:
 * validateParams(['myEnum', myEnum, 'enum', isOptional, ['one', 'two']])
 * ```
 *
 * @example
 * ```
 * // Array of a single type:
 * validateParams(['myStrArr', myStrArr, 'string[]', isOptional])
 * ```
 */
export declare function validateParams(...expects: [string, unknown, ExpectType | ExpectType[], boolean?, unknown[]?][]): boolean[];

/**
 * verify a whether the input binary is a dcx snapshot by
 * checking the dcx snapshot mimetype and return the mimetype
 *
 * @param {ArrayBuffer | Buffer} buff
 * @return {string} mimetype
 */
export declare const verifyandGetSnapshotMimetype: (buff: ArrayBuffer | Buffer) => string;

/**
 *  verify whether the input string is a valid rfc4122 uuid
 * @param {string} inputStr need to be verify whether it follow rfc4122 uuid
 * @returns {boolean}
 */
export declare const verifyUuid: (inputStr: string) => boolean;

export declare type Watchable<T> = _Watchable<T> & T;

/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
declare interface _Watchable<T> {
    watchProperty(prop: keyof T, handler: (next: T[typeof prop], current?: T[typeof prop], target?: T) => void): void;
    revokeWatchProxy(): T;
}

/**
 * Create watch proxy around object.
 * Adds `watchProperty` method to the object.
 * Allows watching changes on properties.
 * Handlers are called just before applying the change to the object.
 *
 * @param {unknown} obj - Object to wrap in proxy
 */
export declare function WatchProxy<T extends object = {}>(obj: T): T & Watchable<T>;

/**
 * perform an async operation with some maximum number of retries.
 * @param fn An async function to retry on rejection
 * @param maxRetries - The maximum number of times to retry the operation, default 3.
 * @returns return type of provided fn.
 */
export declare function withRetries<T extends Promise<any>>(fn: () => T, maxRetries?: number): any;

export { }
