/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { v4 as uuid, validate } from 'uuid';
import { isArray, isObject } from './types';

/**
 * Detect Node.js Environment
 * ref: https://github.com/iliakan/detect-node/
 *
 * @return {boolean}
 */
export const isNode = (): boolean => {
    try {
        return Object.prototype.toString.call(global.process) === '[object process]';
    } catch (e) {
        return false;
    }
};

/**
 * Detect Browser Environment
 *
 * @return {boolean}
 */
export const isBrowser = (): boolean => {
    return typeof self === 'object' && self.self === self;
};

export const isWebWorker = (): boolean => {
    return isBrowser() && !!(self as any).WorkerGlobalScope;
};

/**
 * Detect if SDK is built for browser
 *
 * @return {boolean}
 */
export const isBrowserSDK = (): boolean => {
    return SDKType() === 'browser';
};

/* istanbul ignore next */
export const SDKType = (): string => {
    return process.env.APP_ENV as string;
};

/**
 * Generate UUID
 *
 * @return {string} UUIDv4
 */
/* istanbul ignore next */
export const generateUuid = (): string => uuid();

/**
 *  verify whether the input string is a valid rfc4122 uuid
 * @param {string} inputStr need to be verify whether it follow rfc4122 uuid
 * @returns {boolean}
 */
export const verifyUuid = (inputStr: string): boolean => validate(inputStr);

/**
 * Merges the objects in the argument list, modifying and returning the first.
 * To merge into a new object, pass {} as the first parameter.
 * Only arguments (except the first) that are dictionaries are merged, so passing
 * 'undefined' as a parameter has no effect.
 * @param   {Object} res The object to merge.
 * @returns {Object} The merged object.
 */
// eslint-disable-next-line @typescript-eslint/no-explicit-any
export const merge = (...objs: unknown[]): any => {
    if (!objs || !Array.isArray(objs) || objs.length < 1) {
        return {};
    }
    return objs.reduce((merged, obj) => {
        const target = isObject(merged) ? merged : {};
        if (isObject(obj)) {
            for (const name in obj) {
                if (Object.prototype.hasOwnProperty.call(obj, name)) {
                    target[name] = obj[name];
                }
            }
        }
        return target;
    });
};

/**
 * Merge objects recursively from right to left
 * @param {object} target
 * @param {object} sources
 * @returns {*}
 */
export const mergeDeep = (target: any, ...sources: any[]): any => {
    if (!sources.length) {
        return target;
    }
    const source = sources.shift();

    if (isObject(target) && isObject(source)) {
        for (const key in source) {
            if (isObject(source[key]) && !isArray(source[key])) {
                if (!target[key]) {
                    Object.assign(target, { [key]: {} });
                }
                mergeDeep(target[key], source[key]);
            } else if (isArray(target[key]) && isArray(source[key])) {
                // this line merges arrays
                // target[key].push(...source[key]);

                // this line replaces arrays
                Object.assign(target, { [key]: source[key] });
            } else {
                Object.assign(target, { [key]: source[key] });
            }
        }
    }

    return mergeDeep(target, ...sources);
};

/**
 * Appends path elements, adding a slash between non-null components if they do not already start or end in a slash.
 * NOTE: appending a '/' will ensure the path ends in a slash.
 * @private
 * @returns {the} appended path.
 */
export const appendPathElements = (...objs: (string | undefined)[]): string => {
    if (!objs || !Array.isArray(objs)) {
        // TODO: should throw?
        return '';
    }
    const elements: string[] = [];
    const numArgs = objs.length;
    for (let i = 0; i < numArgs; i++) {
        let element = objs[i];
        if (typeof element === 'string' && element !== '') {
            if (i !== 0 || element.length === 1) {
                // Remove any leading slash on elements after the first
                if (element.charAt(0) === '/') {
                    element = element.slice(1);
                }
            }
            if (i !== numArgs - 1) {
                // Remove any trailing slash on elements other than the last one
                if (element.charAt(element.length - 1) === '/') {
                    element = element.slice(0, element.length - 1);
                }
            }
            elements.push(element);
        }
    }

    return elements.join('/');
};

/* eslint-disable max-len, no-control-regex */
const filenameBlocklistRegex = /^(CON|PRN|AUX|NUL|COM\d|LPT\d)(\..+)?$/;
const unicodeBlocklistRegex = /[\u0000-\u001F\u0022\u002A\u003A\u003C\u003E\u003F\u005C\u007F\u007C]/;
/* eslint-enable max-len, no-control-regex */

/**
 * Returns true if the given path is a valid path for a component or node.
 *
 * A path is valid if all of its components (derived by splitting it with the forward
 * slash / as a separator) fulfill these criteria:
 * - it must be 1 to 255 characters long
 * - it must not end with a . (dot)
 * - it must not contain any of the following characters
 * o U+0022 " QUOTATION MARK
 * o U+002A * ASTERISK
 * o U+007F | vertical bar
 * o U+003A : COLON
 * o U+003C < LESS-THAN SIGN
 * o U+003E > GREATER-THAN SIGN
 * o U+003F ? QUESTION MARK
 * o U+005C \ REVERSE SOLIDUS
 * o The C0 controls, U+0000 through U+001F and U+007F
 * o it does not end with a . (period) or ' ' (space)
 * - it does not equal:
 * o manifest
 * o mimetype
 * o .dcx*
 * o dcx*
 * - it does not equal to any of the following names with or without a file extension:
 * o CON
 * o PRN
 * o AUX
 * o NUL
 * o COM[1-9]
 * o LPT[1-9]
 * @see [schema definition](https://git.corp.adobe.com/caf/xcm/blob/master/schemas/dcx/manifest.schema.json#L295)
 * @param   {String}  path
 * @returns {Boolean}
 */
export const isValidPath = (path: string, pathIsAbsolute = false): boolean => {
    try {
        if (path.length > 65535) {
            return false;
        }

        const components = (pathIsAbsolute ? path.slice(1) : path).split('/');
        if (!components || components.length === 0) {
            return false;
        }
        // let i = 0;
        for (const component of components) {
            if (component.length < 1 || component.length > 255) {
                return false;
            }
            if (unicodeBlocklistRegex.test(component)) {
                return false;
            }

            if (component.endsWith('.') || component.endsWith(' ')) {
                return false;
            }

            if (filenameBlocklistRegex.test(component)) {
                return false;
            }

            if (component.startsWith('dcx') || component.startsWith('.dcx')) {
                return false;
            }
            if (component === 'manifest' || component === 'mimetype') {
                return false;
            }
        }

        if (pathIsAbsolute && (!path.startsWith('/') || path.endsWith('/'))) {
            return false;
        }

        return true;
    } catch (x) {
        return false;
    }
};

export const isValidAbsolutePath = (absPath: string): boolean => {
    return isValidPath(absPath, true);
};

/**
 * Does nothing, returns nothing
 * @private
 */
// eslint-disable-next-line @typescript-eslint/no-empty-function
export const noOp = (): void => {};

/**
 * Consume and discard the data from a stream.
 * @private
 * @param {Object}   stream   the stream to drain.
 * @param {Function} callback if not undefined, called when stream is consumed
 * @param {Function} data     if not undefined, called with received data
 */
export const consumeStream = (stream, callback?, data?): void => {
    stream.on('data', data || noOp);
    stream.on('end', callback || noOp);
};

/**
 * Creates and returns a flat copy of the object passed in. This means that the returned object
 * has the same keys and for each key exactly the same value as the original. This can be used
 * to create copies of lookup tables.
 * @private
 * @param   {Object} obj
 * @returns {Object}
 */
export const flatCopy = <T extends Record<string, unknown> = any>(obj: T): T => {
    const result: Record<string, unknown> = {};
    const keys = Object.keys(obj);
    const count = keys.length;
    for (let i = 0; i < count; i++) {
        const key = keys[i];
        result[key] = obj[key];
    }
    return result as T;
};

/**
 * Creates and returns a deep copy of the object utilizing JSON.
 * @private
 * @param   {Object}   obj
 * @returns {Object}
 */
// eslint-disable-next-line @typescript-eslint/ban-types
export const deepCopy = <T extends object>(obj: T): T => {
    return JSON.parse(JSON.stringify(obj));
};

/**
 * Does a deep compare of two objects.
 * @private
 * @param   {Object}  obj1               One object.
 * @param   {Object}  obj2               The other object.
 * @param   {Object}  propertiesToIgnore Optional. Object with properties that should not be compared.
 * @returns {Boolean}
 */
export const objectsEqual = (
    // eslint-disable-next-line @typescript-eslint/ban-types
    obj1: Object,
    // eslint-disable-next-line @typescript-eslint/ban-types
    obj2: Object,
    propertiesToIgnore?: Record<string, boolean>,
): boolean => {
    const checkedProperties = {};
    let key,
        keys = Object.keys(obj1);
    let count = keys.length;

    // Compare all properties of obj1 with the same property of obj2
    for (let i = 0; i < count; i++) {
        key = keys[i];
        if (!propertiesToIgnore || !propertiesToIgnore[key]) {
            const val1 = obj1[key],
                val2 = obj2[key];

            if (typeof val1 !== typeof val2) {
                return false;
            }
            if (isObject(val1) && isObject(val2)) {
                if (!objectsEqual(val1, val2, propertiesToIgnore)) {
                    return false;
                }
            } else if (val1 !== val2) {
                return false;
            }
        }
        checkedProperties[key] = true;
    }

    keys = Object.keys(obj2);
    count = keys.length;
    // Now check to see whether there are any properties in obj2 that were not in obj1
    for (let i = 0; i < count; i++) {
        key = keys[i];
        if (!checkedProperties[key]) {
            return false;
        }
    }

    return true;
};

// Source for uriParsePattern: http://www.ietf.org/rfc/rfc3986.txt (see Appendix B)
const uriParsePattern = new RegExp('^(([^:/?#]+):)?(//([^/?#]*))?([^?#]*)(\\?([^#]*))?(#(.*))?');

/**
 * Extract the path (query and fragment) from a URL.
 * Should potentially be moved to dcx-js and made internal
 * @public
 * @param   {String} url the url
 * @returns {Object}    { scheme, authority, path, query, fragment }
 */
export const parseURI = (
    url: string,
): { scheme: string; authority: string; path: string; query: string; fragment: string } => {
    const matches = url.match(uriParsePattern) || [];
    return {
        scheme: matches[2],
        authority: matches[4],
        path: matches[5],
        query: matches[7],
        fragment: matches[9],
    };
};

// Parses the given href and returns a string that can be used to compare the endpoint
// of the uri with that of other uris by appending the well-known port number if the
// authority doesn't specify a port number. That way http://foo.bar is equal to http://foo.bar:80.
// Returns undefined if the uri doesn't specify a scheme or authority.
export const endPointOf = (href: string): string | undefined => {
    const uriItems = parseURI(href);

    const scheme = uriItems.scheme;
    const authority = uriItems.authority;
    const defaultPortNumber = scheme === 'https' ? 443 : scheme === 'http' ? 80 : -1;

    let result: string | undefined;

    if (scheme && authority) {
        result = (scheme + '://' + authority).toLowerCase();
        if (defaultPortNumber >= 0 && authority.indexOf(':') < 0) {
            result = result + ':' + defaultPortNumber;
        }
    }

    return result;
};

/**
 * Get domain from URL, does not include subdomains.
 *
 * URL may or may not contain port number, protocol, path or query parameters
 *
 * @param url
 */
export const getDomainFromURL = (url: string): string => {
    if (!url || typeof url !== 'string') {
        return url;
    }

    if (url.indexOf('//') > -1) {
        url = url.split('/')[2];
    } else {
        url = url.split('/')[0];
    }

    // remove query params
    url = url.split('?')[0];

    // remove path
    url = url.split('/')[0];

    // remove port number
    url = url.split(':')[0];

    // remove subdomains
    const splits = url.split('.');
    url = splits.slice(Math.max(splits.length - 2, 0)).join('.');

    return url;
};

/**
 * Ensure that the href begins with a slash if it is a relative href
 * @private
 * @param   {String} href the href
 * @returns {String}    the href
 */
export const ensureRelativeHrefStartsWithSlash = (href: string): string => {
    if (href) {
        const uriItems = parseURI(href);
        const scheme = uriItems.scheme;
        const authority = uriItems.authority;

        if (!scheme && !authority) {
            if (href.charAt(0) !== '/') {
                href = '/' + href;
            }
        }
    }
    return href;
};

const zeroCharCode = '0'.charCodeAt(0);
const oneCharCode = '1'.charCodeAt(0);
const nineCharCode = '9'.charCodeAt(0);
const lowerACharCode = 'a'.charCodeAt(0);
const upperACharCode = 'A'.charCodeAt(0);
const lowerFCharCode = 'f'.charCodeAt(0);
const upperFCharCode = 'F'.charCodeAt(0);

export const isHexChar = (ch: string): boolean => {
    const inputCode = ch.charCodeAt(0);
    return (
        (inputCode >= zeroCharCode && inputCode <= nineCharCode) ||
        (inputCode >= lowerACharCode && inputCode <= lowerFCharCode) ||
        (inputCode >= upperACharCode && inputCode <= upperFCharCode)
    );
};

export const isPctEncoding = (str: string): boolean => {
    return str.length >= 3 && str.charAt(0) === '%' && isHexChar(str.charAt(1)) && isHexChar(str.charAt(2));
};

export const hexVal = (ch: string): number => {
    const inputCode = ch.charCodeAt(0);
    if (inputCode >= zeroCharCode && inputCode <= nineCharCode) {
        return inputCode - zeroCharCode;
    }
    if (inputCode >= lowerACharCode && inputCode <= lowerFCharCode) {
        return 10 + inputCode - lowerACharCode;
    }
    if (inputCode >= upperACharCode && inputCode <= upperFCharCode) {
        return 10 + inputCode - upperACharCode;
    }
    return 0;
};

/**
 * Returns a time stamp in the form of the number of milliseconds since 1 January 1970 00:00:00 UTC.
 * @param   {Boolean} roundDownToFullSeconds Whether to limit the resolutio to full seconds. Can be
 *                                         useful if you want to use the timestamp to compare it
 *                                         with the modification time stamps of files, which are
 *                                         also limited to full seconds.
 * @returns {Number}  The number of milliseconds since 1 January 1970 00:00:00 UTC.
 */
export const timeStamp = (roundDownToFullSeconds = false): number => {
    let ts = new Date().getTime();
    if (roundDownToFullSeconds) {
        ts = ts - (ts % 1000);
    }
    return ts;
};

/** @private */
const _unreservedChars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-_.~';
/** @private */
const _reservedChars = _unreservedChars + ":/?#[]@!$&'()*+,;=";
/** @private */
const _hex = '0123456789ABCDEF';

/** @private */
const _bitmapFor = (chars: string): boolean[] => {
    const bitmap: boolean[] = [];
    for (let i = 0; i < 128; ++i) {
        bitmap.push(chars.indexOf(String.fromCharCode(i)) !== -1);
    }
    return bitmap;
};

/** @private */
const _clearReservedChars = _bitmapFor(_reservedChars);
/** @private */
const _clearUnreservedChars = _bitmapFor(_unreservedChars);
/** @private */
const _clearURLPathChars = _bitmapFor(_unreservedChars + '/');

/** @private */
const _encodeUTF8ByteUsingBitmap = (ch: number, allowedBitmap: boolean[]): string => {
    if (ch < 128 && allowedBitmap[ch]) {
        return String.fromCharCode(ch);
    }
    let result = '%';
    result += _hex.charAt((ch >> 4) & 0xf); //eslint-disable-line no-bitwise
    result += _hex.charAt(ch & 0xf); //eslint-disable-line no-bitwise
    return result;
};

// from https://gist.github.com/joni/3760795
/** @internal */
export const _toUTF8Array = (str: string): number[] => {
    /*eslint no-bitwise: ["error", { "allow": ["~", "|", "&", "<<", ">>"] }] */

    const utf8: number[] = [];
    for (let i = 0; i < str.length; i++) {
        let charcode = str.charCodeAt(i);
        if (charcode < 0x80) {
            utf8.push(charcode);
        } else if (charcode < 0x800) {
            utf8.push(0xc0 | (charcode >> 6), 0x80 | (charcode & 0x3f));
        } else if (charcode < 0xd800 || charcode >= 0xe000) {
            utf8.push(0xe0 | (charcode >> 12), 0x80 | ((charcode >> 6) & 0x3f), 0x80 | (charcode & 0x3f));
        } else if (++i < str.length) {
            //surrogate pair
            // UTF-16 encodes 0x10000-0x10FFFF by
            // subtracting 0x10000 and splitting the
            // 20 bits of 0x0-0xFFFFF into two halves
            charcode = 0x10000 + (((charcode & 0x3ff) << 10) | (str.charCodeAt(i) & 0x3ff));
            utf8.push(
                0xf0 | (charcode >> 18),
                0x80 | ((charcode >> 12) & 0x3f),
                0x80 | ((charcode >> 6) & 0x3f),
                0x80 | (charcode & 0x3f),
            );
        }
    }
    return utf8;
};

/** @private */
const _encodeStringUsingBitmap = (str: string, allowedBitmap: boolean[]): string => {
    // Note: this assumes that the caller has NFC-normalized the string
    const bytes = _toUTF8Array(str);
    let result = '';
    for (let i = 0; i < bytes.length; i++) {
        result += _encodeUTF8ByteUsingBitmap(bytes[i], allowedBitmap);
    }
    return result;
};

/**
 * @private
 * @param str
 */
const _percentEncodeExceptUnreserved = (str: string): string => {
    str = str.normalize('NFC');
    return _encodeStringUsingBitmap(str, _clearUnreservedChars);
};

/**
 * @private
 * @param str
 */
const _percentEncodeExceptReservedForURITemplates = (str: string): string => {
    str = str.normalize('NFC');
    let pos = 0;
    let result = '';
    while (pos < str.length) {
        if (isPctEncoding(str.substr(pos))) {
            result += str.substr(pos, 3);
            pos += 3;
        } else {
            result += _encodeStringUsingBitmap(str.charAt(pos++), _clearReservedChars);
        }
    }
    return result;
};

/**
 * Escapes path characters and normalizes unicode
 *
 * @param {string} path - path to escape
 */
export const escapeURLPath = (path: string): string => {
    if (!String.prototype.normalize) {
        // old browser and no polyfills
        /* istanbul ignore next */
        console.warn('String.normalize not found while escaping URL path. You may be missing a polyfill.');
    } else {
        path = path.normalize('NFC');
    }
    if (path.length > 0 && path.charAt(0) === '/') {
        path = path.substr(1);
    }
    return _encodeStringUsingBitmap(path, _clearURLPathChars);
};

/**
 * Take out leading slash for an absolute path
 *
 * @param {string} path - path to be processed
 */
export const removeLeadingSlashForPath = (path: string): string => {
    if (path.length > 0 && path.charAt(0) === '/') {
        path = path.substr(1);
    }
    return path;
};

/**
 * Returns an expansion of URI template using the supplied values
 * @param {String} template URI template string
 * @param {Object} values   Map from name to values. Values are either strings or arrays or objects.
 *                          A string is a simple value; an array is treated as a sequence of values interpreted
 *                          as strings and an an object is treated as a set of (string, string) pairs.
 * @returns {String} The expanded URI string
 */
// eslint-disable-next-line @typescript-eslint/no-explicit-any, @typescript-eslint/explicit-module-boundary-types
export const expandURITemplate = (
    template: string,
    values?: Record<string, (string | number) | (string | number)[] | Record<string, string | number>>,
): string => {
    let pos = 0;
    let result = '';

    let reserved = false,
        named = false,
        specialEmpty = false,
        firstlet = true,
        topLevel = true,
        explode = false;

    let letName = '',
        prefix = '',
        separator = ',',
        maxPrefixLength = -1;

    const getPrefix = (value, maxLen) => {
        /*eslint no-bitwise: ["error", { "allow": ["~", "|", "&", "<<", ">>"] }] */

        // RFC 6570, 2.4.1
        //    The max-length is a positive integer that refers to a maximum number
        //    of characters from the beginning of the letiable's value as a Unicode
        //    string.  Note that this numbering is in characters, not octets, in
        //    order to avoid splitting between the octets of a multi-octet-encoded
        //    character or within a pct-encoded triplet.  If the max-length is
        //    greater than the length of the letiable's value, then the entire
        //    value string is used.
        //
        // It's strange that we have to parse the %-encodings but they might already
        // have been present in a "restricted-encoding" value.
        //
        let pfxPos = 0;
        while (pfxPos < value.length && maxLen > 0) {
            if (isPctEncoding(value.substr(pfxPos))) {
                let charVal = (hexVal(value.substr(pfxPos + 1)) << 4) + hexVal(value.substr(pfxPos + 2));
                pfxPos += 3;
                // A UTF-8 byte begins a multibyte sequence if its top two bits are 11
                if ((charVal & 0xc0) === 0xc0) {
                    while (pfxPos < value.length && isPctEncoding(value.substr(pfxPos))) {
                        charVal = (hexVal(value.substr(pfxPos + 1)) << 4) + hexVal(value.substr(pfxPos + 2));
                        pfxPos += 3;
                        // ... and ends if its top two bits are not 10
                        if ((charVal & 0xc0) !== 0x80) {
                            break;
                        }
                    }
                }
            } else {
                ++pfxPos;
            }
            --maxLen;
        }
        return value.substr(0, pfxPos);
    };

    const appendValueToResult = (valueStr: string) => {
        if (firstlet) {
            result += prefix;
        } else if (explode || topLevel) {
            result += separator;
        } else {
            result += ',';
        }

        if (named && letName && (firstlet || explode || topLevel)) {
            result += _percentEncodeExceptUnreserved(letName);
            if (!specialEmpty || valueStr.length > 0) {
                result += '=';
            }
        }
        if (valueStr) {
            let encodedVal;
            if (reserved) {
                encodedVal = _percentEncodeExceptReservedForURITemplates(valueStr);
            } else {
                encodedVal = _percentEncodeExceptUnreserved(valueStr);
            }
            if (maxPrefixLength > 0) {
                encodedVal = getPrefix(encodedVal, maxPrefixLength);
            }
            result += encodedVal;
        }
        firstlet = false;
        topLevel = false;
    };

    const appendKeyValueToResult = (key: string, value) => {
        // RFC 6570, 2.4.1: "Prefix modifiers are not applicable to letiables that have composite values"
        //     (maxLength is not used in here)

        if (firstlet) {
            result += prefix;
        } else if (explode || topLevel) {
            result += separator;
        } else {
            result += ',';
        }

        if (named && letName && firstlet && !explode) {
            result += _percentEncodeExceptUnreserved(letName);
            if (!specialEmpty || value.length > 0) {
                result += '=';
            }
        }

        if (key) {
            if (reserved) {
                result += _percentEncodeExceptReservedForURITemplates(key);
            } else {
                result += _percentEncodeExceptUnreserved(key);
            }

            if (explode) {
                result += '=';
            } else {
                result += ',';
            }

            if (value) {
                if (reserved) {
                    result += _percentEncodeExceptReservedForURITemplates(value);
                } else {
                    result += _percentEncodeExceptUnreserved(value);
                }
            }
        }

        firstlet = false;
        topLevel = false;
    };

    while (pos < template.length) {
        if (template.charAt(pos) !== '{') {
            // RFC 6570, 2.1
            //   The characters outside of expressions in a URI Template string are
            //   intended to be copied literally to the URI reference if the character
            //   is allowed in a URI (reserved / unreserved / pct-encoded) or, if not
            //   allowed, copied to the URI reference as the sequence of pct-encoded
            //   triplets corresponding to that character's encoding in UTF-8
            if (isPctEncoding(template.substr(pos))) {
                result += template.substr(pos, 3);
                pos += 3;
            } else {
                result += _encodeStringUsingBitmap(template.charAt(pos++), _clearReservedChars);
            }
            continue;
        }
        if (++pos < template.length) {
            reserved = false;
            named = false;
            prefix = '';
            separator = ',';
            // RFC 6570, 1.3
            //   ... the expansion process for ";" (path-style
            //   parameters) will omit the "=" when the letiable value is empty,
            //   whereas the process for "?" (form-style parameters) will not omit the
            //   "=" when the value is empty
            specialEmpty = false;

            switch (template.charAt(pos++)) {
                case '+':
                    reserved = true;
                    break;
                case '#':
                    prefix = '#';
                    reserved = true;
                    break;
                case '.':
                    prefix = '.';
                    separator = '.';
                    break;
                case '/':
                    prefix = '/';
                    separator = '/';
                    break;
                case ';':
                    prefix = ';';
                    separator = ';';
                    named = specialEmpty = true;
                    break;
                case '?':
                    prefix = '?';
                    separator = '&';
                    named = true;
                    break;
                case '&':
                    prefix = '&';
                    separator = '&';
                    named = true;
                    break;
                default:
                    --pos;
                    break;
            }

            letName = '';
            firstlet = true;
            topLevel = true;
            while (pos < template.length) {
                // This, overly broadly, accepts non "letchar" characters in a letiable name.
                // To simplify things, letiables end at } , * : or the end of the template.
                // See RFC 6570, 2.3 for proper "letchar" definition.
                if (
                    template.charAt(pos) === '}' ||
                    template.charAt(pos) === ',' ||
                    template.charAt(pos) === '*' ||
                    template.charAt(pos) === ':'
                ) {
                    explode = false;
                    maxPrefixLength = -1;
                    if (template.charAt(pos) === '*') {
                        explode = true;
                        if (++pos >= template.length) {
                            break;
                        }
                    } else if (template.charAt(pos) === ':') {
                        if (++pos >= template.length) {
                            break;
                        }
                        // RFC 6570, section 2.4.1, prefix can't start with a 0
                        if (template.charCodeAt(pos) >= oneCharCode && template.charCodeAt(pos) <= nineCharCode) {
                            maxPrefixLength = 0;
                        }
                        // and must be less than 10000
                        while (
                            pos < template.length &&
                            template.charCodeAt(pos) >= zeroCharCode &&
                            template.charCodeAt(pos) <= nineCharCode &&
                            maxPrefixLength < 10000
                        ) {
                            maxPrefixLength = maxPrefixLength * 10 + (template.charCodeAt(pos++) - zeroCharCode);
                        }
                        if (pos >= template.length) {
                            break;
                        }
                    }

                    // We should really be sitting on a , or } here but if not skip the junk
                    while (pos < template.length && template.charAt(pos) !== '}' && template.charAt(pos) !== ',') {
                        ++pos;
                    }

                    if (letName.length > 0 && letName.charAt(letName.length - 1) === '*') {
                        explode = true;
                        letName = letName.substr(0, letName.length - 1);
                    }

                    if (letName.length > 0) {
                        const val = values ? values[letName] : undefined;
                        if (val || val === '') {
                            if (Array.isArray(val)) {
                                // RFC 6570, 2.4.1:
                                // "Prefix modifiers are not applicable to letiables that have composite values"
                                maxPrefixLength = -1;
                                for (let i = 0; i < val.length; i++) {
                                    appendValueToResult(String(val[i]));
                                }
                            } else if (typeof val === 'object' && val !== null) {
                                // appendKeyValueToResult ignores prefixes altogether
                                // so maxPrefixLength is irrelevant here
                                for (const propName in val) {
                                    // ignore prototype properties
                                    if (Object.prototype.hasOwnProperty.call(val, propName)) {
                                        appendKeyValueToResult(propName, String(val[propName]));
                                    }
                                }
                            } else {
                                appendValueToResult(String(val));
                            }
                        }
                    }

                    if (template.charAt(pos++) === '}') {
                        break;
                    }

                    letName = '';
                    topLevel = true;
                } else {
                    letName += template.charAt(pos++);
                }
            }
        }
    }

    return result;
};

// eslint-disable-next-line @typescript-eslint/no-namespace
// export namespace DCXUtil {
//     export const isBrowser = _isBrowser;
// }

export const getSuffixIdxOfMimeType = (type: string, suffix: string): number => {
    if (!type || !suffix) {
        return -1;
    }
    return type.indexOf(suffix);
};

export const getStartIdxOfMimeType = (type: string): number => {
    if (!type) {
        return -1;
    }
    return type.search('(text|image|audio|video|ingredient|document|model|application|font)\\/');
};

/**
 * Returns the first matching group from input string
 * @param {String} responseData response body
 * @param {string | RegExp} regexp   Regex pattern
 * @returns {String} pick the first matched group otherwise return empty string
 */
export const getFirstRegexpCapture = (responseData: string, regexp: string | RegExp): string => {
    if (!responseData || typeof responseData !== 'string' || !regexp) {
        return '';
    }
    let matcher: string[] | null;
    if (typeof regexp === 'string') {
        matcher = responseData.match(regexp);
    } else {
        matcher = regexp.exec(responseData);
    }
    return !matcher ? '' : matcher[1];
};
