/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2023 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { BasicLink, Link, RenditionLink, RenditionOptions, RequireAtLeastOne } from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { expandURITemplate } from './AdobeDCXUtil';
type RenditionLinkSelector = (links: RenditionLink[], options?: RenditionOptions) => RenditionLink;
type BasicLinkSelector<T> = (link: BasicLink[], options?: T) => BasicLink;
type LinkExpander<T> = (link: string, values?: T) => string;

export interface LinkProvider<T extends Record<string, any>> {
    // the type definition on this is a bit more complex than desired due to the optional nature of each of the options arguments
    selector?: T extends RequireAtLeastOne<Exclude<RenditionOptions, keyof BasicLink>, keyof RenditionOptions>
        ? RenditionLinkSelector
        : BasicLinkSelector<T>;
    expander?: LinkExpander<T>;
}

export function provideLink<T extends Record<string, any>>(
    links: BasicLink | BasicLink[],
    linkProvider?: LinkProvider<T>,
    opts?: T,
);
export function provideLink(
    links: RenditionLink | RenditionLink[],
    linkProvider?: LinkProvider<RenditionOptions>,
    opts?: RenditionOptions,
);
export function provideLink<T extends Record<string, any> = any>(
    links: Link | RenditionLink | RenditionLink[],
    linkProvider: LinkProvider<T> = {},
    opts?: T,
): string {
    const selector = typeof linkProvider.selector === 'function' ? linkProvider.selector : defaultSelector;
    const selectedLink = Array.isArray(links) ? selector(links, opts) : links;
    if (!selectedLink) {
        throw new DCXError(DCXError.INVALID_PARAMS, 'Could not select appropriate link for usage');
    }
    if (!selectedLink.templated) {
        return selectedLink.href;
    }

    if (typeof linkProvider.expander === 'function') {
        return linkProvider.expander(selectedLink.href, opts);
    }

    return expandURITemplate(selectedLink.href, opts);
}

function defaultSelector(links: RenditionLink[], opts?: RenditionOptions);
function defaultSelector(links: BasicLink[], opts?: Record<string, any>);
function defaultSelector(links: BasicLink[] | RenditionLink[], opts: Record<string, any> = { mode: 'id' }) {
    let selectedLink = links[0];
    let highestRank = 0;
    for (const link of links) {
        let rank = 0;
        for (const property in link) {
            if (property in opts && link[property] === opts[property]) {
                rank++;
            }
        }
        if (rank > highestRank) {
            highestRank = rank;
            selectedLink = link;
        }
    }
    return selectedLink;
}
