/* eslint-disable @typescript-eslint/ban-types */
/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { isArray } from './types';

type WatchHandler = (next: unknown, current?: unknown) => void;

interface _Watchable<T> {
    watchProperty(prop: keyof T, handler: (next: T[typeof prop], current?: T[typeof prop], target?: T) => void): void;
    revokeWatchProxy(): T;
}

export type Watchable<T> = _Watchable<T> & T;

/**
 * Create watch proxy around object.
 * Adds `watchProperty` method to the object.
 * Allows watching changes on properties.
 * Handlers are called just before applying the change to the object.
 *
 * @param {unknown} obj - Object to wrap in proxy
 */
export function WatchProxy<T extends object = {}>(obj: T): T & Watchable<T> {
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    if ((obj as any).__watch_proxy__) {
        // already proxied, return it
        return obj as T & Watchable<T>;
    }

    if ('watchProperty' in obj) {
        // not proxied, but key exists already
        throw new Error('Cannot proxy object, watchProperty already exists.');
    }

    let listeners = {} as Record<keyof T, WatchHandler[]>;

    const watchProperty = ((prop: keyof T, handler: WatchHandler) => {
        listeners[prop] = listeners[prop] || [];
        listeners[prop].push(handler);
    }).bind({ listeners });

    const trap: ProxyHandler<{}> = {
        set: function (target, prop, val) {
            // eslint-disable-next-line no-useless-catch
            try {
                if (prop in listeners && isArray(listeners[prop])) {
                    listeners[prop].map((h) => h.call(undefined, val, target[prop], target));
                }
            } catch (_) {
                // noop
                throw _;
            }
            target[prop] = val;
            return true;
        },
    };

    const { proxy, revoke } = Proxy.revocable<T>(obj, trap);

    const revokeWatchProxy = () => {
        revoke();

        listeners = {} as Record<keyof T, WatchHandler[]>;

        delete (obj as any).__watch_proxy__;
        delete (obj as any).revokeWatchProxy;
        delete (obj as any).watchProperty;

        return obj;
    };

    Object.defineProperty(proxy, '__watch_proxy__', {
        value: true,
        enumerable: false,
        configurable: true,
    });

    Object.defineProperty(proxy, 'revokeWatchProxy', {
        value: revokeWatchProxy,
        enumerable: false,
        configurable: true,
    });

    Object.defineProperty(proxy, 'watchProperty', {
        value: watchProperty,
        enumerable: false,
        configurable: true,
    });

    return proxy as T & Watchable<T>;
}
