/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { DCXError } from '@dcx/error';
import { _toUTF8Array } from './AdobeDCXUtil';

/**
 * Utility functions for converting between different data types
 */

export function arrayBufferToString(buff: ArrayBuffer | Buffer, numOfBytes?: number): string {
    if (!buff) {
        return '';
    }

    if (numOfBytes != null) {
        buff = buff.slice(0, numOfBytes);
    }

    if (process.env.APP_ENV === 'browser') {
        const dec = new TextDecoder();
        return dec.decode(buff);
    }

    if (buff instanceof Buffer) {
        return buff.toString('utf8');
    }
    try {
        return Buffer.from(buff).toString('utf8');
    } catch (e) {
        throw new DCXError(DCXError.INVALID_DATA, (e as Error).message);
    }
}

export function stringToBuffer(data: string): Buffer | Uint8Array {
    if (process.env.APP_ENV === 'browser') {
        const enc = new TextEncoder();
        return enc.encode(data);
    }

    return Buffer.from(_toUTF8Array(data));
}

export function dataToBuffer(data: Buffer | Blob | ArrayBuffer | string): undefined | Buffer | Blob | ArrayBuffer {
    if (typeof data === 'string') {
        return stringToBuffer(data);
    }

    return data;
}

export function concatUint8Arrays(array1: Uint8Array, array2: Uint8Array): Uint8Array {
    if (process.env.APP_ENV === 'node') {
        return Buffer.concat([array1, array2], array1.length + array2.length);
    }
    const result = new Uint8Array(array1.length + array2.length);
    result.set(array1, 0);
    result.set(array2, array1.length);
    return result;
}

export function objectFromEntries<T = any>(
    entries: IterableIterator<[string, T]> | ReturnType<typeof Object.entries>,
): Record<string, T> {
    const obj = {};
    for (const [k, v] of entries) {
        obj[k] = v;
    }
    return obj;
}

/**
 * Converts an array of items into an array of subarrays. Each subarray should have a maximum length of chunkSize
 * Ordering of subarrays is not guaranteed to match the ordering of the original array.
 * ex: `chunk([1,2,3,4,5], 2) // [[5],[3,4],[1,2]]`
 * @param array
 * @param chunkSize
 * @returns
 */
export function chunkArray<T extends any[]>(array: T, chunkSize: number) {
    return array
        .reduce((chunks, val, i) => {
            if (i % chunkSize === 0) {
                chunks.unshift([val]);
            } else {
                chunks[0].push(val);
            }
            return chunks;
        }, [] as T[])
        .reverse();
}

/**
 * Converts a ReadableStream to an Async Iterable using a Generator
 * This functionality will hopefulyl be built directly into the WHATWG ReadableStreams in the future
 * Some tracking information can be found in the referenced link.
 * @see {@link https://github.com/whatwg/streams/issues/778#issuecomment-461341033 | WHATWG Stream Async Iterator Issue}
 * @param stream the stream to read from
 */
export async function* asyncIterableFromStream(stream) {
    const reader = stream.getReader();

    try {
        while (true) {
            const { done, value } = await reader.read();
            if (done) {
                return;
            }
            yield value;
        }
    } finally {
        reader.releaseLock();
    }
}
