/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { isArray } from './types';

/*jslint regexp: true*/
const parseHeaderPattern = /^([^:]+):(.*)$/;
const whiteSpaceFilterPattern = /^\s+|\s+$/g;
/*jslint regexp: false*/
/**
 * Parses the http headers in allHeadersAsText as per http://www.w3.org/Protocols/rfc2616/rfc2616-sec4.html and returns
 * a dictionary with the lower-cased header fields as keys and the header values as values.
 * @private
 * @param   {String} allHeadersAsText
 * @returns {Object}
 */
export const parseHeaders = (allHeadersAsText: string): Record<string, string> => {
    const headers = {};
    const allHeaderLines = allHeadersAsText.split(/\r?\n/);
    let field, value;
    for (let i = 0; i < allHeaderLines.length; ++i) {
        const line = allHeaderLines[i];
        if (line.length > 0) {
            const charCodeOfFirstCharacter = line.charCodeAt(0);

            if (field && (charCodeOfFirstCharacter === 9 || charCodeOfFirstCharacter === 32)) {
                // If a line starts with a tab or a space then it is a continuation of a previous header
                headers[field] = headers[field] + ' ' + line.replace(whiteSpaceFilterPattern, '');
            } else {
                // A new header
                const m = parseHeaderPattern.exec(line);
                if (m && m.length > 1) {
                    field = m[1].toLowerCase();
                    value = m[2] || '';
                    value = value.replace(whiteSpaceFilterPattern, '');
                    if (headers[field]) {
                        headers[field] = headers[field] + ',' + value;
                    } else {
                        headers[field] = value;
                    }
                }
            }
        }
    }
    return headers;
};

/**
 * Turns header keys to lowercase and removes undefined value entries.
 *
 * @param headers - incoming headers to be normalized
 */
export const normalizeHeaders = (headers: Record<string, string | string[]>): Record<string, string> => {
    if (headers === null || typeof headers !== 'object') {
        return {};
    }
    const normalizedHeaders = {};
    for (const [key, value] of Object.entries(headers)) {
        normalizedHeaders[key.toLowerCase()] = isArray(value) ? value.join(';') : value;
    }
    return normalizedHeaders;
};
