/**
 * @ref https://github.com/chbrown/rfc6902
 */

import { JSONPatchDocument } from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { isArray, isObject } from './types';
import { validateParams } from './validation';

type JSONPatchDiff = (input: any, output: any, ptr: JSONPatchPointer) => JSONPatchDocument;

/**
 * Callback to allow clients to provide a partial `diff()` function.
 * If this callback returns void, we fall back to the built-in `diffAny()`.
 */
export type JSONPatchVoidableDiff = (input: any, output: any, ptr: JSONPatchPointer) => JSONPatchDocument | void;

/**
 * Produce a 'application/json-patch+json'-type patch to get from one object to another.
 * This does not alter `input` or `output` unless they have a property getter with
 * side-effects (which is not a good idea anyway).
 *
 * `diff` is called on each pair of comparable non-primitive nodes in the
 * `input`/`output` object trees, producing nested patches.
 *
 * Return `undefined` to fall back to default behaviour.
 *
 * Returns list of operations to perform on `input` to produce `output`.
 *
 * See {@link https://tools.ietf.org/html/rfc6902 | RFC6902}
 *
 * @param   input   - The "before patch" object
 * @param   output  - The "after patch" object
 * @param   [diff]  - An optional callback for each non-primitive node comparison
 *                    Useful if, for example, you are patching classes.
 */
export function createJSONPatch(input: any, output: any, diff?: JSONPatchVoidableDiff): JSONPatchDocument {
    const ptr = new JSONPatchPointer();
    // a new Pointer gets a default path of [''] if not specified
    return (diff ? wrapVoidableDiff(diff) : diffAny)(input, output, ptr);
}

/**
 * List the keys in `minuend` that are not in `subtrahend`.
 *
 * A key is only considered if it is both:
 * 1) an own-property (o.hasOwnProperty(k)) of the object
 * 2) has a value that is not undefined.
 * This is to match JSON semantics, where JSON object
 * serialization drops keys with undefined values.
 *
 * @param minuend       Object of interest
 * @param subtrahend    Object of comparison
 * @returns Array of keys that are in `minuend` but not in `subtrahend`.
 */
function subtract(minuend: { [index: string]: any }, subtrahend: { [index: string]: any }): string[] {
    // initialize empty object; we only care about the keys, the values can be anything
    const obj: { [index: string]: number } = {};
    // build up obj with all the properties of minuend
    for (const addKey in minuend) {
        if (Object.prototype.hasOwnProperty.call(minuend, addKey) && minuend[addKey] !== undefined) {
            obj[addKey] = 1;
        }
    }
    // now delete all the properties of subtrahend from obj
    // (deleting a missing key has no effect)
    for (const delKey in subtrahend) {
        if (Object.prototype.hasOwnProperty.call(subtrahend, delKey) && subtrahend[delKey] !== undefined) {
            delete obj[delKey];
        }
    }
    // finally, extract whatever keys remain in obj
    return Object.keys(obj);
}

/**
 * List the keys that shared by all `objects`.
 * The semantics of what constitutes a "key" is described in {@link subtract}.
 *
 * @param objects Array of objects to compare
 * @returns Array of keys that are in ("own-properties" of) every object in `objects`.
 */
function intersection(objects: ArrayLike<{ [index: string]: any }>): string[] {
    const length = objects.length;
    // prepare empty counter to keep track of how many objects each key occurred in
    const counter: { [index: string]: number } = {};
    // go through each object and increment the counter for each key in that object
    for (let i = 0; i < length; i++) {
        const object = objects[i];
        for (const key in object) {
            if (Object.prototype.hasOwnProperty.call(object, key) && object[key] !== undefined) {
                counter[key] = (counter[key] || 0) + 1;
            }
        }
    }
    // now delete all keys from the counter that were not seen in every object
    for (const key in counter) {
        if (counter[key] < length) {
            delete counter[key];
        }
    }
    // finally, extract whatever keys remain in the counter
    return Object.keys(counter);
}

interface ArrayAdd {
    op: 'add';
    index: number;
    value: any;
}
interface ArrayRemove {
    op: 'remove';
    index: number;
}
interface ArrayReplace {
    op: 'replace';
    index: number;
    original: any;
    value: any;
}
/**
 * These are not proper JSONPatchOperation objects, but will be converted into
 * JSONPatchOperation objects eventually.
 *
 * {index} indicates the actual target position, never "end of array".
 */
type ArrayOperation = ArrayAdd | ArrayRemove | ArrayReplace;
function isArrayAdd(arrayOperation: ArrayOperation): arrayOperation is ArrayAdd {
    return arrayOperation.op === 'add';
}
function isArrayRemove(arrayOperation: ArrayOperation): arrayOperation is ArrayRemove {
    return arrayOperation.op === 'remove';
}

interface DynamicAlternative {
    operations: ArrayOperation[];
    /**
     * Indicates the total cost of getting to this position.
     */
    cost: number;
}

function appendArrayOperation(base: DynamicAlternative, operation: ArrayOperation): DynamicAlternative {
    return {
        // the new operation must be pushed on the end
        operations: base.operations.concat(operation),
        cost: base.cost + 1,
    };
}

/**
 * Calculate the shortest sequence of operations to get from `input` to `output`,
 * using a dynamic programming implementation of the Levenshtein distance algorithm.
 *
 * @example
 *           output
 *                A   Z
 *                -   -
 *           [0]  1   2
 * input A |  1  [0]  1
 *       B |  2  [1]  1
 *       C |  3   2  [2]
 * 1) start at 0,0 (+0)
 * 2) keep A (+0)
 * 3) remove B (+1)
 * 4) replace C with Z (+1)
 *
 * If the `input` (source) is empty, they'll all be in the top row, resulting in an
 * array of 'add' operations.
 *
 * If the `output` (target) is empty, everything will be in the left column,
 * resulting in an array of 'remove' operations.
 *
 * @param input   - Input array
 * @param output  - Output array
 * @param ptr     - Patch document pointer
 * @param [diff]  - Diff function to use
 * @returns A list of add/remove/replace operations.
 */
function diffArrays<T>(input: T[], output: T[], ptr: JSONPatchPointer, diff: JSONPatchDiff): JSONPatchDocument {
    // set up cost matrix (very simple initialization: just a map)
    const memo: { [index: string]: DynamicAlternative } = {
        '0,0': { operations: [], cost: 0 },
    };
    /**
     * Calculate the cheapest sequence of operations required to get from
     * input.slice(0, i) to output.slice(0, j).
     * There may be other valid sequences with the same cost, but none cheaper.
     * @param i The row in the layout above
     * @param j The column in the layout above
     * @returns An object containing a list of operations, along with the total cost
     *          of applying them (+1 for each add/remove/replace operation)
     */
    function dist(i: number, j: number): DynamicAlternative {
        // memoized
        const memoKey = `${i},${j}`;
        let memoized = memo[memoKey];
        if (memoized === undefined) {
            // TODO: this !diff(...).length usage could/should be lazy
            if (i > 0 && j > 0 && !diff(input[i - 1], output[j - 1], new JSONPatchPointer()).length) {
                // equal (no operations => no cost)
                memoized = dist(i - 1, j - 1);
            } else {
                const alternatives: DynamicAlternative[] = [];
                if (i > 0) {
                    // NOT topmost row
                    const removeBase = dist(i - 1, j);
                    const removeOperation: ArrayRemove = {
                        op: 'remove',
                        index: i - 1,
                    };
                    alternatives.push(appendArrayOperation(removeBase, removeOperation));
                }
                if (j > 0) {
                    // NOT leftmost column
                    const addBase = dist(i, j - 1);
                    const addOperation: ArrayAdd = {
                        op: 'add',
                        index: i - 1,
                        value: output[j - 1],
                    };
                    alternatives.push(appendArrayOperation(addBase, addOperation));
                }
                if (i > 0 && j > 0) {
                    // TABLE MIDDLE
                    // supposing we replaced it, compute the rest of the costs:
                    const replaceBase = dist(i - 1, j - 1);
                    // okay, the general plan is to replace it, but we can be smarter,
                    // recursing into the structure and replacing only part of it if
                    // possible, but to do so we'll need the original value
                    const replaceOperation: ArrayReplace = {
                        op: 'replace',
                        index: i - 1,
                        original: input[i - 1],
                        value: output[j - 1],
                    };
                    alternatives.push(appendArrayOperation(replaceBase, replaceOperation));
                }
                // the only other case, i === 0 && j === 0, has already been memoized

                // the meat of the algorithm:
                // sort by cost to find the lowest one (might be several ties for lowest)
                // [4, 6, 7, 1, 2].sort((a, b) => a - b) -> [ 1, 2, 4, 6, 7 ]
                const best = alternatives.sort((a, b) => a.cost - b.cost)[0];
                memoized = best;
            }
            memo[memoKey] = memoized;
        }
        return memoized;
    }
    // handle weird objects masquerading as Arrays that don't have proper length
    // properties by using 0 for everything but positive numbers
    const inputLength = isNaN(input.length) || input.length <= 0 ? 0 : input.length;
    const outputLength = isNaN(output.length) || output.length <= 0 ? 0 : output.length;
    const arrayOperations = dist(inputLength, outputLength).operations;
    const [paddedOperations] = arrayOperations.reduce<[JSONPatchDocument, number]>(
        ([operations, padding], arrayOperation) => {
            if (isArrayAdd(arrayOperation)) {
                const paddedIndex = arrayOperation.index + 1 + padding;
                const indexToken = paddedIndex < inputLength + padding ? String(paddedIndex) : '-';
                const operation = {
                    op: arrayOperation.op,
                    path: ptr.add(indexToken).toString(),
                    value: arrayOperation.value,
                };
                // padding++ // maybe only if array_operation.index > -1 ?
                return [operations.concat(operation), padding + 1];
            } else if (isArrayRemove(arrayOperation)) {
                const operation = {
                    op: arrayOperation.op,
                    path: ptr.add(String(arrayOperation.index + padding)).toString(),
                };
                // padding--
                return [operations.concat(operation), padding - 1];
            }
            // replace
            const replacePtr = ptr.add(String(arrayOperation.index + padding));
            const replaceOperations = diff(arrayOperation.original, arrayOperation.value, replacePtr);
            return [operations.concat(...replaceOperations), padding];
        },
        [[], 0],
    );
    return paddedOperations;
}

function diffObjects(input: any, output: any, ptr: JSONPatchPointer, diff: JSONPatchDiff): JSONPatchDocument {
    // if a key is in input but not output -> remove it
    const operations: JSONPatchDocument = [];
    subtract(input, output).forEach((key) => {
        operations.push({ op: 'remove', path: ptr.add(key).toString() });
    });
    // if a key is in output but not input -> add it
    subtract(output, input).forEach((key) => {
        operations.push({ op: 'add', path: ptr.add(key).toString(), value: output[key] });
    });
    // if a key is in both, diff it recursively
    intersection([input, output]).forEach((key) => {
        operations.push(...diff(input[key], output[key], ptr.add(key)));
    });
    return operations;
}

/**
 * `diffAny()` returns an empty array if `input` and `output` are materially equal
 * (i.e., would produce equivalent JSON); otherwise it produces an array of patches
 * that would transform `input` into `output`.
 *
 * > Here, "equal" means that the value at the target location and the
 * > value conveyed by "value" are of the same JSON type, and that they
 * > are considered equal by the following rules for that type:
 * > o  strings: are considered equal if they contain the same number of
 * >    Unicode characters and their code points are byte-by-byte equal.
 * > o  numbers: are considered equal if their values are numerically
 * >    equal.
 * > o  arrays: are considered equal if they contain the same number of
 * >    values, and if each value can be considered equal to the value at
 * >    the corresponding position in the other array, using this list of
 * >    type-specific rules.
 * > o  objects: are considered equal if they contain the same number of
 * >    members, and if each member can be considered equal to a member in
 * >    the other object, by comparing their keys (as strings) and their
 * >    values (using this list of type-specific rules).
 * > o  literals (false, true, and null): are considered equal if they are
 * >    the same.
 */
function diffAny(input: any, output: any, ptr: JSONPatchPointer, diff: JSONPatchDiff = diffAny): JSONPatchDocument {
    // strict equality handles literals, numbers, and strings (a sufficient but not necessary cause)
    if (input === output) {
        return [];
    }

    const inIsArr = isArray(input);
    const outIsArr = isArray(output);
    if (inIsArr && outIsArr) {
        return diffArrays(input, output, ptr, diff);
    }
    if (!inIsArr && !outIsArr && isObject(input) && isObject(output)) {
        return diffObjects(input, output, ptr, diff);
    }
    // at this point we know that input and output are materially different;
    // could be array -> object, object -> array, boolean -> undefined,
    // number -> string, or some other combination, but nothing that can be split
    // up into multiple patches: so `output` must replace `input` entirely.
    return [{ op: 'replace', path: ptr.toString(), value: output }];
}
/**
 * Unescape token part of a JSON Pointer string
 * `token` should *not* contain any '/' characters.
 * > Evaluation of each reference token begins by decoding any escaped
 * > character sequence.  This is performed by first transforming any
 * > occurrence of the sequence '~1' to '/', and then transforming any
 * > occurrence of the sequence '~0' to '~'.  By performing the
 * > substitutions in this order, an implementation avoids the error of
 * > turning '~01' first into '~1' and then into '/', which would be
 * > incorrect (the string '~01' correctly becomes '~1' after
 * > transformation).
 *
 * Assumes:
 * ~1 is unescaped with higher priority than ~0 because it is a lower-order escape character.
 * "lower order" because '/' needs escaping due to the JSON Pointer serialization technique.
 * Whereas, '~' is escaped because escaping '/' uses the '~' character.
 */
function unescape(token: string): string {
    return token.replace(/~1/g, '/').replace(/~0/g, '~');
}

/**
 * Escape token part of a JSON Pointer string
 * > '~' needs to be encoded as '~0' and '/'
 * > needs to be encoded as '~1' when these characters appear in a
 * > reference token.
 *
 * This is the exact inverse of `unescape()`, so the reverse
 * replacements must take place in reverse order.
 */
function escape(token: string): string {
    return token.replace(/~/g, '~0').replace(/\//g, '~1');
}

interface JSONPatchPointerEvaluation {
    parent: any;
    key: string;
    value: any;
}

/**
 * JSON Pointer representation
 */
export class JSONPatchPointer {
    constructor(public tokens = ['']) {}
    /**
     * `path` *must* be a properly escaped string.
     */
    static fromJSON(path: string): JSONPatchPointer {
        const tokens = path.split('/').map(unescape);
        if (tokens[0] !== '') {
            throw new DCXError(DCXError.INVALID_DATA, `Invalid JSON Pointer: ${path}`);
        }
        return new JSONPatchPointer(tokens);
    }
    toString(): string {
        return this.tokens.map(escape).join('/');
    }
    /**
     * Returns an object with 'parent', 'key', and 'value' properties.
     * In the special case that this Pointer's path == "",
     * this object will be {parent: null, key: '', value: object}.
     * Otherwise, parent and key will have the property such that parent[key] == value.
     */
    evaluate(object: any): JSONPatchPointerEvaluation {
        let parent: any = null;
        let key = '';
        let value = object;
        for (let i = 1, l = this.tokens.length; i < l; i++) {
            parent = value;
            key = this.tokens[i];
            /* istanbul ignore next */
            value = (parent || {})[key];
        }
        return { parent, key, value };
    }
    get(object: any): any {
        return this.evaluate(object).value;
    }
    set(object: any, value: any): void {
        let cursor: any = object;
        for (let i = 1, l = this.tokens.length - 1, token = this.tokens[i]; i < l; i++) {
            /* istanbul ignore next */
            cursor = (cursor || {})[token];
        }
        if (cursor) {
            cursor[this.tokens[this.tokens.length - 1]] = value;
        }
    }
    /* istanbul ignore next */
    push(token: string): void {
        // mutable
        this.tokens.push(token);
    }
    /**
     * `token` should be a String. It'll be coerced to one anyway.
     * immutable (shallowly)
     */
    add(token: string): JSONPatchPointer {
        const tokens = this.tokens.concat(String(token));
        return new JSONPatchPointer(tokens);
    }
}

/* istanbul ignore next */
function wrapVoidableDiff(diff: JSONPatchVoidableDiff): JSONPatchDiff {
    function wrappedDiff(input: any, output: any, ptr: JSONPatchPointer): JSONPatchDocument {
        const customPatch = diff(input, output, ptr);
        // ensure an array is always returned
        return isArray(customPatch) ? customPatch : diffAny(input, output, ptr, wrappedDiff);
    }
    return wrappedDiff;
}

class PatchDocumentBuilder {
    /**
     * The operations included in the current document.
     */
    private _operations: JSONPatchDocument;

    /**
     * Get the document operations as an array of objects
     */
    public get operations(): JSONPatchDocument {
        return this._operations;
    }

    constructor(initialDocument: JSONPatchDocument = []) {
        this._operations = initialDocument;
    }

    /**
     * Get the document as a string
     */
    public getDocument(): string {
        return JSON.stringify(this._operations);
    }

    /**
     * Add "add" operation to document.
     *
     * @param {string}  path        Path to add value to.
     * @param {unknown} value       Value to add.
     */
    public add(path: string, value: unknown): PatchDocumentBuilder {
        validateParams(['path', path, 'string']);

        this._operations.push({ op: 'add', path, value });

        return this;
    }

    /**
     * Add a value to a list.
     * Handles adding the index to the end of the path.
     *
     * @param {string} path                     Path to the array to add to.
     * @param {unknown} value                   Value to add to the list.
     * @param {number | 'end' | '-'} index      If a number is provided, used as the index at which to add the value.
     *                                          Otherwise adds the value to the end of the list.
     */
    public addToList(path: string, value: unknown, index: number | 'end' | '-'): PatchDocumentBuilder {
        validateParams(['path', path, 'string']);

        if (typeof index !== 'number' && index !== 'end' && index !== '-') {
            throw new DCXError(
                DCXError.INVALID_PARAMS,
                'Index must be a number, or either "end"/"-" to append to end.',
            );
        }

        this._operations.push({
            op: 'add',
            value,
            path: `${path.endsWith('/') ? path : path + '/'}${index === 'end' ? '-' : index}`,
        });

        return this;
    }

    /**
     * Add "remove" operation to document.
     *
     * @param {string}  path        Path to property to remove.
     */
    public remove(path: string): PatchDocumentBuilder {
        validateParams(['path', path, 'string']);

        this._operations.push({ op: 'remove', path });

        return this;
    }

    /**
     * Add "replace" operation to document.
     * @param {string}  path        Path to property to replace.
     * @param {unknown} value       New value to place at the path.
     */
    public replace(path: string, value: unknown): PatchDocumentBuilder {
        validateParams(['path', path, 'string']);

        this._operations.push({ op: 'replace', path, value });

        return this;
    }

    /**
     * Add "move" operation to document.
     *
     * @param {string}  source          Path to property to move.
     * @param {string}  destination     Path to move the property to.
     */
    public move(source: string, destination: string): PatchDocumentBuilder {
        validateParams(['source', source, 'string'], ['destination', destination, 'string']);

        this._operations.push({ op: 'move', from: source, path: destination });

        return this;
    }

    /**
     * Add "copy" operation to document.
     *
     * @param {string}  source          Path to property to copy.
     * @param {string}  destination     Path to copy the property to.
     */
    public copy(source: string, destination: string): PatchDocumentBuilder {
        validateParams(['source', source, 'string'], ['destination', destination, 'string']);

        this._operations.push({ op: 'copy', from: source, path: destination });

        return this;
    }
}

/**
 * Create an instance of PatchDocumentBuilder
 */
export function createPatchDocumentBuilder(initialDocument?: JSONPatchDocument) {
    return new PatchDocumentBuilder(initialDocument);
}

interface InvalidPropertyIssue {
    property: string;
    issue: string;
}

function validationIssue(property: string, issue: string): InvalidPropertyIssue {
    return {
        property,
        issue,
    };
}

/**
 * Perform simple validation on a JSON Patch document.
 *
 * This only validates that the individual operations are valid,
 * it does not validate that more complex ordering will fail, even
 * if that is statically known.
 *
 * @param doc
 * @returns {true | InvalidPropertyIssue[]}
 */
export function validateJSONPatchDocument(doc: unknown): true | InvalidPropertyIssue[] {
    if (!isArray(doc)) {
        return [validationIssue('doc', 'Invalid type.')];
    }
    const issues: InvalidPropertyIssue[] = [];
    doc.forEach((o, i) => {
        const res = validateJSONPatchOperation(o);
        if (res === true) {
            return;
        }
        res.forEach((issue) => {
            issue.property = `doc[${i}].${issue.property}`;
            issues.push(issue);
        });
    });
    return issues.length > 0 ? issues : true;
}

/**
 * Validate individual JSON Patch operation.
 *
 * @param op
 * @returns {boolean}
 */
export function validateJSONPatchOperation(op: any): true | InvalidPropertyIssue[] {
    const issues: InvalidPropertyIssue[] = [];

    if (typeof op !== 'object' || op == null || isArray(op)) {
        issues.push(validationIssue('*', 'Operation not an object.'));
        return issues;
    }

    if (typeof op.path !== 'string') {
        issues.push(validationIssue('path', 'Missing or invalid.'));
    }

    switch (op.op) {
        case 'remove':
            break;
        case 'add':
        case 'replace':
            // value must be set
            if (op.value == null) {
                issues.push(validationIssue('value', 'Missing or invalid. Required for add/replace.'));
            }
            break;
        case 'move':
        case 'copy':
            // from must be set
            if (typeof op.from !== 'string') {
                issues.push(validationIssue('from', 'Missing or invalid. Required for move/copy.'));
            }
            break;
        default:
            issues.push(validationIssue('op', 'Missing or invalid.'));
            break;
    }
    return issues.length > 0 ? issues : true;
}
