/**
 * ref: https://github.com/jhermsmeier/node-http-link-header/blob/master/LICENSE.md
 */

/* eslint-disable no-control-regex */
/* eslint-disable no-bitwise */

import { AdobeAssetWithLinks, LinkMode, LinkSet } from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { expandURITemplate } from './AdobeDCXUtil';
import { pruneUndefined } from './object';
import { isArray, isObject } from './types';

const COMPATIBLE_ENCODING_PATTERN = /^utf-?8|ascii|utf-?16-?le|ucs-?2|base-?64|latin-?1$/i;
const WS_TRIM_PATTERN = /^[\s\uFEFF\xA0]+|[\s\uFEFF\xA0]+$/g;
const WS_CHAR_PATTERN = /\s|\uFEFF|\xA0/;
const WS_FOLD_PATTERN = /\r?\n[\x20\x09]+/g;
const DELIMITER_PATTERN = /[;,"]/;
const WS_DELIMITER_PATTERN = /[;,"]|\s/;

const STATE = {
    IDLE: 1 << 0,
    URI: 1 << 1,
    ATTR: 1 << 2,
};

function trim(value) {
    return value.replace(WS_TRIM_PATTERN, '');
}

function hasWhitespace(value) {
    return WS_CHAR_PATTERN.test(value);
}

function skipWhitespace(value, offset) {
    while (hasWhitespace(value[offset])) {
        offset++;
    }
    return offset;
}

function needsQuotes(value) {
    return WS_DELIMITER_PATTERN.test(value);
}

export class Link {
    /** @type {Array} URI references */
    public refs: Record<string, any>[] = [];

    /**
     * Link
     * @constructor
     * @param {String} [value]
     * @returns {Link}
     */
    constructor(value?: string) {
        if (value) {
            this.parse(value);
        }
    }

    /**
     * Get refs with given relation type
     * @param {String} value
     * @returns {Array<Object>}
     */
    rel(value) {
        const links: Record<string, unknown>[] = [];

        for (let i = 0; i < this.refs.length; i++) {
            if (this.refs[i].rel === value) {
                links.push(this.refs[i]);
            }
        }

        return links;
    }

    /**
     * Get refs where given attribute has a given value
     * @param {String} attr
     * @param {String} value
     * @returns {Array<Object>}
     */
    get(attr: string, value: string): Record<string, any>[] {
        attr = attr.toLowerCase();

        const links: Record<string, unknown>[] = [];

        for (let i = 0; i < this.refs.length; i++) {
            if (this.refs[i][attr] === value) {
                links.push(this.refs[i]);
            }
        }

        return links;
    }

    set(link: Record<string, any>): Link {
        this.refs.push(link);
        return this;
    }

    has(attr: string, value: string): boolean {
        attr = attr.toLowerCase();

        for (let i = 0; i < this.refs.length; i++) {
            if (this.refs[i][attr] === value) {
                return true;
            }
        }

        return false;
    }

    parse(pValue: string, pOffset = 0) {
        let value = pOffset ? pValue.slice(pOffset) : pValue;

        // Trim & unfold folded lines
        value = trim(value).replace(WS_FOLD_PATTERN, '');

        let state = STATE.IDLE;
        const length = value.length;
        let offset = 0;
        let ref: null | Record<string, unknown | unknown[]> = null;

        while (offset < length) {
            if (state === STATE.IDLE) {
                if (hasWhitespace(value[offset])) {
                    offset++;
                    continue;
                } else if (value[offset] === '<') {
                    const end = value.indexOf('>', offset);
                    if (end === -1) {
                        throw new Error('Expected end of URI delimiter at offset ' + offset);
                    }
                    ref = { uri: value.slice(offset + 1, end) };
                    this.refs.push(ref);
                    offset = end;
                    state = STATE.URI;
                } else {
                    throw new Error('Unexpected character "' + value[offset] + '" at offset ' + offset);
                }
                offset++;
            } else if (state === STATE.URI) {
                if (hasWhitespace(value[offset])) {
                    offset++;
                    continue;
                } else if (value[offset] === ';') {
                    state = STATE.ATTR;
                    offset++;
                } else if (value[offset] === ',') {
                    state = STATE.IDLE;
                    offset++;
                } else {
                    throw new Error('Unexpected character "' + value[offset] + '" at offset ' + offset);
                }
            } else if (state === STATE.ATTR) {
                if (value[offset] === ';' || hasWhitespace(value[offset])) {
                    offset++;
                    continue;
                }
                const end = value.indexOf('=', offset);
                if (end === -1) {
                    throw new Error('Expected attribute delimiter at offset ' + offset);
                }
                const attr = trim(value.slice(offset, end)).toLowerCase();
                let attrValue = '';
                offset = end + 1;
                offset = skipWhitespace(value, offset);
                if (value[offset] === '"') {
                    offset++;
                    while (offset < length) {
                        if (value[offset] === '"') {
                            offset++;
                            break;
                        }
                        if (value[offset] === '\\') {
                            offset++;
                        }
                        attrValue += value[offset];
                        offset++;
                    }
                } else {
                    let end = offset + 1;
                    while (!DELIMITER_PATTERN.test(value[end]) && end < length) {
                        end++;
                    }
                    attrValue = value.slice(offset, end);
                    offset = end;
                }
                if (ref && ref[attr] && isSingleOccurenceAttr(attr)) {
                    // Ignore multiples of attributes which may only appear once
                } else if (ref && attr[attr.length - 1] === '*') {
                    ref[attr] = parseExtendedValue(attrValue);
                } else {
                    attrValue = attr === 'rel' || attr === 'type' ? attrValue.toLowerCase() : attrValue;
                    if (ref && ref[attr] != null) {
                        const maybeArr = ref[attr];
                        if (isArray(maybeArr)) {
                            maybeArr.push(attrValue);
                        } else {
                            ref[attr] = [ref[attr], attrValue];
                        }
                    } else if (ref) {
                        ref[attr] = attrValue;
                    } else {
                        throw new Error('Unexpected null ref');
                    }
                }
                switch (value[offset]) {
                    case ',':
                        state = STATE.IDLE;
                        break;
                    case ';':
                        state = STATE.ATTR;
                        break;
                }
                offset++;
            } else {
                throw new Error('Unknown parser state "' + state + '"');
            }
        }

        ref = null;

        return this;
    }

    toString() {
        const refs: string[] = [];
        let link = '';
        let ref: Record<string, any>;

        for (let i = 0; i < this.refs.length; i++) {
            ref = this.refs[i];
            // eslint-disable-next-line no-loop-func
            link = Object.keys(this.refs[i]).reduce(function (link, attr) {
                if (attr === 'uri') {
                    return link;
                }
                return link + '; ' + formatAttribute(attr, ref[attr]);
            }, '<' + ref.uri + '>');
            refs.push(link);
        }

        return refs.join(', ');
    }
}

/**
 * Determines whether an encoding can be
 * natively handled with a `Buffer`
 * @param {String} value
 * @returns {Boolean}
 */
const isCompatibleEncoding = (value) => {
    return COMPATIBLE_ENCODING_PATTERN.test(value);
};

const isSingleOccurenceAttr = (attr) => {
    return attr === 'rel' || attr === 'type' || attr === 'media' || attr === 'title' || attr === 'title*';
};

const isTokenAttr = (attr) => {
    return attr === 'rel' || attr === 'type' || attr === 'anchor';
};

const escapeQuotes = (value) => {
    return value.replace(/"/g, '\\"');
};

/**
 * Parses an extended value and attempts to decode it
 * @internal
 * @param {String} value
 * @return {Object}
 */
const parseExtendedValue = (value) => {
    const parts = /([^']+)?(?:'([^']+)')?(.+)/.exec(value) || [];
    return {
        language: parts[2].toLowerCase(),
        encoding: isCompatibleEncoding(parts[1]) ? null : parts[1].toLowerCase(),
        value: isCompatibleEncoding(parts[1]) ? decodeURIComponent(parts[3]) : parts[3],
    };
};

/**
 * Format a given extended attribute and it's value
 * @param {String} attr
 * @param {Object} data
 * @return {String}
 */
const formatExtendedAttribute = (attr, data) => {
    const encoding = (data.encoding || 'utf-8').toUpperCase();
    const language = data.language || 'en';

    let encodedValue = '';

    if (Buffer.isBuffer(data.value) && isCompatibleEncoding(encoding)) {
        encodedValue = data.value.toString(encoding);
    } else if (Buffer.isBuffer(data.value)) {
        encodedValue = data.value.toString('hex').replace(/[0-9a-f]{2}/gi, '%$1');
    } else {
        encodedValue = encodeURIComponent(data.value);
    }

    return attr + '=' + encoding + "'" + language + "'" + encodedValue;
};

/**
 * Format a given attribute and it's value
 * @param {String} attr
 * @param {String|Object} value
 * @return {String}
 */
const formatAttribute = (attr, value) => {
    if (Array.isArray(value)) {
        return value
            .map((item) => {
                return formatAttribute(attr, item);
            })
            .join('; ');
    }

    if (attr[attr.length - 1] === '*' || typeof value !== 'string') {
        return formatExtendedAttribute(attr, value);
    }

    if (isTokenAttr(attr)) {
        value = needsQuotes(value) ? '"' + escapeQuotes(value) + '"' : escapeQuotes(value);
    } else if (needsQuotes(value)) {
        value = encodeURIComponent(value);
        // We don't need to escape <SP> <,> <;> within quotes
        value = value.replace(/%20/g, ' ').replace(/%2C/g, ',').replace(/%3B/g, ';');

        value = '"' + value + '"';
    }

    return attr + '=' + value;
};

/**
 * Generic link parser to get a specific property value from a link relationship
 * @param links             A LinkSet provided in any of the formats in which they may be stored on a dcx-js class
 * @param relationship      The relationship key
 * @param property          The property to pull from the relationship
 * @param linkMode          If the relationship contains an array of links, allows specifying the exact mode to return the property for.
 * @return                  The property if the relationship and property exist on the provided links. Else undefined
 */

export function getLinkProperty(
    links: AnyLinkSetInput,
    relationship: string,
    property: string,
    linkMode: LinkMode = 'id',
): string | undefined {
    const _links = _getLinks(links);

    const link = _links[relationship];
    if (link) {
        if (Array.isArray(link)) {
            const modeLink = link.filter((basicLink) => {
                return basicLink.mode === linkMode;
            });
            return modeLink.length > 0 ? modeLink[0][property] : link.length > 0 ? link[0][property] : undefined;
        }
        return link[property];
    }
    return undefined;
}

/**
 * Generic link parser to get the href property of a link relationship
 * @param links             A LinkSet provided in any of the formats in which they may be stored on a dcx-js class
 * @param relationship      The relationship key
 * @param linkMode          If the relationship contains an array of links, allows specifying the exact mode to return the property for.
 * @return                  The href if the relationship and href exists on the provided links. Else undefined
 */
export function getLinkHref(links: AnyLinkSetInput, relationship: string, linkMode: LinkMode = 'id'): string {
    const _links = _getLinks(links);
    const link = _links[relationship];

    let href: string | undefined;

    if (isObject(link) && typeof link.href === 'string') {
        href = link.href;
    } else if (Array.isArray(link)) {
        href = getLinkProperty(_links, relationship, 'href', linkMode);
    }

    if (typeof href !== 'string') {
        throw new DCXError(DCXError.INVALID_PARAMS, 'Missing or invalid link href.');
    }
    return href;
}

/**
 * Generic link parser to get the result href from a templated link with values.
 *
 * @param links             A LinkSet provided in any of the formats in which they may be stored on a dcx-js class
 * @param relationship      The relationship key
 * @param values            The key/values to bind to the template. If any value is undefined, it is stripped from the values object.
 * @param linkMode          If the relationship contains an array of links, allows specifying the exact mode to return the property for.
 * @return                  The expanded href template if the relationship exists and the link is marked as templated=true. Else undefined
 */
export function getLinkHrefTemplated(
    links: AnyLinkSetInput,
    relationship: string,
    values: Record<
        string,
        (string | number | undefined) | (string | number)[] | Record<string, string | number | undefined>
    >,
    linkMode: LinkMode = 'id',
): string {
    const _links = _getLinks(links);

    const href = getLinkHref(_links, relationship, linkMode);

    return expandURITemplate(href, pruneUndefined(values));
}

export const parse = (value: string, offset = 0) => {
    return new Link().parse(value, offset);
};

function _getLinks(links: AnyLinkSetInput): LinkSet {
    if (!isObject(links)) {
        return {};
    }
    if ('_links' in links) {
        return (links as UnderscoreLinksObject)._links;
    }
    if ('links' in links) {
        return (links as LinksObject).links;
    }
    return links;
}

// @internal
type UnderscoreLinksObject = { _links: LinkSet };
// @internal
type LinksObject = { links: LinkSet };
// @internal
type AnyLinkSetInput = LinkSet | UnderscoreLinksObject | LinksObject | AdobeAssetWithLinks | undefined;
