/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { merge } from './AdobeDCXUtil';
import { isObject } from './types';

/**
 * Object (POJO) utilities
 */

// eslint-disable-next-line @typescript-eslint/ban-types
export function safeGet<T extends object, K extends keyof T>(obj: T, key: K): T[K] | undefined {
    if (Object.prototype.hasOwnProperty.call(obj, key)) {
        return obj[key];
    }
    return undefined;
}

/**
 * Combines objects, mutating them in place to proxy to each object.
 * Assigns to the left, so later objects in the argument list may overwrite existing properties.
 * Does not combine deeply, properties that are objects themselves will not be combined.
 *
 * TODO: Also assign getters/setters in the case of class instances.
 *
 * @param {unknown} sources - objects to combine
 */
export function combineInPlace(...sources: unknown[]): void {
    if (sources.length < 2) {
        return;
    }
    const first = sources.shift();

    merge(first, ...sources);

    sources.map((s) => {
        Object.assign(first as any, s);
    });
}

/**
 * Prune undefined entries from an object.
 * Returns new object, does not mutate existing object.
 * Does not recurse, only prunes first level.
 *
 * @param {Object} obj - object to prune
 */
export function pruneUndefined<T = Record<string, unknown>>(obj: unknown): T {
    if (!isObject(obj)) {
        throw new Error('Expecting object');
    }

    const newObj = {};
    for (const key in obj) {
        if (obj[key] != null) {
            newObj[key] = obj[key];
        }
    }
    return newObj as T;
}
