/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { PlatformReadableStream } from '@dcx/common-types';
import type { Readable } from 'stream';

// export const hasOwnProperty = Object.prototype.hasOwnProperty;

export const isDefined = <T extends unknown>(o: T | undefined | null): o is T => {
    return o != null;
};

export const isUndefined = <T extends unknown>(o: T | undefined | null): o is undefined | null => {
    return o == null;
};

/**
 * Determine if a value is a Stream
 *
 * @param {Object} val The value to test
 * @returns {boolean} True if value is a Stream, otherwise false
 */
export const isReadableStream = (val: unknown): val is PlatformReadableStream => {
    return isObject(val) && (isFunction(val.pipe) || isFunction(val.pipeTo));
};

export const isNodeReadableStream = (val: unknown): val is Readable => {
    return isReadableStream(val) && isFunction((val as Readable).on);
};

export const isWHATWGReadableStream = (val: unknown): val is ReadableStream => {
    return isReadableStream(val) && isFunction((val as ReadableStream).pipeTo);
};

export const isObject = (val: unknown): val is Record<string, unknown> => {
    return val != null && typeof val === 'object';
};

/**
 * @deprecated in v7 - removed in v8 (use isAnyFunction instead)
 */
// eslint-disable-next-line @typescript-eslint/ban-types
export const isFunction = <OUT = unknown, IN = unknown>(val: unknown): val is AsyncFunction<OUT, IN> | Function => {
    return typeof val === 'function';
};

// eslint-disable-next-line @typescript-eslint/ban-types
type AsyncFunction<OUT = unknown, IN = unknown> = Function & ((...args: IN[]) => Promise<OUT>);

/**
 * This will narrow down to any function which returns a promise.
 * It can additionally accept parameter types to allow for more specific
 * narrowing of the value under test.
 *
 * i.e.
 * ```typescript
 * if (isAsyncFunction<number>(maybeFn)) {
 *   const valPlus5 = 5 + (await maybeFn());
 * }
 *```

 * Note that the actual return value is not directly tested.
 * The generics cannot be used for further discrimination/matching.
 * As an example, the following code will ALWAYS result in the first branch being taken.
 *
 * ```typescript
 * const booleanAsyncFn = async () => false;
 *
 * if (isAsyncFunction<number>(booleanAsyncFn)) {
 *   // This branch will be executed as return types are not evaluated against generic type parameters.
 * } else if (isAsyncFunction<boolean>(booleanAsyncFn)) {
 *   // does not get executed -- return values are not inspected
 * }
 * ```
 * 
 * @deprecated in v7 - removed in v8 (use isAnyFunction instead)
 */
export const isAsyncFunction = <OUT = unknown, IN = unknown>(val: unknown): val is AsyncFunction<OUT, IN> => {
    return toString.call(val) === '[object AsyncFunction]';
};

// eslint-disable-next-line @typescript-eslint/ban-types
export const isAnyFunction = <OUT = unknown, IN = unknown>(val: unknown): val is AsyncFunction<OUT, IN> | Function => {
    return typeof val === 'function';
};

export const isPromise = (p: unknown): p is Promise<unknown> => {
    return p != null && isObject(p) && isFunction(p.then);
};

export const isArray = <T = unknown>(p: unknown): p is Array<T> => {
    return Array.isArray(p);
};

export const isVoid = (p: unknown): p is void => {
    return p == null;
};

export const isBuffer = (p: unknown): p is Buffer => {
    return isObject(p) && isFunction(p.constructor) && p.constructor.name === 'Buffer' && isFunction(p.slice);
};

export const isArrayBuffer = (p: unknown): p is Array<unknown> => {
    return isObject(p) && isFunction(p.constructor) && p.constructor.name === 'ArrayBuffer' && isFunction(p.slice);
};

export const isArrayBufferView = (p: unknown): p is Array<unknown> => {
    return (
        isObject(p) &&
        isFunction(p.constructor) &&
        p.constructor.name === 'DataView' &&
        isObject(p.buffer) &&
        isFunction(p.buffer.slice)
    );
};

/**
 * Returns true if string matches:
 * `application/json`
 * `application/<anything>+json`
 * `<anything>;application/json;<anything>`
 * `<anything>;application/<anything>+json;<anything>`
 *
 * @param str
 */
export const isJsonContentType = (str: string): boolean => {
    if (typeof str !== 'string') {
        return false;
    }

    const lower = str.toLowerCase();
    const spl = lower.split('application/');
    if (spl.length < 2) {
        return false;
    }

    const afterSlash = spl[1].split(';')[0].trim();
    return afterSlash === 'json' || afterSlash.endsWith('+json');
};

/**
 * Check if string matches a Shared Cloud asset's URN
 *
 * Matches a string with the format:
 * `urn:aaid:{two chars of a-Z}:{to chars of a-Z}:{valid URNv1-5}`
 * ie. `urn:aaid:sc:US:48a91bdd-d956-5ac1-b45e-c1ab1b6bd9f7`
 * OR `urn:aaid:xy:EU:48a91bdd-d956-5ac1-b45e-c1ab1b6bd9f7`
 *
 * An alternate form, matching only the prefix urn:aaid:sc:{2 char region}:
 * `/^urn:aaid:sc:[a-zA-Z]{2}:[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i`
 *
 * @param {string} str
 * @deprecated in v7 - removed in v8
 */
export const isValidAdobeURN = (str: string): boolean => {
    // eslint-disable-next-line max-len
    return /^urn:aaid:[a-zA-Z]{2}:[a-zA-Z0-9]{2,6}:[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i.test(
        str,
    );
};
