/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { DCXError } from '@dcx/error';
import { isArray } from './types';

/**
 * Validation Utilities
 * @internal
 */

export type ExpectType =
    | 'function'
    | 'function[]'
    | 'boolean'
    | 'boolean[]'
    | 'number'
    | 'number[]'
    | 'string'
    | 'string[]'
    | 'integer'
    | 'integer[]'
    | 'object'
    | 'object[]'
    | 'null'
    | 'undefined'
    | 'nullish'
    | '+number'
    | '-number'
    | 'enum'
    | 'array';

const errorTemplate = (name: string, expected: ExpectType | ExpectType[], possibleVals?: unknown[]) => {
    if (isArray(expected)) {
        return new DCXError(
            DCXError.INVALID_PARAMS,
            `Param '${name}' type must be one of: [${expected.join(',')}].${
                possibleVals && possibleVals.length > 0 ? ' Possible values: ' + possibleVals.join(', ') + '.' : ''
            }`,
        );
    }

    return new DCXError(
        DCXError.INVALID_PARAMS,
        `Param '${name}' must be of type '${expected}'.${
            possibleVals && possibleVals.length > 0 ? ' Possible values: ' + possibleVals.join(', ') + '.' : ''
        }`,
    );
};

/**
 * Validate a single parameter, throw error if not pass
 * @param {string} paramName - The parameter name to validate, used in the error
 * @param {unknown} paramVal - The parameter value itself, this is validated
 * @param {ExpectType} expectType - The expected type, if set to enum will ONLY check possibleVals array
 * @param {boolean} [optional = false] - If true, null/undefined is a valid value
 * @param {unknown[]} [possibleVals = []] - Array of concrete values the param must match
 */
export const validateParam = (
    paramName: string,
    paramVal: unknown,
    expectType: ExpectType | ExpectType[],
    optional = false,
    possibleVals: unknown[] = [],
): boolean => {
    if (optional && paramVal == null) {
        return true;
    }

    if (isArray(expectType)) {
        // validate, accept at least one valid type
        for (const i in expectType) {
            const et = expectType[i];
            try {
                validateParam(paramName, paramVal, et, optional, possibleVals);
                return true;
            } catch (_) {
                // noop
            }
        }
        throw errorTemplate(paramName, expectType, possibleVals);
    }

    if (
        (expectType === 'null' && paramVal !== null) ||
        (expectType === 'undefined' && paramVal !== undefined) ||
        (expectType === 'nullish' && paramVal != null)
    ) {
        throw errorTemplate(paramName, expectType, possibleVals);
    } else if (expectType === 'null' || expectType === 'undefined' || expectType === 'nullish') {
        return true;
    }

    if (!optional && paramVal == null) {
        throw errorTemplate(paramName, expectType, possibleVals);
    }

    if (expectType.endsWith('[]')) {
        // validating an array of some type
        if (!isArray(paramVal)) {
            throw errorTemplate(paramName, expectType, possibleVals);
        }

        paramVal.forEach((v, ind) => {
            validateParam(`${paramName}[${ind}]`, v, expectType.substr(0, expectType.length - 2) as ExpectType);
        });

        return true;
    }

    let expectTypePrim: string = expectType.toLowerCase();
    switch (expectType) {
        case 'integer': {
            expectTypePrim = 'number';
            break;
        }
        case '+number': {
            expectTypePrim = 'number';
            break;
        }
        case '-number': {
            expectTypePrim = 'number';
            break;
        }
    }

    if (expectTypePrim === 'array') {
        if (!isArray(paramVal)) {
            throw errorTemplate(paramName, expectType, possibleVals);
        }
    } else if (expectTypePrim !== 'enum') {
        if (typeof paramVal !== expectTypePrim) {
            throw errorTemplate(paramName, expectType, possibleVals);
        }
        if (expectType === 'integer' && (typeof paramVal !== 'number' || !Number.isInteger(paramVal))) {
            throw errorTemplate(paramName, expectType, possibleVals);
        }
        if (expectType === '+number' && (typeof paramVal !== 'number' || paramVal < 0)) {
            throw errorTemplate(paramName, expectType, possibleVals);
        }
        if (expectType === '-number' && (typeof paramVal !== 'number' || paramVal > 0)) {
            throw errorTemplate(paramName, expectType, possibleVals);
        }
    }

    if (possibleVals.length > 0) {
        const len = possibleVals.length;
        let exists = false;
        for (let i = 0; i < len; i++) {
            if (possibleVals[i] === paramVal) {
                exists = true;
                break;
            }
        }
        if (!exists) {
            throw errorTemplate(paramName, expectType, possibleVals);
        }
    }
    return true;
};

/**
 * Validate multiple parameter types specified as tuples
 * 
 * @note
 * If optional not set, defaults to `false`.
 * 
 * @note
 * If expectedType is set to `enum`, only will validate that the value matches one of `possibleVals`.
 *
 * @param {[paramName: string,
            paramVal: unknown,
            expectedType: ExpectType,
            optional?: boolean,
            possibleVals?: unknown[]][]} expects expected type and value tuple
 *
 * @example
 * ```js
 * validateParams(['paramName', paramVal, 'type', isOptional])
 * ```
 * 
 * @example
 * ```
 * // Multiple params to validate:
 * validateParams(['param', param, 'string'], ['param2', param2, 'number', true])
 * ```
 * 
 * @example
 * ```
 * // Multiple valid types:
 * validateParams(['stringOrNumber', stringOrNumber, ['string', 'number'], isOptional])
 * ```
 * 
 * @example
 * ```
 * // Enum of valid types:
 * validateParams(['myEnum', myEnum, 'enum', isOptional, ['one', 'two']])
 * ```
 * 
 * @example
 * ```
 * // Array of a single type:
 * validateParams(['myStrArr', myStrArr, 'string[]', isOptional])
 * ```
 */
export function validateParams(
    ...expects: [string, unknown, ExpectType | ExpectType[], boolean?, unknown[]?][]
): boolean[] {
    return expects.map((p) => {
        return validateParam(...p);
    });
}

/**
 * Validate multiple object properties from an object.
 * If optional not set, defaults to `false`.
 * If expectedType is set to `enum`, only will validate that the value matches one of `possibleVals`.
 *
 * @note
 * Do not specify generic types, except possibly the first one.
 * They should be inferred from use.
 *
 * @note
 * This function asserts and sets properties as required on validated object.
 * IF OPTIONAL, THE VALIDATED PROPERTY SHOULD STILL BE TREATED AS OPTIONAL/POSSIBLY UNDEFINED.
 *
 * @param {unknown} opts options object to validate
 * @param {[
 *  paramName: string,
 *  expectedType: ExpectType,
 *  optional?: boolean,
 *  possibleVals?: unknown[]
 *  ][]} expects expected type tuple
 *
 * @example
 * validateObject(obj, 'objectName', ['prop1', 'string', false], ['prop2', 'number', true])
 */
export function validateObject<
    // eslint-disable-next-line @typescript-eslint/ban-types
    T extends Object,
    U extends Extract<keyof T, string>,
    V extends T = T & Pick<Required<T>, U>,
>(obj: T, objectName?: string, ...expects: [U, ExpectType | ExpectType[], boolean?, unknown[]?][]): asserts obj is V {
    objectName = objectName ? ' ' + objectName + ' ' : ' ';
    if (!obj || typeof obj !== 'object') {
        throw new DCXError(DCXError.INVALID_PARAMS, `Object${objectName}is invalid.`);
    }

    const name = 0;
    const type = 1;
    const optional = 2;
    const possibleVals = 3;
    try {
        expects.forEach((p) => {
            validateParam(p[name], obj[p[name] as U], p[type], p[optional] || false, p[possibleVals] || []);
        });
    } catch (e) {
        throw new DCXError(
            DCXError.INVALID_PARAMS,
            `Object${objectName}is invalid. ${(e as any).message.replace('Param', 'Property')}`,
            e as any,
        );
    }
}

/**
 * Assert a generic condition.
 * If it returns false, throw an INVALID_PARAMS error with the provided message.
 * If it returns anything else, return undefined.
 *
 * @throws {AdobeDCXError}
 *
 * @param {Function} condition - function that returns a boolean and takes no arguments
 * @param {string} message
 */
export const assert = (condition: () => boolean, message: string) => {
    if (condition() === false) {
        throw new DCXError(DCXError.INVALID_PARAMS, message);
    }
};
