/**
 * @ref https://github.com/chbrown/rfc6902
 */
import { JSONPatchDocument } from '@dcx/common-types';
/**
 * Callback to allow clients to provide a partial `diff()` function.
 * If this callback returns void, we fall back to the built-in `diffAny()`.
 */
export type JSONPatchVoidableDiff = (input: any, output: any, ptr: JSONPatchPointer) => JSONPatchDocument | void;
/**
 * Produce a 'application/json-patch+json'-type patch to get from one object to another.
 * This does not alter `input` or `output` unless they have a property getter with
 * side-effects (which is not a good idea anyway).
 *
 * `diff` is called on each pair of comparable non-primitive nodes in the
 * `input`/`output` object trees, producing nested patches.
 *
 * Return `undefined` to fall back to default behaviour.
 *
 * Returns list of operations to perform on `input` to produce `output`.
 *
 * See {@link https://tools.ietf.org/html/rfc6902 | RFC6902}
 *
 * @param   input   - The "before patch" object
 * @param   output  - The "after patch" object
 * @param   [diff]  - An optional callback for each non-primitive node comparison
 *                    Useful if, for example, you are patching classes.
 */
export declare function createJSONPatch(input: any, output: any, diff?: JSONPatchVoidableDiff): JSONPatchDocument;
interface JSONPatchPointerEvaluation {
    parent: any;
    key: string;
    value: any;
}
/**
 * JSON Pointer representation
 */
export declare class JSONPatchPointer {
    tokens: string[];
    constructor(tokens?: string[]);
    /**
     * `path` *must* be a properly escaped string.
     */
    static fromJSON(path: string): JSONPatchPointer;
    toString(): string;
    /**
     * Returns an object with 'parent', 'key', and 'value' properties.
     * In the special case that this Pointer's path == "",
     * this object will be {parent: null, key: '', value: object}.
     * Otherwise, parent and key will have the property such that parent[key] == value.
     */
    evaluate(object: any): JSONPatchPointerEvaluation;
    get(object: any): any;
    set(object: any, value: any): void;
    push(token: string): void;
    /**
     * `token` should be a String. It'll be coerced to one anyway.
     * immutable (shallowly)
     */
    add(token: string): JSONPatchPointer;
}
declare class PatchDocumentBuilder {
    /**
     * The operations included in the current document.
     */
    private _operations;
    /**
     * Get the document operations as an array of objects
     */
    get operations(): JSONPatchDocument;
    constructor(initialDocument?: JSONPatchDocument);
    /**
     * Get the document as a string
     */
    getDocument(): string;
    /**
     * Add "add" operation to document.
     *
     * @param {string}  path        Path to add value to.
     * @param {unknown} value       Value to add.
     */
    add(path: string, value: unknown): PatchDocumentBuilder;
    /**
     * Add a value to a list.
     * Handles adding the index to the end of the path.
     *
     * @param {string} path                     Path to the array to add to.
     * @param {unknown} value                   Value to add to the list.
     * @param {number | 'end' | '-'} index      If a number is provided, used as the index at which to add the value.
     *                                          Otherwise adds the value to the end of the list.
     */
    addToList(path: string, value: unknown, index: number | 'end' | '-'): PatchDocumentBuilder;
    /**
     * Add "remove" operation to document.
     *
     * @param {string}  path        Path to property to remove.
     */
    remove(path: string): PatchDocumentBuilder;
    /**
     * Add "replace" operation to document.
     * @param {string}  path        Path to property to replace.
     * @param {unknown} value       New value to place at the path.
     */
    replace(path: string, value: unknown): PatchDocumentBuilder;
    /**
     * Add "move" operation to document.
     *
     * @param {string}  source          Path to property to move.
     * @param {string}  destination     Path to move the property to.
     */
    move(source: string, destination: string): PatchDocumentBuilder;
    /**
     * Add "copy" operation to document.
     *
     * @param {string}  source          Path to property to copy.
     * @param {string}  destination     Path to copy the property to.
     */
    copy(source: string, destination: string): PatchDocumentBuilder;
}
/**
 * Create an instance of PatchDocumentBuilder
 */
export declare function createPatchDocumentBuilder(initialDocument?: JSONPatchDocument): PatchDocumentBuilder;
interface InvalidPropertyIssue {
    property: string;
    issue: string;
}
/**
 * Perform simple validation on a JSON Patch document.
 *
 * This only validates that the individual operations are valid,
 * it does not validate that more complex ordering will fail, even
 * if that is statically known.
 *
 * @param doc
 * @returns {true | InvalidPropertyIssue[]}
 */
export declare function validateJSONPatchDocument(doc: unknown): true | InvalidPropertyIssue[];
/**
 * Validate individual JSON Patch operation.
 *
 * @param op
 * @returns {boolean}
 */
export declare function validateJSONPatchOperation(op: any): true | InvalidPropertyIssue[];
export {};
//# sourceMappingURL=json_patch.d.ts.map