/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
/**
 * Validation Utilities
 * @internal
 */
export type ExpectType = 'function' | 'function[]' | 'boolean' | 'boolean[]' | 'number' | 'number[]' | 'string' | 'string[]' | 'integer' | 'integer[]' | 'object' | 'object[]' | 'null' | 'undefined' | 'nullish' | '+number' | '-number' | 'enum' | 'array';
/**
 * Validate a single parameter, throw error if not pass
 * @param {string} paramName - The parameter name to validate, used in the error
 * @param {unknown} paramVal - The parameter value itself, this is validated
 * @param {ExpectType} expectType - The expected type, if set to enum will ONLY check possibleVals array
 * @param {boolean} [optional = false] - If true, null/undefined is a valid value
 * @param {unknown[]} [possibleVals = []] - Array of concrete values the param must match
 */
export declare const validateParam: (paramName: string, paramVal: unknown, expectType: ExpectType | ExpectType[], optional?: boolean, possibleVals?: unknown[]) => boolean;
/**
 * Validate multiple parameter types specified as tuples
 *
 * @note
 * If optional not set, defaults to `false`.
 *
 * @note
 * If expectedType is set to `enum`, only will validate that the value matches one of `possibleVals`.
 *
 * @param {[paramName: string,
            paramVal: unknown,
            expectedType: ExpectType,
            optional?: boolean,
            possibleVals?: unknown[]][]} expects expected type and value tuple
 *
 * @example
 * ```js
 * validateParams(['paramName', paramVal, 'type', isOptional])
 * ```
 *
 * @example
 * ```
 * // Multiple params to validate:
 * validateParams(['param', param, 'string'], ['param2', param2, 'number', true])
 * ```
 *
 * @example
 * ```
 * // Multiple valid types:
 * validateParams(['stringOrNumber', stringOrNumber, ['string', 'number'], isOptional])
 * ```
 *
 * @example
 * ```
 * // Enum of valid types:
 * validateParams(['myEnum', myEnum, 'enum', isOptional, ['one', 'two']])
 * ```
 *
 * @example
 * ```
 * // Array of a single type:
 * validateParams(['myStrArr', myStrArr, 'string[]', isOptional])
 * ```
 */
export declare function validateParams(...expects: [string, unknown, ExpectType | ExpectType[], boolean?, unknown[]?][]): boolean[];
/**
 * Validate multiple object properties from an object.
 * If optional not set, defaults to `false`.
 * If expectedType is set to `enum`, only will validate that the value matches one of `possibleVals`.
 *
 * @note
 * Do not specify generic types, except possibly the first one.
 * They should be inferred from use.
 *
 * @note
 * This function asserts and sets properties as required on validated object.
 * IF OPTIONAL, THE VALIDATED PROPERTY SHOULD STILL BE TREATED AS OPTIONAL/POSSIBLY UNDEFINED.
 *
 * @param {unknown} opts options object to validate
 * @param {[
 *  paramName: string,
 *  expectedType: ExpectType,
 *  optional?: boolean,
 *  possibleVals?: unknown[]
 *  ][]} expects expected type tuple
 *
 * @example
 * validateObject(obj, 'objectName', ['prop1', 'string', false], ['prop2', 'number', true])
 */
export declare function validateObject<T extends Object, U extends Extract<keyof T, string>, V extends T = T & Pick<Required<T>, U>>(obj: T, objectName?: string, ...expects: [U, ExpectType | ExpectType[], boolean?, unknown[]?][]): asserts obj is V;
/**
 * Assert a generic condition.
 * If it returns false, throw an INVALID_PARAMS error with the provided message.
 * If it returns anything else, return undefined.
 *
 * @throws {AdobeDCXError}
 *
 * @param {Function} condition - function that returns a boolean and takes no arguments
 * @param {string} message
 */
export declare const assert: (condition: () => boolean, message: string) => void;
//# sourceMappingURL=validation.d.ts.map