import { isArray } from '@dcx/util';

class XMLNodeBase {
    protected _data: Partial<XNode> & Pick<XNode, 'children'>;
    constructor(data: XNode) {
        this._data = data;
    }
    getElementsByTagName(tagName: string): XMLElement[] {
        return this._data.children
            .filter((n): n is XNode => typeof n !== 'string' && n.tagName === tagName)
            .map(createXMLElement);
    }
}

export class XMLElement extends XMLNodeBase {
    protected _data: XNode;
    constructor(data: XNode) {
        super(data);
        this._data = data;
    }
    setAttribute(name: string, value: string) {
        this._data.attributes[name] = value;
    }
    getAttribute(name: string): unknown | undefined {
        return this._data.attributes[name];
    }
    appendChild(
        tagName: string,
        attributes: Record<string, string> = {},
        children: (XNode | string)[] = [],
    ): XMLElement {
        const child = { tagName, attributes, children };
        this._data.children.push(child);
        return createXMLElement(child);
    }
}

function createXMLElement(data: XNode): XMLElement {
    return new XMLElement(data);
}

export class XMLParser extends XMLNodeBase {
    constructor(data?: XNode) {
        super(data as XNode);
    }

    parse(str: string): XMLParser {
        const children = parse(str);
        this._data = { children: isArray(children) ? children : [children] };
        return this;
    }

    getElementsByTagName(tagName: string): XMLElement[] {
        return super.getElementsByTagName(tagName);
    }

    toString(): string {
        return stringify(this._data.children as XNode[]);
    }
}

export function parseXML(str: string): XMLParser {
    const parser = new XMLParser();
    return parser.parse(str);
}

export interface XNode {
    tagName: string;
    attributes: Record<string, string>;
    children: (string | XNode)[];
    pos?: number;
}

interface XMLParseOptions {
    pos?: number;
    noChildNodes?: string[];
    keepComments?: boolean;
    keepWhitespace?: boolean;
    simplify?: boolean;
    parseNode?: boolean;
    attrName?: string;
    attrValue?: string;
}

/* istanbul ignore next */
function parse(str: string, options: XMLParseOptions = {}): (string | XNode)[] | XNode {
    let pos = options.pos || 0;
    const keepComments = !!options.keepComments;
    const keepWhitespace = !!options.keepWhitespace;

    const openBracket = '<';
    const openBracketCC = '<'.charCodeAt(0);
    const closeBracket = '>';
    const closeBracketCC = '>'.charCodeAt(0);
    const minusCC = '-'.charCodeAt(0);
    const slashCC = '/'.charCodeAt(0);
    const exclamationCC = '!'.charCodeAt(0);
    const singleQuoteCC = "'".charCodeAt(0);
    const doubleQuoteCC = '"'.charCodeAt(0);
    const openCornerBracketCC = '['.charCodeAt(0);
    const closeCornerBracketCC = ']'.charCodeAt(0);

    /**
     * parsing a list of entries
     */
    function parseChildren(tagName): (string | XNode)[] {
        const children: (string | XNode)[] = [];
        while (str[pos]) {
            if (str.charCodeAt(pos) === openBracketCC) {
                if (str.charCodeAt(pos + 1) === slashCC) {
                    const closeStart = pos + 2;
                    pos = str.indexOf(closeBracket, pos);

                    const closeTag = str.substring(closeStart, pos);
                    if (closeTag.indexOf(tagName) === -1) {
                        const parsedText = str.substring(0, pos).split('\n');
                        throw new Error(
                            'Unexpected close tag\nLine: ' +
                                (parsedText.length - 1) +
                                '\nColumn: ' +
                                (parsedText[parsedText.length - 1].length + 1) +
                                '\nChar: ' +
                                str[pos],
                        );
                    }

                    if (pos + 1) {
                        pos += 1;
                    }

                    return children;
                } else if (str.charCodeAt(pos + 1) === exclamationCC) {
                    if (str.charCodeAt(pos + 2) === minusCC) {
                        //comment support
                        const startCommentPos = pos;
                        while (
                            pos !== -1 &&
                            !(
                                str.charCodeAt(pos) === closeBracketCC &&
                                str.charCodeAt(pos - 1) === minusCC &&
                                str.charCodeAt(pos - 2) === minusCC &&
                                pos !== -1
                            )
                        ) {
                            pos = str.indexOf(closeBracket, pos + 1);
                        }
                        if (pos === -1) {
                            pos = str.length;
                        }
                        if (keepComments) {
                            children.push(str.substring(startCommentPos, pos + 1));
                        }
                    } else if (
                        str.charCodeAt(pos + 2) === openCornerBracketCC &&
                        str.charCodeAt(pos + 8) === openCornerBracketCC &&
                        str.substr(pos + 3, 5).toLowerCase() === 'cdata'
                    ) {
                        // cdata
                        const cdataEndIndex = str.indexOf(']]>', pos);
                        if (cdataEndIndex === -1) {
                            children.push(str.substr(pos + 9));
                            pos = str.length;
                        } else {
                            children.push(str.substring(pos + 9, cdataEndIndex));
                            pos = cdataEndIndex + 3;
                        }
                        continue;
                    } else {
                        // doctypesupport
                        const startDoctype = pos + 1;
                        pos += 2;
                        let encapsuled = false;
                        while ((str.charCodeAt(pos) !== closeBracketCC || encapsuled === true) && str[pos]) {
                            if (str.charCodeAt(pos) === openCornerBracketCC) {
                                encapsuled = true;
                            } else if (encapsuled === true && str.charCodeAt(pos) === closeCornerBracketCC) {
                                encapsuled = false;
                            }
                            pos++;
                        }
                        children.push(str.substring(startDoctype, pos));
                    }
                    pos++;
                    continue;
                }
                const node = parseNode();
                children.push(node);
                if (node.tagName[0] === '?') {
                    children.push(...node.children);
                    node.children = [];
                }
            } else {
                const text = parseText();
                if (keepWhitespace || text.trim().length > 0) {
                    children.push(text);
                }
                pos++;
            }
        }
        return children;
    }

    /**
     *    returns the text outside of texts until the first '<'
     */
    function parseText(): string {
        const start = pos;
        pos = str.indexOf(openBracket, pos) - 1;
        if (pos === -2) {
            pos = str.length;
        }
        return str.slice(start, pos + 1);
    }
    /**
     *    returns text until the first nonAlphabetic letter
     */
    const nameSpacer = '\r\n\t>/= ';

    function parseName(): string {
        const start = pos;
        while (nameSpacer.indexOf(str[pos]) === -1 && str[pos]) {
            pos++;
        }
        return str.slice(start, pos);
    }
    /**
     *    is parsing a node, including tagName, Attributes and its children,
     * to parse children it uses the parseChildren again, that makes the parsing recursive
     */
    const NoChildNodes = options.noChildNodes || ['img', 'br', 'input', 'meta', 'link', 'hr'];

    function parseNode(): XNode {
        pos++;
        const tagName = parseName();
        const attributes = {};
        let children: (XNode | string)[] = [];

        // parsing attributes
        while (str.charCodeAt(pos) !== closeBracketCC && str[pos]) {
            const c = str.charCodeAt(pos);
            if ((c > 64 && c < 91) || (c > 96 && c < 123)) {
                //if('abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ'.indexOf(S[pos])!==-1 ){
                const name = parseName();
                let value: string | null;
                // search beginning of the string
                let code = str.charCodeAt(pos);
                while (
                    code &&
                    code !== singleQuoteCC &&
                    code !== doubleQuoteCC &&
                    !((code > 64 && code < 91) || (code > 96 && code < 123)) &&
                    code !== closeBracketCC
                ) {
                    pos++;
                    code = str.charCodeAt(pos);
                }
                if (code === singleQuoteCC || code === doubleQuoteCC) {
                    value = parseString();
                    if (pos === -1) {
                        return {
                            tagName,
                            attributes,
                            children,
                        };
                    }
                } else {
                    value = null;
                    pos--;
                }
                attributes[name] = value;
            }
            pos++;
        }
        // optional parsing of children
        if (str.charCodeAt(pos - 1) !== slashCC) {
            if (tagName === 'script') {
                const start = pos + 1;
                pos = str.indexOf('</script>', pos);
                children = [str.slice(start, pos)];
                pos += 9;
            } else if (tagName === 'style') {
                const start = pos + 1;
                pos = str.indexOf('</style>', pos);
                children = [str.slice(start, pos)];
                pos += 8;
            } else if (NoChildNodes.indexOf(tagName) === -1) {
                pos++;
                children = parseChildren(tagName);
            } else {
                pos++;
            }
        } else {
            pos++;
        }
        return {
            tagName,
            attributes,
            children,
        };
    }

    function parseString(): string {
        const startChar = str[pos];
        const startpos = pos + 1;
        pos = str.indexOf(startChar, startpos);
        return str.slice(startpos, pos);
    }

    function findElements() {
        const r = new RegExp('\\s' + options.attrName + '\\s*=[\'"]' + options.attrValue + '[\'"]').exec(str);
        if (r) {
            return r.index;
        }
        return -1;
    }

    let out: (string | XNode)[] | XNode;
    if (options.attrValue !== undefined) {
        options.attrName = options.attrName || 'id';
        out = [];

        while ((pos = findElements()) !== -1) {
            pos = str.lastIndexOf('<', pos);
            if (pos !== -1) {
                out.push(parseNode());
            }
            str = str.substr(pos);
            pos = 0;
        }
    } else if (options.parseNode) {
        out = parseNode();
    } else {
        out = parseChildren('');
    }

    if (options.simplify) {
        return simplify(Array.isArray(out) ? out : [out]);
    }

    return out;
}

/**
 * transform the DomObject to an object that is like the object of PHP`s simple_xmp_load_*() methods.
 * this format helps you to write that is more likely to keep your program working, even if there a small changes in the XML schema.
 * be aware, that it is not possible to reproduce the original xml from a simplified version, because the order of elements is not saved.
 * therefore your program will be more flexible and easier to read.
 *
 * @param {tNode[]} children the childrenList
 */
function simplify(children): any {
    const out = {};
    if (!children.length) {
        return '';
    }

    if (children.length === 1 && typeof children[0] == 'string') {
        return children[0];
    }
    // map each object
    children.forEach(function (child) {
        if (typeof child !== 'object') {
            return;
        }
        if (!out[child.tagName]) {
            out[child.tagName] = [];
        }
        const kids = simplify(child.children);
        out[child.tagName].push(kids);
        if (Object.keys(child.attributes).length) {
            kids._attributes = child.attributes;
        }
    });

    for (const i in out) {
        if (out[i].length === 1) {
            out[i] = out[i][0];
        }
    }

    return out;
}

/**
 * stringify a previously parsed string object.
 * this is useful,
 *  1. to remove whitespace
 * 2. to recreate xml data, with some changed data.
 * @param {tNode} O the object to Stringify
 */
function stringify(O: XNode | XNode[], prettify = false) {
    let out = '';
    let indent = 0;

    function writeChildren(O) {
        indent += 1;
        if (O) {
            if (prettify && O.length > 0 && typeof O[0] !== 'string') {
                out += '\r\n';
            }
            for (let i = 0; i < O.length; i++) {
                if (typeof O[i] === 'string') {
                    out += O[i].trim();
                } else {
                    writeNode(O[i]);
                }
            }
            if (prettify && O.length > 0 && typeof O[0] !== 'string' && !out.endsWith('\r\n')) {
                out += '\r\n';
            }
        }
        indent -= 1;
    }

    function writeNode(N: XNode) {
        if (prettify) {
            out += '<'.padStart(indent) + N.tagName;
        } else {
            out += '<' + N.tagName;
        }

        for (const i in N.attributes) {
            if (N.attributes[i] === null) {
                out += ' ' + i;
            } else if (N.attributes[i].indexOf('"') === -1) {
                out += ' ' + i + '="' + N.attributes[i].trim() + '"';
            } else {
                out += ' ' + i + "='" + N.attributes[i].trim() + "'";
            }
        }
        if (N.tagName[0] === '?') {
            out += prettify ? '?>\r\n' : '?>';
            return;
        }
        out += '>';
        writeChildren(N.children);
        if (prettify && out.endsWith('\r\n')) {
            out += '</'.padStart(indent) + N.tagName + '>\r\n';
        } else if (prettify) {
            out += '</' + N.tagName + '>\r\n';
        } else {
            out += '</' + N.tagName + '>';
        }
    }
    writeChildren(O);

    return out;
}
